# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Arnet
from ArnetModel import IpGenericAddress, IpGenericPrefix, Ip4Address
from IntfModels import Interface
from CliCommon import CliModelNotDegradable
from CliModel import DeferredModel, Model, Submodel
from CliModel import List, Bool, Dict, Enum, Int, Float, Str, GeneratorDict
from DeviceNameLib import kernelIntfToEosIntf, eosIntfToKernelIntf
import Tac
import PimCountersLib
import IpUtils
from prettytable import PrettyTable
import functools

def timeFormat( t ):
   t = int( t )
   s = t % 60
   t //= 60
   m = t % 60
   t //= 60
   h = t % 24
   t //= 24
   d = t

   if d:
      return '%dd%02dh' % ( d, h )
   else:
      return '%02d:%02d:%02d' % ( h, m, s )

def printTable( table ):
   table.border = False
   table.align = 'l'
   table.padding_width = 0
   table.right_padding_width = 2
   print( table.get_string().strip() )

class PimMode( Model ):
   ''' Pim mode based on mode and multicast boundary router flag. '''
   mode = Enum( help="Pim operational mode",
                values=( 'Sparse',
                         'Bidir',
                         'Sparse/Bidir',
                         'None' ) )
   borderRouter = Bool( help="Multicast border router is enabled" )

   def fromTacc( self, intf ):
      typeDict = {
            'modePimSm' : 'Sparse',
            'modePimBidir' : 'Bidir',
            'modePimSmAndBidir' : 'Sparse/Bidir',
            'modePimNone' : 'None'
            }

      self.mode = typeDict[ intf.mode ]
      self.borderRouter = intf.borderRouter

   def isSparse( self ):
      if self.mode in [ 'Sparse', 'Sparse/Bidir' ]:
         return True
      if self.borderRouter:
         return True
      return False

   def toString( self ):
      strDict = {
            'Sparse' : 'sparse',
            'Bidir' : 'bidir',
            'Sparse/Bidir' : 'sparse/bidir',
            'None' : 'none'
            }
      if self.mode == 'None':
         if self.borderRouter:
            return "mbr"
         else:
            return strDict[ self.mode ]
      else:
         if self.borderRouter:
            return strDict[ self.mode ] + "/" + "mbr"
         else:
            return strDict[ self.mode ]


class PimKernelInterface( Model ):
   ''' Kernel interface information to be added to each Pim interface. '''
   virtualInterface = Int( help="Virtual interface number", optional=True )
   name = Str( help="Kernel interface name" )
   bytesIn = Int( help="Bytes received" )
   pktsIn = Int( help="Packets received" )
   bytesOut = Int( help="Bytes sent" )
   pktsOut = Int( help="Packets received" )
   pktsQueued = Int( help="Number of packets queued by kernel" )
   pktsDropped = Int( help="Number of packets dropped by kernel" )

   def toEosIntfName( self ):
      return kernelIntfToEosIntf( self.name )

   def fromKernel( self, kernelIntfName, af, netnsName=None ):
      if af == 'ipv4':
         cmdStr = 'cat /proc/net/ip_mr_vif'
      else:
         cmdStr = 'cat /proc/net/ip6_mr_vif'
      lines = Arnet.NsLib.runMaybeInNetNs( netnsName, cmdStr.split(),
                  stdout=Tac.CAPTURE ).splitlines()

      self.virtualInterface = 0
      self.name = ""
      self.bytesIn = 0
      self.pktsIn = 0
      self.bytesOut = 0
      self.pktsOut = 0
      self.pktsQueued = 0
      self.pktsDropped = 0

      for line in lines:
         tokens = line.split()

         # Search for by interface name
         if tokens[ 1 ] != kernelIntfName:
            continue

         self.virtualInterface = int( tokens[ 0 ] )
         self.name = tokens[ 1 ]
         self.bytesIn = int( tokens[ 2 ] )
         self.pktsIn = int( tokens[ 3 ] )
         self.bytesOut = int( tokens[ 4 ] )
         self.pktsOut = int( tokens[ 5 ] )

         # Ugly backward compatibility with unpatched kernel
         if not "PktsQed" in lines[ 0 ]:
            self.pktsQueued = 0
            self.pktsDropped = 0
         else:
            self.pktsQueued = int( tokens[ 6 ] )
            self.pktsDropped = int( tokens[ 7 ] )

   def fromBessIntfCtrs( self, kernelIntfName, intfCtrs ):
      self.name = kernelIntfName
      if not intfCtrs:
         self.bytesIn = 0
         self.pktsIn = 0
         self.bytesOut = 0
         self.pktsOut = 0
         self.pktsQueued = 0
         self.pktsDropped = 0
         return
      self.bytesIn = intfCtrs.bytesIn
      self.pktsIn = intfCtrs.pktsIn
      self.bytesOut = intfCtrs.bytesOut
      self.pktsOut = intfCtrs.pktsOut
      self.pktsQueued = intfCtrs.pktsQueued
      self.pktsDropped = intfCtrs.pktsDropped

class PimInterfaceDetails( Model ):
   ''' Render the PimInterface model differently with more information. '''
   helloDelay = Int( help="My triggered hello delay in seconds" )
   helloHoldTime = Int( help="My hello hold time in seconds" )
   drPriority = Int( help="My DR priority" )
   generationId = Int( help="My generation id" )
   propagationDelay = Int( help="My propagation delay in milliseconds" )
   usedPropagationDelay = Int( help="Effective propagation delay in milliseconds" )
   overrideInterval = Int( help="My override interval in milliseconds" )
   usedOverrideInterval = Int( help="Effective propagation delay in milliseconds" )
   trackingSupport = Bool( help="Tracking support and disabling join suppression" )
   usedTrackingSupport = Bool( help="Effective tracking support" )
   joinPrunePeriod = Int( help="Join/prune interval in seconds" )
   joinPruneHoldTime = Int( help="Join/prune hold time in seconds" )
   assertTimeout = Int( help="Assert timeout in seconds" )
   assertOverrideInterval = Int( help="Assert override interval in seconds" )
   secondaryAddress = List( valueType=IpGenericAddress,
                            help="Secondary addresses",
                            optional=True )
   def fromTacc( self, intf ):
      self.helloDelay = intf.helloDelay
      self.helloHoldTime = intf.helloHoldTime
      self.drPriority = intf.drPriority
      self.generationId = intf.generationId
      self.propagationDelay = intf.propagationDelay
      self.usedPropagationDelay = intf.usedPropagationDelay
      self.overrideInterval = intf.overrideInterval
      self.usedOverrideInterval = intf.usedOverrideInterval
      self.trackingSupport = intf.tBit
      self.usedTrackingSupport = intf.usedTBit
      self.joinPrunePeriod = intf.joinPrunePeriod
      self.joinPruneHoldTime = intf.joinPruneHoldTime
      self.assertTimeout = intf.assertTimeout
      self.assertOverrideInterval = intf.assertOverrideInterval
      for addr in sorted( intf.secondaryAddress ):
         self.secondaryAddress.append( addr )

class PimInterface( Model ):
   ''' Holds Pim interface and associated kernel information. '''
   __revision__ = 2
   address = IpGenericAddress( help="Interface address" )
   intf = Interface( help="Interface name" )
   mode = Submodel( help="Pim mode", valueType=PimMode )
   neighborCount = Int( help="Number of neighbors on this interface" )
   helloPeriod = Int( help="My time interval between sending Hellos in seconds" )
   usedDrPriority = Int( help="Effective DR priority", optional=True )
   usedDrAddress = IpGenericAddress( help="Current DR", optional=True )

   kernelIntf = Submodel( help="Interface information from the kernel",
                          valueType=PimKernelInterface, optional=True )

   details = Submodel( help="Pim message detailed counters",
                       valueType=PimInterfaceDetails, optional=True )


   def fromTacc( self, intf, detail=False, bessIntfCtrs=None, netnsName=None ):
      self.address = intf.address
      self.intf = intf.intfId
      self.mode = PimMode()
      self.mode.fromTacc( intf )
      self.neighborCount = len ( intf.neighbor )
      self.helloPeriod = intf.helloPeriod
      if self.mode.isSparse():
         # Bidir does not generate a DR
         self.usedDrPriority = intf.usedDrPriority
         self.usedDrAddress = intf.usedDr

         # Bidir does not use the kernel
         self.kernelIntf = PimKernelInterface()
         if bessIntfCtrs:
            self.kernelIntf.fromBessIntfCtrs( self.toKernelIntfName(),
                  bessIntfCtrs.get( self.intf.stringValue ) )
         else:
            self.kernelIntf.fromKernel( self.toKernelIntfName(), self.address.af,
                                        netnsName )

      if detail:
         self.details = PimInterfaceDetails()
         self.details.fromTacc( intf )

   def toKernelIntfName( self ):
      name = self.intf.stringValue
      return eosIntfToKernelIntf( name )

   def renderTableRow( self, allRows=True ):
      if allRows:
         if self.mode.isSparse():
            return [ self.address.stringValue, self.intf.stringValue,
                     self.mode.toString(), self.neighborCount,
                     self.helloPeriod, self.usedDrPriority,
                     self.usedDrAddress.stringValue, self.kernelIntf.pktsQueued,
                     self.kernelIntf.pktsDropped ]
         else:
            return [ self.address.stringValue, self.intf.stringValue,
                     self.mode.toString(), self.neighborCount, self.helloPeriod,
                     "", "", "", "" ]
      else:
         return [ self.address.stringValue, self.intf.stringValue,
                  self.mode.toString(), self.neighborCount, self.helloPeriod ]

   def renderDetail( self ):
      ''' Generate Detailed output. '''

      print( "Interface {} address is {}".format( self.intf.stringValue,
            self.address ) )

      if self.mode.isSparse() and self.kernelIntf.virtualInterface is not None:
         print( "Vif number is %d" % ( self.kernelIntf.virtualInterface ) )

      print( "PIM: enabled" )
      print( "PIM version: 2, mode: %s" % ( self.mode.toString() ) )
      print( "PIM neighbor count: %d" % ( self.neighborCount ) )

      if self.mode.isSparse():
         drMessage = ""
         if self.address == self.usedDrAddress:
            drMessage = "(this system)"
         print( f"PIM Effective DR: {self.usedDrAddress} {drMessage}" )
         print( "PIM Effective DR Priority: %s" % ( self.usedDrPriority ) )

      print( "PIM Effective Propagation Delay: %d milliseconds" %
             ( self.details.usedPropagationDelay ) )
      print( "PIM Effective Override Interval: %d milliseconds" %
             ( self.details.usedOverrideInterval ) )
      tSupportMessage = "enabled"
      if not self.details.usedTrackingSupport:
         tSupportMessage = "disabled"
      print( "PIM Effective Tracking Support: %s" % ( tSupportMessage ) )

      print( "PIM Hello Interval: %d seconds" % ( self.helloPeriod ) )
      print( "PIM Hello Hold Time: %d seconds" % ( self.details.helloHoldTime ) )
      print( "PIM Hello Priority: %d" % ( self.details.drPriority ) )
      print( "PIM Hello Lan Delay: %d milliseconds" %
             ( self.details.propagationDelay ) )
      print( "PIM Hello Override Interval: %d milliseconds" %
             ( self.details.overrideInterval ) )
      tSupportMessage = "enabled"
      if not self.details.trackingSupport:
         tSupportMessage = "disabled"
      print( "PIM Hello Tracking Support: %s" % ( tSupportMessage ) )
      print( "PIM Hello Generation ID: 0x%x" % ( self.details.generationId ) )
      print( "PIM Hello Generation ID is not required" )

      print( "PIM Triggered Hello Delay: %d seconds" % ( self.details.helloDelay ) )
      print( "PIM Join-Prune Interval: %d seconds" %
             ( self.details.joinPrunePeriod ) )
      print( "PIM Join-Prune Hold Time: %d seconds" %
             ( self.details.joinPruneHoldTime ) )
      print( "PIM Assert Timeout: %d seconds" % ( self.details.assertTimeout ) )
      print( "PIM Assert Override Interval: %d seconds" %
             ( self.details.assertOverrideInterval ) )
      print( "PIM Interface Secondary Addresses: " )
      for addr in self.details.secondaryAddress:
         print( "  %s " % addr )

class PimInterfaces( Model ):
   '''Cli Model for show ip pim interface.'''
   interfaces = Dict( help="Pim interfaces", keyType=Interface,
                      valueType=PimInterface )
   def renderDetail( self ):
      for key in Arnet.sortIntf( self.interfaces ):
         self.interfaces[ key ].renderDetail()

   def render( self ):
      # pylint: disable-next=use-implicit-booleaness-not-len
      if not len( self.interfaces ):
         return
      if list( self.interfaces.values() )[ 0 ].details:
         self.renderDetail()
      else:
         self.renderTable()

   def renderTable( self ):
      printSmHeader = False
      for intf in self.interfaces.values():
         if intf.mode.isSparse():
            printSmHeader = True

      if printSmHeader:
         table = PrettyTable( [ 'Address', 'Interface', 'Mode', 'Neighbor',
                                'Hello', 'DR', 'DR Address', 'PktsQed',
                                'PktsDropped' ] )
         table.add_row( [ '', '', '', 'Count', 'Intvl', 'Pri', '', '', '' ] )
      else:
         table = PrettyTable( [ 'Address', 'Interface', 'Mode', 'Neighbor',
                                'Hello' ] )
         table.add_row( [ '', '', '', 'Count', 'Intvl' ] )
      # Print interfaces
      for key in Arnet.sortIntf( self.interfaces ):
         table.add_row( self.interfaces[ key ].renderTableRow(
            allRows=printSmHeader ) )
      printTable( table )


class PimNeighbor( Model ):
   ''' Holds Pim neighbor information. '''
   address = IpGenericAddress( help="Neighbor address" )
   intf = Interface( help="Interface through which neighbor was discovered" )
   creationTime = Float( help="UTC time when neighbor was created" )
   lastRefreshTime = Float( help="UTC time when neighbor last sent Hello message" )
   holdTime = Int( help="Holdtime for neighbor in seconds" )
   mode = Submodel( help="Interface mode", valueType=PimMode )
   bfdState = Enum( help="BFD session state with neighbor",
                    values=( 'disabled', 'down', 'init', "up" ) )
   transport = Enum( help="The transport mechanism used for " + 
                     "sending/receiving or join/prune messages.", 
                     values=( 'datagram', 'tcp', 'sctp' ) )
   detail = Bool( help="Display detailed neighbor information", optional=True )
   routerId = Ip4Address( help="Router ID", optional=True )
   multipathColor = Int( help="Multipath Color", optional=True )
   secondaryAddress = List( valueType=IpGenericAddress,
                            help="Pim neighbor secondary addresses",
                            optional=True )
   maintenanceReceived = Bool( help="Neighbor is undergoing maintenance" )
   maintenanceSent = Bool( help="Communicated to neighbor that we are undergoing "
                                "maintenance" )

   def fromTacc( self, intf, neighbor, detail=False ):
      self.address = neighbor.address
      self.intf = neighbor.intfId
      self.creationTime = neighbor.creationTime + Tac.utcNow() - Tac.now()
      self.lastRefreshTime = neighbor.lastRefreshTime + Tac.utcNow() - Tac.now()
      self.holdTime = neighbor.holdTime
      self.mode = PimMode()
      self.mode.fromTacc( intf )
      if neighbor.transportProtocol == 'tcp':
         self.transport = 'tcp'
      elif neighbor.transportProtocol == 'sctp':
         self.transport = 'sctp'
      else:
         self.transport = 'datagram'

      if intf.bfdEnabled:
         if neighbor.bfdState == "up":
            self.bfdState = "up"
         elif neighbor.bfdState in [ "down", "adminDown" ]:
            self.bfdState = "down"
         else:
            self.bfdState = "init"
      else:
         self.bfdState = "disabled"

      self.detail = detail
      if detail:
         for addr in sorted( neighbor.secondaryAddress ):
            self.secondaryAddress.append( addr )

         rtId = Tac.Value( "Arnet::IpAddr" )
         rtId.value = neighbor.portInfo.interfaceId.routerId
         self.routerId = rtId
         self.multipathColor = neighbor.multipathColor
         self.maintenanceReceived = neighbor.maintenanceInitiator
         self.maintenanceSent = neighbor.maintenanceReceiver

   def renderDetail( self ):
      ''' Generate Detailed output. '''
      print( "Interface {} neighbor address is {}".format( self.intf.stringValue,
            self.address ) )

      uptimeStr = timeFormat( abs( Tac.utcNow() - self.creationTime ) )
      if self.holdTime > abs( Tac.utcNow() - self.lastRefreshTime ):
         expiresStr = timeFormat( self.holdTime - abs( Tac.utcNow() -
                                 self.lastRefreshTime ) )
      else:
         expiresStr = timeFormat( 0 )
      print( "Neighbor Uptime: %s " % uptimeStr )
      print( "Neighbor Expires: %s " % expiresStr )
      print( "Neighbor Mode: %s " % self.mode.toString() )
      print( "Neighbor Transport: %s " % self.transport )
      print( "Neighbor Router ID: %s " % self.routerId )
      print( "Neighbor Multipath Color: %d " % self.multipathColor )
      if self.maintenanceReceived or self.maintenanceSent:
         maintString = "Neighbor Maintenance Option:"
         if self.maintenanceReceived:
            maintString += " received"
         if self.maintenanceSent:
            if self.maintenanceReceived:
               maintString += ","
            maintString += " sent"
         print( maintString )
      print( "Neighbor Secondary Addresses: " )
      for addr in self.secondaryAddress:
         print( "  %s " % addr )

class IntfPimNeighbors( Model ):
   ''' BFD-Independent model to use as an intermediate level of interfaces
       to store "PimNeighbor"s '''
   neighbors = Dict( help='Pim neighbors', keyType=IpGenericAddress,
                     valueType=PimNeighbor )

class VrfPimNeighbors( Model ):
   ''' Cli Model for show ip pim neighbor. '''
   __revision__ = 2
   interfaces = Dict( help="Pim neighbors by interfaces", keyType=str,
                     valueType=IntfPimNeighbors )

   def renderTable( self ):
      header = [ 'Neighbor Address', 'Interface', 'Uptime', 'Expires',
                 'Mode', 'Transport' ]
      table = PrettyTable( header )
      if self.interfaces:
         # Print neighbors
         for _, interface in sorted( self.interfaces.items() ):
            for _, neighbor in sorted( interface.neighbors.items() ):
               uptimeStr = timeFormat( abs( Tac.utcNow() - neighbor.creationTime ) )
               if neighbor.holdTime > abs( Tac.utcNow() - neighbor.lastRefreshTime ):
                  expiresStr = timeFormat( neighbor.holdTime -
                                           abs( Tac.utcNow() -
                                                neighbor.lastRefreshTime ) )
               else:
                  expiresStr = timeFormat( 0 )
               table.add_row( [ neighbor.address.stringValue,
                                neighbor.intf.stringValue, uptimeStr,
                                expiresStr, neighbor.mode.toString(),
                                neighbor.transport ] )
      else:
         ## Adding empty row to print table
         table.add_row( [ '' ] * len( header ) )
      printTable( table )

   def renderDetail( self, af ):
      hashFunc = IpUtils.compareIpAddress if af == 'ipv4'else \
         IpUtils.compareIp6AddressStr

      for _, interface in sorted( self.interfaces.items() ):
         for key in sorted( interface.neighbors,
                            key=functools.cmp_to_key( hashFunc ) ):
            interface.neighbors[ key ].renderDetail()

   def render( self ):
      af = 'ipv4'
      detail = False
      if self.interfaces:
         intf = list( self.interfaces.values() )[ 0 ]
         nbr = list( intf.neighbors.values() )[ 0 ]
         af = nbr.address.af
         detail = nbr.detail

      if detail:
         self.renderDetail( af )
      else:
         self.renderTable()

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         tdict = {}
         for interface in dictRepr[ 'interfaces' ].values():
            for neighbor in interface[ 'neighbors' ].values():
               tdict[ neighbor[ 'address' ] ] = neighbor
         dictRepr[ 'neighbors' ] = tdict
         del dictRepr[ 'interfaces' ]
      return dictRepr

class PimNeighbors( Model ):
   '''Cli Model to wrap up PimNeighbor for show ip pim vrf all neighbor.'''
   __revision__ = 2
   vrfs = Dict( help="A mapping between VRF name and PimNeighbors",
                          keyType=str, valueType=VrfPimNeighbors )
   _isAll = Bool( help="A boolean value to determine if current model is for all" )
   def render( self ):
      for key in sorted( self.vrfs ):
         neighbors = self.vrfs[ key ]
         print( "PIM Neighbor Table for " + key + " VRF" )
         neighbors.render()

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         if self._isAll is True:
            raise CliModelNotDegradable( "Current revision does not support all" )
         # No more than one vrf will be populated when _isAll is false
         if len( self.vrfs ) == 1:
            dictRepr[ "neighbors" ] = \
               dictRepr[ "vrfs" ][ next(
                     iter( dictRepr[ "vrfs" ] ) ) ][ "neighbors" ]
         else:
            dictRepr[ "neighbors" ] = {}
         del dictRepr[ "vrfs" ]
      return dictRepr

class MulticastRibRoute( Model ):
   '''Cli Model for show ip route multicast'''
   vias = Dict( help="set of next-hop addresses that this route is reachable via",
         keyType=IpGenericAddress, valueType=str )
   routeProtocol = Str( help="Routing protocol through which this route was learnt" )
   routePreference = Int( help="Route preference" )
   routeMetric = Int( help="Route metric" )
   routeCreationTime = Str( help="Route creation time" )

class MulticastRibRoutes( Model ):
   routes = Dict( help="Multicast static routes", keyType=IpGenericPrefix,
         valueType=MulticastRibRoute )
   def render( self ):
      print( '''Codes: C - connected, S - static, K - kernel,
       O - OSPF, IA - OSPF inter area, E1 - OSPF external type 1,
       E2 - OSPF external type 2, N1 - OSPF NSSA external type 1,
       N2 - OSPF NSSA external type2, B I - iBGP, B E - eBGP,
       R - RIP, I - IS-IS, A B - BGP Aggregate, A O - OSPF Summary,
       NG - Nexthop Group Static Route''' )
      print()

      for route in sorted( self.routes ):
         routeInfo = self.routes[ route ] # this way or itervalues()?
         outputStr = "%-6s %-17s [%d/%d]" % ( routeInfo.routeProtocol,
               route, routeInfo.routeMetric, routeInfo.routePreference )
         print( outputStr, end=' ' )
         indent = ""
         for via in sorted( routeInfo.vias ):
            print( indent + "via {}, {}, {}".format( via,
                  routeInfo.routeCreationTime, routeInfo.vias[ via ] ) )
            indent = " " * ( len( outputStr ) + 1 )


class VrfPimBfdNeighbors( Model ):
   ''' Cli Model for show ip pim neighbor bfd. '''
   __revision__ = 2
   interfaces = Dict( help="Pim neighbors by interfaces", keyType=str,
                     valueType=IntfPimNeighbors )

   def render( self ):
      def bfdStateString( bfdState ):
         bfdStateDict = {
               "up" : "U",
               "down" : "D",
               "init" : "I",
               "disabled" : "N"
         }

         return bfdStateDict[ bfdState ]

      print( "U - BFD is enabled and is UP" )
      print( "I - BFD is enabled and is INIT" )
      print( "D - BFD is enabled and is DOWN" )
      print( "N - Not running BFD" )
      header = [ 'Neighbor Address', 'Interface', 'Uptime',
                 'Expires', 'Mode', 'Flags' ]

      if not self.interfaces:
         print( "%-18s %-16s %-10s %-10s %-17s %-5s" % tuple( header ) )
         return
      table = PrettyTable( header )

      # Print neighbors with BFD state.
      for _, interface in sorted( self.interfaces.items() ):
         for _, neighbor in sorted( interface.neighbors.items() ):
            uptimeStr = timeFormat( abs( Tac.utcNow() - neighbor.creationTime ) )
            if neighbor.holdTime > abs( Tac.utcNow() - neighbor.lastRefreshTime ):
               expiresStr = timeFormat( neighbor.holdTime -
                                        abs( Tac.utcNow() -
                                             neighbor.lastRefreshTime ) )
            else:
               expiresStr = timeFormat( 0 )

            table.add_row( [ neighbor.address.stringValue,
                             neighbor.intf.stringValue, uptimeStr, expiresStr,
                             neighbor.mode.toString(),
                             bfdStateString( neighbor.bfdState ) ] )
      printTable( table )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         tdict = {}
         for interface in dictRepr[ 'interfaces' ].values():
            for neighbor in interface[ 'neighbors' ].values():
               tdict[ neighbor[ 'address' ] ] = neighbor
         dictRepr[ 'neighbors' ] = tdict
         del dictRepr[ 'interfaces' ]
      return dictRepr

class PimBfdNeighbors( Model ):
   '''Cli Model to wrap up PimBfdNeighbors for show ip pim vrf all neighbor bfd.'''
   __revision__ = 2
   vrfs = Dict( help="A mapping between VRF name and PimBfdNeighbors",
                          keyType=str, valueType=VrfPimBfdNeighbors )
   _isAll = Bool( help="A boolean value to determine if current model is for all" )
   def render( self ):
      for key in sorted( self.vrfs ):
         neighbors = self.vrfs[ key ]
         print( "PIM Neighbor Table for " + key + " VRF" )
         neighbors.render()

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         if self._isAll is True:
            raise CliModelNotDegradable( "Current revision does not support all" )
         # No more than one vrf will be populated when _isAll is false
         if len( self.vrfs ) == 1:
            dictRepr[ "neighbors" ] = \
               dictRepr[ "vrfs" ][ next(
                     iter( dictRepr[ "vrfs" ] ) ) ][ "neighbors" ]
         else:
            dictRepr[ "neighbors" ] = {}
         del dictRepr[ "vrfs" ]
      return dictRepr

class PimMessageType( Model ):
   messageType = Enum( help="Pim message type",
                       values=( 'Assert',
                                'BootstrapRouter',
                                'CrpAdvertisement',
                                'DFElection',
                                'Graft',
                                'GraftAck',
                                'Hello',
                                'JoinPrune',
                                'Register',
                                'RegisterStop' ) )

   def fromTacc( self, mType ):
      ''' Convert from message type number to message type key. '''
      typeDict = {
         0 : 'Hello',
         1 : 'Register',
         2 : 'RegisterStop',
         3 : 'JoinPrune',
         4 : 'BootstrapRouter',
         5 : 'Assert',
         6 : 'Graft',
         7 : 'GraftAck',
         8 : 'CrpAdvertisement',
         10 : 'DFElection'
      }

      if mType in typeDict:
         self.messageType = typeDict[ mType ]
         return self.messageType
      else:
         return None

   def toString( self ):
      ''' Convert from message type key to a formatted string. '''
      typeDict = {
         'Hello' : 'Hello',
         'Register' : 'Register',
         'RegisterStop' : 'Register Stop',
         'JoinPrune' : 'Join/Prune',
         'BootstrapRouter' : 'Bootstrap Router',
         'Assert' : 'Assert',
         'Graft' : 'Graft',
         'GraftAck' : 'Graft Ack',
         'CrpAdvertisement' : 'CRP Advertisement',
         'DFElection' : 'DF Election'
      }

      if self.messageType in typeDict:
         return typeDict[ self.messageType ]
      else:
         return ""

class PimMessageCounter( Model ):
   rxPackets = Int( help="Received packets" )
   rxInvalid = Int( help="Invalid/errored received packets" )
   rxFiltered = Int( help="Filtered received packets" )
   txPackets = Int( help="Sent packets" )
   txInvalid = Int( help="Invalid/errored sent packets" )
   txFiltered = Int( help="Filtered sent packets" )

   def init( self ):
      self.rxPackets = 0
      self.rxInvalid = 0
      self.rxFiltered = 0
      self.txPackets = 0
      self.txInvalid = 0
      self.txFiltered = 0

class PimMessageSubtypeCounters( Model ):
   subtypeCounter = Dict( help="Subtype message counters",
                          valueType=PimMessageCounter )

class PimMessageCountersDetails( Model ):
   subtypeCounters = Dict( help="Map from message type to its subtype counters",
                           valueType=PimMessageSubtypeCounters )

   def init( self ):
      for mtype in PimCountersLib.messageTypeDict:
         typeModel = PimMessageType()
         if not typeModel.fromTacc( mtype ):
            continue

         subtypeDict = PimCountersLib.messageSubtypeDict( mtype )
         self.subtypeCounters[ typeModel.messageType ] = PimMessageSubtypeCounters()
         for stype in subtypeDict: # pylint: disable=consider-using-dict-items
            counterModel = PimMessageCounter()
            counterModel.init()
            self.subtypeCounters[ typeModel.messageType ].\
                  subtypeCounter[ subtypeDict[ stype ] ] = counterModel

   def fromTacc( self, allCounterList, intfId, detail ):
      self.init()

      for pimCounters in allCounterList:
         counterWrapper = PimCountersLib.PimCountersWrapper( pimCounters )
         for mtype in PimCountersLib.messageTypeDict:
            typeModel = PimMessageType()
            if not typeModel.fromTacc( mtype ):
               continue

            # Handle subtype counters.
            subtypeDict = PimCountersLib.messageSubtypeDict( mtype )
            for stype in subtypeDict: # pylint: disable=consider-using-dict-items
               counterModel = self.subtypeCounters[ typeModel.messageType ].\
                     subtypeCounter[ subtypeDict[ stype ] ]

               txCounter = counterWrapper.getCounter( 'tx', mtype, stype, intfId )
               counterModel.txPackets += txCounter.success
               counterModel.txInvalid += txCounter.invalid
               counterModel.txFiltered += txCounter.filtered

               rxCounter = counterWrapper.getCounter( 'rx', mtype, stype, intfId )
               counterModel.rxPackets += rxCounter.success
               counterModel.rxInvalid += rxCounter.invalid
               counterModel.rxFiltered += rxCounter.filtered

               self.subtypeCounters[ typeModel.messageType ].\
                     subtypeCounter[ subtypeDict[ stype ] ] = counterModel

class PimMessageCounters( Model ):
   counters = Dict( help="Map from message type to its counters",
                    valueType=PimMessageCounter )

   details = Submodel( help="Detailed interface information",
                       valueType=PimMessageCountersDetails, optional=True )

   def init( self ):
      # Initialize all model counters to 0.
      for mtype in PimCountersLib.messageTypeDict:
         typeModel = PimMessageType()
         if not typeModel.fromTacc( mtype ):
            continue
         counterModel = PimMessageCounter()
         counterModel.init()
         self.counters[ typeModel.messageType ] = counterModel

   def fromTacc( self, allCounterList, intfId, detail ):
      self.init()

      # Add up all counters and put them in the model.
      for pimCounters in allCounterList:
         counterWrapper = PimCountersLib.PimCountersWrapper( pimCounters )
         for mtype in PimCountersLib.messageTypeDict:
            typeModel = PimMessageType()
            if not typeModel.fromTacc( mtype ):
               continue

            counterModel = self.counters[ typeModel.messageType ]

            txCounter = counterWrapper.getCounter( 'tx', mtype, None, intfId )
            counterModel.txPackets += txCounter.success
            counterModel.txInvalid += txCounter.invalid
            counterModel.txFiltered += txCounter.filtered

            rxCounter = counterWrapper.getCounter( 'rx', mtype, None, intfId )
            counterModel.rxPackets += rxCounter.success
            counterModel.rxInvalid += rxCounter.invalid
            counterModel.rxFiltered += rxCounter.filtered

      if detail:
         self.details = PimMessageCountersDetails()
         self.details.fromTacc( allCounterList, intfId, detail )

   def render( self ):
      print( "PIM Control Counters" )
      print( "%-20s %12s %12s %12s %12s %12s %12s" %
            ( '', 'Received', 'Rx Error', 'Rx Filtered',
                  'Sent', 'Tx Error', 'Tx Filtered' ) )

      for key in sorted( self.counters ):
         mtype = PimMessageType()
         mtype.messageType = key
         totalCounter = self.counters[ key ]
         print( "%-20s %12d %12d %12d %12d %12d %12d" %
               ( mtype.toString(), totalCounter.rxPackets,
                 totalCounter.rxInvalid,
                 totalCounter.rxFiltered, totalCounter.txPackets,
                 totalCounter.txInvalid, totalCounter.txFiltered ) )
         if self.details:
            subtypeCounters = self.details.subtypeCounters[ key ].subtypeCounter
            for subkey in sorted( subtypeCounters ):
               subCounter = subtypeCounters[ subkey ]
               print( "  %-18s %12d %12d %12d %12d %12d %12d" %
                  ( subkey, subCounter.rxPackets,
                    subCounter.rxInvalid,
                    subCounter.rxFiltered, subCounter.txPackets,
                    subCounter.txInvalid, subCounter.txFiltered ) )


#----------------------------------------------------------------------------------
# PimBidirModel
#----------------------------------------------------------------------------------

class Rpf( DeferredModel ):
   rpfNeighbor = IpGenericAddress( help="RPF neighbor", optional=True )
   rpfPrefix = IpGenericPrefix( help="RPF route prefix" )
   rpfMetric = Int( help="RPF route metric" )
   rpfPreference = Int( help="RPF route preference" )
   rpfRib = Enum( help="RPF routing table",
         values=( "none", "U", "M" ) )

class MrouteInterface( DeferredModel ):
   address = IpGenericAddress( help="Interface IP address", optional=True )
   iAmDf = Bool( help="This Pim router is the designated router on this interface" )
   forward = Bool( help="Member of outgoing interface set" )

   downstreamJoinState = Enum( help="Downstream join state",
         values=( "noInfo",
            "prunePending",
            "joined" ) )
   lastJoinReceived = Float( help="UTC time at which the last downstream join "
         "was received", optional=True )
   localReceiverInclude = Bool( help="Local receiver include ( *,G,I )" )

class MrouteInterfaceDetails( DeferredModel ):
   interfaces = Dict( help="Map of interfaces for multicast route",
         keyType=Interface, valueType=MrouteInterface )

class Mroute( DeferredModel ):
   rp = IpGenericAddress( help="RP(G): The Rendezvous Point for this group",
         optional=True )

   rpf = Submodel( help="Reverse path forwarding information", valueType=Rpf,
         optional=True )

   rpfInterface = Interface( help="Reverse path forwarding interface, packets for "
         "this mroute must arrive on this interface to be "
         "accepted/forwarded. If this value is set to "
         "'Register', the RPF interface is the register tunnel",
         optional=True )

   rpfFrr = Submodel( help="Reverse path forwarding information for MoFRR",
                      valueType=Rpf, optional=True )

   rpfFrrInterface = Interface(
      help="Reverse path forwarding interface for MoFRR",
      optional=True )

   creationTime = Float( help="UTC time at which the route was created" )
   oifList = List( help="Outgoing interface list", valueType=Interface )
   upstreamJoinState = Enum( help="Upstream joined state",
         values=( "upNotJoined", "upJoined" ) )
   active = Bool( help="Group active", optional=True )
   joinDesired = Bool( help="Join Desired (*, G)" )
   details = Submodel( help="Detailed information for this multicast route",
         valueType=MrouteInterfaceDetails, optional=True )

class Groups( DeferredModel ):
   groups = Dict( help="Map of bidirectional mode PIM multicast groups",
         keyType=IpGenericAddress, valueType=Mroute )
class MrouteBidirCount( DeferredModel ):
   groups = Int( help="Number of groups" )

#-----------------------------------------------------------------------------------
# PimsmModel
#----------------------------------------------------------------------------------

class OifPrime( DeferredModel ):
   """ Interfaces not in the OIF set of an mroute """
   # Possible reasons why this interface isn't in the OIF set
   interface = Interface( help="Name of this interface" )
   incomingInterface = Bool( help="The incoming interface for the route" )
   notDr = Bool( help="Not a designated router on this interface" )
   assertLoser = Bool( help="Assert loser on this interface" )
   rptPruned = Bool( help="Pruned from the RP tree" )
   locallyExcluded = Bool( help="Locally excluded" )

class WcMroute( DeferredModel ):
   """(*,G) specific state"""
   joinDesiredWc = Bool( help="Join Desired (*,G)" )
   joinDesiredWcRp = Bool( help="Join Desired (*,*,G)" )
   rptJoinDesiredWc = Bool( help="RPT Join Desired (*,G)" )
   nullImmediateOlistWc = Bool( help="No interfaces in immediate olist (*,G)" )
   nullImmediateOlistWcRp = Bool( help="No interfaces in immediate olist (*,*,G)" )

class SgMroute( DeferredModel ):
   """(S,G) speficic state"""
   sourceActive = Bool( help="If true, there is active traffic for the source" )
   couldRegisterSg = Bool( help="CouldRegister(S,G): If true, we may send register"
         " messages for this (S,G)" )
   registerStopSg = Bool( help="If true, we should send a register stop for this "
         "(S,G) in response to a register message" )
   joinDesiredSg = Bool( help="Join Desired (S,G), if true we should send joins "
         "for this source if a valid upstream neighbor exists" )
   pruneDesiredSgRpt = Bool( help="Prune Desired (S,G,Rpt), if true, we should "
         " prune ourselves from the RP tree for this source. "
         " This is to avoid receiving traffic both from the RP "
         " tree and source tree." )
   nullImmediateOlistSg = Bool( help="No interfaces in immediate olist (S,G)" )
   nullInheritedOlistSg = Bool( help="No interfaces in immediate olist (S,G)" )

class EvpnInfo( DeferredModel ):
   # Evpn related information
   vrf = Str( help="VRF the interface belongs to", optional=True )
   vni = Int( help="VNI of the interface", optional=True )
   sbdVlan = Bool( help="This is SBD VLAN interface", optional=True )
   pegDr = Bool( help="This interface is PEG DR", optional=True )

class MrouteDetails( DeferredModel ):
   oifPrimes = Dict( help="Map of interfaces not in OIF set",
         valueType=OifPrime )
   upstreamJoinState = Enum( help="Upstream joined state",
         values=( "upNotJoined",
            "upJoined" ) )

   upstreamRptState = Enum( help="Upstream RPT joined state",
                     values=( "upRptNotPruned",
                              "upRptNotJoined",
                              "upRptPruned" ) )

   sgState = Submodel( help="State specific to (S,G) routes",
         valueType=SgMroute, optional=True )
   wcState = Submodel( help="State specific to (*,G) routes",
         valueType=WcMroute, optional=True )
   fastFailovers = List( help="This field is not used", 
                         valueType=str, 
                         optional=True )
   evpnInfo = Submodel( help="EVPN Information",
                        valueType=EvpnInfo, optional=True )

class RpfSm( DeferredModel ):
   rpfNeighbor = IpGenericAddress( help="RPF neighbor", optional=True )
   rpfPrefix = IpGenericPrefix( help="RPF route prefix" )
   rpfMetric = Int( help="RPF route metric" )
   rpfPreference = Int( help="RPF route preference" )
   rpfRib = Enum( help="RPF routing table",
         values=( "none",
            "U",
            "M" ) )
   rpfAttached = Bool( help="If true, the source/RP is directly connected to this "
         "router" )
   rpfEvpnTenantDomain = Bool( help="If true, the source/RP is inside EVPN Tenant "
                               " Domain ", optional=True )
   rpfMvpn = Bool( help="If true, the source/RP is inside MVPN Domain ",
                   optional=True )

class MrouteSm( DeferredModel ):
   sourceAddress = IpGenericAddress( help="Source address" )
   rp = IpGenericAddress( help="RP(G): The Rendezvous Point for this group",
         optional=True )
   rpf = Submodel( help="Reverse path forwarding information", valueType=RpfSm,
         optional=True )
   rpfInterface = Interface( help="Reverse path forwarding interface, packets for "
         "this mroute must arrive on this interface to be "
         "accepted/forwarded. If this value is set to "
         "'Register', the RPF interface is the register tunnel",
         optional=True )
   rpfFrr = Submodel( help="Reverse path forwarding information for MoFrr",
                      valueType=RpfSm, optional=True )
   rpfFrrInterface = Interface(
      help="Reverse path forwarding interface for MoFRR",
      optional=True )

   routeFlags = Str( help="Route flags" )
   creationTime = Float( help="UTC time at which the route was created" )
   oifList = List( help="Outgoing interface list", valueType=Interface )
   nonDrOifList = List( help="List of outgoing interfaces with egress drop rule "
                        "in effect on non-DR", 
                        valueType=Interface, 
                        optional=True )
   registerInOifList = Bool( help="If true, the register tunnel is part of the "
         " outgoing interface list", optional=True )
   details = Submodel( help="Detailed information for this multicast route",
         valueType=MrouteDetails, optional=True )

class GroupSm( DeferredModel ):
   groupSources = Dict( help="Map of multicast routes for this group",
               keyType=IpGenericAddress, valueType=MrouteSm )

class GroupSms( DeferredModel ):
   groups = Dict( help="Map of sparse-mode Pim multicast groups",
         keyType=IpGenericAddress, valueType=GroupSm )

class SourceGroupCollection( DeferredModel ):
   sourceOrGroupAddrs = GeneratorDict( help="List of IP address",
         keyType=IpGenericAddress, valueType=str )

class SourcesGroups( DeferredModel ):
   isSource = Bool( help="If true, we print S-> no of G" )
   sourcegroup = GeneratorDict( help="S-> no of G or G-> no of S mapping ",
         keyType=int, valueType=SourceGroupCollection )

class AssertMetric( DeferredModel ):
   rptBit = Bool( help="Assert metric is for the RP tree" )
   preference = Int( help="Route preference to the source/RP" )
   metric = Int( help="Route metric to the source/RP" )
   address = IpGenericAddress( help="Address of router that generated the assert" )

class SgMrouteInterface( DeferredModel ):
   """(S,G,I) state"""
   couldAssertSg = Bool( help="Could assert (S,G,I)" )
   assertTrackingDesiredSg = Bool( help="Assert tracking desired (S,G,I)" )
   lostAssertSg = Bool( help="Member of lost assert (S,G)" )
   lostAssertSgRpt = Bool( help="Member of lost assert (S,G,Rpt)" )
   myAssertMetricSg = Submodel( help="My assert metric (S,G,I)",
         valueType=AssertMetric, optional=True )
   sptAssertMetric = Submodel( help="SPT assert metric (S,I)",
         valueType=AssertMetric, optional=True )
   # State summarization macros
   localReceiverIncludeSg = Bool( help="Local receiver include (S,G,I)" )
   localReceiverExcludeSg = Bool( help="Local receiver exclude (S,G,I)" )
   joinsSg = Bool( help="Member of joins (S,G)" )
   prunesSgRpt = Bool( help="Member of prunes (S,G,Rpt)" )
   includeSg = Bool( help="Member of pim include (S,G)" )
   excludeSg = Bool( help="Member of pim exclude (S,G)" )
   immediateOlistSg = Bool( help="Member of immediate olist (S,G)" )
   inheritedOlistSg = Bool( help="Member of inherited olist (S,G)" )
   inheritedOlistSgRpt = Bool( help="Member of inherited olist (S,G,Rpt)" )

class WcMrouteInterface( DeferredModel ):
   """(*,G,I) state"""
   # Assert state
   couldAssertWc = Bool( help="Could assert (*,G,I)" )
   assertTrackingDesiredWc = Bool( help="Assert tracking desired (*,G,I)" )
   lostAssertWc = Bool( help="Member of lost assert (*,G)" )
   myAssertMetricWc = Submodel( help="My assert metric (*,G,I)",
         valueType=AssertMetric, optional=True )
   rptAssertMetric = Submodel( help="RPT assert metric (G,I)",
         valueType=AssertMetric, optional=True )
   # State summarization macros
   # Specific to (*,G) routes (and therefore optional)
   localReceiverIncludeWc = Bool( help="Local receiver include (*,G,I)" )
   joinsWc = Bool( help="Member of joins (*,G)" )
   joinsWcRp = Bool( help="Member of joins (*,*,G)" )
   includeWc = Bool( help="Member of pim include (*,G)" )
   immediateOlistWc = Bool( help="Member of immediate olist (*,G)" )
   immediateOlistWcRp = Bool( help="Member of immediate olist (*,*,G)" )


class MrouteSmInterfaceDetails( DeferredModel ):
   # Detailed assert information
   lastAssertAction = Int( help="Last assert action", optional=True )
   assertWinnerMetric = Submodel( help="Assert winner metric",
         valueType=AssertMetric, optional=True )
   sgState = Submodel( help="Interface state specific to (S,G) routes",
         valueType=SgMrouteInterface, optional=True )
   wcState = Submodel( help="Interface state specific to (*,G) routes",
         valueType=WcMrouteInterface, optional=True )
   evpnInfo = Submodel( help="EVPN Information",
                        valueType=EvpnInfo, optional=True )

class MrouteSmInterface( DeferredModel ):
   address = IpGenericAddress( help="Interface IP address", optional=True )
   iAmDr = Bool( help="This Pim router is the designated router on this interface" )
   forward = Bool( help="Member of outgoing interface set" )
   markedForDeletion = Bool( help="If true, this interface has no (S,G,I)/(*,G,I)"
         " state and is scheduled to be removed from the "
         " route's interface list" )
   downstreamJoinState = Enum( help="Downstream join state",
         values=( "noInfo",
            "prunePending",
            "joined" ) )
   lastJoinReceived = Float( help="UTC time at which the last downstream join "
         "was received", optional=True )
   rptState = Enum( help="Downstream RPT state",
         values=( "rptNoInfo",
            "rptPruned",
            "rptPrunePending",
            "rptPruneTemp",
            "rptPrunePendingTemp" ) )
   lastRptPruneReceived = Float( help="UTC time at which the last downstream RPT "
         "prune was received", optional=True )

   assertState = Enum( help="Assert state",
         values=( "assertNoInfo",
            "iAmAssertWinner",
            "iAmAssertLoser" ) )
   details = Submodel( help="Detailed mroute interface information",
         valueType=MrouteSmInterfaceDetails, optional=True )

class MrouteSmInterfaces( DeferredModel ):
   source = IpGenericAddress( help="Source address of multicast route" )
   group = IpGenericAddress( help="Group address of multicast route" )
   interfaces = Dict( help="Map of interfaces for multicast route",
         keyType=Interface, valueType=MrouteSmInterface )

class MrouteCount( DeferredModel ):
   groups = Int( help="Number of groups" )
   sources = Int( help="Number of sources" )
   wcRoutes = Int( help="Number of (*,G) routes" )
   sgRoutes = Int( help="Number of (S,G) routes" )
   unresolved = Int( help="Number of routes for which RPF information has not been"
         "determined" )
   toBeDeleted = Int( help="Number of routes currently scheduled for deletion" )
   switching = Int( help="Number of (S,G)s currently in the midst of switching from"
         " the RP tree to the SP tree" )
   couldRegister = Int( help="Number of (S,G)s for which CouldRegister(S,G) is "
         "True" )
   registerStop = Int( help="Number of (S,G)s for which the RP should generate"
         " register stops on receipt of a register/null register" )
   registerOif = Int( help="Number of (S,G)s with the register tunnel in the OIF "
         "set" )
   mbrDiscovered = Int( help="Number of (S,G)s discovered via MBR" )
   drDiscovered = Int( help="Number of routes discovered on the RP from a DR via a"
         "a register message" )
   anycastRpDiscovered = Int( help="Number of routes discovered on the RP from from"
         " an anycast RP via a register message" )
   msdpDiscovered = Int( help="Number of routes discovered on the RP via an MSDP "
         "source advertisement" )
