# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=redefined-builtin

import Tac, sys, time, Pci

pcieGpioUndefined = "PcieGpioUndefined"
pcieGpioDirectionReg = "PcieGpioDirectionControl"
pcieGpioOutputReg = "PcieGpioOutputReg"
pcieGpioInputReg = "PcieGpioInputReg"

class GpioError( Exception ):
   pass

class GpioPin:
   def __init__( self, bit, name, activeHi, isInput ):
      self.bit = bit
      self.name = name
      self.activeHi = activeHi
      self.input = isInput

#There is a separate PLXPcieSwitchGpio class for the switch as it is a
#different device than the PCA92555 for which the the Gpio class defined later
class PLXPcieSwitchGpio:

   def __init__( self, switch):
      self.switch = switch

   def set( self, name, value):
      try:
         id = int(name)
      except ValueError:
         id = -1
      p = self.switch.pinByName_.get(name) or self.switch.pinById_.get(id)
      if not p:
         raise GpioError( f"Pin with name/id {name} not found" )
      id = p.bit
      directionBit = self.switch.getGpioPinBit(id, pcieGpioDirectionReg)
      directionReg = self.switch.getGpioPinReg(id, pcieGpioDirectionReg)
      
      def gpioSet(bitPosition, reg):
         oldVal = self.switch.readReg(0, reg)
         #output is 1 input is 0
         if value == 'output':
            writeValue = 1
         elif value == 'input':
            writeValue = 0
         else:
            writeValue = (value == 'on') ^ (not p.activeHi)
         newVal = (oldVal & ~(0x1 << bitPosition)) | (writeValue << bitPosition)
         self.switch.writeReg(0, reg, newVal)
          
      if value in ['on', 'off']:
         outputBit = self.switch.getGpioPinBit(id, pcieGpioOutputReg)
         outputReg = self.switch.getGpioPinReg(id, pcieGpioOutputReg)

         if p.input:
            print( "Pin is input pin" )
         gpioSet(outputBit, outputReg)
      elif value in ['input', 'output']:
         gpioSet(directionBit, directionReg)
      else:
         raise GpioError( f"illegal value '{value}'" )

   def read( self, name, returnLogicalValue=True, reg=None ):
      p = self.switch.pinByName_.get(name) or self.switch.pinById_.get(int(name))
      if not p:
         raise GpioError( f"Pin with name/id {name} not found" )
      id = p.bit
      if reg is not None:
         readReg = reg
      elif p.input:
         readReg = pcieGpioInputReg
      else:
         readReg = pcieGpioOutputReg

      readBit = self.switch.getGpioPinBit(id, readReg)
      readReg = self.switch.getGpioPinReg(id, readReg)
      value = self.switch.readReg(0, readReg)
      value = (value >> readBit) & 0x1
      if returnLogicalValue and not p.activeHi:
         value = int( not value )
      return value

   def scan( self, pinId=None ):
      def scanPin( pin ):
         # output == 0; input == 1
         dirConfig = pin.input
         dirConfigStr = "input" if dirConfig else "output"
         dirStatus = not self.read(
            pin.name, returnLogicalValue=False, reg=pcieGpioDirectionReg )
         dirStatusStr = "input" if dirStatus else "output"
         if dirConfig != dirStatus:
            detail = "expected " + dirConfigStr
         else:
            detail = "ok"

         activeType = "hi" if pin.activeHi else "lo"

         reg = pcieGpioInputReg if dirStatus else pcieGpioOutputReg
         value = self.read( pin.name, returnLogicalValue=False, reg=reg )
         logicalValue = value if pin.activeHi else not value
         logicalValueStr = "on" if logicalValue else "off"

         # pylint: disable-next=consider-using-f-string
         print( "%2d %15s is %3s (%1s) %6s (active %s) %s" % (
            pin.bit,
            pin.name,
            logicalValueStr,
            int( value ),
            dirStatusStr,
            activeType,
            detail ) )

      if pinId is not None:
         pin = self.switch.pinById_[ pinId ]
         scanPin( pin )
      else:
         pins = self.switch.pinById_
         for pin in pins.values():
            scanPin( pin )

class PciePort:
   def __init__( self, id, direction, width, desc, expectedSpeed ):
      self.id = id
      self.direction = direction
      self.width = width
      self.desc = desc
      self.expectedSpeed = expectedSpeed

class PcieSwitch:
   """ Base class for the PLX86xx family of switches """

   #Electrical idle mask registers for 86xx family
   electricalIdleMaskEvenReg = 0x200
   electricalIdleMaskOddReg = 0x204
   erratum12Reg = 0x23c
   
   def __init__( self, smbusDevice, gpioPins, pciPorts, idNum ):
      self.pinByName_ = {}
      self.pinById_ = {}
      self.portById_ = {}
      self.idNum_ = idNum
      for (i, name, activeHi, direction) in gpioPins:
         gp = GpioPin( i, name, activeHi, direction=='input' )
         self.pinByName_[name] = gp
         self.pinById_[i] = gp
      for (id, direction, width, desc, expectedSpeed ) in pciPorts:
         port = PciePort( id, direction, width, desc, expectedSpeed )
         self.portById_[id] = port
      self.device_ = smbusDevice
      
   def writeReg( self, port, addr, data ):
      # convert the data into a string
      # TODO check the byte ordering
      addrVal = Tac.newInstance( "HalPlxPcieSwitch::Smbus8600RegAddrWrapper" )
      addrVal.portSelector = port
      addrVal.regAddr = addr
      addrVal.regByteMask = 0xf
      addrVal.command = 'write'
      writeBuf = ""
      for i in range(3, -1, -1):
         writeBuf += chr( ( data >> (i * 8) ) & 0xff )
      self.device_.write(addrVal.value, writeBuf, accessSize = 4)

   def readReg( self, port, addr ):
      addrVal = Tac.newInstance( "HalPlxPcieSwitch::Smbus8600RegAddrWrapper" )
      addrVal.portSelector = port
      addrVal.regAddr = addr
      addrVal.regByteMask = 0xf
      addrVal.command = 'read'
      res = self.device_.read(addrVal.value, count = 4, accessSize = 4)
      rc = 0
      for i in range(4):
         rc = ( rc << 8 ) + res[i]
      return rc
      
   def getVendorIdDeviceID( self, port ):
      val = self.readReg(port, 0x0)
      # vendorID, deviceID
      return val & 0xffff, val >> 16

   def getPciBusNumbers( self, port ):
      val = self.readReg(port, 0x18)
      # primary bus, secondary bus, subordinate bus
      return val & 0xff, (val >> 8) & 0xff, (val >> 16) & 0xff
      
   def getErrorCount( self, swHamImpl, port ):
      regBase = 0xb88
      regOffset = ( ( (port // 8) * 2 ) + ( port % 2 ) ) * 4
      reg = regBase + regOffset
      regMask = 0xff << ( ( (port % 8) // 2 ) * 8 )
      if swHamImpl:
         val = swHamImpl.data[ reg ]
      else:
         val = self.readReg(0, reg)
      print( "errorcount: " + hex( val ) )
      return ( val & regMask ) >> ( ( (port % 8) // 2 ) * 8 )

   def resetErrorCount( self, swHamImpl, port ):
      regBase = 0xb88
      regOffset = ( ( (port // 8) * 2 ) + ( port % 2 ) ) * 4
      reg = regBase + regOffset
      if swHamImpl:
         regVal = swHamImpl.data[ reg ]
      else:
         regVal = self.readReg(0, reg)
      mask = regVal & ( 0xff << ( ( (port % 8) // 2 ) * 8 ) )
      regVal = regVal & ~mask
      if swHamImpl:
         swHamImpl.data[ reg ] = regVal
      else:
         self.writeReg(0, reg, regVal)
         #self.writeReg(0, reg, 0xffffffff)

   def cdrClear( self, swHamImpl, port ):
      serdes = self.getSerdes( port )
      reg = 0xbd0 + ( (serdes // 8) * 4 )
      if swHamImpl:
         swHamImpl.data[ reg ] = 0
      else:
         self.writeReg(0, reg, 0)

   def cdrFreeze( self, swHamImpl, port, reset ):
      self.cdrClear( swHamImpl, port )
      serdes = self.getSerdes( port )
      reg = 0xbd0 + ( (serdes // 8) * 4 )
      shift = ( serdes % 8 ) * 4
      if reset:
         regVal = 0xc << shift
      else:
         regVal = 0x4 << shift
      if swHamImpl:
         swHamImpl.data[ reg ] = regVal
      else:
         self.writeReg(0, reg, regVal)

   stepDown = 0x1
   stepUp = 0x2      
   dirStr = { 0x1: "down", 0x2: "up" }
   maxStepOutIterations = 50
      
   def cdrGetStepOutCount( self, swHamImpl, port, direction, maxCount ):
      serdes = self.getSerdes( port )
      reg = 0xbd0 + ( (serdes // 8) * 4 )
      shift = ( serdes % 8 ) * 4
      if swHamImpl:
         regVal = (direction << shift) | swHamImpl.data[ reg ]
      else:
         regVal = self.readReg(0, reg)
      sys.stdout.write(f"Step out, direction {PcieSwitch.dirStr[direction]} .")
      sys.stdout.flush()
      count = 0
      for i in range(maxCount):
         if swHamImpl:
            swHamImpl.data[ reg ] = regVal
         else:
            self.writeReg(0, reg, regVal)
         # allow some time for the errors to build up
         time.sleep( 3 )
         errCnt = self.getErrorCount( swHamImpl, port )
         if errCnt:
            break
         sys.stdout.write(".")
         sys.stdout.flush()
         count = i
      print( f"OK, step out count {count}" )
      return count

   def cdrGetStepBackCount( self, swHamImpl, port, direction, maxCount ):
      serdes = self.getSerdes( port )
      reg = 0xbd0 + ( (serdes // 8) * 4 )
      shift = ( serdes % 8 ) * 4
      if swHamImpl:
         regVal = (direction << shift) | swHamImpl.data[ reg ]
      else:
         regVal = self.readReg(0, reg)
      sys.stdout.write(f"Step in, direction {PcieSwitch.dirStr[direction]} .")
      sys.stdout.flush()
      for i in range(maxCount):
         self.resetErrorCount( swHamImpl, port )
         if swHamImpl:
            swHamImpl.data[ reg ] = regVal
         else:
            self.writeReg(0, reg, regVal)
         # allow some time for the errors to build up
         time.sleep( 3 )
         errCnt = self.getErrorCount( swHamImpl, port )
         if errCnt == 0:
            break
         sys.stdout.write(".")
         sys.stdout.flush()
      print( f"OK, step back count {maxCount - i}" )
      return maxCount - i

   def pciAddress( self, port ):
      busNum, _, _ = self.getPciBusNumbers( port )
      return Pci.Address( 0, busNum, port, 0 )

   def getSerdesEyeWidth( self, port, useSmbus=True ):
      _, status = self.getLinkStatus( port )
      if not status:
         return f"port {port} not connected, cannot measure eye"
      if not useSmbus:
         # get a ham to talk to the switch root port, some of the registers are not
         # available over i2c, all mmio accesses go via port 0
         import Fru # pylint: disable=import-outside-toplevel
         pciAddress = self.pciAddress( 0 )
         swHam = Tac.newInstance( "Fru::Ham",
                                        *Fru.pciDeviceHam( "swHam", pciAddress, 0 ) )
         swHamImpl = swHam.hamImpl
         assert swHamImpl
      else:
         swHamImpl = None
      self.cdrFreeze( swHamImpl, port, False )
      self.getErrorCount( swHamImpl, port )
      self.resetErrorCount( swHamImpl, port )
      self.getErrorCount( swHamImpl, port )
      upCount = self.cdrGetStepOutCount( swHamImpl, port, PcieSwitch.stepUp, 
                                                  PcieSwitch.maxStepOutIterations )
      if upCount and upCount <= PcieSwitch.maxStepOutIterations:
         upCount = self.cdrGetStepBackCount( swHamImpl, port, PcieSwitch.stepDown, 
                                                                            upCount )
      self.cdrFreeze( swHamImpl, port, False )
      self.getErrorCount( swHamImpl, port )
      self.resetErrorCount( swHamImpl, port )
      self.getErrorCount( swHamImpl, port )
      downCount = self.cdrGetStepOutCount( swHamImpl, port, PcieSwitch.stepDown, 
                                                   PcieSwitch.maxStepOutIterations)
      if downCount and downCount <= PcieSwitch.maxStepOutIterations:
         downCount = self.cdrGetStepBackCount( swHamImpl, port, PcieSwitch.stepUp, 
                                                                          downCount )
      # the 6 is an artifact of the plx example code, and means a number we
      # generate is compatible with the values their software produces
      self.cdrClear( swHamImpl, port )
      self.resetErrorCount( swHamImpl, port )
      return f"eye width {(upCount + downCount) * 6}"

   def showSerdesEyeWidth( self, port ):
      if port == "upstream":
         for i, portObj in self.portById_.items():
            if portObj.direction == "upstream":
               self.getSerdesEyeWidth( i )
      elif port in self.portById_:
         return self.getSerdesEyeWidth( port )
      print( f"Invalid port {port}\n" )
      return None

   def setSerdesParams( self, port, drive, deEmphasis, equalisation ):
      serdes = self.getSerdes( port )
      reg = 0xb98 + ( ( serdes // 4) * 4 )
      #Save the other 24 bits in the register. 8 bits per port
      oldVal = self.readReg(0, reg)
      oldVal = oldVal & ~(0xff << ( ( serdes % 4 ) * 8 ) )
      val = ((0x80 | drive) << ( ( serdes % 4 ) * 8 )) | oldVal
      self.writeReg(0, reg, val)
      reg = 0xba8 + ( ( serdes // 4) * 4 )
      oldVal = self.readReg(0, reg)
      oldVal = oldVal & ~(0xff << ( ( serdes % 4 ) * 8 ) )
      val = ((0x80 | deEmphasis) << ( ( serdes % 4 ) * 8 )) | oldVal
      self.writeReg(0, reg, val)
      reg = 0xbb8 + ( ( serdes // 8) * 4 )
      oldVal = self.readReg(0, reg)
      #4 bits per port
      oldVal = oldVal & ~(0xf << ( ( serdes // 8 ) * 4 ) )
      val = (equalisation << ( ( serdes % 8 ) * 4 )) | oldVal
      self.writeReg(0, reg, val)

   def enableAllPorts( self, enable=True ):
      for port in self.portById_:
         self.enablePort( port, enable )

   def enablePort( self, port, enable=True ):
      reg = 0x230 + ( 4 * ( port % 2 ) )
      oldVal = self.readReg( 0, reg )
      mask = 1 << ( port // 2 )
      if not enable:
         val = oldVal | mask
      else:
         val = oldVal & ~mask
      self.writeReg( 0, reg, val )

   def enableHotplugPort( self, port, enable=True ):
      # only valid for downstream ports
      reg = 0x7c
      # Bit 5 is hot plug surprise, bit 6 is hot plug capability
      mask = 0x60
      oldVal = self.readReg( port, reg )
      if enable:
         val = oldVal | mask
      else:
         val = oldVal & ~mask
      self.writeReg( port, reg, val )

   def enableHotplug( self, enable=True ):
      # disable upstream ports
      for port, portObj in self.portById_.items():
         if portObj.direction == "upstream":
            self.enablePort( port, False )

      # enable hotplug on all downstream ports
      for port, portObj in self.portById_.items():
         if portObj.direction == "downstream":
            self.enableHotplugPort( port, enable )

      # reenable upstream ports
      for port, portObj in self.portById_.items():
         if portObj.direction == "upstream":
            self.enablePort( port, True )

   def loopbackEnable( self, analog ):
      if analog:
         # enables detection of training control bits and analog loopback
         self.writeReg(0, 0x23c, 0x00249c80 )
      else:
         # enables detection of training control bits and analog loopback
         self.writeReg(0, 0x23c, 0x00049c80 )
         
   def loopbackDisable( self ):
      # enables detection of training control bits and analog loopback
      self.writeReg(0, 0x23c, 0x00149c80 )

   def loopbackTest( self, port, duration, analog ):
      serdes = self.getSerdes( port )
      # pylint: disable-next=unused-variable
      speed, status = self.getLinkStatus( port )
      print( speed )
      #if not status:
      #   return ( "port %d not connected, cannot run loopback test" % port, 1 )
      # 1) put port into loopback mode
      self.loopbackEnable(analog)
      enableReg = 0x220 + ( (port % 2) * 4 )
      enableShift = ( ( port // 2) * 4 )
      # we assume all other ports have loopback disabled
      val = 1 << enableShift
      self.writeReg(0, enableReg, val)
      # 2) verify port in correct mode
      sys.stdout.write("Waiting for loopback mode")
      sys.stdout.flush()
      count = 0
      while ( self.readReg(0, enableReg) >> enableShift ) & 0xf != 0x9:
         #print "reg:"+hex(enableReg)+" value:"+hex(self.readReg(0, enableReg))
         sys.stdout.write(".")
         sys.stdout.flush()
         time.sleep( 1 )
         count += 1
         if count > 10:
            print( " failed to enter loopback mode" )
            # take port out of loopback mode
            self.writeReg(0, enableReg, 0)
            self.loopbackDisable()
            return ( " error during test setup", 1 )
      print( " " )
      # 3) set diag counter up
      errRegs = [ 0x240, 0x248, 0x244, 0x24c ]
      errorRegister = errRegs[ serdes // 4 ]
      # set error register to point to serdes we are testing
      errRegVal = ( serdes % 4 ) << 24
      self.writeReg(0, errorRegister, errRegVal)

      # for digital need to set up the user test data
      if not analog:
         self.writeReg(0, 0x210, 0x4a4a4a4a)
         self.writeReg(0, 0x214, 0x4a4a4a4a)
         self.writeReg(0, 0x218, 0xb5b5b5b5)
         self.writeReg(0, 0x21c, 0xb5b5b5b5)

      # 4) start test
      if not analog:
         testBit = 16 # User Test Pattern
      else:
         testBit = 24 # PRBS
      controlReg = 0x258 + ( 4 * (serdes // 8) )
      controlVal = 1 << ( ( serdes % 8 ) + testBit )
      self.writeReg(0, controlReg, controlVal)
      assert self.readReg(0, controlReg) == controlVal
      #print "control reg 0x%x, value 0x%x, expected 0x%x" % ( controlReg,
      #                                                 self.readReg(0, controlReg),
      #                                                 controlVal )
      
      # 5) wait some time monitoring the errors and provide user feedback
      sys.stdout.write("Test running ")
      sys.stdout.flush()
      prevErrCount = 0
      for _ in range(duration*4):
         time.sleep(0.25)
         regVal = self.readReg( 0, errorRegister ) 
         errCount = ( regVal >> 16 ) & 0xff
         if errCount != prevErrCount:
            print( f"e{errCount} ", end=' ' )
            prevErrCount = errCount
            if errCount == 0xff:
               # too many errors to continue
               break
         else:
            sys.stdout.write(".")
         sys.stdout.flush()
      print( " " )
      self.writeReg(0, errorRegister, 0)

      # 6) stop test
      self.writeReg(0, controlReg, 0)
      
      # 7) take port out of loopback mode
      self.writeReg(0, enableReg, 0)
      sys.stdout.write("Waiting for loopback to end .")
      sys.stdout.flush()
      while ( self.readReg(0, enableReg) >> enableShift ) & 0xf != 0:
         sys.stdout.write(".")
         sys.stdout.flush()
         time.sleep( 1 )
      print( "" )
      self.loopbackDisable()
      if errCount == 0xff:
         return ( "test failed. Error count overflowed", 1 )
      elif errCount: 
         return ( f"test failed. Error count {errCount}", 1 )
      else:
         return ( "test passed!", 0 )
         
   def getLinkStatus( self, port, easyMode = False ):
      # The easy parameter is to qualify 
      val = self.readReg(port, 0x78)
      status = (val >> 20) & 0x3f
      speed = (val >> 16) & 0xf

      if not status or status == 0x3f:
         serdes = self.getSerdes( port )
         reg = 0x200 + ( ( serdes // 8 ) * 4 )
         val = ( self.readReg( 0, reg ) >> ( ( serdes % 8 ) + 24 ) ) & 0x1
         if val == 1:
            receivedDetected = "is"
         else:
            receivedDetected = "is not"
         good = False
         # if it's not good and a supe link make sure it's not the active one
         if self.portById_[port].desc.startswith( "xx1 supe" ):
            import Fru # pylint: disable=import-outside-toplevel
            good = Fru.slotId() != int( self.portById_[port].desc[-1] )
         # If the port is expected to be not be connected (expectedSpeed is 0 )
         # or we are doing the easy version of this test with bannocks it is
         # good if we have a receiver detected
         elif self.portById_[port].expectedSpeed == [0] or \
                  val == 1 and easyMode: 
            good = True
         return ( f"Link down, receiver {receivedDetected} detected, "
                  f"({self.portById_[port].desc})", good )

      if speed == 1:
         speedStr = "2.5GT/s"
      elif speed == 2:
         speedStr = "5.0GT/s"
      else:
         speedStr = f"unknown ({speed})"
      # determine if we got the expected width and speed
      good = self.portById_[port].width == status and \
             ( self.portById_[port].expectedSpeed == [0] or \
             speed in self.portById_[port].expectedSpeed )
      # if it's not good and a supe link make sure it's not the active one
      return ( f"Link up: Speed {speedStr}: Num Lanes {status}, "
               f"({self.portById_[port].desc})", good )

   def identify( self ):
      try:
         _, did = self.getVendorIdDeviceID( 0 )
      except: # pylint: disable=bare-except
         print( "FAIL: Device not present" )
         return 1
      if self.idNum_ == did:
         # pylint: disable-next=consider-using-f-string
         print( "PASS: PLX-PEX%x confirmed" % self.idNum_ )
         return 0
      else:
         # pylint: disable-next=consider-using-f-string
         print( "FAIL: Detected a PLX-PEX%x instead of expected PLX-PEX%x" %
                ( did, self.idNum_ ) )
         return 1
   
   def scan( self, verbose = True, msg = "", easyMode = False ):
      _, did = self.getVendorIdDeviceID( 0 )
      # pylint: disable-next=consider-using-f-string
      rc = "%s/scd/%d/%d/%d PLX-PEX%x \n" % ( msg, self.device_.accelId, 
                                              self.device_.busId,
                                              self.device_.deviceId, did )
      allGood = True
      if verbose:
         for id in self.portById_:
            a, b, c = self.getPciBusNumbers( id )
            linkStatus, good = self.getLinkStatus( id, easyMode )
            rc += f"   Port {id}\n      {linkStatus}\n"
            rc += f"      Bus Numbers, Primary {a}, Secondary {b}, Subordinate {c}\n"
            allGood = allGood and good
      return rc, 0 if allGood else 1

   def getSerdes( self, port ):
      # convert from port to serdes (link)
      if port == 0:
         return 0
      if port == 1:
         return 8
      if port == 2:
         return 4
      if port == 3:
         return 12
      if port == 4:
         return 1
      if port == 5:
         return 9
      if port == 6:
         return 2
      if port == 7:
         return 10
      if port == 8:
         return 3
      if port == 9:
         return 11
      if port == 10:
         return 5
      if port == 11:
         return 13
      if port == 12:
         return 6
      if port == 13:
         return 14
      if port == 14:
         return 7
      if port == 15:
         return 15
      assert 0


   # register info
   # registerAddr: ( { bitOffset: { width, name, type, typeInfo, upstream),
   #                                 ...
   #                 }, ValidMask (which bit in the register are valid),
   #                   usePort, (whether to use the port number or 0 )
   #                   portOffset (for registers with multiple ports, the number
   #                               of bits the ports are separated by in this
   #                               register)
   regMap = {
   # 0x068 pcie capability (all ports)
   0x068: ( { 24: (1, "slot implemented", "bool", None, False) 
            }, 0x3fffffff, True ),
   # 0x078 Link status & control (all ports)
   0x078: ( { 4:  (1, "link disable", "bool", None, False), 
              16: (4, "link speed", "mapped", {1:"2.5GT/s", 2:"5.0GT/s"}, True),
              20: (4, "link width", "mapped", {0:"down", 1:"x1", 2:"x2", 4:"x4",
                                               8:"x8"}, True),
              27: (1, "link in training", "bool", None, False),
              29: (1, "Data link active", "bool", None, False)
            }, 0xfbff0cfd, True ),
   # 0x07c Slot capability (all downstream ports)
   0x07c: ( { 5: (1, "hotplug surprise", "bool", None, False),
              6: (1, "hotplug capable", "bool", None, False),
              7: (7, "slot power limit", "int", None, False),
              19:(13,"slot number", "int", None, False)
            }, 0xfffbffff, True ),
   # 0x080 Slot status & control (all downstream ports)
   0x080: ( { 3: (1, "presence detect irq enable", "bool", None, False),
              5: (1, "hotplug detect irq enable", "bool", None, False),
              10:(1, "power controller control", "bool", None, False),
              17:(1, "power fault detected", "bool", None, False),
              19:(1, "presence detect changed", "bool", None, False),
              22:(1, "presence detected", "bool", None, False),
            }, 0xffff1fff, True ),
   # 0x098 Link status & control 2 (all ports)
   0x098: ( { 0: (3, "target link speed", "mapped", {1:"2.5GT/s", 2:"5.0GT/s"},True),
              16:(1, "current de-emphasis", "bool", None, False)
            }, 0x00011fff, True ),
   # 0x1dc Debug control (Port 0)(selects the upstream port)
   0x1dc: ( { 8: (4, "Upstream port ID", "int", None, True),
              18:(1, "NT mod enabled", "bool", None, True),
              24:(4, "NT port", "int", None, True),
            }, 0xff37bf00, False ),
   # 0x200/4 serdes  
   0x200: ( { 0: (1, "electrical idle causes complience state", "bool", None, True),
              8: (1, "mask electrical idle detect", "bool", None, True),
              16:(1, "mask receiver not detected", "bool", None, True),
              24:(1, "receiver detected", "bool", None, True)
            }, 0x01010101, False, 1 ),
   # 0x230(even ports)/0x234(odd ports) Port disable/quiet (masked on port 0)
   0x230: ( { 0: (1, "Port disabled", "bool", None, True),
              8: (1, "Port held quiet", "bool", None, True),
            }, 0x00010101, False ),
   # 0xb98-0xba4 serdes drive levels (masked on port 0)
   0xb98: ( { 0: (5, "drive level", "hex", None, True),
              7: (1, "auto load disable", "bool", None, True),
            }, 0x0000009f, False, 8 ),
   # 0xba8-0xbb4 de-empahsis levels (masked on port 0)
   0xba8: ( { 0: (5, "post cursor emphasis level", "hex", None, True),
              7: (1, "auto load disable", "bool", None, True),
            }, 0x0000009f, False, 8 ),
   # 0xbb8-0xbbc rx equalization (masked on port 0)
   0xbb8: ( { 0: (4, "rx equalisation", "hex", None, True),
            }, 0x0000000f, False, 4 ),
   }
   
   def showBits( self, value, mask ):
      res = ""
      for i in range(31, -1, -1):
         if (1 << i) & mask:
            res += str(int((value >> i) & 1))
         else:
            res += '.'
         if not i % 8:
            res += ' '
      return res
   
   def showRegister( self, reg, value, upstream, offset = 0 ):
      regKeys = list( reg[ 0 ] )
      regKeys.sort()
      # offset allows for registers that repeat in a single U32
      res = self.showBits( value, (reg[1] << offset) ) + " ---- Valid bits ----\n"
      for field in regKeys:
         regWidth, txt, regType, typeInfo, _ = reg[0][field]
         val = (value >> (field + offset)) & ((1 << regWidth) - 1)
         res += self.showBits( value, ((1 << regWidth) - 1) << (field + offset) )
         if regType == "bool":
            res += txt+":"+str(val != 0)+"\n"
         elif regType == "mapped":
            if val in typeInfo:
               valTxt = typeInfo[val]
            else:
               valTxt = f"INVALID({val})"
            res += txt+":"+valTxt+"\n"
         elif regType == "int":
            res += txt+":"+str(val)+"\n"
         else:
            res += txt+":"+hex(val)+"\n"
      return res+"\n"

   def showMultiPortReg( self, regList, port ):
      portPerReg = 16 // len( regList )
      regVal = self.readReg( 0 , regList[ port // portPerReg ] )
      # pylint: disable-next=consider-using-f-string
      res = "============ 0x%03x : 0x%08x ============\n" % (
         regList[ port // portPerReg ],
         regVal )
      res += self.showRegister(
         PcieSwitch.regMap[ regList[ 0 ] ],
         regVal, False,
         ( port % portPerReg ) * PcieSwitch.regMap[ regList[ 0 ] ][ 3 ] )
      return res

   def showSerdesSettings( self, port ):
      serDes = self.getSerdes(port)
      res = f"********** Port width:{self.portById_[ port ].width} **********\n"
      for i in range( serDes, serDes + self.portById_[ port ].width ):
         res += f"************ Serdes:{i} ************\n"
         res += self.showMultiPortReg( [ 0x200, 0x204 ], i)
         res += self.showMultiPortReg( [ 0xb98, 0xb9c, 0xba0, 0xba4 ], i)
         res += self.showMultiPortReg( [ 0xba8, 0xbac, 0xbb0, 0xbb4 ], i)
         res += self.showMultiPortReg( [ 0xbb8, 0xbbc ], i)
      return res

   def showPcieRegister( self, port, reg ):
      if port not in self.portById_:
         return f"Invalid port {port}\n"
      if PcieSwitch.regMap[reg][2]:
         readPort = port
      else:
         readPort = 0
      regVal = self.readReg( readPort , reg )
      # pylint: disable-next=consider-using-f-string
      return "0x%08x\n" % (regVal)

   def showPcieRegistersInternal( self, port ):
      ''' Dumps out a load of registers useful for debugging ports '''
      upstream = (self.portById_[ port ].direction == "upstream")
      res = "\n     *************************\n"
      res += "     *************************\n"
      res += f"************* Port {port} **************\n"
      res += f"************* Port {port} **************\n"
      res += f"************* Port {port} **************\n"
      res += "     *************************\n"
      res += "     *************************\n"
      # interesting registers
      interestingRegisters = [ 0x068, 0x078, 0x07c, 0x080, 0x098 ]
      if upstream:
         interestingRegisters.extend( [0x1dc] )
      for intReg in interestingRegisters:
         if PcieSwitch.regMap[intReg][2]:
            readPort = port
         else:
            readPort = 0
         regVal = self.readReg( readPort , intReg )
         # pylint: disable-next=consider-using-f-string
         res += "============ 0x%03x : 0x%08x ============\n" % (intReg , regVal)
         res += self.showRegister( PcieSwitch.regMap[intReg], regVal, upstream )
      regList = [ 0x230, 0x234 ] # even/odd split
      reg = regList[ port % 2 ] # bits 0-7 are interesting
      regVal = self.readReg( 0 , reg )
      # pylint: disable-next=consider-using-f-string
      res += "============ 0x%03x : 0x%08x ============\n" % ( reg, regVal )
      res += self.showRegister(PcieSwitch.regMap[0x230], regVal, upstream, port // 2)
      res += self.showSerdesSettings( port )
      return res

   def showPcieRegisters( self, port ):
      if port == "upstream":
         res = ""
         for i, portObj in self.portById_.items():
            if portObj.direction == "upstream":
               res += self.showPcieRegistersInternal( i )
      elif port in self.portById_:
         res = self.showPcieRegistersInternal( port )
      else:
         res = f"Invalid port {port}\n"
      return res

   def showPcieErrCountersInternal( self, port ):
      res = f"\nPort {port}:\n"
      val = self.readReg( port, 0xfc4 )
      res += "   CESta: "
      if val & 1:
         res += "RxErr "
      if val & 0x100:
         res += "Rollover "
      res += "\n"
      res += f"   Bad TLP Count: {self.readReg( port, 0x1e8 )}\n"
      res += f"   Bad DLLP Count: {self.readReg( port , 0x1ec )}\n"
      return res

   def pcieRead( self, port, regaddr ):
      if port not in self.portById_:
         return f"Invalid port {port}\n"
      # pylint: disable-next=consider-using-f-string
      return "0x%x" % self.readReg( port, regaddr )

   def pcieWrite( self, port, regaddr, regval ):
      if port not in self.portById_:
         return f"Invalid port {port}\n"
      self.writeReg( port, regaddr, regval )
      return None

   def showPcieErrCounters( self, port ):
      if port not in self.portById_:
         return f"Invalid port {port}\n"
      return self.showPcieErrCountersInternal( port )

   def clearPcieErrCountersInternal( self, port ):
      # clear AER CESta bits
      self.writeReg( port , 0xfc4, 0xffffffff)
      # clear badTLP counter
      self.writeReg( port , 0x1e8, 0 )
      # clear badDLLP counter
      self.writeReg( port , 0x1ec, 0 )

   def clearPcieErrCounters( self, port ):
      if port not in self.portById_:
         return f"Invalid port {port}\n"
      self.clearPcieErrCountersInternal( port )
      return None

   def getPortById( self, port):
      if port not in self.portById_:
         return None
      return self.portById_[port]
   
   def gpio(self):
      if len(self.pinByName_) == 0 or len(self.pinById_) == 0:
         print("Gpio pins not used for this chip")
         return None
      gpio = PLXPcieSwitchGpio(self)
      return gpio

   def workaroundErratum12( self ):
      origVal = self.readReg( 0, self.erratum12Reg )
      self.writeReg( 0, self.erratum12Reg, origVal | 1 << 20 )

   def setElectricalMask(self):
      self.writeReg(0, self.electricalIdleMaskEvenReg, 0xffff)
      self.writeReg(0, self.electricalIdleMaskOddReg, 0xffff)

   def getGpioPinReg(self, gpioId, regType):
      if regType == pcieGpioDirectionReg:
         assert gpioId >= 0
         if gpioId < 16:
            return 0x62c
         else:
            return 0x630
      elif regType == pcieGpioOutputReg:
         if 0 <= gpioId < 16:
            return 0x644
         else:
            return 0x648
      else:
         if 0 <= gpioId < 16:
            return 0x63c
         else:
            return 0x640
         
   def getGpioPinBit(self, gpioId, regType):
      if regType == pcieGpioDirectionReg:
         return ( gpioId % 16 ) * 2 + 1
      elif regType in ( pcieGpioOutputReg, pcieGpioInputReg ):
         return gpioId % 16
      return None
   def mountEntity( self, sysdb, path, typ, attr ):
      mg = sysdb.mountGroup()
      entity = mg.mount(path, typ, attr)
      mg.close(blocking=True)
      return entity
