# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import defaultdict

import BothTrace
from Toggles import RcfLibToggleLib
from Assert import assertEqual, assertGreater, assertIn, assertTrue

import RcfAst
import RcfMetadata
import RcfTypeFuture as Rcf
import RoutingTypeFuture as Routing
from RcfAstVisitor import Visitor
from RcfDebugLocationLib import getLocation, updateAetNodeKeysInDebugSymbols
from RcfImmediateValueHelper import AetCollectionGen, AetValueGen
from RcfInstantiatingCollHelpers import InstantiatingCollectionGV

bv = BothTrace.Var
bt0 = BothTrace.tracef0
bt8 = BothTrace.tracef8

RcfTypeSystem = RcfMetadata.RcfTypeSystem
BT = RcfMetadata.RcfBuiltinTypes

class AetGenPhase( Visitor ):
   """ Generation of the Abstract Evaluation Tree (AET) from the AST.

   During this phase:
      - we assume that the AST is fully valid.
      - we build the AET node representing the AST we traverse.
      - we build the metadata information required for the evaluation of the AET.
      - we dedup compiler AET against published AET.

   Attributes:
      aetDedupHelper (RcfAetDedupHelper):
         Helper object responsible for deduping generated AETs against already
         published AETs from the last compilation or other AETs generated in this
         compilation.
      metadataHelper (RcfAetMetadataHelper):
         Helper object responsible for managing the indexes associated with external
         references.
      rcfCodeVersion (int):
         Version of the RCF text being compiled.
      instantiatingCollectionCleaner (InstantiatingCollectionCleaner):
         Cleaner responsible for reming instantiated AET nodes from the instantiating
         collections.
      currentFunction (RcfAst.Function):
         Ast object for the function whose AET is currently being generated.
      currentFunctionAetKeys (set):
         Collection of AET keys generated for the current function.
      debugSymbols (dict of Rcf.Debug.FunctionSymbol keyed by function name):
         Storage object for the Debug Symbols being generated for the current
         generation of AETs.
      aetNodeSymbolContainerHead (Rcf.Debug.SymbolContainer):
         As Debug Symbols are generated for an AET they are added to a linked list.
         This is the head of that linked list.
      binOpExpressionTypeMap(dict of AET types):
         Mapping to look up the AET type for a given operator.
   """

   def __init__( self, publishedAfs, rcfCodeVersion,
                 instantiatingCollectionCleaner ):
      super().__init__()
      self.aetDedupHelper = RcfAetDedupHelper( publishedAfs,
                                               instantiatingCollectionCleaner )
      self.metadataHelper = RcfAetMetadataHelper( publishedAfs )
      self.rcfCodeVersion = rcfCodeVersion
      self.instantiatingCollectionCleaner = instantiatingCollectionCleaner
      self.currentFunction = None
      self.currentFunctionAetKeys = None
      self.debugSymbols = {}
      self.aetNodeSymbolContainerHead = None
      self.binOpExpressionTypeMap = {
         "and": Rcf.Eval.AndOperator,
         "or": Rcf.Eval.OrOperator,
         "??": Rcf.Eval.SequentialOperator,
      }

   def getOpAndArgs( self, op, attribute, value ):
      """ Helper function to get op AET type and ctor args for assignment and
      relational statements where the op and rhs AET types may or may not be
      overriden in the metadata

      Possible cases:
      - op and rhs AET types are all overriden in the metadata
         - e.g. "community = empty"
      - op type is overriden in the metadata
         - e.g. "community add community_list LIST" vs "community add { 1, 2, 3 }"
      - no AET types are overriden in the metadata
         - this means the lhs type's `operatorToAetType` dict in the metadata should
           be used to get the op AET type
         - e.g. "med = 100"
      """
      if op == 'is_not':
         # 'is_not' is special. The AST phase has enclosed this in a not so
         # a regular 'is' operator is used here.
         op = 'is'
      ctorArgs = [ self.visit( attribute )( attribute ) ]
      opAetType, rhsAetType, rhsAetCtorArgs = RcfTypeSystem.aetTypes.get(
            ( attribute.symbol.rcfType, op, value.evalType ),
            ( None, None, None ) )
      # All overriden statements have the op AET type overridden,
      # so we can use the presence of `opAetType` to determine
      # whether or not this statement is overriden in any way.

      if opAetType and rhsAetType:
         # Both the AET op node type and RHS AET node type are overridden in
         # the metadata, so use that here.
         ctorArgs.append( rhsAetType( *rhsAetCtorArgs ) )
      else:
         # Either the AET op node type is overridden or there is no overriding.
         # This block handles both of those cases.
         if isinstance( value, RcfAst.ExternalRef ):
            ctorArgs.append( value.name )
            index = self.extRefIndex( value )
            ctorArgs.append( index )
         else:
            valNode = self.visit( value )( value )
            if valNode is not None:
               # It'll be None in cases like 'empty'/'none'
               ctorArgs.append( valNode )

            if isinstance( value, RcfAst.Collection ) and value.ranges:
               rangeValueLists = AetCollectionGen(
                  aetGenVisitor=self ).buildRangeLists( value )
               ctorArgs += rangeValueLists
               opAetType = Rcf.Eval.IntInImmediateSetWithRanges

      # If opAetType is not None (indicating that there is an override for this
      # operation in the medatada), use that.  Otherwise use the general AET op
      # node for the base rcfType/op combination.
      aetOp = opAetType or attribute.symbol.rcfType.aetOpTypes[ op ]
      return aetOp, ctorArgs

   def getBuiltinFunctionBody( self, fnName ):
      """ Generates the body for the specified builtin function """
      name = RcfMetadata.metadataProcessor.allBuiltinFunctions[ fnName ][ 'aetType' ]
      aetTypeFunc = RcfMetadata.getAetType( name )
      # The body of the function simply contains an instance of the aetType
      # specified in RcfMetadata.yaml for this built-in function.
      builtinStmt = Rcf.Eval.ExpressionList( aetTypeFunc(), None )
      return Rcf.Eval.Block( builtinStmt )

   def visitRoot( self, root, **kwargs ):
      '''
      Visit all functions bottom up.
      The requirement of this collection is that every callee is specified before
      the respective caller
      '''
      for function in root.validFunctions():
         self.visit( function )( function )

      RcfAetMetadataHelper( self.aetDedupHelper.resultAfs ).verifySanity()
      return self.aetDedupHelper.resultAfs, self.debugSymbols

   def addAetNodeSymbolToList( self, symbol ):
      # The container is a linked list, add a new node storing the current
      # head as next.
      self.aetNodeSymbolContainerHead = Rcf.Debug.SymbolContainer(
         symbol, self.aetNodeSymbolContainerHead )

   def addAetNodeSymbol( self, astNode, key, fragmentType, location,
                         nodeSpecificData=None ):
      nodeSpecificDataList = [ nodeSpecificData ]
      if isinstance( astNode, RcfAst.Expr ):
         nodeSpecificDataList.append( self.handleParens( astNode ) )
      if isinstance( astNode, RcfAst.StmtExpr ):
         nodeSpecificDataList.append(
            self.handleStatmentExpressionDelimiters( astNode ) )

      nodeSpecificDataHead = None
      for data in nodeSpecificDataList:
         if data is None:
            continue
         nodeSpecificDataHead = Rcf.Debug.SymbolContainer( data,
                                                           nodeSpecificDataHead )
      symbol = Rcf.Debug.AetNodeSymbol( key, fragmentType, location,
                                        nodeSpecificDataHead )
      self.addAetNodeSymbolToList( symbol )

   def getLocation( self, node, excludeChildNodes=None ):
      return getLocation( node, excludeChildNodes=excludeChildNodes )

   def handleParens( self, node ):
      # If one is present then both must be.
      assert bool( node.openingParens ) == bool( node.closingParens )

      if not node.openingParens:
         return None
      explicitParens = node.explicitParens
      openingParensLocation = self.getLocation( node.openingParens )
      closingParensLocation = self.getLocation( node.closingParens )
      return Rcf.Debug.ParensData( explicitParens, openingParensLocation,
                                   closingParensLocation )

   def handleStatmentExpressionDelimiters( self, node ):
      # If one is present then both must be.
      assert bool( node.openStmtPoint ) == bool( node.closeExprPoint )

      if not node.openStmtPoint:
         return None
      openingExprPoint = self.getLocation( node.openStmtPoint )
      closingExprPoint = self.getLocation( node.closeExprPoint )
      return Rcf.Debug.StatementExpressionData( openingExprPoint, closingExprPoint )

   def visitFunction( self, function, **kwargs ):
      """ Build an AET for the given function.

      Args: function (RcfAst.Function): the AST function we want to transform in
                                        AET.

      Note:
         This function can be called when we're walking the functions definitions
         of the AST's Root, or when we want to get the AET address of a function
         from an AST's function call perspective.

         If we never saw this function before during this compilation, we build it.
         Otherwise we use the dedup'd AET stored inside aetDedupHelper.
      """
      fnName = function.name

      self.currentFunction = function
      self.aetNodeSymbolContainerHead = None
      function.metadata = Rcf.Eval.MetadataType()
      function.features = Routing.Policy.RouteMapFeatures()

      if fnName in RcfMetadata.metadataProcessor.allBuiltinFunctions:
         assertEqual( function.block.stmts, [] )
         assertEqual( function.context, None )
         body = self.getBuiltinFunctionBody( fnName )
         # We may have route map features installed for this builtin.
         function.symbol.updateRouteMapFeatures( function.features )
         functionDefinition = None
      else:
         # Create the function definition
         functionStartLine = function.context.start.line
         functionEndLine = function.context.stop.line
         functionStartCol = function.context.start.column
         functionEndCol = function.context.stop.column

         functionDefinition = Rcf.Debug.FunctionDefinition(
            function.codeUnitKey.codeUnitName, functionStartLine, functionStartCol,
            functionEndLine, functionEndCol )
         body = self.visit( function.block )( function.block )

      if not RcfLibToggleLib.toggleRcfAccessAttrsEnabled():
         function.accessibleAttributes = Rcf.Eval.AccessibleAttributes()
      argumentsExpected = bool( function.funcParams )
      compiledAet = Rcf.Eval.Function( fnName,
                                       function.metadata,
                                       function.features,
                                       function.accessibleAttributes,
                                       argumentsExpected,
                                       body )
      function.metadata = None
      function.features = None

      if fnName in RcfMetadata.metadataProcessor.allBuiltinFunctions:
         aetNodeSymbols = None
      else:
         # Add the root function definition node.
         location = self.getLocation(
            function.context, excludeChildNodes=[ function.block.context ] )
         self.addAetNodeSymbol( function, compiledAet.key, "definition",
                                location )
         aetNodeSymbols = self.aetNodeSymbolContainerHead

      compiledAetKeys = InstantiatingCollectionGV.logAetKeysUsed( compiledAet )
      dedupedAet = self.aetDedupHelper.dedupAet( fnName, compiledAet )

      if aetNodeSymbols and dedupedAet != compiledAet:
         # We're not using the AET just created.
         # The debugSymbols created have the wrong AET keys.
         # We need to recreate them with the corresponding AET keys of the
         # deduped AET.
         dedupedAetKeys = InstantiatingCollectionGV.getLoggedAetKeysForAet(
            dedupedAet )
         aetNodeSymbols = updateAetNodeKeysInDebugSymbols(
            aetNodeSymbols, compiledAetKeys, dedupedAetKeys,
            self.instantiatingCollectionCleaner )
      if function.codeUnitKey.domain == Rcf.Metadata.FunctionDomain.BUILTIN:
         # Builtin functions are not defined anywhere, but all other function types
         # have a function definition.
         assert not ( aetNodeSymbols or functionDefinition )
      else:
         assert aetNodeSymbols and functionDefinition
      debugSymbols = Rcf.Debug.FunctionSymbol( fnName,
                                               functionDefinition,
                                               aetNodeSymbols,
                                               function.codeUnitKey.domain,
                                               self.rcfCodeVersion,
                                               dedupedAet.key )
      InstantiatingCollectionGV.logDebugSymbolKeysUsed( debugSymbols )
      self.debugSymbols[ fnName ] = debugSymbols
      return dedupedAet

   def constructLinkedList( self, nodeList, listNodeType, visitNodes=False ):
      """
      Helper method for constructing linked lists.
      Arguments:
         - nodeList: list of AST/AET nodes to be contained in the linked list
         - listNodeType: linked list node type to contain the entries
         - visitNodes: bool indicating if the nodes in nodeList need to have
                       self.visit called on them (they are AST nodes) or not (they
                       are already AET nodes)
      """
      aetNodes = []
      if visitNodes:
         # Visit the nodes forwards. This way structures are allocated in the order
         # you would expect (for instance, external reference indices).
         for astNode in nodeList:
            aetNodes.append( self.visit( astNode )( astNode ) )
            if isinstance( astNode, RcfAst.Collection ):
               # If we see a collection within a linked list then we are handling
               # function arguments. This means we need to handle the possibilty of
               # ranges within the collection.
               # We split out the ranges into two lists (lower and upper bound
               # values) and add them after the collection.
               if astNode.ranges:
                  aetNodes += AetCollectionGen(
                     aetGenVisitor=self ).buildRangeLists( astNode )
               else:
                  # If no ranges are present then add two empty linked lists
                  # so that a consistent number of arguments are present in the
                  # arguments list at runtime. This avoids trying to figure out what
                  # arguments are present at runtime.
                  cachable = True
                  aetNodes += [ Rcf.Eval.IntListImmediate( None, cachable ),
                                Rcf.Eval.IntListImmediate( None, cachable ) ]
      else:
         aetNodes = nodeList
      head = None
      # Construct the list backwards because `next` is a ctor argument
      for entry in reversed( aetNodes ):
         head = listNodeType( entry, head )
      return head

   def visitBlock( self, block, **kwargs ):
      stmts = self.constructLinkedList( block.stmts, Rcf.Eval.ExpressionList,
                                        visitNodes=True )
      blockNode = Rcf.Eval.Block( stmts )
      openingBraceLocation = self.getLocation( block.context.openingBrace() )
      closingBraceLocation = self.getLocation( block.context.closingBrace() )
      blockLabelLocation = None
      if block.label:
         blockLabelLocation = self.getLocation( block.label.token )
      blockData = Rcf.Debug.BlockData( openingBraceLocation,
                                       closingBraceLocation,
                                       blockLabelLocation )
      # There is no 'block' fragment but we need to specify something.
      # Just use 'openBlock'
      self.addAetNodeSymbol( block, blockNode.key, 'openBlock', None,
                             nodeSpecificData=blockData )
      return blockNode

   def visitIfStmt( self, ifStmt, **kwargs ):
      condition = self.visitExpr( ifStmt.condition )( ifStmt.condition )
      thenBlock = self.visit( ifStmt.thenBlock )( ifStmt.thenBlock )
      elseBlock = None
      elsePartLocation = None
      if ifStmt.elseBlock:
         elseBlock = self.visit( ifStmt.elseBlock )( ifStmt.elseBlock )
         elsePartLocation = self.getLocation( ifStmt.context.elsePart )
      ifLblLocation = None
      if ifStmt.ifLbl:
         ifLblLocation = self.getLocation( ifStmt.ifLbl.token )
      elseLblLocation = None
      if ifStmt.elseLbl:
         elseLblLocation = self.getLocation( ifStmt.elseLbl.token )
      aetIfStmt = Rcf.Eval.IfExpression( condition, thenBlock, elseBlock )
      ifPartLocation = self.getLocation( ifStmt.context.ifPart )
      nodeSpecificData = Rcf.Debug.IfData( ifLblLocation, ifPartLocation,
                                           elseLblLocation, elsePartLocation )
      self.addAetNodeSymbol( ifStmt, aetIfStmt.key, 'ifStatement', None,
                             nodeSpecificData=nodeSpecificData )
      return aetIfStmt

   def constructArgList( self, call ):
      if not call.funcArgs:
         return None

      funcArgs = []
      if call.functionSelf:
         # Methods are only expected for builtin functions
         assert ( call.symbol.node.codeUnitKey.domain ==
                  Rcf.Metadata.FunctionDomain.BUILTIN ), (
                     "Unexpected method in non-builtin function" )
         funcArgs += [ call.functionSelf ]
      funcArgs += call.funcArgs
      return self.constructLinkedList( funcArgs, Rcf.Eval.FunctionArgumentList,
                                       visitNodes=True )

   def visitCall( self, call, **kwargs ):
      # Since we're about to descend into the callee we need to 'pop' state
      # associated with the current function, visit the callee, and then 'push'
      # back the current functions / callers state.
      # For example, the instantiating collections keys are logged for each function,
      # so the keys used so far for the current function must be preserved.
      calleeName = call.funcName
      calleePtr = self.aetDedupHelper.getAet( calleeName )
      self.currentFunction.metadata.updateWith( calleePtr.metadata )
      self.currentFunction.features.updateFrom( calleePtr.features )
      if RcfLibToggleLib.toggleRcfAccessAttrsEnabled():
         self.currentFunction.accessibleAttributes.updateWith(
            calleePtr.accessedAttributes )
      argList = self.constructArgList( call )
      aetFuncCall = Rcf.Eval.FunctionCallExpression( calleeName, calleePtr, argList )
      functionCallData = Rcf.Debug.FunctionCallData( calleeName )
      location = self.getLocation( call.context )
      self.addAetNodeSymbol( call, aetFuncCall.key, 'functionCall', location,
                             nodeSpecificData=functionCallData )
      return aetFuncCall

   def visitExternalRefOp( self, externalRefOp, **kwargs ):
      aetAttr = self.visit( externalRefOp.attribute )( externalRefOp.attribute )
      attrSym = externalRefOp.attribute.symbol
      opAetType = attrSym.rcfType.aetOpTypes[ externalRefOp.op ]
      # BUG833158 - Update all external reference types to use RHS nodes
      rhsResolveType = externalRefOp.rhs.resolveType()
      if rhsResolveType == BT.PrefixList:
         assert not externalRefOp.additionalAttributes, (
            "Unexpected additional attributes for extRef type: %s" % rhsResolveType )
         rhsAetNode = self.visit( externalRefOp.rhs )( externalRefOp.rhs )
         opAetNode = opAetType( aetAttr, rhsAetNode )
      else:
         assert isinstance( externalRefOp.rhs, RcfAst.ExternalRef ), (
            "Variable RHS is not supported for %s" % rhsResolveType )
         ctorArgs = []
         ctorArgs.append( aetAttr )
         ctorArgs.append( externalRefOp.rhs.name )
         index = self.extRefIndex( externalRefOp.rhs )
         ctorArgs.append( index )
         ctorArgs += [ self.visit( attr )( attr )
                       for attr in externalRefOp.additionalAttributes ]
         opAetNode = opAetType( *ctorArgs )
      location = self.getLocation( externalRefOp.context )
      self.addAetNodeSymbol( externalRefOp, opAetNode.key, 'condition', location )
      return opAetNode

   def visitExternalRef( self, extRef, **kwargs ):
      return self.visitValue( extRef )

   def visitConstant( self, constant, **kwargs ):
      return self.visitValue( constant )

   def visitCommunityValue( self, commVal, **kwargs ):
      return self.visitValue( commVal )

   def visitCollection( self, collection, **kwargs ):
      return AetCollectionGen( aetGenVisitor=self ).build( collection )

   def visitAssign( self, assign, **kwargs ):
      op = assign.op
      attribute = assign.attribute
      value = assign.value
      # For some assign statements, the metadata may provide specific op and/or
      # rhs AET types as a way of overriding the AET construction. Additionally, the
      # metadata may provide specific ctor args to be used when instantiating the rhs
      # AET node.
      aetOp, ctorArgs = self.getOpAndArgs( op, attribute, value )
      aetNode = aetOp( *ctorArgs )
      location = self.getLocation( assign.context )
      self.addAetNodeSymbol( assign, aetNode.key, "assignment", location )
      return aetNode

   def visitReturn( self, returnOperation, **kwargs ):
      aetExpr = self.visitExpr( returnOperation.expr )( returnOperation.expr )
      aetReturnExpr = Rcf.Eval.ReturnExpression( aetExpr )
      location = self.getLocation(
         returnOperation.context,
         excludeChildNodes=[ returnOperation.context.expr() ] )
      self.addAetNodeSymbol( returnOperation, aetReturnExpr.key, 'statement',
                             location )
      return aetReturnExpr

   def visitExit( self, exitOperation, **kwargs ):
      aetExpr = self.visitExpr( exitOperation.expr )( exitOperation.expr )
      aetExitExpr = Rcf.Eval.ExitExpression( aetExpr )
      location = self.getLocation(
         exitOperation.context,
         excludeChildNodes=[ exitOperation.context.expr() ] )
      self.addAetNodeSymbol( exitOperation, aetExitExpr.key, 'statement', location )
      return aetExitExpr

   def visitAttribute( self, attribute, **kwargs ):
      attrAetType = attribute.symbol.aetType
      attribute.symbol.updateRouteMapFeatures( self.currentFunction.features )
      attrAetNode = attrAetType()
      return attrAetNode

   def visitLogicalOp( self, logicalOp, **kwargs ):
      opType = self.binOpExpressionTypeMap[ logicalOp.operator ]
      astNode = logicalOp

      # There is always one more operand than operator.
      expr = logicalOp.expressionList[ 0 ]
      exprList = [ self.visitExpr( expr )( expr ) ]
      opList = []

      for expr, op in zip( logicalOp.expressionList[ 1 : ], logicalOp.operatorList ):
         # visitExpr will handle the expressions debugSymbol
         exprList.append( self.visitExpr( expr )( expr ) )

         # The operator is handled second so that it is constructed after both
         # operands.
         opAetNode = opType()
         opList.append( opAetNode )
         location = self.getLocation( op )
         self.addAetNodeSymbol( astNode, opAetNode.key, 'logicalOperator', location )
         # Only the first operator is associated with the AST node for adding
         # parenthesis, etc, so unset it after the first iteration.
         astNode = None

      expressionList = self.constructLinkedList( exprList, Rcf.Eval.ExpressionList )
      operatorList = self.constructLinkedList( opList, Rcf.Eval.LogicalOperatorList )

      aetNode = Rcf.Eval.LogicalOperatorExpression( expressionList, operatorList )
      # The LogicalOperatorExpression node is evaluated and logged, however
      # it is an internal construct and not tied to any of the RCF text.
      # The individual operators have debugSymols tied to the RCF text instead.
      self.addAetNodeSymbolToList( Rcf.Debug.InternalAetNodeSymbol( aetNode.key ) )
      return aetNode

   def visitBinOp( self, binOp, **kwargs ):
      aetNode = None
      fragmentType = None

      # For some attribute operation statements, the metadata may provide
      # specific lhs, op, and rhs AET types as a way of overriding the AET
      # construction. Additionally, the metadata may provide specific ctor args to
      # be used when instantiating the rhs AET node.
      aetOp, ctorArgs = self.getOpAndArgs( binOp.operator, binOp.lhs, binOp.rhs )
      aetNode = aetOp( *ctorArgs )
      fragmentType = 'condition'

      if binOp.operator == 'is_not':
         # 'is_not' is special. It's treated like an 'is' operator enclosed in
         # a 'not'.
         # The 'is_not' text is tied to a debugSybmol for the enclosing 'not' as
         # that operator has the final result. A special internal symbol is used for
         # the 'is' part (this node) so that a symbol is found marking the node as
         # internal.
         self.addAetNodeSymbolToList(
            Rcf.Debug.InternalAetNodeSymbol( aetNode.key ) )
      else:
         location = self.getLocation( binOp.context )
         self.addAetNodeSymbol( binOp, aetNode.key, fragmentType, location )
      return aetNode

   def visitNot( self, notExpr, **kwargs ):
      expr = self.visitExpr( notExpr.expr )( notExpr.expr )
      aetNode = Rcf.Eval.NotExpression( expr )
      if hasattr( notExpr, 'representing_is_not' ):
         # is_not is a special case and we consider the whole expression to
         # be part of the artificial 'not'.
         location = self.getLocation( notExpr.context )
      else:
         location = self.getLocation( notExpr.context.children[ 0 ] )
      self.addAetNodeSymbol( notExpr, aetNode.key, 'logicalOperator', location )
      return aetNode

   def visitVariable( self, variable, **kwargs ):
      aetTypeFunc = RcfMetadata.getAetType( variable.evalType.variableAetTypename )
      assert aetTypeFunc is not None
      aetNode = aetTypeFunc( variable.name, variable.symbol.argVectorIndex )
      return aetNode

   #-- custom
   def visitExpr( self, expr ):
      """
      The function pointer is returned to reduce the call stack depth required
      when traversing the tree. The caller is responsible for calling the returned
      method.
      This halves our call stack depth by removing this visitExpr call from the
      stack. This matters when handling expressions with a very large number of
      operators.
      """
      if isinstance( expr, RcfAst.Constant ):
         aetExpr = self.visitTriStateBoolExpression
      else:
         aetExpr = self.visit( expr )
      return aetExpr

   def visitTriStateBoolExpression( self, expr ):
      validTypes = [ RcfAst.Constant.Type.boolean, RcfAst.Constant.Type.trilean ]
      assert expr.type in validTypes, ( f"Unexpected type {expr.type} when building "
                                        "the tri state bool expression AET" )
      aetTriStateValue = AetValueGen.aetTriStateNameToValueMap[ expr.value ]
      aetNode = Rcf.Eval.TriStateBoolExpression(
         Rcf.Eval.TriStateBoolImmediate( aetTriStateValue ) )
      location = self.getLocation( expr.context )
      self.addAetNodeSymbol( expr, aetNode.key, 'condition', location )
      return aetNode

   def visitValue( self, value ):
      valueVisitor = AetValueGen( aetGenVisitor=self )
      return valueVisitor.build( value )

   def extRefIndex( self, extRef ):
      extRefTypeToUsedListMap = {
         "prefix_list_v4": self.currentFunction.metadata.v4PrefixListsUsed,
         "prefix_list_v6": self.currentFunction.metadata.v6PrefixListsUsed,
         "as_path_list": self.currentFunction.metadata.asPathAccessListsUsed,
         "community_list": self.currentFunction.metadata.commListsUsed,
         "ext_community_list": self.currentFunction.metadata.extCommListsUsed,
         "large_community_list": self.currentFunction.metadata.largeCommListsUsed,
         "roa_table": self.currentFunction.metadata.roaTablesUsed,
         "dynamic_prefix_list": self.currentFunction.metadata.dynPrefixListsUsed,
      }
      extRefTypeToUsedListNameMap = {
         "prefix_list_v4": "v4PrefixListsUsed",
         "prefix_list_v6": "v6PrefixListsUsed",
         "as_path_list": "asPathAccessListsUsed",
         "community_list": "commListsUsed",
         "ext_community_list": "extCommListsUsed",
         "large_community_list": "largeCommListsUsed",
         "roa_table": "roaTablesUsed",
         "dynamic_prefix_list": "dynPrefixListsUsed",
      }

      metadataExtRefUsedList = extRefTypeToUsedListMap[ extRef.etype ]
      index = self.metadataHelper.getIndex(
         extRef.name,
         extRefTypeToUsedListNameMap[ extRef.etype ] )
      metadataExtRefUsedList[ str( extRef.name ) ] = index
      return index

class RcfAetMetadataTypeHelper:
   """This class helps manage index re-use and allocation for the index values that
   need to be allocated for specific external constructs (prefix lists etc).

   Attributes:
      afs: AllFunctionStatus that has the AETs.
      metadataType: Name of metadata attribute inside Rcf::Eval::FunctionMetadata.
      afsInUseIndicesDict: A dict of ExternalEntityName -> index for all functions
                           in self.afs.
      freeIndexGenerator: The generator for generating new index values.
   """
   def __init__( self, afs, metadataType ):
      self.afs = afs
      self.metadataType = metadataType
      self.afsInUseIndicesDict = self.getAfsInUseIndicesDict( self.afs )
      self.freeIndexGenerator = self.freeIndexGeneratorFunction()

   def getAfsInUseIndicesDict( self, afs ):
      """Returns dict of mappings from external entity name to index"""
      ret = {}
      for aet in afs.aet.values():
         metadataDict = getattr( aet.metadata, self.metadataType )
         for name, index in metadataDict.items():
            assertGreater( index, 0 )
            if ret.get( name ):
               # The name/index mapping should be the same across all AETs
               assertEqual( ret[ name ], index )
            else:
               ret[ name ] = index
      return ret

   def freeIndexGeneratorFunction( self ):
      # Start the range from 1, so 0 can be used an invalid value. Zero is the
      # default value for an int and the code can assert if it sees a zero index.
      index = 1
      while True:
         inUseIndices = set( self.afsInUseIndicesDict.values() )
         while index in inUseIndices:
            index += 1
         yield index
         index += 1

   def getIndex( self, name ):
      """Get an index for an external construct.  Re-use the index collected
      from afs, if it exists there."""
      if self.afsInUseIndicesDict.get( name ) is not None:
         # Reuse the existing index
         index = self.afsInUseIndicesDict[ name ]
         bt8( "Rcf:RAMTH:getIndex: re-used index" )
      else:
         index = next( self.freeIndexGenerator )
         self.afsInUseIndicesDict[ name ] = index
      bt8( "Rcf:RAMTH:getIndex|",
           bv( str( self.metadataType ) ),
           bv( name ),
           bv( index ) )

      assertGreater( index, 0 )
      return index

   def verifySanity( self ):
      """Verify that the same index value is used in every reference to an external
      construct, across all RCF functions.
      """
      # The below operation has the necessary asserts to verify consistency
      self.getAfsInUseIndicesDict( self.afs )

class RcfAetMetadataHelper:
   """This wrapper class helps manage index re-use and allocation for all the
   external constructs (prefix lists etc).

   Attributes:
      afs: AllFunctionStatus that has the AETs.
      metadataTypes: The names of external construct types that are supported.
      helpers: Dict of metadataType -> RcfAetMetadataTypeHelper instance
   """
   def __init__( self, afs ):
      self.afs = afs
      self.metadataTypes = [ 'v4PrefixListsUsed',
                             'v6PrefixListsUsed',
                             'asPathAccessListsUsed',
                             'commListsUsed',
                             'extCommListsUsed',
                             'largeCommListsUsed',
                             'roaTablesUsed',
                             'dynPrefixListsUsed',
                             'tunnelRibsUsed',
      ]
      self.helpers = {}
      for metadataType in self.metadataTypes:
         self.helpers[ metadataType ] = \
               RcfAetMetadataTypeHelper( self.afs, metadataType )

   def getIndex( self, name, metadataType ):
      """Get an index for an external construct"""
      assertIn( metadataType, self.metadataTypes )
      return self.helpers[ metadataType ].getIndex( name )

   def verifySanity( self ):
      """Verifies consistency in all the helpers"""
      for helper in self.helpers.values():
         helper.verifySanity()
      for aet in self.afs.aet.values():
         assertTrue( aet.verifySanity( aet.metadata ) )

class RcfAetDedupHelper:
   """This manages AET dedup, when the compiler generates the AETs.
   There should be one instance of RcfAetDedupHelper for every batch compilation.

   Attributes:
      publishedAfs:  AllFunctionStatus that has the already published AETs.
      instantiatingCollectionCleaner: Cleaner responsible for reming instantiated AET
                                      nodes from the instantiating collections.
      resultAfs:     The AllFunctionStatus result that is produced by this class.
      aetDedupTable: Used to store known AETs that new AETs can be deduped against.
                     It is a dictionary, keyed by an AET hash, of sets, where the set
                     is all of the known AETs with that hash. This provides optimal
                     lookup time when deduping. This structure is constructed afresh
                     on each compiler invocation so that reference counts for AETs in
                     the table do not need to be maintained.
   """

   def __init__( self, publishedAfs, instantiatingCollectionCleaner ):
      self.publishedAfs = publishedAfs
      self.instantiatingCollectionCleaner = instantiatingCollectionCleaner
      self.resultAfs = Rcf.AllFunctionStatus( "resultAfs" )
      # Use a set per hash so that published and deduped AETs can be
      # added without needing to check if they're already in the table.
      self.aetDedupTable = defaultdict( set )
      # It is important to pre populate the dedupTable with published AETs to
      # ensure that we will reuse published AETs if possible
      bt0( "Rcf:RADH:init: populating dedupTable with published functions" )
      for aet in self.publishedAfs.aet.values():
         self.aetDedupTable[ aet.hash( 0 ) ].add( aet )

   def functionExistsInResult( self, funcName ):
      """Returns whether the function already exists in the deduped result."""
      return funcName in self.resultAfs.aet

   def getAet( self, funcName ):
      """Returns the deduped AET, if it exists in the result.  This should be
      called only after checking if functionExistsInResult().
      """
      assert self.functionExistsInResult( funcName )
      return self.resultAfs.aet[ funcName ]

   def getDedupByContent( self, newAet ):
      for aet in self.aetDedupTable[ newAet.hash( 0 ) ]:
         if aet and aet.compare( newAet ):
            bt0( "Rcf:RADH:dedupAet: re-use", bv( aet.funcName ),
                 "for", bv( newAet.funcName ) )
            return aet
      # If the new AET was not found in the dedup table then insert it and return
      self.aetDedupTable[ newAet.hash( 0 ) ].add( newAet )
      return None

   def dedupAet( self, funcName, newAet ):
      """Does AET Dedup.
      If AET comparison between newAet and the AET in self.publishedAfs (if it is
      present there) returns true, it reuses the AET present in self.publishedAfs
      and discards newAet.
      Stores the deduped AET in self.resultAfs and returns the deduped AET.
      """
      # funcName should not already exist in result, when this function is called.
      assert not self.functionExistsInResult( funcName )
      assert newAet

      dedupedAet = None
      # Look through all AETs, regardless of the function name, from the
      # previous AFS and the current AFS for one to reuse.
      dedupedAet = self.getDedupByContent( newAet )

      if dedupedAet:
         self.instantiatingCollectionCleaner.cleanup( newAet )
      else:
         bt0( "Rcf:RADH:dedupAet: new|", bv( funcName ) )
         dedupedAet = newAet

      assert dedupedAet
      self.resultAfs.aet[ funcName ] = dedupedAet
      return dedupedAet

def genAbstractEvalTree( rcfAst, publishedAfs, rcfCodeVersion,
                        instantiatingCollectionCleaner ):
   generateAet = AetGenPhase( publishedAfs, rcfCodeVersion,
                              instantiatingCollectionCleaner )
   return generateAet( rcfAst )
