#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

""" Translate the Antlr Parse tree into the Rcf AST.
"""

import RcfAst
from ArnetLib import asnStrToNum
from RcfParser import RcfParser
from RcfVisitor import RcfVisitor
from RcfImmediateValueHelper import RcfImmediateValueHelper
from RcfMetadata import metadataProcessor
from RcfDebugLocationLib import getLeftmostPointToken, getRightmostPointToken
from RcfAgentHeartbeat import RcfAgentHeartbeat
from itertools import chain
from Toggles import RcfLibToggleLib

contextStrLen = len( "Context" )

class AstGenVisitor( RcfVisitor ):
   """ This visitor converts the Antlr parse tree to the Rcf AST.

   !Rules (don't change unless discussing with authors first)

      - Define the visit method following same the order in which
        the parser rules are defined in the grammar file.

   Args:
      codeUnitKey: code unit key of the current parse tree

   Attributes:
      self.currentFunction (RcfFunction): the current function.

   @author: matthieu (rcf-dev)
   """
   def __init__( self, codeUnitKey ):
      self.codeUnitKey = codeUnitKey
      self.currentFunction = None

   def __call__( self, parseTree ):
      """ Visitor instances are functors.

      Primes the visitor to walk over the parse tree.

      Args:
         parseTree: the parse tree from Antlr.

      Returns:
         The AST for this parse tree.
      """
      return self.visit( parseTree )( parseTree )

   def visit( self, tree ):
      """
      Determines the method a ParseRuleContext object should be handled with.
      The function pointer is returned to reduce the call stack depth required
      when traversing the tree. The caller is responsible for calling the returned
      method.
      This halves our call stack depth by removing this visit call from the stack.
      As we can recurse deeply when dealing with very large expressions with many
      operators it is important to minimize our call stack depth.

      Args:
         tree: Named tree to match the overridden method, ctx is more appropriate.
               This is the ParseRuleContext for a given node.

      Returns:
         A pointer to the method on this visitor the caller should invoke with the
         given tree.
      """
      typeName = tree.__class__.__name__
      assert typeName.endswith( "Context" )
      methodName = "visit" + typeName[ : -contextStrLen ]
      assert hasattr( self, methodName ), ( "The AstGenVisitor must implement a "
                                               "method for " + typeName )
      return getattr( self, methodName )

   def visitRcf( self, ctx ):
      """ Visit a parse tree produced by RcfParser#rcf
      """
      return [ self.visitFuncDecl( funcDecl ) for funcDecl in ctx.funcDecl() ]

   def visitFuncDecl( self, ctx ):
      """ Visit a parse tree produced by RcfParser#funcDecl.
      """
      RcfAgentHeartbeat.punch()

      # Function name
      if self.codeUnitKey.isPoaWrapper():
         # For POA wrapper functions, we swap in the unique name after passing
         # through the parser phase of the RCF compiler. This is to avoid having to
         # special case characters that come from the POA description which are not
         # supported in the grammar.
         name = self.codeUnitKey.getPoaWrapperFunctionName()
      else:
         # 'function <name>' will be two tokens, extract <name>
         assert len( ctx.funcDef().children ) == 2
         name = ctx.funcDef().children[ 1 ].getText()


      self.currentFunction = name
      block = self.visitBlock( ctx.block() )
      if ctx.funcParams():
         funcParams = self.visitFuncParams( ctx.funcParams() )
      else:
         funcParams = []
      function = RcfAst.Function( ctx, name=name, codeUnitKey=self.codeUnitKey,
                                  block=block, funcParams=funcParams )
      return function

   def visitFuncParams( self, ctx ):
      """ Visit a parse tree produced by RcfParser#funcParams
      """
      funcParams = []
      for funcParam in ctx.funcParam():
         funcParams.append( self.visitFuncParam( funcParam ) )
      return funcParams

   def visitFuncParam( self, ctx ):
      """ Visit a parse tree produced by RcfParser#funcParam
      """
      # 'FUNC_PARAM_TYPE VARIABLE' will be two tokens
      assert len( ctx.children ) == 2
      fptype = ctx.children[ 0 ].getText()
      fpname = ctx.children[ 1 ].getText()
      astNode = RcfAst.FunctionParam( ctx, name=fpname, typeStr=fptype )
      return astNode

   def visitBlockImpl( self, blockCtx ):
      """ Get all stmts from a block
      """
      stmts = []
      for stmt in blockCtx.stmt():
         stmts.append( self.visit( stmt )( stmt ) )
      return stmts

   def visitBlock( self, ctx ):
      """ Visit a parse tree produced by RcfParser#block.
      """
      stmts = self.visitBlockImpl( ctx )
      astBlock = RcfAst.Block( ctx, stmts, label=None )
      return astBlock

   def visitLabeledBlock( self, ctx ):
      """ Visit a parse tree produced by RcfParser#labeledBlock.
      """
      stmts = self.visitBlockImpl( ctx.block() )
      label = RcfAst.Label( ctx.LABEL().getPayload() )
      astBlock = RcfAst.Block( ctx.block(), stmts, label=label )
      return astBlock

   def visitStmt( self, ctx ):
      """ Visit a parse tree produced by RcfParser#stmt.
      """
      if ctx.ifStmt():
         return self.visitIfStmt( ctx.ifStmt() )
      elif ctx.assignOperation():
         return self.visitAssignOperation( ctx.assignOperation() )
      elif ctx.returnOperation():
         return self.visitReturnOperation( ctx.returnOperation() )
      elif ctx.exitOperation():
         return self.visitExitOperation( ctx.exitOperation() )
      elif ctx.directive():
         return self.visitDirective( ctx.directive() )
      elif ctx.expr():
         return self.visit( ctx.expr() )( ctx.expr() )
      elif ctx.labeledBlock():
         return self.visitLabeledBlock( ctx.labeledBlock() )
      else:
         raise NotImplementedError( 'Unknown statement node' )

   def visitAssignOperation( self, ctx ):
      """ Visit a parse tree produced by RcfParser#assignOperation.
      """
      attribute = self.visitAttribute( ctx.attribute() )
      op = ctx.op.getText()
      value = self.visit( ctx.value() )( ctx.value() )

      # For certain assignment operations to as_path we want to convert
      # RHS singletons to collections for uniformity later in the complier.
      transformSingletonToCollectionForOperators = [ 'prepend' ]
      if RcfLibToggleLib.toggleRcfAsPathImmediateAssignEnabled():
         transformSingletonToCollectionForOperators.append( '=' )
      if ( attribute.name == 'as_path' and
           op in transformSingletonToCollectionForOperators and
           not isinstance( value, RcfAst.Collection ) ):
         value = RcfAst.Collection( value.context, [ value ],
                                    RcfAst.Collection.Type.AsPathImmediate )

      astNode = RcfAst.Assign( ctx, attribute, value, op )
      return astNode

   def visitString( self, ctx ):
      # a string could be an enum, interface, or attribute.
      cvalue = ctx.getText()
      enumType = RcfImmediateValueHelper.getEnumType( cvalue )
      interfaceType = RcfImmediateValueHelper.getInterfaceType( cvalue )
      if enumType is not None:
         return RcfAst.Constant( ctx, cvalue, enumType )
      elif interfaceType is not None:
         return RcfAst.Constant( ctx, cvalue, interfaceType )
      else:
         return self.visitAttribute( ctx )

   def visitValue( self, ctx ):
      if ctx.string():
         return self.visit( ctx.string() )( ctx.string() )
      elif ctx.immediate():
         return self.visit( ctx.immediate() )( ctx.immediate() )
      elif ctx.extRef():
         return self.visit( ctx.extRef() )( ctx.extRef() )
      elif ctx.variable():
         return self.visit( ctx.variable() )( ctx.variable() )
      else:
         assert False, "Unknown Value Type: %s" % type( ctx )
      return None

   def asPathFromAsPathContext( self, asPathContext ):
      """ Get an ordered list of as numbers given the asPathContext object built by
      the parser.

      Args:
         context (RcfParser.AsPathContext): the parsed, asPath object from Antlr.

      Returns:
         list (ordered) of RcfAst.Constant objects

      """
      asPathListResult = []
      for asn in asPathContext.asn():
         node = None
         if asn.integer():
            asNum = int( asn.integer().getText() )
            node = RcfAst.Constant( asn, asNum,
                                    RcfAst.Constant.Type.integer )
         elif asn.asDot():
            asDot = asnStrToNum( asn.asDot().getText() )
            node = RcfAst.Constant( asn, asDot,
                                    RcfAst.Constant.Type.asDot )
         elif asn.attribute():
            astAttr = self.visitAttribute( asn.attribute() )
            node = astAttr
         elif asn.variable():
            astAttr = self.visitVariable( asn.variable() )
            node = astAttr

         assert node, "Unknown ASN parsed type: " + str( type( asn ) )
         asPathListResult.append( node )

      return asPathListResult

   def visitRangeValue( self, ctx ):
      """ Visit a parse tree produced by RcfParser#immediateRange
      """
      if ctx.integer():
         return self.visit( ctx.integer() )( ctx.integer() )
      elif ctx.asDot():
         return self.visit( ctx.asDot() )( ctx.asDot() )
      else:
         assert False, "Unknown Range Value Type: %s" % type( ctx )
      return None

   def visitImmediateRange( self, ctx ):
      """ Visit a parse tree produced by RcfParser#SetRange
      """
      lowerBoundNode = self.visit( ctx.lowerBound )( ctx.lowerBound )
      upperBoundNode = self.visit( ctx.upperBound )( ctx.upperBound )
      return RcfAst.Range( ctx, lowerBoundNode, upperBoundNode )

   def visitCommColon( self, ctx ):
      """ Visit a parse tree produced by RcfParser#CommColon.
      """
      high = int( ctx.integer()[ 0 ].getText() )
      low = int( ctx.integer()[ 1 ].getText() )
      return RcfAst.Constant( ctx, ( high << 16 ) + ( low ),
                              RcfAst.Constant.Type.immediateCommunityValue )

   def visitLargeComm( self, ctx ):
      """ Visit a parse tree produced by RcfParser#LargeComm.
      """
      high = int( ctx.integer()[ 0 ].getText() )
      med = int( ctx.integer()[ 1 ].getText() )
      low = int( ctx.integer()[ 2 ].getText() )
      return RcfAst.Constant( ctx, ( high, med, low ),
                              RcfAst.Constant.Type.immediateLargeCommunityValue )

   def visitBandwidth( self, ctx ):
      return RcfAst.Constant( ctx, ctx.getText(), RcfAst.Constant.Type.bandwidth )

   def visitIpV4Address( self, ctx ):
      return RcfAst.Constant( ctx, ctx.getText(), RcfAst.Constant.Type.ipAddress )

   def visitExtCommSection( self, ctx ):
      if ctx.integer():
         return self.visit( ctx.integer() )( ctx.integer() )
      elif ctx.ipV4Address():
         return self.visit( ctx.ipV4Address() )( ctx.ipV4Address() )
      elif ctx.asDot():
         return self.visit( ctx.asDot() )( ctx.asDot() )
      elif ctx.bandwidth():
         return self.visit( ctx.bandwidth() )( ctx.bandwidth() )
      elif ctx.attribute():
         return self.visitString( ctx.attribute() )
      else:
         assert False, "Unknown Value Type: %s" % type( ctx )

   @staticmethod
   def getExtCommType( ctx ):
      cvalue = ctx.text
      assert cvalue[ -1 ] == ":"
      cvalue = cvalue[ : -1 ]

      enumType = RcfImmediateValueHelper.getEnumType( cvalue )
      # pylint: disable=no-member
      assert enumType == RcfAst.Constant.Type.ImmediateExtCommunity
      return RcfAst.Constant( ctx, cvalue, enumType )

   @staticmethod
   def getExtCommSection2None( section1, ctx ):
      if section1.value == "COLOR":
         # pylint: disable=no-member
         return RcfAst.Constant( ctx, "CO_MATCH_EXACT",
                                 RcfAst.Constant.Type.ImmediateExtCommunityColor )
      elif section1.value == "LINK-BANDWIDTH-AS":
         return RcfAst.Constant( ctx, 0, RcfAst.Constant.Type.integer )
      else:
         return RcfAst.Constant( ctx, None, RcfAst.Constant.Type.none )

   def visitExtComm( self, ctx ):
      """ Visit a parse tree produced by RcfParser#ExtComm.
      """
      section1 = self.getExtCommType( ctx.section1 )

      if ctx.section2:
         section2 = self.visit( ctx.section2 )( ctx.section2 )
      else:
         section2 = self.getExtCommSection2None( section1, ctx )

      if ctx.section3:
         section3 = self.visit( ctx.section3 )( ctx.section3 )
      else:
         section3 = RcfAst.Constant( ctx, 0, RcfAst.Constant.Type.none )

      return RcfAst.CommunityValue( ctx, ( section1, section2, section3 ),
               RcfAst.CommunityValue.Type.immediateExtCommunityValue )

   def visitResolutionRib( self, ctx ):
      return self.visitExtRef( ctx )

   def visitExtCommEOF( self, ctx ):
      """ Visit a parse tree produced by RcfParser#ExtCommEOF.

      Used by RcfAstTestLib.py to turn a string representation of an extended
      community to the corresponding AST node. This "EOF" parse context is a wrapper
      around the ExtComm parse rule with a mandatory 'EOF' token at the end. As long
      as the parse rule has been matched by antlr, we only care about the first token
      from here.
      """
      assert len( ctx.children ) == 2
      assert ctx.children[ 1 ].getText() == '<EOF>'
      return self.visitExtComm( ctx.children[ 0 ] )

   def immediateSetFromImmediateSetContext( self, immediateSetContext ):
      """ Get an ordered, python friendly list of immediate values given the
      immediateSetContext object built by the parser
      along with a type associated with it

      Args:
         context (RcfParser.ImmediateSetContext): the parsed,
            immediateSet object from Antlr.

      Returns:
         tuple of:
            - type of community
               - type is determined by the type of the first element in the list
            - list (ordered) of integer and string constants:
               - <U32> communities are translated to integer constants
               - <U16>:<U16> communities are translated to integer constants
               - <U32>:<U32>:<U32> large communities are translated to tuple of three
                                  integers with constant type immediateCommunityValue
               - <string> communities are translated to string constants
               - <string> enums are translated to string constants

         "100 200:200 INTERNET" would map to:
         e.g [ Constant( type=integer, value=100 ),
               Constant( type=integer, value=13107400 ),
               Constant( type=string, value='INTERNET' ) ]

         "1:1:1 200:400:600" would map to:
         [ Constant( type=immediateCommunityValue, value=( 1, 1, 1 ),
           Constant( type=immediateCommunityValue, value=( 200, 400, 600 ) ]

         "LEVEL1 LEVEL2" would map to:
         e.g [ Constant( type=string, value='LEVEL1' ),
               Constant( type=string, value='LEVEL2' ) ]
      """
      immediateSetResult = []
      if immediateSetContext.immediateSetValue():
         for valueContext in immediateSetContext.immediateSetValue():
            immediateSetResult.append( self.visit( valueContext )( valueContext ) )
      return immediateSetResult

   def immediateListFromImmediateListContext( self, immediateListContext ):
      immediateListResult = []
      if immediateListContext.immediateListValue():
         for valueContext in immediateListContext.immediateListValue():
            immediateListResult.append( self.visit( valueContext )( valueContext ) )
      return immediateListResult

   def visitInteger( self, ctx ):
      return RcfAst.Constant( ctx, int( ctx.getText() ),
                              RcfAst.Constant.Type.integer )

   def visitAsDot( self, ctx ):
      return RcfAst.Constant( ctx, asnStrToNum( ctx.getText(), minValue=0 ),
                              RcfAst.Constant.Type.asDot )

   def visitImmediate( self, ctx ):
      if ctx.integer():
         return self.visit( ctx.integer() )( ctx.integer() )
      elif ctx.bandwidth():
         return self.visit( ctx.bandwidth() )( ctx.bandwidth() )
      elif ctx.ipPrefix():
         ctype = RcfAst.Constant.Type.prefix
         cvalue = ctx.ipPrefix().getText()
      elif ctx.ipAddress():
         ctype = RcfAst.Constant.Type.ipAddress
         cvalue = ctx.ipAddress().getText()
      elif ctx.macAddress():
         ctype = RcfAst.Constant.Type.macAddress
         cvalue = ctx.macAddress().getText()
      elif ctx.esi():
         ctype = RcfAst.Constant.Type.esi
         cvalue = ctx.esi().getText()
      elif ctx.asPath():
         return RcfAst.Collection(
            ctx, self.asPathFromAsPathContext( ctx.asPath() ),
            RcfAst.Collection.Type.AsPathImmediate )
      elif ctx.asDot():
         return self.visit( ctx.asDot() )( ctx.asDot() )
      elif ctx.trilean():
         return self.visitTrilean( ctx.trilean() )
      elif ctx.extComm():
         return self.visit( ctx.extComm() )( ctx.extComm() )
      elif ctx.immediateSet():
         return RcfAst.Collection(
            ctx, self.immediateSetFromImmediateSetContext( ctx.immediateSet() ),
            RcfAst.Collection.Type.ImmediateSet )
      elif ctx.immediateList():
         return RcfAst.Collection(
            ctx, self.immediateListFromImmediateListContext( ctx.immediateList() ),
            RcfAst.Collection.Type.ImmediateList )
      elif ctx.EMPTY():
         return RcfAst.Collection( ctx, [], RcfAst.Collection.Type.empty )
      elif ctx.NONE():
         ctype = RcfAst.Constant.Type.none
         cvalue = None
      else:
         assert False, "Unknown constant type: %s" % type( ctx )
      constant = RcfAst.Constant( ctx, cvalue, ctype )
      return constant

   def visitReturnOperation( self, ctx ):
      """ Visit a parse tree produced by RcfParser#returnOperation.
      """
      returnExpr = ctx.expr()
      expr = self.visit( returnExpr )( returnExpr )
      openReturnPoint = getLeftmostPointToken( ctx )
      closeExprPoint = getRightmostPointToken( returnExpr )
      ast = RcfAst.Return( ctx, expr, openReturnPoint, closeExprPoint )
      return ast

   def visitExitOperation( self, ctx ):
      """ Visit a parse tree produced by RcfParser#exitOperation.
      """
      exitExpr = ctx.expr()
      expr = self.visit( exitExpr )( exitExpr )
      openExitPoint = getLeftmostPointToken( ctx )
      closeExprPoint = getRightmostPointToken( exitExpr )
      ast = RcfAst.Exit( ctx, expr, openExitPoint, closeExprPoint )
      return ast

   def visitDirective( self, ctx ):
      """ Visit a parse tree produced by RcfParser#directive
      """
      directiveValue = ctx.DIRECTIVE().getText()
      ast = RcfAst.Directive( ctx, directiveValue )
      return ast

   def visitIfStmt( self, ctx ):
      """ Visit a parse tree produced by RcfParser#ifStmt.
      """
      conditionExpr = ctx.expr()
      condition = self.visit( conditionExpr )( conditionExpr )
      thenBlock = self.visitBlock( ctx.block()[ 0 ] )
      if ctx.ifStmt(): # if then else if ...
         elseBlock = self.visit( ctx.ifStmt() )( ctx.ifStmt() )
         assert len( ctx.block() ) == 1
      elif len( ctx.block() ) == 2: # if then else
         elseBlock = self.visitBlock( ctx.block()[ 1 ] )
      else: # if then
         elseBlock = None
      openStmtPoint = getLeftmostPointToken( ctx )
      closeExprPoint = getRightmostPointToken( conditionExpr )
      ast = RcfAst.IfStmt( ctx, condition, thenBlock, elseBlock,
                           openStmtPoint, closeExprPoint )
      return ast

   def visitRelational( self, ctx ):
      """ Visit a parse tree produced by RcfParser#relational.
      """
      operator = ctx.BIN_REL().getText()
      attribute = self.visitAttribute( ctx.attribute() )
      constant = self.visit( ctx.value() )( ctx.value() )
      astNode = RcfAst.BinOp( ctx, operator, attribute, constant )
      if operator == "is_not":
         # 'is_not' is treated as 'is' enclosed in a 'not'.
         astNode = RcfAst.Not( ctx, astNode )
         astNode.representing_is_not = True
      return astNode

   def visitAttribute( self, ctx ):
      """ Visit a parse tree produced by RcfParser#attribute.
      """
      name = '.'.join( part.getText() for part in ctx.ID() )
      ast = RcfAst.Attribute( ctx, name )
      return ast

   def visitExtRef( self, ctx ):
      """ Visit a parse tree produced by RcfParser#extRef
      """
      # 'EXT_REF_TYPE <name>' will be two tokens, extract both
      assert len( ctx.children ) == 2
      etype = ctx.children[ 0 ].getText()
      name = ctx.children[ 1 ].getText()
      astNode = RcfAst.ExternalRef( ctx, name, etype )
      return astNode

   def handleImplicitPrecedence( self, ctx, astNode ):
      # Add parens to indicate implicit precedence only when the parent
      # ctx is also for an operator and the parent ctx operator is different.
      # Implicit precedence is indicated when there is a change in operators.
      operatorTypes = ( RcfParser.AndContext, RcfParser.NotContext,
                        RcfParser.OrContext, RcfParser.SeqContext )
      assert isinstance( ctx, operatorTypes )
      if ( isinstance( ctx.parentCtx, operatorTypes ) and
           not isinstance( ctx, type( ctx.parentCtx ) ) ):
         astNode.openingParens.append( getLeftmostPointToken( ctx ) )
         astNode.closingParens.append( getRightmostPointToken( ctx ) )
         astNode.explicitParens = False

   def visitNot( self, ctx ):
      """ Visit a parse tree produced by RcfParser#Not
      """
      expr = self.visit( ctx.expr() )( ctx.expr() )
      astNode = RcfAst.Not( ctx, expr )
      self.handleImplicitPrecedence( ctx, astNode )
      return astNode

   def visitParens( self, ctx ):
      """ Visit a parse tree produced by RcfParser#Parens
      """
      astNode = self.visit( ctx.expr() )( ctx.expr() )
      astNode.openingParens.append( ctx.openingParens() )
      astNode.closingParens.append( ctx.closingParens() )
      astNode.explicitParens = True
      return astNode

   def visitExternalRefOp( self, ctx ):
      """ Visit a parse tree produced by RcfParser#ExternalRefOp
      """
      attribute = self.visitAttribute( ctx.attribute() )
      isExact = ctx.EXT_REF_OPERATOR().getText() == 'match_exact'
      isMatchCovered = ctx.EXT_REF_OPERATOR().getText() == 'match_covered'
      additionalAttributes = []
      if ctx.extRef():
         rhs = self.visitExtRef( ctx.extRef() )
         if rhs.etype == 'as_path_list':
            # 'as_path match as_path_list NAME' depends on the current origin
            # value during the evaluation. Add the additional origin attribute
            # so an accessor to the origin is included in the AET node
            # The accessor should refer to the input.origin if input.as_path
            # is being matched against.
            originAttr = 'origin'
            if 'input.' in attribute.name:
               originAttr = 'input.' + originAttr
            additionalAttributes += [ RcfAst.Attribute( None, originAttr ) ]
      elif ctx.variable():
         # BUG840103 - Handle adding an additional origin attribute for RHS variables
         # which may or may not refer to a symbol which requires it. For phase 1 of
         # function arguments, as_path_list is not a support function parameter type
         # so we can defer on this handling for now.
         rhs = self.visitVariable( ctx.variable() )
      else:
         assert False, "Unknown External Reference Type: %s" % type( ctx )
      astNode = RcfAst.ExternalRefOp( ctx, attribute, isExact, isMatchCovered, rhs )
      astNode.additionalAttributes += additionalAttributes
      return astNode

   def visitLogicalOperator( self, ctx, operator,
                             getLhs=lambda ctx: ctx.lhs,
                             getRhs=lambda ctx: ctx.rhs,
                             ctxForImplicitPrecedence=lambda ctx: ctx ):
      '''
      A sequence of operators of the same type form a tree like the following
           op
      expr         op
              expr    expr
      where op is in [ and, or, ?? ]
      A sequence of operators of the same type is flattened into a list of
      operands and operators in the AST.
      Note the tree is right associative, so operators are always to the right.

      Args:
         ctx (RcfParser.*Context): context of the top most operator in the parse
                                   tree.
         operator (String): string representation of the operator. ([ and, or, ?? ])
         getLhs (function): function taking an operator ctx as an argument and
                            returning the lhs expression context.
         getRhs (function): function taking an operator ctx as an argument and
                            returning the rhs context, which is either another
                            operator or an expression.
         ctxForImplicitPrecedence (function): function taking an operator ctx as an
                                              argument and returning it's parent
                                              context for the purpose or adding
                                              implicit precedence.
      '''
      operatorType = type( ctx )
      expressionList = []
      operatorList = []

      rhsCtx = ctx
      while isinstance( rhsCtx, operatorType ):
         lhs = getLhs( rhsCtx )
         expressionList.append( self.visit( lhs )( lhs ) )
         operatorList.append( rhsCtx.op )

         # The next operator will be on the RHS
         rhsCtx = getRhs( rhsCtx )

      # A rhsCtx has been found on the RHS that is not another operator.
      expressionList.append( self.visit( rhsCtx )( rhsCtx ) )
      astNode = RcfAst.LogicalOp( ctx, operator, expressionList, operatorList )
      self.handleImplicitPrecedence( ctxForImplicitPrecedence( ctx ), astNode )
      return astNode

   def visitAnd( self, ctx ):
      """ Visit a parse tree produced by RcfParser#And.
      """
      return self.visitLogicalOperator( ctx, 'and' )

   def visitOr( self, ctx ):
      """ Visit a parse tree produced by RcfParser#Or.
      """
      return self.visitLogicalOperator( ctx, 'or' )

   def visitRel( self, ctx ):
      """ Visit a parse tree produced by RcfParser#Rel
      """
      astNode = self.visitRelational( ctx.relational() )
      return astNode

   def visitTri( self, ctx ):
      """ Visit a parse tree produced by RcfParser#Tri
      """
      astNode = self.visitTrilean( ctx.trilean() )
      return astNode

   def visitSeq( self, ctx ):
      """ Visit a parse tree produced by RcfParser#Seq
      """
      astNode = self.visitSequentialExpr( ctx.sequentialExpr() )
      return astNode

   def visitTrilean( self, ctx ):
      """ Visit a parse tree produced by RcfParser#Trilean
      """
      cvalue = ctx.getText()
      if cvalue == "unknown":
         ctype = RcfAst.Constant.Type.trilean
      else:
         ctype = RcfAst.Constant.Type.boolean
      astNode = RcfAst.Constant( ctx, cvalue, ctype )
      return astNode

   def visitVariable( self, ctx ):
      """ Visit a parse tree produced by RcfParser#variable
      """
      name = ctx.VARIABLE().getText()
      ast = RcfAst.Variable( ctx, name )
      return ast


   def visitFuncCallExpr( self, ctx ):
      """ Visit a parse tree produced by RcfParser#funcCallExpr
      """
      calleeName = ctx.funcName().getText()
      funcArgs = ctx.funcArgs()
      argNodes = []
      if funcArgs:
         argNodes = [ self.visit( val )( val ) for val in funcArgs.value() ]
      astNode = RcfAst.Call( ctx, calleeName, argNodes )
      return astNode

   def visitFuncCall( self, ctx ):
      """ Visit a parse tree produced by RcfParser#funcCall
      """
      return self.visitFuncCallExpr( ctx.funcCallExpr() )

   def visitSequentialExpr( self, ctx ):
      """ Visit a parse tree produced by RcfParser#sequentialExpr
      """

      def getLhs( ctx ):
         return ctx.funcCallExpr()[ 0 ]

      def getRhs( ctx ):
         if len( ctx.funcCallExpr() ) > 1:
            return ctx.funcCallExpr()[ 1 ]
         elif ctx.sequentialExpr():
            return ctx.sequentialExpr()
         elif ctx.trilean():
            return ctx.trilean()
         else:
            assert False, "Unknown Sequential Expr RHS Type: %s" % type( ctx )
         return None

      def ctxForImplicitPrecedence( ctx ):
         # This is a SequentialExpr ctx. It's parent is a SeqExpr ctx because of
         # how the grammar is structured. Pass the SeqExpr ctx so it's parent
         # can be examined to see if implicit precedence is required as the SeqExpr
         # ctx parent may be another operator, whereas the SequentialExpr ctxs parent
         # will always be a SeqExpr.
         return ctx.parentCtx

      return self.visitLogicalOperator(
         ctx, '??', getLhs=getLhs, getRhs=getRhs,
         ctxForImplicitPrecedence=ctxForImplicitPrecedence )

def genAst( parseTree, codeUnitKey ):
   """ Build an AST given a valid parse tree.

   Args:
      parseTree, the parse tree we got from the parsing phase.
      codeUnitKey: code unit key of the current parse tree

   Return:
      AstNode (Ast.Root): The root of the AST.
   """
   walkParseTreeAndBuildAst = AstGenVisitor( codeUnitKey )
   return walkParseTreeAndBuildAst( parseTree )

def getBuiltinFunctions( codeUnitKey, builtInFunctionMock ):
   builtInFunctionMock = builtInFunctionMock or {}
   builtInFunctions = []
   for name, attrs in chain( metadataProcessor.allBuiltinFunctions.items(),
                             builtInFunctionMock.items() ):
      block = RcfAst.Block( None, [], label=None )
      # Function parameters for built-in functions do not have a name
      funcParams = [ RcfAst.FunctionParam( None, None, funcParamType )
                     for funcParamType in attrs.get( 'funcParamTypes', [] ) ]
      func = RcfAst.Function( None, name=name, codeUnitKey=codeUnitKey, block=block,
                              funcParams=funcParams )
      builtInFunctions.append( func )
   return builtInFunctions
