#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''
Module to render a section of RCF code with annotations to underline a particular
part.

A use of this is to render the code section where a compilation error or warning was
found to highlight the erroneous code.

For example, say the RCF text had a missing operator (after the last false on the
     first line and before the true)
```
function foo() {
   return false or false or false
           # some comment

          true;
}
```

and the following compilation was produced
```
error occured: Error line 5:8: extraneous input 'true' expecting ';'
```

this module will facilitate rendering the relevant code section
```
2    return false or false or false
3            # some comment
4
5         true;
          ^~~~
```
Note that it supports rendering lines above and below the actual error
to provide additional context.

'''

from collections import namedtuple
import functools

from CliPlugin.RcfDebugRenderLib import (
   Annotations,
   InvocationContext,
   InvocationCodeManager,
)
from CliPlugin.RcfDebugRender import (
   InvocationPrinter,
   InvocationRenderer,
)
from RcfDebugLocationLib import (
   Location,
   getLocation,
)

class DiagCodeManager( InvocationCodeManager ):
   def getCodeLine( self, lineNumber ):
      return self.context[ lineNumber - 1 ]

class StartUnderlineBar( Annotations.PrefixedBar ):
   prefix = '^'
   lineChar = '~'

class ContinuationUnderlineBar( Annotations.PrefixedBar ):
   prefix = '\\'
   lineChar = '~'

class DiagFragmentHandler:
   @staticmethod
   def handleSectionStart( fragment, codeManager, printer ):
      printer.expressionInProgress = True
      printer.writeToScratchpads( '', Annotations.BlankBar )

   @staticmethod
   def handleUnderline( fragment, codeManager, printer ):
      underlineBar = StartUnderlineBar
      if fragment.continuation:
         underlineBar = ContinuationUnderlineBar
      codeChunk = ' '
      if fragment.location.length:
         codeChunk = codeManager.popCode( fragment.location.length )
      printer.writeToScratchpads( codeChunk, underlineBar )

   @staticmethod
   def handleSectionEnd( fragment, codeManager, printer ):
      assert printer.expressionInProgress is True
      printer.expressionInProgress = False
      printer.writeToScratchpads( '', Annotations.BlankBar )

   @staticmethod
   def handle( fragment, codeManager, printer ):
      DiagFragmentHandler.dispatch[ fragment.fragmentType ]( fragment, codeManager,
                                                             printer )

DiagFragmentHandler.dispatch = {
   "sectionStart": DiagFragmentHandler.handleSectionStart,
   "underline": DiagFragmentHandler.handleUnderline,
   "sectionEnd": DiagFragmentHandler.handleSectionEnd,
}

# required to provide callStackDepth of 0 to the InvocationPrinter
DiagInvocationContext = functools.partial( InvocationContext, functionName=None,
                                           invocationIndex=None, callStackDepth=0,
                                           renderContext=None )

class DiagCodeRenderer( InvocationRenderer ):
   Fragment = namedtuple( "Fragment", "fragmentType continuation location" )
   fillLinesWithoutCode = True
   fillEmptyAnnotationsLineBetweenCode = False

   def __init__( self, codeUnitMapping, codeUnitKey, partsToUnderline,
                 codeSectionStart, codeSectionEnd ):
      invocationContext = DiagInvocationContext( codeUnitKey=codeUnitKey )
      self.printer = InvocationPrinter( invocationContext,
                                        annotationsBelowCode=True )
      self.codeManager = DiagCodeManager(
         codeUnitMapping.rcfTextLinesForCodeUnit( codeUnitKey ) )
      self.partsToUnderline = partsToUnderline
      self.codeSectionStart = codeSectionStart
      self.codeSectionEnd = codeSectionEnd
      self.fragmentHandler = DiagFragmentHandler

   def getFragmentIterator( self ):
      if self.codeSectionStart:
         yield DiagCodeRenderer.Fragment(
            'sectionStart', False,
            Location( self.codeSectionStart.line, 0, 0 ) )
      for context in self.partsToUnderline:
         contextLocation = getLocation( context, useNamedTuples=True )
         continuation = False
         for location in contextLocation:
            yield DiagCodeRenderer.Fragment( 'underline', continuation, location )
            continuation = True
      if self.codeSectionEnd:
         yield DiagCodeRenderer.Fragment(
            'sectionEnd', False,
            Location( self.codeSectionEnd.line,
                      self.codeSectionEnd.column, 0 ) )
      else:
         self.printer.expressionInProgress = False

   def fillCodeUnitDetails( self ):
      # code unit details are rendered separately
      pass

def renderCodeSection( codeUnitMapping, codeUnitKey, partsToUnderline=None,
                       codeSectionStart=None, codeSectionEnd=None ):
   '''
   Helper to render a section of RCF code.

   Args:
   - codeUnitMapping (CodeUnitMapping):
       Required to provide the RCF text of the compilation.
   - codeUnitKey (CodeUnitKey):
       The key of the code unit getting rendered
   - partsToUnderline (RcfParseContext or antrl4.Token in a list or on it's own)
       parts of the code section that should be underlined within the code section.
   - codeSectionStart (PointToken)
       The location the code section should start at.
   - codeSectionEnd (PointToken)
       The location the code section should end at.
   '''
   if partsToUnderline is None:
      partsToUnderline = []
   if not isinstance( partsToUnderline, list ):
      partsToUnderline = [ partsToUnderline ]
   renderer = DiagCodeRenderer( codeUnitMapping, codeUnitKey,
                                partsToUnderline, codeSectionStart,
                                codeSectionEnd )
   return renderer.render()
