# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import defaultdict

import BothTrace
from Assert import assertGreaterEqual, assertIn

bv = BothTrace.Var
bt0 = BothTrace.tracef0
bt8 = BothTrace.tracef8

class RcfFunctionVariableTypeHelper:
   """This class helps manage index allocation for the index values that
   need to be allocated for a specific variable type (e.g. "IntVariable").

   Attributes:
      inUseArgVectorIndiciesDict: A dict of variable name -> allocated index
   """
   def __init__( self ):
      self.inUseArgVectorIndiciesDict = {}

   def getArgVectorIndex( self, varName ):
      """Get an argVector index for a variable of this type in this scope"""
      assert varName not in self.inUseArgVectorIndiciesDict, (
         "%s has already been defined and assigned an argVector index" % varName )
      argVectorIndex = len( self.inUseArgVectorIndiciesDict )
      self.inUseArgVectorIndiciesDict[ varName ] = argVectorIndex
      bt8( "Rcf:RFVTH:getArgVectorIndex|",
           bv( varName ),
           bv( argVectorIndex ) )

      assertGreaterEqual( argVectorIndex, 0 )
      return argVectorIndex

class RcfFunctionVariableHelper:
   """This wrapper class helps manage index allocation for variables of all supported
   variable types within a single function. The variable helper object for a function
   will allocate and maintain a per variable type mapping of variable name to index.
   This index is used at eval time to lookup the value represnted by the variable
   name in the stack data passed to the callee function by the caller function.

   For example, taking the below function foo:

      function foo( int_type $varA, prefix_list_type $varB, int_type $varC )

   results in a variable helper object which looks like:

      { 'IntVariable'        : { '$varA' : 0,
                                 '$varC' : 1
                               },
        'PrefixListVariable' : { '$varB' : 0 }
      }

   Attributes:
      variableTypes: set of variable typenames which are supported
      helpers: Dict of variableType -> RcfFunctionVariableTypeHelper instance
   """
   def __init__( self, variableTypes ):
      self.variableTypes = variableTypes
      self.helpers = defaultdict( RcfFunctionVariableTypeHelper )

   def getArgVectorIndex( self, varName, variableType ):
      """Get a argVector index for a given variable name and variable type"""
      assertIn( variableType, self.variableTypes )
      return self.helpers[ variableType ].getArgVectorIndex( varName )
