# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from enum import Enum
from collections import defaultdict, namedtuple

import re

from CodeUnitMapping import CodeUnitKey
from RcfDiag import RcfDiag, RcfSyntaxDiag
from RcfLabelLocGen import LabelGenVisitor
from RcfSyntaxParser import runRcfLexer, runRcfParser
import RcfTypeFuture as Rcf

class EditTextRequest:
   """ Class representing the request for editing Rcf text.

   Attributes:
      - codeUnitName (str): code unit name of the RCF text.
      - codeUnitText (str): the text to edit.
      - lloc (LabelLoc): the label in original text to act on.
      - llocs (LabelLoc): all the labels present in this function
      - action (Action enum): the type of actions
      - userText (str): the multiline input the user entered.
   """
   class Action( Enum ):
      REPLACE_BLOCK = 1
      DELETE_BLOCK = 2
      BLOCK_INSERT_BEFORE = 3
      BLOCK_INSERT_AFTER = 4

   def __init__( self, codeUnitName, codeUnitText, lloc, llocs, action, userText ):
      self.codeUnitName = codeUnitName
      self.codeUnitText = codeUnitText
      self.lloc = lloc
      self.llocs = llocs
      self.action = action
      self.userText = userText

class EditTextResponse:
   """ Class representing the response for editing Rcf text.

   Attributes:
      - resultCodeUnitText (str): the resulting codeUnitText after edition.
      - codeUnitTextErrors ( [CompilerDiagMessageBase] ): all parsing error found in
                                                          codeUnitText.
      - userTextErrors ( [CompilerDiagMessageBase] ): all parsing error found
                                                      in user's multi line input.
      - badLabelName (str): if the provided label doesn't match, the provided label
      - surroundingTextFound (bool): whether surrounding text (comments) was found.
   """
   def __init__( self, resultCodeUnitText, codeUnitTextErrors, userTextErrors,
                 badLabelName, surroundingTextFound ):
      self.resultCodeUnitText = resultCodeUnitText
      self.codeUnitTextErrors = codeUnitTextErrors
      self.userTextErrors = userTextErrors
      self.badLabelName = badLabelName
      self.surroundingTextFound = surroundingTextFound

   def failed( self ):
      return (
         bool( self.codeUnitTextErrors ) or
         bool( self.userTextErrors ) or
         self.badLabelName is not None or
         self.surroundingTextFound
      )

class LabelLoc:
   """ Represents a label location in the code.

   Attributes:
      - ctx (Parser context): parser context for the label and the associated code.
      - blockType (LabelLoc.LabelType): type of label (only block so far)
      - name (str): identifier of the label (e.g @label1)
      - indent (str): indentation of this label in the code (lazy).
      - leading (str): space before the label, up to the first non space in the line
                       (lazy)

        e.g:
          "    @label1 {}": label1 indent / leading?
           ^~~  indent(4) == leading(4)

          "    @label1 {}  @label2 {}": label2 indent / leading?
           ^~~ indent(4) ^~~~ leading(2)
   """
   class LabelType( Enum ):
      block = 1

   def __init__( self, ctx, labelType, name ):
      self.ctx = ctx
      self.type = labelType
      self.name = name
      self.indent_ = None  # computed lazily
      self.leading_ = None # computed lazily

   def findIndentAndLeading_( self ):
      tokenStream = self.ctx.parser.getTokenStream()
      labelToken = self.ctx.LABEL().getPayload()
      self.indent_, self.leading_ = indentAndLeading( tokenStream, labelToken )

   @property
   def indent( self ):
      if self.indent_ is None:
         self.findIndentAndLeading_()
      return self.indent_

   @property
   def leading( self ):
      if self.leading_ is None:
         self.findIndentAndLeading_()
      return self.leading_

class LabelLocRequest:
   """ Request to build the label information

   Attributes:
      - codeUnitName(string): the name of the code unit the codeUnitText is in
      - codeUnitText (string): the rcf code (text)
   """
   def __init__( self, codeUnitName, codeUnitText ):
      self.codeUnitName = codeUnitName
      self.codeUnitText = codeUnitText

class LabelLocResult:
   """ Result of building the label information

   Attributes:
      - fatalErroList [CompilerDiagMessageBase]: list of fatal error encountered.
      - llocPerFunction { str: [LabelLoc] }: all LabelLocs indexed per function name.
   """
   def __init__( self, llocRequest, fatalErrorList, llocPerFunction ):
      self.llocRequest = llocRequest
      self.fatalErrorList = fatalErrorList
      self.llocPerFunction = llocPerFunction

   def failed( self ):
      """ Whether the request failed or not.
      """
      return bool( self.fatalErrorList )

def indentAndLeading( tokenStream, token ):
   """ Given a token stream, and a token, returns the text on the token's line, up to
   the token itself.

   Args:
      - tokenStream (CommonTokenStream): the token stream
      - token (CommonToken): the token

   Returns:
     indent(str), leading(str)
   """
   def reversedFrom( iterable, index ):
      # walk back from index, to index -1, index - 2... down to 0
      for i in range( index, -1, -1 ):
         yield iterable[ i ]

   # Walk back the token stream, up to the point where the current token's line
   # no longer is the same as our token line.
   lineStartToken = token
   leading = ""
   leadingFinished = False
   for t in reversedFrom( tokenStream.tokens, token.tokenIndex - 1 ):
      if not leadingFinished and t.text in [ ' ', '\t' ]:
         leading += str( t.text )
      else:
         leadingFinished = True
      if t.line != token.line:
         break
      lineStartToken = t

   line = tokenStream.getText( lineStartToken.tokenIndex, token.tokenIndex )
   indent = re.match( r'(?P<indent>^\s*)', line ).group( 'indent' )
   # Get all the text in the lexer from the lineStart token to our token.
   # Note: if our token is the line start: indentation and leading are empty string.
   return indent, leading

def parseTreeFrom( codeUnitName, text, rule ):
   """ Given a text, and a parse rule, get the associated parseTree

   Args:
      codeUnitName (str): the code unit name of the text
      text (str): the text we want to parse
      rule (str): the parser rule to invoke

   Returns:
      tuple( parseTree, diag, errors ).
      errors is None when parseTree isn't (and vice versa).
   """
   # Labeled editing is only performed on user-defined code units
   codeUnitKey = CodeUnitKey( Rcf.Metadata.FunctionDomain.USER_DEFINED,
                              codeUnitName )
   diag = RcfSyntaxDiag( codeUnitKey=codeUnitKey )

   _, tokenStream = runRcfLexer( diag, text )
   parseTree = runRcfParser( diag, tokenStream, rule=rule )

   if diag.hasErrors():
      return None, diag.allErrors
   return parseTree, None

def textFromContext( parserContext ):
   """ Given a parse tree, get the associated text.

   Args:
      parseContext (RcfContext): parse tree object.

   Retunrs:
      str: the associated text in its original form.
   """
   tokenStream = parserContext.parser.getTokenStream()
   sourceInterval = parserContext.getSourceInterval()
   return tokenStream.getText( sourceInterval[ 0 ], sourceInterval[ 1 ] )

def buildLabelLoc( labelLocRequest ):
   """ Given Rcf text, build a dictionnary of RcfLabeling.LabelLoc indexed by the
   function name they belong to.

   Args:
      labelLocRequest (LabelBuildingRequest): the request.

   Returns:
      LabelBuildingResult object.
   """
   # Labeled editing is only performed on user-defined code units
   codeUnitKey = CodeUnitKey( Rcf.Metadata.FunctionDomain.USER_DEFINED,
                              labelLocRequest.codeUnitName )
   syntaxDiag = RcfSyntaxDiag( codeUnitKey=codeUnitKey )
   _, tokenStream = runRcfLexer( syntaxDiag, labelLocRequest.codeUnitText )
   rcfRoot = runRcfParser( syntaxDiag, tokenStream )

   if syntaxDiag.hasErrors():
      llocPerFunction = defaultdict( lambda: defaultdict( lambda: None ) )
      return LabelLocResult( labelLocRequest, syntaxDiag.allErrors, llocPerFunction )

   diag = RcfDiag( strict=False )
   # walk the parse tree looking for labels
   llocPerFunction = labelLocFromParseTree( rcfRoot, diag, codeUnitKey )

   if diag.hasErrors():
      llocPerFunction = defaultdict( lambda: defaultdict( lambda: None ) )
      return LabelLocResult( labelLocRequest, diag.allErrors,
            llocPerFunction )

   return LabelLocResult( labelLocRequest, fatalErrorList=[],
         llocPerFunction=llocPerFunction )

def labelLocFromParseTree( rcfRoot, diag, codeUnitKey ):
   """ Given the parse tree, get all the label locations.

   Args:
      - labelLocRequest (LabelBuildingRequest): the request.
      - rcfRoot (RcfContext) : Rcf root object of the parse tree.
      - diag (RcfDiag): the RcfDiag object
      - codeUnitKey (CodeUnitKey): The CodeUnitKey identifying the unit that the
      rcfRoot is in.

   Returns:
      LabelLocResult.
   """
   # walk over the parse tree, walk each function in reasearch
   # of labels.
   # If duplicated labels are found within a function, inform through the diag.
   generateLabels = LabelGenVisitor( diag, codeUnitKey )
   return generateLabels( rcfRoot )

def textBeforeBlockLabel( rcfRoot, label ):
   """ Return the text from the root to the start of the label.

   Args:
      - rcfRoot (RcfContext): rcf root of the parsetree.
      - label (LabelLoc): label location object (block)
   We make sure not to omit spaces, or comments.

   eg:
         |'''
         |function foo() {
         |   return true;
         |}

         |function bar {
         |# this is a comment
         |   @l1 {
         |      return false;
         |   }
         |}
         |'''

   would return:

         |'''
         |function foo() {
         |   return true;
         |}

         |function bar {
         |# this is a comment
         |   < everything before @l1 as well.
         |'''

   Returns:
      the string before the label block position.
   """
   tokenStream = rcfRoot.parser.getTokenStream()
   tokenLabel = label.ctx.LABEL().getPayload()
   return tokenStream.getText( 0, tokenLabel.tokenIndex - 1 )

def textAfterBlockLabel( rcfRoot, label ):
   """ Return the text from the end of the block label, down to the rest of the code.

   Args:
      - rcfRoot (RcfContext): rcfroot of the parsetree.
      - label (LabelLoc): label location object (block)
   We make sure not to omit spaces, or comments.

   eg: (| delimits begining of the line)
         |'''
         |function foo() {
         |   return true;
         |}

         |function bar {
         |# this is a comment
         |   @l1 {
         |      return false;
         |   }
         |}
         |'''

   would return:
         |''' # this is another comment
         |}
         |'''

   Returns:
      the string after the label block position.
   """
   tokenStream = rcfRoot.parser.getTokenStream()
   _, labelBlockEndTokenIndex = label.ctx.getSourceInterval()
   end = len( tokenStream.tokens ) - 1
   return tokenStream.getText( labelBlockEndTokenIndex + 1, end )

def reIndentUserBlockLabel( indent, userBlockLabelText ):
   """ Given a label indentation, re-indent the user code.

   note: first line is not re-indented by design, because we keep the
   indentation of where the label was in the original input.

   Args:
      - indent (str): the indent string.
      - userLabelBlockText: the user block label text.

      e.g:
         with userBlockLabelText:
            |'''@label1 {
            |   med = 42;
            |}'''

        indent: |'''   '''

      would return:
            |'''@label1 {
            |      med = 42;
            |   }'''
   Returns:
      string: indented userBlockLabelText
   """
   userBlockLabelText = deindent( userBlockLabelText )
   return indent.join( userBlockLabelText.splitlines( True ) )

def deindent( text ):
   """ Finds the common indentation of each line in the given text, return a text
   removed of this common indentation.

   Args:
      - text (str) the text to de-indent.

   Returns:
      str
   """
   assert not text.isspace()
   linesIndentRe = re.compile( r'(^[ \t]*)(?:\S)', re.MULTILINE )
   linesIndent = linesIndentRe.findall( text )
   commonIndent = None

   for lineIndent in linesIndent:
      if lineIndent == '': # minor optim that should hit most of the time.
         commonIndent = ''
         break
      if commonIndent is None:
         commonIndent = lineIndent
         continue
      if not lineIndent.startswith( commonIndent ):
         # Indent isn't common anymore. Find the longest common indent
         # between currentIndent and the line indent.
         #
         # Walk over each character ( Common Indent Char, and Current Line Char )
         # until they no longer match, get the substring that matches.
         for i, ( cic, clc ) in enumerate( zip( commonIndent, lineIndent ) ):
            if cic != clc:
               commonIndent = commonIndent[ : i ]
               break # adjusted to the new common indent found
   if commonIndent:
      text = re.sub( '^' + commonIndent, '', text, flags=re.MULTILINE )
   return text

def hasSurroundingText( userTextParseTree ):
   """ Tells whether a userParseTree has leading or trailing comments.

   Args:
      - userTextParseTree( ParserContext ): the user text parse tree.

   Returns:
      bool: whether the userTextParseTree is surrounded by text other than
      whitespaces.
   """
   iStream = userTextParseTree.parser.getInputStream()
   lbBegin, lbEnd = userTextParseTree.getSourceInterval()
   COMMENT_CHANNEL = 50
   DEFAULT_CHANNEL = 0
   textChannels = { DEFAULT_CHANNEL, COMMENT_CHANNEL }
   tChannelsBeforeLabel = { t.channel for t in iStream.tokens[ : lbBegin ] }
   tChannelsAfterLabel = { t.channel for t in iStream.tokens[ lbEnd + 1 : -1 ] }
   return ( len( textChannels & tChannelsBeforeLabel ) or
            len( textChannels & tChannelsAfterLabel ) )

def validateUserBlockLabelTextReplace( codeUnitName, userBlockLabelText, labelName ):
   """ Make sure the input userBlockLabelText does parse correctly.

   Args:
      codeUnitName (str): The code unit name of the user text
      userBlockLabelText (str): the user input block label text.
      labelName (str): the label name the user wants to replace.

   Invalid reasons:
      - parseErrors: userBlockLabelText parses incorrectly
      - badLabelName: the user label provided doesn't match the one we're replacing

   Returns:
      (invalid, parseErrors, badLabelName)
   """
   Result = namedtuple( 'Result', [ 'invalid', 'parseErrors', 'badLabelName',
                                    'surroundingTextFound' ] )

   userTextParseTree, userTextErrors = (
      parseTreeFrom( codeUnitName, userBlockLabelText, 'labeledBlockInput' )
   )
   if userTextParseTree is None:
      return Result( invalid=True,
                     parseErrors=userTextErrors,
                     badLabelName=None,
                     surroundingTextFound=False )

   userLabel = str( userTextParseTree.LABEL() )
   surroundingTextFound = hasSurroundingText( userTextParseTree )

   if userLabel != labelName:
      return Result( invalid=True,
                     parseErrors=userTextErrors,
                     badLabelName=userLabel,
                     surroundingTextFound=surroundingTextFound )

   if surroundingTextFound:
      return Result( invalid=True,
                     parseErrors=userTextErrors,
                     badLabelName=None,
                     surroundingTextFound=surroundingTextFound )

   return Result( invalid=bool( userTextErrors ),
      parseErrors=userTextErrors,
      badLabelName=None,
      surroundingTextFound=False )

def validateUserBlockLabelTextInsert( codeUnitName, userBlockLabelText, llocs ):
   """ Make sure the input userBlockLabelText does parse correctly and
   can be safely inserted in the function.

   Args:
      codeUnitName (str): The code unit the text is in
      userBlockLabelText (str): the user input block label text.
      llocs ([str]): the list of labels present in this function.

   Invalid reasons:
      - bad block parsing (parseErrors is set)
      - bad label name (label about to be inserted
                        already exists in function)
   Returns:
      (invalid, parseErrors, badLabelName)
   """

   Result = namedtuple( 'Result', [ 'invalid', 'parseErrors', 'badLabelName',
                                    'surroundingTextFound' ] )

   userTextParseTree, userTextErrors = (
      parseTreeFrom( codeUnitName, userBlockLabelText, 'labeledBlockInput' )
   )
   if userTextParseTree is None:
      return Result( invalid=True,
                     parseErrors=userTextErrors,
                     badLabelName=None,
                     surroundingTextFound=False )

   userLabel = str( userTextParseTree.LABEL() )
   surroundingTextFound = hasSurroundingText( userTextParseTree )

   if userLabel in llocs:
      return Result( invalid=True,
                     parseErrors=userTextErrors,
                     badLabelName=userLabel,
                     surroundingTextFound=surroundingTextFound )

   if surroundingTextFound:
      return Result( invalid=True,
                     parseErrors=userTextErrors,
                     badLabelName=None,
                     surroundingTextFound=surroundingTextFound )

   return Result( invalid=bool( userTextErrors ),
      parseErrors=userTextErrors,
      badLabelName=None,
      surroundingTextFound=False )

def editRcfText( editTextReq ):
   """ Edit RCF text according to the request.

   Args:
      editTextReq (EditTextRequest): the request.

   Returns:
      EditTextResponse.
   """

   handlerPerAction = {
      EditTextRequest.Action.REPLACE_BLOCK: replaceBlockLabelText,
      EditTextRequest.Action.DELETE_BLOCK: deleteBlockLabelText,
      EditTextRequest.Action.BLOCK_INSERT_BEFORE: insertBeforeLabelText,
      EditTextRequest.Action.BLOCK_INSERT_AFTER: insertAfterLabelText,
   }
   return handlerPerAction[ editTextReq.action ]( editTextReq )

def replaceBlockLabelText( editTextReq ):
   """ Perform the block-label replace given an editTextRequest.

   Args:
      - editTextReq (EditTextRequest): the edit request details.

   Returns:
      EditTextResponse that may, or may not have succeded.
   """
   codeUnitTextRootCtx, codeUnitTextErrors = parseTreeFrom( editTextReq.codeUnitName,
                                                            editTextReq.codeUnitText,
                                                            'rcf' )
   userTextValidation = validateUserBlockLabelTextReplace(
         codeUnitName=editTextReq.codeUnitName,
         userBlockLabelText=editTextReq.userText,
         labelName=editTextReq.lloc.name )

   if codeUnitTextErrors or userTextValidation.invalid:
      return EditTextResponse(
               resultCodeUnitText=None,
               codeUnitTextErrors=codeUnitTextErrors,
               userTextErrors=userTextValidation.parseErrors,
               badLabelName=userTextValidation.badLabelName,
               surroundingTextFound=userTextValidation.surroundingTextFound,
            )
   reIndentedUserText = reIndentUserBlockLabel( editTextReq.lloc.indent,
         editTextReq.userText )

   userLabeledBlockCtx, errors = parseTreeFrom( editTextReq.codeUnitName,
                                                reIndentedUserText,
                                                'labeledBlockInput' )
   # We just indented the userText, this shouldn't cause any parse error.
   assert errors is None
   userLabeledBlockText = textFromContext( userLabeledBlockCtx )

   codeUnitTextBeforeBlock = textBeforeBlockLabel( codeUnitTextRootCtx,
                                                   editTextReq.lloc )
   codeUnitTextAfterBlock = textAfterBlockLabel( codeUnitTextRootCtx,
                                                 editTextReq.lloc )
   resultCodeUnitText = ( codeUnitTextBeforeBlock + userLabeledBlockText +
                     codeUnitTextAfterBlock )
   return EditTextResponse(
            resultCodeUnitText=resultCodeUnitText,
            codeUnitTextErrors=None,
            userTextErrors=None,
            badLabelName=None,
            surroundingTextFound=False )

def deleteBlockLabelText( editTextReq ):
   """ Perform the block-label delete given an editTextRequest.

   Args:
      - editTextReq (EditTextRequest): the edit request details.

   Returns:
      EditTextResponse that may, or may not have succeded.
   """
   codeUnitTextRootCtx, codeUnitTextErrors = parseTreeFrom( editTextReq.codeUnitName,
                                                            editTextReq.codeUnitText,
                                                            'rcf' )
   if codeUnitTextErrors:
      return EditTextResponse(
               resultCodeUnitText=None,
               codeUnitTextErrors=codeUnitTextErrors,
               userTextErrors=None,
               badLabelName=None,
               surroundingTextFound=None,
            )

   codeUnitTextBeforeBlock = textBeforeBlockLabel( codeUnitTextRootCtx,
                                                   editTextReq.lloc )
   codeUnitTextAfterBlock = textAfterBlockLabel( codeUnitTextRootCtx,
                                                 editTextReq.lloc )

   if editTextReq.lloc.leading:
      codeUnitTextBeforeBlock = \
         codeUnitTextBeforeBlock[ : -len( editTextReq.lloc.leading ) ]

   codeUnitTextAfterBlock = re.sub( r'^[ \t]*\n?', '', codeUnitTextAfterBlock )

   resultCodeUnitText = ( codeUnitTextBeforeBlock + codeUnitTextAfterBlock )

   return EditTextResponse(
            resultCodeUnitText=resultCodeUnitText,
            codeUnitTextErrors=None,
            userTextErrors=None,
            badLabelName=None,
            surroundingTextFound=None )

def insertBeforeLabelText( editTextReq ):
   """ Perform the block-label replace given an editTextRequest.

   Args:
      - editTextReq (EditTextRequest): the edit request details.

   Returns:
      EditTextResponse that may, or may not have succeded.
   """
   codeUnitTextRootCtx, codeUnitTextErrors = parseTreeFrom( editTextReq.codeUnitName,
                                                            editTextReq.codeUnitText,
                                                            'rcf' )
   userTextValidation = validateUserBlockLabelTextInsert(
         codeUnitName=editTextReq.codeUnitName,
         userBlockLabelText=editTextReq.userText,
         llocs=editTextReq.llocs )

   if codeUnitTextErrors or userTextValidation.invalid:
      return EditTextResponse(
               resultCodeUnitText=None,
               codeUnitTextErrors=codeUnitTextErrors,
               userTextErrors=userTextValidation.parseErrors,
               badLabelName=userTextValidation.badLabelName,
               surroundingTextFound=userTextValidation.surroundingTextFound,
            )
   reIndentedUserText = reIndentUserBlockLabel( editTextReq.lloc.indent,
         editTextReq.userText )

   reIndentedUserTextStrippedCtx, errors = parseTreeFrom( editTextReq.codeUnitName,
                                                          reIndentedUserText,
                                                          'labeledBlock' )
   # We just indented the userText, this shouldn't cause any parse error.
   assert errors is None
   reIndentedUserTextStripped = textFromContext( reIndentedUserTextStrippedCtx )

   textBeforeAnchorBlock = textBeforeBlockLabel( codeUnitTextRootCtx,
                                                 editTextReq.lloc )
   anchorTextBlock = textFromContext( editTextReq.lloc.ctx )
   textAfterAnchorBlock = textAfterBlockLabel( codeUnitTextRootCtx,
                                               editTextReq.lloc )
   resultCodeUnitText = ( textBeforeAnchorBlock +
                     reIndentedUserTextStripped + '\n' +
                     editTextReq.lloc.indent + anchorTextBlock +
                     textAfterAnchorBlock )
   return EditTextResponse(
            resultCodeUnitText=resultCodeUnitText,
            codeUnitTextErrors=None,
            userTextErrors=None,
            badLabelName=None,
            surroundingTextFound=None )

def insertAfterLabelText( editTextReq ):
   """ Inserts a labeled block after a given label.

   Args:
      - editTextReq (EditTextRequest): the edit request details.

   Returns:
      EditTextResponse that may, or may not have succeded.
   """
   codeUnitTextRootCtx, codeUnitTextErrors = parseTreeFrom( editTextReq.codeUnitName,
                                                            editTextReq.codeUnitText,
                                                            'rcf' )
   userTextValidation = validateUserBlockLabelTextInsert(
         codeUnitName=editTextReq.codeUnitName,
         userBlockLabelText=editTextReq.userText,
         llocs=editTextReq.llocs )

   if codeUnitTextErrors or userTextValidation.invalid:
      return EditTextResponse(
               resultCodeUnitText=None,
               codeUnitTextErrors=codeUnitTextErrors,
               userTextErrors=userTextValidation.parseErrors,
               badLabelName=userTextValidation.badLabelName,
               surroundingTextFound=userTextValidation.surroundingTextFound,
            )
   reIndentedUserText = reIndentUserBlockLabel( editTextReq.lloc.indent,
         editTextReq.userText )

   reIndentedUserTextStrippedCtx, errors = parseTreeFrom( editTextReq.codeUnitName,
                                                          reIndentedUserText,
                                                          'labeledBlock' )
   # We just indented the userText, this shouldn't cause any parse error.
   assert errors is None
   reIndentedUserTextStripped = textFromContext( reIndentedUserTextStrippedCtx )

   textBeforeAnchorBlock = textBeforeBlockLabel( codeUnitTextRootCtx,
                                                 editTextReq.lloc )
   anchorTextBlock = textFromContext( editTextReq.lloc.ctx )
   textAfterAnchorBlock = textAfterBlockLabel( codeUnitTextRootCtx,
                                               editTextReq.lloc )
   resultCodeUnitText = ( textBeforeAnchorBlock +
                     anchorTextBlock + '\n' +
                     editTextReq.lloc.indent + reIndentedUserTextStripped +
                     textAfterAnchorBlock )
   return EditTextResponse(
            resultCodeUnitText=resultCodeUnitText,
            codeUnitTextErrors=None,
            userTextErrors=None,
            badLabelName=None,
            surroundingTextFound=None )
