# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re

# (?P<foo>...) denotes a named capture group which can be retrieved via
# match.group( 'foo' ). This regex matches:
# - +N (as inc)
# - -N (as dec)
# - N +IGP (as nPlusIgp)
# - N +METRIC (as nPlusMetric)
# - +IGP (as plusIgp)
# - +METRIC (as plusMetric)
# - METRIC (as metric)
# BUG982631: U32 assignment is not supported until optional unions are implemented.
# To work around this, we support string-encoding integers.
# - N (as set)
setMedRe = re.compile(
      r"(?P<set>[0-9]+)|" +
      r"\+(?P<inc>[0-9]+)|-(?P<dec>[0-9]+)|" +
      r"(?P<nPlusIgp>[0-9]+) \+IGP|(?P<nPlusMetric>[0-9]+) \+METRIC|" +
      r"(?P<plusIgp>\+IGP)|(?P<plusMetric>\+METRIC)|" +
      r"(?P<metric>METRIC)" )

def parseSetMed( data ):
   """Return the metric type and any parsed integers from data using setMedRe.

   Args:
      data (str): data to parse
   Returns:
      tuple of ( metricType, value ) if data parses, or ( None, None ) otherwise.
   """
   metricType = None
   value = None
   if match := setMedRe.fullmatch( data ):
      if strMatch := match.group( 'set' ):
         metricType = "medNormal"
         value = int( strMatch )
      elif strMatch := match.group( 'inc' ):
         metricType = "medAdditive"
         value = int( strMatch )
      elif strMatch := match.group( 'dec' ):
         metricType = "medSubtractive"
         value = int( strMatch )
      elif strMatch := match.group( 'nPlusIgp' ):
         metricType = "medValueAddIgpNexthopCost"
         value = int( strMatch )
      elif strMatch := match.group( 'nPlusMetric' ):
         metricType = "medValueAddIgpMetric"
         value = int( strMatch )
      elif strMatch := match.group( 'plusIgp' ):
         metricType = "medAddIgpNexthopCost"
      elif strMatch := match.group( 'plusMetric' ):
         metricType = "medAddIgpMetric"
      elif strMatch := match.group( 'metric' ):
         metricType = "medIgpMetric"
      else:
         # Unreachable
         assert False
   return metricType, value
