#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import RcfMetadata
from RcfCodeAnalysis import FunctionScopeCodeAnalysisPhase
from RcfTypeBinding import (
   FunctionScopeTypeBindingPhase,
   LinkerTypeBindingPhase
)
from RcfValueValidation import (
   FunctionScopeValueValidation,
   LinkerValueValidation,
)
from RcfSymbolDefinitionPhase import (
   FunctionScopeDefinitionPhase,
   LinkerDefinitionPhase
)
from RcfSymbolResolutionPhase import (
   FunctionScopeResolutionPhase,
   LinkerResolutionPhase
)
from RcfLibCallgraph import (
   Callgraph,
   CallGraphNode,
   findAllCycles,
)
from RcfBottomUpOrder import setFunctionsInBottomUpOrder

BT = RcfMetadata.RcfBuiltinTypes
RcfBuiltinSymbols = RcfMetadata.RcfBuiltinSymbols
RcfTypeSystem = RcfMetadata.RcfTypeSystem
MAX_U32 = 0xFFFFFFFF

def runFunctionScopeSemanticAnalysis( functions, diag ):
   """ Runs the per function semantic analysis over the AST.

        The following phases will follow (in order).

        - SymbolTable generation / validation:
           - Definition
           - Resolution

        - Type binding / validation

   Args:
      functions: A list for function AST nodes
      diag: the RCF diagnostics object.
   """
   for function in functions:
      functionDiag = diag.emptyFunctionDiag()

      FunctionScopeDefinitionPhase( functionDiag )( function )

      if not functionDiag.hasErrors():
         FunctionScopeResolutionPhase( functionDiag )( function )

      if not functionDiag.hasErrors():
         FunctionScopeTypeBindingPhase( functionDiag )( function )

      if not functionDiag.hasErrors():
         FunctionScopeValueValidation( functionDiag )( function )

      if not functionDiag.hasErrors():
         FunctionScopeCodeAnalysisPhase( functionDiag )( function )

      if functionDiag.hasErrors():
         function.valid = False

      diag.update( functionDiag )

def addCallsToCallgraph( function, callgraph ):
   '''
   Add all function calls in the function to the callgraph.
   At this point the function calls should be valid.
   '''
   for call in function.functionCalls:
      callgraph.add( CallGraphNode( function.name ),
                     # Include the location of this functions definition
                     # and the call site. This will allow any compilation errors
                     # to include the locations of the functions and
                     # function call sites involved in any recursion found.
                     CallGraphNode( call.funcName,
                                    callerNode=function,
                                    callSiteNode=call ) )

def runLinker( rcfAst, diag, rcfExternalConfig ):
   """ Runs the cross function semantic analysis over the AST.

        The following phases will follow (in order).

        - SymbolTable generation / validation:
           - Definition
           - Resolution
           - Cycle detection

        - Type binding / validation

   Args:
      rcfAst: the RCF Abstract Syntax Tree root.
      diag: the RCF diagnostics object.
      rcfExternalConfig : RcfHelperTypes.RcfExternalConfig
         This python object holds:
            aclConfig (Acl::AclListConfig)
            roaTableStatusDir (Rpki::RoaTableStatusDir)
            dynPfxListConfigDir (DynamicPrefixList::Config)
         These are used to validate
         external references during symbol table generation.
   """
   callgraph = Callgraph()

   LinkerDefinitionPhase( diag )( rcfAst )

   for func in rcfAst.validFunctions():
      funcDiag = diag.emptyFunctionDiag()

      LinkerResolutionPhase( funcDiag, rcfExternalConfig )( func )

      if not funcDiag.hasErrors():
         LinkerTypeBindingPhase( funcDiag )( func )

      if not funcDiag.hasErrors():
         LinkerValueValidation( funcDiag )( func )

      if not funcDiag.hasErrors():
         addCallsToCallgraph( func, callgraph )
      else:
         func.valid = False

      diag.update( funcDiag )

   for cycle in findAllCycles( callgraph ):
      diag.cycleError( cycle )
      for node in cycle:
         node.callerNode.valid = False

   setFunctionsInBottomUpOrder( rcfAst, callgraph )
