# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from functools import partial

from RcfAbstractScope import AbstractScope
from RcfFunctionVariableHelper import RcfFunctionVariableHelper

class Symbol:
   """ A symbol in Rcf has a name and an associated type.

   Each symbol points to the node in the AST where it is defined.

   Note that a Symbol is not a type.

   Attributes:
      name (str): name of the symbol
      rcfType (RcfType, NIY): rcf type of this symbol
      node (RcfAST Node): AST Node defining this symbol
      const (bool): whether this symbol is const or not
      aetType (Tacc): the AET type for this symbol
      routeMapFeature (dict): dict of value indexed by RouteMap feature
                              (attribute name of Routing::Policy::RouteMapFeatures)
      inputVariant (bool): Whether the symbol is for a "input." variant of an
                           attribute.
      rootAttribute (RcfAst Node): Ast node of the root attribute for this
                                   subattribute
   """
   def __init__( self, name, rcfType=None, node=None, const=False,
                 aetType=None, routeMapFeatures=None, inputVariant=False,
                 rootAttribute=None ):
      self.name = name
      self.rcfType = rcfType
      self.node = node
      self.const = const
      self.aetType = aetType
      self.routeMapFeatures = routeMapFeatures or set()
      self.inputVariant = inputVariant
      self.rootAttribute = rootAttribute

   def updateRouteMapFeatures( self, routeMapFeature ):
      for feature in self.routeMapFeatures:
         setattr( routeMapFeature, feature, True )

ConstSymbol = partial( Symbol, const=True )

class Label( Symbol ):
   pass

class ScopeSymbol( Symbol, AbstractScope ):
   """ Super symbols, that can act as a scope itself.

   e.g: function, struct etc..

   * Why not inherit from both Symbol and Scope?

     To keep multiple inheritance sane, we should only inherit from
     one concrete class. Following this rule helps reasoning
     more clearly about the class hierarchy, at the cost of minimal
     code duplication:

     ScopedSymbol is a Symbol that can "act-as" a Scope.

   * Why not inherit from Scope instead of Symbol?

     Functionnaly, we want to treat this more like a super symbol
     rather that a super scope. A struct, for instance, is a super
     symbol, but a dumb down scope.
   """

   def __init__( self, name, rcfType=None, node=None, routeMapFeatures=None,
                 enclosingScope=None ):
      Symbol.__init__( self, name=name, rcfType=rcfType, node=node,
                       routeMapFeatures=routeMapFeatures )
      self.enclosingScope = enclosingScope
      self.members = {}

   def define( self, symbol ):
      """ Define a symbol within this symbol.

      Args:
         symbol (RcfSymbol.Symbol): the symbol we want to include.
      """
      self.members[ symbol.name ] = symbol

   def getScopeName( self ):
      return self.name

   def getEnclosingScope( self ):
      return self.enclosingScope

   def resolve( self, reference ):
      raise NotImplementedError

class Function( ScopeSymbol ):
   """ Function are a special kind of ScopeSymbol, with a name and
   a return type. Built in functions provide route map features directly because
   they do not contain expressions with attributes to provide them.
   """
   def __init__( self, name, rcfType, retType, node, enclosingScope,
                 routeMapFeatures, functionSelf, allowedVariableTypenames ):
      super().__init__( name=name,
                        rcfType=rcfType,
                        node=node,
                        routeMapFeatures=routeMapFeatures,
                        enclosingScope=enclosingScope )
      self.retType = retType
      self.functionSelf = functionSelf
      # Function parameter types are not part of the __init__() because they are
      # added by DefinitionPhase.visitFuncParam() after the creation of this function
      # symbol in DefinitionPhase.visitFunction().
      self.funcParamTypes = []
      self.variableHelper = RcfFunctionVariableHelper(
            variableTypes=allowedVariableTypenames )

   def resolve( self, reference ):
      """ Resolve a reference (string) in this scope, and above.

      Args:
         reference (str): a string from the rcf code like "foo".

      Returns:
         The symbol associated to this reference according to the scope
         search, or None.
      """
      symbol = self.members.get( reference )
      if symbol is None:
         if self.enclosingScope:
            return self.enclosingScope.resolve( reference )
      return symbol

   def getVariableArgVectorIndex( self, variableName, variableTypename ):
      """ Get a unique argVector index for a variable of the given name and type in
      this function's scope.

      Args:
         variableName (str): name of the variable, e.g. "$var1"
         variableTypename (str): name of the variable type, e.g. "IntVariable"

      Returns:
         The index to be used at eval time to get the value represented by this
         symbol.
      """
      return self.variableHelper.getArgVectorIndex( variableName, variableTypename )

class FunctionParam( Symbol ):
   """ Function parameter symbols are constant. Variables in the function body
   resolve to a function parameter symbol.

   Attributes:
      argVectorIndex (int): Index into per function, per variable type vector at eval
   """
   def __init__( self, name, rcfType, node, argVectorIndex ):
      # Function parameters are const
      super().__init__( name=name, rcfType=rcfType, node=node, const=True )
      self.argVectorIndex = argVectorIndex

