#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from TypeFuture import TacLazyType

from RcfAbstractScope import AbstractScope
from RcfHelperTypes import BuiltinRibs
import RcfMetadata

BT = RcfMetadata.RcfBuiltinTypes

class Scope( AbstractScope ):
   """ A scope holds a dictionnary of symbols indexed by their name.

   We define object in a scope, and we resolve them through the scopes.
   A Scope may, or may not have an enclosing scope.

   e.g:
   - global scope doesn't have an enclosing scope.
   - a function scope has the global scope enclosing it.

   Attributes:
      name (str): the name of the scope e.g 'global'.
      symbols (dict): dict of symbols defined at this scope, indexed by
                      their name
      enclosing (Scope, optional): the parent (aka 'enclosing') scope.
   """
   def __init__( self, name, enclosing=None ):
      super().__init__()
      self.name = name
      self.enclosing = enclosing
      self.symbols = {}

   def define( self, symbol ):
      """ Define a symbol in this scope.

      This function will also set the scope (self) where this symbol is defined.

      Args:
         symbol (Symbol): the symbol to define.
      """
      self.symbols[ symbol.name ] = symbol

   def resolve( self, reference ):
      """ Find the symbol associated to this name, in this scope, or
      in its enclosing scope (recursively).

      Args:
         reference (str): the name of the symbol to find.

      Returns:
         the symbol (Symbol) if found, None otherwise.
      """
      symbol = self.symbols.get( reference )
      if symbol is None:
         if self.enclosing:
            return self.enclosing.resolve( reference )
      return symbol

   def getScopeName( self ):
      return self.name

   def getEnclosingScope( self ):
      return self.enclosing

class BlockScope( Scope ):
   """ A block scope is created to keep track of Rcf block and their nesting.
   """
   def __init__( self, enclosingScope ):
      super().__init__( name="Local", enclosing=enclosingScope )

class GlobalScope( Scope ):
   """ The Global scope is unique in Rcf and has no enclosing scope.
   """
   def __init__( self ):
      super().__init__( name="Global", enclosing=None )

   def undefine( self, functionName ):
      self.symbols.pop( functionName, None )

class AttributeScope:
   """ Class that we use to lookup whether an attribute is defined or not.
   """
   def resolve( self, attrName ):
      return RcfMetadata.RcfBuiltinSymbols.builtInAttributes.get( attrName, None )

class ExternalScope:
   """ Class that we use to lookup whether an external symbol is defined or not.

   Constructor arguments:
      rcfExternalConfig : RcfHelperTypes.RcfExternalConfig
         This python object holds:
            aclConfig (Acl::AclListConfig)
            roaTableStatusDir (Rpki::RoaTableStatusDir)
            dynPfxListConfigDir (DynamicPrefixList::Config)
         These are used to validate
   Attributes:
      aclListConfig: Acl::AclListConfig used for extref validation
      roaTableStatusDir : Rpki::RoaTableStatusDir used for extref validation
      dynPfxListConfigDir : DynamicPrefixList::Config used for extref validation
      dispatchOnType: mapping from extref keywords to validation methods
   """
   AclCommunityListType = TacLazyType( "Acl::CommunityListType" )

   def __init__( self, rcfExternalConfig ):
      self.aclListConfig = rcfExternalConfig.aclListConfig
      self.roaTableStatusDir = rcfExternalConfig.roaTableStatusDir
      self.dynPfxListConfigDir = rcfExternalConfig.dynPfxListConfigDir
      self.tunnelRibNameIdMap = rcfExternalConfig.tunnelRibNameIdMap
      self.dispatchOnType = {
            'prefix_list_v4': self.resolvePrefixListV4,
            'prefix_list_v6': self.resolvePrefixListV6,
            'as_path_list': self.resolveAsPathList,
            'community_list': self.resolveCommunityList,
            'ext_community_list': self.resolveExtCommunityList,
            'large_community_list': self.resolveLargeCommunityList,
            'roa_table': self.resolveRoaTable,
            'dynamic_prefix_list': self.resolveDynamicPrefixList,
            'rib_tunnel': self.resolveTunnelRib
      }

   def define( self, symbol ):
      raise TypeError( "Rcf does not define external symbols" )

   def resolve( self, extRef ):
      """ Check whether a given external ref is defined in Sysdb or not.

      Args:
         externalRef (Ast.ExternalRef): the external reference to check.

      Returns:
         true if extRef is defined already, false otherwise.
      """
      if not self.aclListConfig:
         return False
      return self.dispatchOnType[ extRef.etype ]( extRef )

   def resolvePrefixListV4( self, extRef ):
      name = extRef.name
      assert self.aclListConfig
      return name in self.aclListConfig.prefixList

   def resolvePrefixListV6( self, extRef ):
      name = extRef.name
      assert self.aclListConfig
      return name in self.aclListConfig.ipv6PrefixList

   def resolveAsPathList( self, extRef ):
      name = extRef.name
      assert self.aclListConfig
      return name in self.aclListConfig.pathList

   def resolveAnyCommunityList( self, extRef, commListEntryType ):
      """ Check whether a given external ref exists in AclList Config and contains
      entries for the commList type we care about.

      Args:
         extRef (Ast.ExternalRef): the external reference to check.
         commListEntryType (RcfSymbol.Type): RCF builtin extref type we care about

      Returns:
         Symbol object if the requested commList was found in AclListConfig
         with entries of the type we requested, None otherwise.
      """
      name = extRef.name
      assert self.aclListConfig
      if not name in self.aclListConfig.communityList:
         # no name found, not defined
         return False
      commList = self.aclListConfig.communityList[ name ]
      # Get the set of comm list types (standard, extended, large) which have entries
      # in the specified external reference
      typeOptions = self.communityListRcfTypes( commList )
      return commListEntryType in typeOptions

   def resolveCommunityList( self, extRef ):
      return self.resolveAnyCommunityList( extRef, BT.CommunityList )

   def resolveExtCommunityList( self, extRef ):
      return self.resolveAnyCommunityList( extRef, BT.ExtCommunityList )

   def resolveLargeCommunityList( self, extRef ):
      return self.resolveAnyCommunityList( extRef, BT.LargeCommunityList )

   def communityListRcfTypes( self, communityList ):
      types = set()
      for entry in communityList.communityListEntry.values():
         # RCF does not make a distinction between regular and expanded community
         # list entries at the time of external reference resolution. At runtime,
         # expanded (regex) entries are ignored in add/assign operations.
         if entry.listType in [ self.AclCommunityListType.communityStandard,
                                self.AclCommunityListType.communityExpanded ]:
            types.add( BT.CommunityList )
         elif entry.listType in [ self.AclCommunityListType.extCommunityStandard,
                                  self.AclCommunityListType.extCommunityExpanded ]:
            types.add( BT.ExtCommunityList )
         elif entry.listType in [ self.AclCommunityListType.largeCommunityStandard,
                                  self.AclCommunityListType.largeCommunityExpanded ]:
            types.add( BT.LargeCommunityList )
         else:
            continue # no support
      return types

   def resolveRoaTable( self, extRef ):
      name = extRef.name
      assert self.roaTableStatusDir
      return name in self.roaTableStatusDir.roaTableStatus

   def resolveTunnelRib( self, extRef ):
      name = extRef.name
      assert self.tunnelRibNameIdMap, \
             'tunnelRibNameIdMap must be provided to verify tunnel rib names'
      return name in self.tunnelRibNameIdMap.nameToId

   def resolveDynamicPrefixList( self, extRef ):
      name = extRef.name
      assert self.dynPfxListConfigDir
      return name in self.dynPfxListConfigDir.dynamicPrefixList

class BuiltInExternalScope:
   """ Class that we use to lookup whether a value that is supposed to have a
       built-in name contains a valid one.
   """

   def __init__( self ):
      self.dispatchOnType = {
         'rib_ip': self.resolveIpRib,
         'rib_colored_tunnel': self.resolveColoredTunnelRib
      }

   def resolve( self, extRef ):
      # Check whether a given external ref has a valid built-in name
      return self.dispatchOnType[ extRef.etype ]( extRef )

   def resolveIpRib( self, extRef ):
      return extRef.name in BuiltinRibs[ 'rib_ip' ]

   def resolveColoredTunnelRib( self, extRef ):
      return extRef.name in BuiltinRibs[ 'rib_colored_tunnel' ]
