# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import os.path
import Cell

def isAsuSupported( product, asuSupportedProducts, asuManifestLib ):
   for asuSupportedProduct, asuSupported in asuSupportedProducts.items():
      if product == asuSupportedProduct:
         asuSupported, toggleName = asuSupported
         # check associated toggle state optionally
         return asuSupported or asuManifestLib.getToggleEnabled( toggleName )
   return None

def isDynamicAsuBootupUnsupported( ctx ):
   asuManifest = None
   if "ASU_MANIFEST_LOCAL" in os.environ or "ASU_PSTORE_PLUGIN_PATH" in os.environ:
      from ReloadPolicyPlugin import AsuManifestLib
   else:
      from . import AsuManifestLib # pylint: disable-msg=relative-beyond-top-level
   try:
      if "ASU_MANIFEST_NEXT" in os.environ:
         asuSupportedProducts = AsuManifestLib.getManifestAttribute(
               os.environ[ "ASU_MANIFEST_NEXT" ],
               'asuSupportedProductsList')
      else:
         asuManifest = AsuManifestLib.extractAsuManifest( ctx.nextImage )
         asuSupportedProducts = AsuManifestLib.getManifestAttribute( asuManifest,
               'asuSupportedProductsList' )

      # asuSupportedProducts are keyed by Cell.product(). Use it to perform an exact
      # match.
      product = Cell.product()
      if product:
         asuSupported = isAsuSupported( product, asuSupportedProducts,
                                        AsuManifestLib )
         if asuSupported is None:
            return False
         else:
            return not asuSupported
         return False
   except AsuManifestLib.ManifestCheckError as e:
      ctx.addError( str( e ) )
      return False
   except ( ImportError, NameError, TypeError ):
      # Accounts for the cases where AsuUtil isn't found, AsuUtil doesn't contain
      # doGetFeatureKeys, or AsuUtil arguments have changed.
      ctx.addWarning( "ASU supported products manifest check fail: could not"
                      " perform ASU supported check." )
      return False
   finally:
      AsuManifestLib.safeRemove( asuManifest )

def isAsuBootupUnsupported( ctx ):
   if isDynamicAsuBootupUnsupported( ctx ):
      return True

   return False

def checkAsuSupport( ctx ):
   '''Determines if the next image supports Asu (SSU) reload bootup path. Mostly,
   both Asu shutdown and bootup path are supported or not supported at the same time
   but in some cases when Asu shudown path is supported but the next image does not
   support the Asu bootup path, the Asu reload Cli is available to test the Asu
   shutdown path but the Asu reload will not succeed, so we need to block it when the
   command is issued by the user in a non-testing environment.
   See AID 8350 for more details.
   '''
   errMsg = "SSU reload from %s to %s is not supported on the current platform." % (
               ctx.currentVersion.version(), ctx.nextVersion.version() )

   if isAsuBootupUnsupported( ctx ):
      ctx.addError( errMsg )
      return False

   return True

def Plugin( ctx ):
   category = [ 'ASU' ]
   ctx.addPolicy( checkAsuSupport, category )
