# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import inspect
import os

def compareFeatureKeysDicts( currDict, nextDict ):
   '''Perform the feature-keys check by comparing the feature-keys dictionaries of
   the current and next image. If the check passes, return nothing. If the check
   fails, raise an exception.
   '''

   if "ASU_MANIFEST_LOCAL" in os.environ or "ASU_PSTORE_PLUGIN_PATH" in os.environ:
      from ReloadPolicyPlugin import AsuManifestLib
   else:
      from . import AsuManifestLib # pylint: disable-msg=relative-beyond-top-level
   if not isinstance( currDict, dict ) or not isinstance( nextDict, dict ):
      raise AsuManifestLib.ManifestCheckError( 'Invalid input type.' )
   for feature, featureKeys in currDict.items():
      if feature not in nextDict:
         raise AsuManifestLib.ManifestCheckError( "PStore manifest check fail: "
                              "feature %s not in manifest." % repr( feature ) )
      nextKeys = nextDict[ feature ]
      diffSet = set( featureKeys ).difference( set( nextKeys ) )
      if diffSet:
         diffSetStr = "    " + ", ".join( str( key ) for key in diffSet )
         raise AsuManifestLib.ManifestCheckError( "PStore manifest check fail: keys "
                                 "in feature %s not supported:\n%s"
                                    % ( repr( feature ),
                                        diffSetStr ) )

def checkFeatureKeys( ctx ):
   '''Check that the next image supports the required features and keys for an ASU
   upgrade. The ReloadPolicy plugin context provides the next image information and
   the mode information.
   '''

   asuManifest = None
   if "ASU_MANIFEST_LOCAL" in os.environ or "ASU_PSTORE_PLUGIN_PATH" in os.environ:
      from ReloadPolicyPlugin import AsuManifestLib
   else:
      from . import AsuManifestLib # pylint: disable-msg=relative-beyond-top-level

   try:
      # Get the next image feature-keys.
      if "ASU_MANIFEST_NEXT" in os.environ:
         nextFeatureKeysDict = AsuManifestLib.getManifestAttribute(
                                 os.environ[ "ASU_MANIFEST_NEXT" ],
                                 'pStoreFeatureKeysDict' )
      else:
         asuManifest = AsuManifestLib.extractAsuManifest( ctx.nextImage )
         nextFeatureKeysDict = AsuManifestLib.getManifestAttribute( asuManifest,
                                                   'pStoreFeatureKeysDict' )

      # Get the current image feature-keys.
      if "ASU_MANIFEST_LOCAL" in os.environ:
         currFeatureKeysDict = AsuManifestLib.getManifestAttribute(
                                 os.environ[ "ASU_MANIFEST_LOCAL" ],
                                 'pStoreFeatureKeysDict' )
      else:
         assert ctx.mode
         import AsuUtil
         if not hasattr( AsuUtil, 'doGetFeatureKeys' ):
            raise NameError
         # Check that doGetFeatureKeys is expecting one argument.
         if not len( inspect.getargspec( AsuUtil.doGetFeatureKeys ).args ) == 1:
            raise TypeError
         currFeatureKeysDict = AsuUtil.doGetFeatureKeys( ctx.mode.entityManager )

      compareFeatureKeysDicts( currFeatureKeysDict, nextFeatureKeysDict )
      return True

   except AsuManifestLib.ManifestCheckError as e:
      ctx.addError( str( e ) )
      return False
   except ( ImportError, NameError, TypeError ):
      # Accounts for the cases where AsuUtil isn't found, AsuUtil doesn't contain
      # doGetFeatureKeys, or AsuUtil arguments have changed.
      ctx.addWarning( "PStore manifest check fail: could not perform feature-keys"
                      " check." )
      return None
   finally:
      AsuManifestLib.safeRemove( asuManifest )

def Plugin( ctx ):
   category = [ 'ASU' ]
   ctx.addPolicy( checkFeatureKeys, category )
