# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import os
import imp

# ASU package shouldnt depend on any platform specific package, so use
# ArTogglesPyAgent to get all toggle states.
# All the available toggles are presented when 'reload fast-boot' is run.
def getToggleEnabled( name ):
   # import at top might fail in older product, only import for products using
   # the function.
   import ArTogglesPyAgent as tg
   toggles = tg.pyGetAllToggles()
   for toggle in toggles:
      if toggle.get( 'name', '' ) == name:
         return toggle.get( 'enabled', False )

   # If the toggle doens't exist, returns False for the following cases:
   # - in an older release, rely solely on new release to indicate asuSupport value
   # - where the toggle is removed, but not enabled -> return False
   # Note: we won't be in this chcek if asuSupportForProduct is True.
   return False

class ManifestCheckError( Exception ):
   pass

def extractAsuManifest( nextImagePath ):
   '''Extract the 'asu-manifest' file from the next image to /tmp. '''

   import zipfile
   with zipfile.ZipFile( nextImagePath ) as nextImage:
      if 'asu-manifest' not in nextImage.namelist():
         raise ManifestCheckError( 'New image missing ASU manifest file.' )
      return nextImage.extract( 'asu-manifest', '/tmp' )

def safeRemove( filename ):
   '''Try to remove the specified file. Catch any errors. '''

   try:
      os.remove( filename )
   except ( OSError, TypeError ):
      pass

def getManifestAttribute( manifestFile, manifestAttribute ):
   '''Extract and return the given manifestAttribute from the specified asu-manifest
   file. Raise an exception if an error occurs.
   '''

   try:
      with open( manifestFile ) as manifest:
         newAsuManifest = imp.new_module( 'newAsuManifest' )
         exec( manifest.read(), newAsuManifest.__dict__ )
         return getattr( newAsuManifest, manifestAttribute, {} )
   except ( AttributeError, IOError ):
      raise ManifestCheckError( 'Failed to open ASU manifest file.' )
   except SyntaxError:
      raise ManifestCheckError( 'ASU manifest file corrupted.' )
