# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division

import os

def is32BitArch( arch ):
   return arch in [ 'i686', 'i386' ]

def is64BitArch( arch ):
   return not is32BitArch( arch )

def meminfoFile():
   if 'SIMULATION_VMID' in os.environ:
      return '/tmp/eos64bitplugintest-meminfo'
   return '/proc/meminfo'

def getMemUsage():
   """Read the meminfo file and calculate the memory usage from total and available
   memory.
   """
   memTotal = 0
   memAvailable = 0
   memUsage = 0.0

   with open( meminfoFile() ) as f:
      for line in f:
         if line.startswith( 'MemTotal:' ):
            memTotal = int( line.split()[ 1 ] )
         elif line.startswith( 'MemAvailable:' ):
            memAvailable = int( line.split()[ 1 ] )
         if memTotal and memAvailable:
            break

   # If memory sizes can't be read, just return 0. The meminfo file is in a standard
   # format so this shouldn't happen in practice.
   if memTotal:
      memUsage = ( memTotal - memAvailable ) / memTotal
   return memUsage

def checkEos64MemoryUsage( ctx ):
   """If the current image is 32-bit and the next image is 64-bit, check the
   current memory usage and warn the user if upgrading to the 64-bit image
   may cause OOM errors.
   """
   try:
      if ( is32BitArch( ctx.currentVersion.architecture() ) and
           is64BitArch( ctx.nextVersion.architecture() ) ):
         memUsage = getMemUsage()
         assert memUsage
         if memUsage > 0.6:
            # pylint: disable=consider-using-f-string
            ctx.addWarning(
               "Memory usage is high ({:.0f}%). Loading a 64 bit EOS image may "
               "cause the switch to run out of memory due to increased memory usage."
               .format( memUsage * 100 )
            )
            return False
   # Catch IOError for python2.x compatibility.
   except ( AssertionError, IOError, OSError, ValueError ):
      ctx.addWarning( "Unable to check platform memory usage." )
      return None
   return True

def Plugin( ctx ):
   category = [ "ASU", "ASU+", "General" ]
   ctx.addPolicy( checkEos64MemoryUsage, category )
