# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# Note the plugin must be python2/3 compatible due to being installed on older images
from __future__ import absolute_import, print_function

import re

ssuFmt = (
   'Reloading the system with fast-boot is unsupported while there are pending or '
   'errored L1 Profiles to apply.\nPlease configure the following modules to the '
   'current operational profiles before proceeding with fast-boot:\n\n{modules}'
)

downgradeFmt = (
   'Downgrading to a version that does not support the currently configured L1 '
   'Profiles will result in the affected modules returning to their default profile '
   'configuration on boot.\nThe following modules are configured with non-default '
   'profiles:\n\n{modules}'
)

# Amur was the first release that had L1 Profiles infrastructure defined
AMUR_VERSION = "4.29.2F"

# Note that these constant values are directly copied from
# /src/L1Profile/CliPlugin/L1ProfileCliCommon.py rather than imported due to this
# plugin running in the context of the release prior to upgrade which might not have
# any of the types defined for L1 Profiles.
PROFILE_NAME_NA = 'n/a'
PROFILE_STATUS_ERROR = 'error'
PROFILE_STATUS_PENDING = 'pending'
# Represents the statuses that indicate we have not applied the configured profile
PROFILE_STATUS_NOT_APPLIED = ( PROFILE_STATUS_PENDING, PROFILE_STATUS_ERROR )
# The pre-tokenized command to get the profile status that we need to run.
PROFILE_STATUS_CMD_TOKENIZED = ( 'show', 'l1', 'modules', 'profile', 'status' )

# Common helper functions
# =======================
def getProfileStatus( ctx ):
   # To the future readers of this code: forgive me father for I have CLI-ed. The
   # code below relies on us being able to run a "hidden" CLI command to be able to
   # calculate what the current state of the L1 Profiles configuration is, as parsing
   # the raw objects in Sysdb is a rather complex endeavour. This should be safe as
   # we rely on the contract that CLI models can't remove attributes once they've
   # been created to ensure that this API will be stable going forward. However, to
   # perform this feat we need to dig into the CLI mode object to get the current CLI
   # session and mess around with the session a bit to ensure that can get access to
   # the computed CLI model without printing it to the user. If any of this internal
   # CLI infra changes, this plugin will somehow need to adapt to deal with both the
   # previous and new infra designs.
   session = ctx.mode.session

   # We need to cache the current value of whether we expect to print to the CLI or
   # not so that we can restore the value after we modify it below.
   prevShouldPrint = session.shouldPrint()
   try:
      # We don't want to print anything to the CLI while we're psuedo-scraping it
      # here in the reload plugin, so disable all printing temporarily.
      session.shouldPrintIs( False )
      # Note that we must use the "tokenized" run command to be able to get the CLI
      # model back from the session.
      return session.runTokenizedCmd( PROFILE_STATUS_CMD_TOKENIZED, aaa=False )
   except: # pylint: disable=bare-except
      # We're overly cautious here and catch all exceptions to return back that we
      # don't support profiles if we weren't able to read the status for any reason.
      return None
   finally:
      # Ensure we always restore the shouldPrint configuration before leaving
      session.shouldPrintIs( prevShouldPrint )

def getVerNums( ver ):
   '''Given a version string, parse the string and return a list of version numbers
   in the string.
   Arguments:
      ver: (str) EOS version string

   Returns:
      (list of int) EOS version numbers
   '''
   return [ int( n ) for n in re.split( r'(\d+)', ver ) if n.isdigit() ]

def reloadIsDowngrade( ctx ):
   currentVersion = getVerNums( ctx.currentVersion.version() )
   nextVersion = getVerNums( ctx.nextVersion.version() )

   # We want a strict less-than since if they're equal its self-to-self
   return nextVersion < currentVersion

# Check implementations
# ====================

def checkL1ProfileSsuImpact( ctx ):
   # Step 0: Fetch the current L1 Profile status from the CLI
   # ===============================================================================
   profileStatus = getProfileStatus( ctx )

   # Step 1: Determine if the product supports L1 Profiles at all. If not, then we
   #         can't possibly affect the reload's up path.
   # ===============================================================================
   if not profileStatus:
      return None # Report this plugin as not even running at all

   # Step 2: Determine if we have any profiles pending application. If we don't have
   #         any then there won't be any impact on reload so we can just let the
   #         reload go through. Note that for SSU we don't need to worry about
   #         downgrades so we don't have to worry about losing support for a
   #         configured profile at all.
   # ===============================================================================
   pendingModules = sorted( module for module, moduleInfo
                            in profileStatus.modules.items()
                            if moduleInfo.status in PROFILE_STATUS_NOT_APPLIED )
   if not pendingModules:
      return True

   # Step 3: If we have a pending profile and are in SSU ( which this function
   #         assumes ), then we need to completely block SSU from proceeding
   #         otherwise we'll break things on the startup path.
   # ===============================================================================
   ctx.addError( ssuFmt.format( modules=', '.join( pendingModules ) ) )

   return False

def checkL1ProfileDowngradeImpact( ctx ):
   # Step 0: Fetch the current L1 Profile status from the CLI
   # ===============================================================================
   profileStatus = getProfileStatus( ctx )

   # Step 1: Determine if the product supports L1 Profiles at all. If not, then we
   #         can't possibly affect the reload's up path.
   # ===============================================================================
   if not profileStatus:
      return None # Report this plugin as not even running at all

   # Step 2: Determine if we're downgrading the release. If we aren't then there is
   #         no need to block the reload as we can't remove support for a profile
   #         once it has shipped.
   # ===============================================================================
   if not reloadIsDowngrade( ctx ):
      return True

   # Step 3: Determine if we've configured any non-default profiles on modules. If we
   #         haven't, then its safe to upgrade as we always support the default
   #         profile on all versions that support L1 Profiles, and for those that
   #         don't this will be identical to the initial shipping configuration.
   # ===============================================================================
   configuredModules = sorted( module for module, moduleInfo
                               in profileStatus.modules.items()
                               if moduleInfo.configured )
   if not configuredModules:
      return True

   # Step 4: If there are modules with configured profiles, we want to warn the user
   #         about the downgrade as we don't know if the configured profile will
   #         exist on the old release. If they know the profile exists they can
   #         override the warning and proceed at their own risk.
   # ===============================================================================
   ctx.addWarning( downgradeFmt.format( modules=', '.join( configuredModules ) ) )

   return False

def Plugin( ctx ):
   # L1 Profiles was first developed and released in Amur, so any source release
   # prior to that doesn't have any possible way to affect reloads.
   if getVerNums( ctx.currentVersion.version() ) < getVerNums( AMUR_VERSION ):
      return

   # For ASU cases we need to check to see if we have any pending configurations
   ctx.addPolicy( checkL1ProfileSsuImpact, [ 'ASU', 'ASU+' ] )

   # For cold boot we need to check to see if we are downgrading with non-default
   # profiles configured that might not be supported on the old release.
   ctx.addPolicy( checkL1ProfileDowngradeImpact, [ 'General' ] )
