# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=raise-missing-from
# pylint: disable=consider-using-f-string
from __future__ import absolute_import, division, print_function
from types import MethodType, FunctionType

from PyRibAmiClient import sendAmiDgetRequest
from PyRibAmiClient import readAmiDgetResponse
from PyRibAmiClient import cleanupAmiClient
from PyRibAmiClient import handleDamiCommand
from PyRibAmiClient import SocketException
from PyRibAmiClient import AmiRequestException
from PyRibAmiClient import AmiInputParamsException
from PyRibAmiClient import AmiResponseException
from PyRibAmiClient import AmiServerConnectException
from PyRibAmiClient import CBindingException
from PyRibAmiClient import KeyboardInterruptException

from CliModel import _AttributeType
from CliModel import Dict
from CliModel import GeneratorDict
from CliModel import GeneratorList
from CliModel import List
from CliModel import Model
from CliModel import Submodel
from CliParser import InformationalShowCmdError, InvalidInputError
from RibAgent import RIB_AGENT_CTX
from IsisAgt import ISIS_AGENT_CTX
from OspfAgt import OSPF_AGENT_CTX
from Ospf3Agt import OSPF3_AGENT_CTX
from BasicCliSession import setCommandHandlerCleanupCallback
import Tracing
import time
from datetime import datetime
from ArnetModel import IpGenericAddress

traceHandle = Tracing.Handle( 'RibCapiLib' )
t2 = traceHandle.trace2 # Errors
t4 = traceHandle.trace4 # Model Info
t5 = traceHandle.trace5 # Info
t6 = traceHandle.trace6 # AMI Library
t7 = traceHandle.trace7 # Verbose Debug
t8 = traceHandle.trace8 # Function calls

def tException( name, exception ):
   t2( name, 'Caught', type( exception ).__name__, exception )

# ribd End of Message indicator
EOM = -1

class EmptyResponseException( Exception ):
   pass

class MissingMioAttrIdException( Exception ):
   pass

class ModelContext( object ): # pylint: disable=useless-object-inheritance
   '''
   Class to capture the parent-child dependency between cli models

   This class would effectively represent a tree of models, with the
   topLevel model being the vertex of the tree. From any given node
   in the tree we can traverse all the child nodes.

   For any given command the tree is populated using the parseModel()
   '''
   def __init__( self, level, modelClass, parentAttrName, collectionType, mioId ):
      # Level of the node, used in getResponseForLevel()
      self.level = level
      # The MIO_DGET_XXX value of the node. Used to differentiate between multiple
      # child nodes for a parent
      self.mioAttrId = mioId
      # The CliModel class, used to instantiate the model in createGenerator()
      self.cliModelClass  = modelClass
      # The attribute name in the Parent class, used to pass the generated handle
      self.parentAttrName = parentAttrName
      # The collection type in parent class, dictionary / list used to return
      # the key, model pair
      self.parentAttrCollType = collectionType
      self.isDictionary = isinstance( collectionType, ( Dict, GeneratorDict ) )

      # The attrributes in the node
      self.modelAttrs = []
      self.processData = False
      self.overrideHierarchy = False

      # The child Models for this model. This would correspond  to
      # childGenModel - GeneratorDict / GeneratorList
      # childItrModel - Lists / Dicts
      # childSubmodel - Submodels
      self.childGenModel = []
      self.childItrModel = []
      self.childSubmodel = []
      self.modelAccumulate = False

   def modelClassName( self ):
      return '{}.{}'.format( self.cliModelClass.__module__,
                             self.cliModelClass.__name__ )

   def prints( self ):
      '''
      Print the model tree ( for debugging )
      '''
      t4( "  " * self.level,
          "Level:", self.level, "CliModel Class:", self.cliModelClass,
          "Genrator Child nodes:",  "Yes" if self.modelAccumulate else "No" )
      t4( "  " * self.level,
          "Parent Attr Name:", self.parentAttrName, "Parent Collection Type:",
          self.parentAttrCollType, "MioId:", self.mioAttrId )
      t4( "  " * self.level, self.modelAttrs )

      t4( "  " * self.level,
          "Number of generated child nodes:", len( self.childGenModel ) )
      if self.childGenModel:
         t4( "" )
         for child in self.childGenModel:
            child.prints()

      t4( "  " * self.level,
          "Number of iterator child nodes:", len( self.childItrModel ) )
      if self.childItrModel:
         t4( "" )
         for child in self.childItrModel:
            child.prints()

      t4( "  " * self.level,
          "Number of sudmodel child nodes:", len( self.childSubmodel ) )
      if self.childSubmodel:
         t4( "" )
         for child in self.childSubmodel:
            child.prints()

class ResponseBuffer( object ): # pylint: disable=useless-object-inheritance
   ''' Wrapper class to cache the ami response.'''
   def __init__( self ):
      self.response = None

def CapiAssert( cond, reason="" ):
   if not cond:
      t2( "Capi assert:", reason )
      cleanupRibCapi()
      assert False, reason

def _sendRequest( mode, command, args, highPerf, clientName,
                  agentCtx=RIB_AGENT_CTX ):
   '''
   Wrapper function for sendAmiDgetRequest
   '''
   # if args is an empty sequence, pretend it doesn't exist
   if not args:
      args = None

   t6( 'sendAmiDgetRequest() with command:', command, 'args:', args,
       'highPerf', highPerf, 'agentCtx:', agentCtx )

   # sendAmiDgetRequest will delete vrfName from args,
   # so we are storing vrfName here to be used in the warning later
   vrfNameString = " vrf-%s" % ( args[ "vrfName" ] ) \
                   if args and ( "vrfName" in args ) else ""

   try:
      sendAmiDgetRequest( command, args, highPerf, agentCtx )
   except ( SocketException, AmiRequestException, CBindingException ) as e:
      tException( 'sendAmiDgetRequest()', e )
      raise
   except AmiInputParamsException as e:
      tException( 'sendAmiDgetRequest()', e )
      raise InvalidInputError( ": " + str( e ) )
   except AmiServerConnectException as e:
      tException( 'sendAmiDgetRequest()', e )
      # This can happen if we issue a show command on a VRF and
      # Rib-vrf isn't running. Raise an EmptyResponseException
      # as it is more gracefully handled by all callers of
      # showRibCapiCommand() (mio-relay path would also raise
      # the empty response exception). Just log an error to the
      # mode as-well
      if mode:
         errorMsg = clientName + vrfNameString + ' inactive'
         mode.addWarning( errorMsg )
      raise EmptyResponseException
   except KeyboardInterruptException as e:
      tException( 'sendAmiDgetRequest()', e )
      cleanupRibCapi()
      raise KeyboardInterrupt

def _readResponse():
   '''
   Wrapper function for readAmiDgetResponse
   '''
   t6( 'readAmiDgetResponse()' )
   try:
      return readAmiDgetResponse()
   except ( SocketException, AmiResponseException, CBindingException ) as e:
      tException( 'readAmiDgetResponse()', e )
      raise
   except KeyboardInterruptException as e:
      tException( 'readAmiDgetResponse()', e )
      cleanupRibCapi()
      raise KeyboardInterrupt
   return None

def cleanupRibCapi():
   '''
   Wrapper function for cleanupAmiClient
   '''
   t6( 'cleanupAmiClient()' )
   try:
      cleanupAmiClient()
   except ( SocketException, AmiResponseException, CBindingException ) as e:
      tException( 'cleanupAmiClient()', e )
      raise

def _handleInitialResponse( ):
   '''
   Helper function to read the toplevel response
   '''
   t8( '_handleInitialResponse()' )
   response = _readResponse()

   if not response:
      # Response is completed for some reason!
      cleanupRibCapi()
      raise EmptyResponseException()

   return response

def showRibCapiCommand( mode, topModel, command, args=None, highPerf=False,
                        clientName='Rib Agent', skipNullResponses=True,
                        l3Config=None ):
   '''
   topModel - The top level model which is expected by the first AMI response
   command - The string value of the MIO_DGET command
   args - A dictionary of arguments which is used in sendAmiDgetRequest. Each
      possible argument is listed in the parameter vtable inside gated-ctk.
      There is a special case vrfName used for vrfs and will never be a part of
      the vtable.
   highPerf = Set to True if high performance is desired for this show command --
      RibCapi client will connect to port 9889 on the server that will trigger
      spawning another ribd process to handle the show command
   clientName - Name of the Agent used to call this function
   skipNullResponses - Originally the RibCapi module did not support NULL vtables,
      but some commands need NULL vtables. This flag should be set to FALSE, if there
      are NULL vtables in the response at the topLevel that should not be ignored.
   l3Config - l3/config/protocolAgentModel's value (multi-agent/ribd) is used select
      select the port on which agent (clientName) is listening and our CLI instance
      should connect to.

   This is the entry point for the RibCapi commands which will send the
   corresponding dget request to ribd and return the appropriate populated
   CAPI model used for displaying Cli/JSON output.
   See AID 1726 for more info
   '''
   t8( 'showRibCapiCommand() with topModel:',
       '{}.{}'.format( topModel.__module__, topModel.__name__ ),
       'command:', command, 'args:', args, 'highPerf:', highPerf,
       'clientName:', clientName, 'skipNullResponses:', skipNullResponses )
   topLevelOffset = 1

   assert issubclass( topModel, Model )
   assert isinstance( command, str )
   assert args is None or isinstance( args, dict )

   agentCtx = RIB_AGENT_CTX
   igpAgentToCtx = { "ISIS" : ISIS_AGENT_CTX,
                     "OSPF" : OSPF_AGENT_CTX,
                     "OSPF3" : OSPF3_AGENT_CTX }
   if clientName in igpAgentToCtx:
      assert l3Config or args.get( 'sham_link', None ),\
         "l3Config is needed to distinguish between " \
         "ribd/multi-agent world"
      if args.get( 'sham_link', None ) or\
            l3Config.protocolAgentModel == "multi-agent":
         agentCtx = igpAgentToCtx[ clientName ]

   t7( 'Setting up cleanup handler for CLI' )
   setCommandHandlerCleanupCallback( cleanupRibCapi )

   t7( 'Send request message to ribd to begin' )
   _sendRequest( mode, command, args, highPerf, clientName, agentCtx )

   t7( 'Get initial response' )
   response = _handleInitialResponse()

   if skipNullResponses:
      t7( 'Skipping initial null responses' )
      # Our top level reponse could be a NULL table, incase the
      # model does not expect any NULL tables, keep reading till
      # we get the first non-NULL vtable response
      while not 'data' in response:
         response = _handleInitialResponse()
      topLevelOffset = response[ 'level' ]

   t7( 'Generate the Model Context tree' )
   modelCtxs = parseModel( topLevelOffset, topModel, "TopLevelModel" )
   modelCtxs.prints()

   responseBuffer = ResponseBuffer()

   try:
      # We support two modes:
      # 1) Read all the responses from gated, populate the complete
      #    model & child models and return. This approach is ok for
      #    smaller response (populateFullModel).
      # 2) Read the responses from gated in chunks, this returns
      #    a toplevel model with python generators for each child node.
      #    This is the typical use-case, and the more efficient
      #    of the two approaches (populateModel).
      # We decide between the modes, based on if the first child model
      # is a GeneratedList/GeneratedDict or not. If we do have a
      # GeneratedList/Dict child node we use approach 2.
      if not modelCtxs.childGenModel:
         model = populateFullModel( response, modelCtxs, responseBuffer )
      else:
         _, model, _ = populateModel( response, modelCtxs, responseBuffer )
   except:
      cleanupRibCapi()
      raise

   return model

def parseModel( level, modelType, modelName, collectionType=None, mioId=0 ):
   '''
   The function recursively parses a model and generated a model context tree.
   Each node in the tree respresents a Cli Model, and stores all the information
   required to populate / process the model.
   - childGenModel - A list of all GeneratorDicts / GeneratorLists in the model
   - childItrModel - A list of all List / Dict in the model
   - childSubmodel - A lits of submodels in the model

   level - The level of the top model
   modelType - CliModel.Model subclass type
   modelName - The attributeName in the parent class. Will need it to set the
      generators correctly.
   collectionType - The collectionTYpe in the parent Model. This is required to
      decide between returning key, model / or just the model
   mioId - The MIO_DGET_XXX value of the child node, used to differentiate between
      the child nodes for the same parent.
   '''
   t8( 'parseModel() with level:', level, 'modelType:', modelType.__name__,
       'model:', modelName, 'collectionType:', collectionType, 'mioId:', mioId )

   # Create a instance of ModelContext, and populate with available info
   currentModel = ModelContext( level, modelType , modelName, collectionType, mioId )
   attrs = dir( modelType )

   # Populate the child nodes and other attributes in the model
   # The sorted.. code ensures that we iterate through the
   # attributes in the order that they are defined in the class
   # This ordering is important to ensure the Json conversion
   # and submodel support work as expected
   for fieldName, field in modelType.__attributes__.items():

      # Check if field is a CliModel collection, if yes generate the tree
      # for it as well, and add to the childModel list
      if ( isinstance( field, ( GeneratorDict, GeneratorList ) ) and
           issubclass( field.valueType, Model ) ):
         currentModel.childGenModel.append(
            parseModel( level + 1, field.valueType, fieldName, field, mioId )
         )

      # Get the list of simple Lists/Dicts for the model and generate the subtree
      # for them
      elif ( isinstance( field, ( List, Dict ) ) and
             issubclass( field.valueType, Model ) ):
         currentModel.childItrModel.append(
            parseModel( level + 1, field.valueType, fieldName, field, mioId )
         )

      # Get the list of submodels for the Model and generate its subtree
      elif isinstance( field, Submodel ):
         currentModel.childSubmodel.append(
            parseModel( level + 1, field.valueType, fieldName, field, mioId )
         )
         # Submodel is a special case and needs to be present in attribute list as
         # well for backward compatability
         currentModel.modelAttrs.append( fieldName )

      # Add the relevant attributes to a list
      elif isinstance( field, _AttributeType ):
         currentModel.modelAttrs.append( fieldName )


   for fieldName in attrs:
      field = getattr( modelType, fieldName )
      fieldType = type( field )
      # Get the mioAttrId value for the Model
      if fieldName == "getMioAttrId":
         CapiAssert( ( fieldType ) in [ MethodType, FunctionType ],
         '"getMioAttrId" in %s must be a function' % modelName )
         currentModel.mioAttrId = modelType.getMioAttrId( modelType() )

      # Check if overrideHierarchy is there, and if it is there it is a function
      # For py2, value of fieldType is 'instancemethod' whereas for py3, its value is
      # 'function'. Adding a check for both values for compatibility with both py2/3.
      # We can consider removing MethodType once we fully cutoff py2 support.
      elif fieldName == "overrideHierarchy":
         CapiAssert( ( fieldType ) in [ MethodType, FunctionType ],
         '"overrideHierarchy" in %s must be a function' % modelName )
         currentModel.overrideHierarchy = True

      # Check if processData is there, and if it is there it is a function
      elif fieldName == "processData":
         CapiAssert( ( fieldType ) in [ MethodType, FunctionType ],
         '"processData" in %s must be a function' % modelName )
         currentModel.processData = True

   # Some sanity checks / limitation checks for the model
   # Check we have a mioId for the child nodes, if we have more than 1 child node
   if ( len( currentModel.childGenModel ) >= 2 and not
        currentModel.overrideHierarchy ):

      for childCtx in currentModel.childGenModel:
         if childCtx.mioAttrId == 0:
            cleanupRibCapi()
            errorString = "Missing information for model %s" % \
                          childCtx.modelClassName()
            raise MissingMioAttrIdException( errorString )

   # Submodel support:
   # A submodel typically is populated by the parent model.
   # ** But ** If the submodel does come in a separate response from
   # gated. And it satisfies the below conditions, then the RibCapiLib.py
   # can try and populate the submodel ( this could avoid some code in
   # a few models)
   # A submodel can/will be populated by RibCapiLib if:
   # - It does not have any generated child nodes (no GeneratorDicts/ Lists)
   # - It has the function getMioAttrId() defined ( this is to support
   #   backward compatability, older models that have Submodels in them
   #   would not have getMioAttrId defined, hence will work without any change )
   # - OverrideHiearachy should not be set. OverrideHiearachy means the parent
   #   model knows what it is doing, and will take care of the submodels
   # - All its child nodes satisfy the above condition ( this would basically
   #   imply that, for RibCapiLib to resolve a submodel, no node under the submodel
   #   hierarchy should have GeneratedDicts / GeneratedList )
   #
   # ** There is however a caveat to this. In case a very huge node ( potentially
   #    running to 1000s of gated messages) is made a submodel, the benfit of
   #    using RibCapiLib gets beaten. For example: A single LSA in case of Ospf3
   #    is a good candidate for making a submodel. But the whole Area hierachy in
   #    output is not a good candidate for making it a submodel (it could potentialy
   #    run into a 1000s of responses from gated). The area hierarchy would be
   #    better as a GeneratorList(), with a length of 1.
   modelAccumulate = True
   if currentModel.childGenModel:
      modelAccumulate = False
   for childCtx in currentModel.childItrModel:
      if not childCtx.modelAccumulate:
         modelAccumulate = False
   for childCtx in currentModel.childSubmodel:
      if not childCtx.modelAccumulate:
         modelAccumulate = False
   currentModel.modelAccumulate = modelAccumulate

   return currentModel

def createGenerator( modelInfo, responseBuffer ):
   '''
   This is python generator function. It reads a response from gated,
   and tries to map the response to the modelInfo.cliModelClass.
   The function gets responses from gated which match modelInfo.level
   and modelInfo.mioAttrId, but if overrideHierarchy is set it reads
   any response and passes the same to the model.
   The function calls populateModel to do the actual work of setting the
   model attributes etc.

   modelInfo - The generator will return python models corresponding to
               modelInfo.cliModelClass
   responseBuffer - Buffer used to cache ami response
   '''
   key = None
   response = True

   level = modelInfo.level
   mioAttrId = modelInfo.mioAttrId

   # Check if overrideHierarchy is set, in which case we would need
   # to ignore the level / mioId information for the response
   overrideHierarchy = modelInfo.overrideHierarchy

   traceHeader = 'createGenerator( {} level={} mioAttrId={} overrideHierarchy={} )' \
         .format( modelInfo.modelClassName(), level, mioAttrId, overrideHierarchy )
   t8( traceHeader )

   try:
      while response:
         t7( traceHeader, 'Looking for a new response' )
         response = getResponseForLevel( level, mioAttrId, responseBuffer,
                                         overrideHierarchy )
         if response is None:
            t7( traceHeader, 'No valid response, completed generator' )
            return

         t7( traceHeader, 'Populating model with response' )
         key, model, readNext = populateModel( response, modelInfo, responseBuffer )
         prevKey = key
         prevModel = model

         t7( traceHeader, 'populateModel() returned key:', key,
             'model:', type( model ).__name__, 'and readNext:', readNext )
         # If the model has overrideHierarchy set, then the model will take care of
         # populating the child node, we just need to provide the response to model
         while readNext:
            t7( traceHeader, 'Reading next model for overrideHierarchy' )
            response = getResponseForLevel( level, mioAttrId, responseBuffer,
                                            overrideHierarchy )

            # Either exception or no more responses
            if response is None: # pylint: disable=no-else-break
               t7( traceHeader, 'No valid response during readNext, breaking' )
               break
            # Skip any null vtables if any
            elif 'data' not in response:
               t7( traceHeader, 'Found a null vtable, skipping' )
               continue
            else:
               t7( traceHeader, 'Populating model with a readNext response' )
               key, model, readNext = populateModel( response, modelInfo,
                                                     responseBuffer, prevModel,
                                                     readNext )

            if model != prevModel:
               t7( traceHeader, 'Yielding prevModel', type( prevModel ).__name__ )
               if key is not None:
                  yield prevKey, prevModel
               else:
                  yield prevModel
               prevKey = key
               prevModel = model

         if model:
            t7( traceHeader, 'Yielding model', type( model ).__name__ )
            if key is not None:
               yield key, model
            else:
               yield model
         else:
            return
   except Exception as e:
      tException( traceHeader, e )
      cleanupRibCapi()
      raise

def getResponseForLevel( level, mioDgetType, responseBuffer,
                         overrideHierarchy=False ):
   '''
   Given a level and mioAttrId, return a response corresponding to the
   same if available else return None.

   - If the mioDgetType is 0, we do not check for it. This is to simplify
     the model for cases where there is only 1 child node

   responseBuffer - Buffer used to cache ami response
   '''
   traceHeader = 'getResponseForLevel( level={}, mioDgetType={}, ' \
                 'overrideHierarchy={} )'.format( level, mioDgetType,
                                                  overrideHierarchy )
   t8( traceHeader )

   respType = 0
   response = None

   while not response:
      try:
         if responseBuffer.response is not None:
            t7( traceHeader, 'Taking response from buffer' )
            response = responseBuffer.response
         else:
            response = _readResponse()
            t7( traceHeader, 'Received response:', response )
      except Exception as e:
         tException( traceHeader, e )
         cleanupRibCapi()
         raise

      if not response:
         t7( traceHeader, 'No valid response, returning' )
         return None

      respLevel = response[ 'level' ]
      if 'miotype' in response:
         respType = response[ 'miotype' ]
      t7( traceHeader, 'Response is level:', level, 'miotype:', respType )

      # Check if the level and mioId match for the response
      # pylint: disable-next=consider-using-in
      if respLevel == level and ( mioDgetType == 0 or respType == mioDgetType ):
         if mioDgetType:
            t7( traceHeader,
                'Correct level and matching miotype, returning response' )
         else:
            t7( traceHeader, 'Same level and ignoring miotype, returning response' )
         responseBuffer.response = None
         return response
      # If overrideHierarchy, ignore the level mismatch and return response
      elif overrideHierarchy and respLevel > level:
         t7( traceHeader, 'Overriding hierarchy and response level is higher '
             'than current level, returning response' )
         responseBuffer.response = None
         return response
      # Assuming the user forgets to render for some data at Level X,
      # lets not just completely stall. If we get a request for
      # response level Y, but have a response at level X such that
      # X > Y, lets just ignore the response
      elif level < respLevel:
         t7( traceHeader, 'Not overriding hierarchy and got a higher response '
             'level, deleting response.' )
         responseBuffer.response = None
         response = None
         t5( "Deleting response as there is a request for a lower level,",
             "Response level:", respLevel, "Requested level:", level, response )
         continue
      # No matching response for level and mioId, return none
      else:
         t7( traceHeader, 'Response level is not valid for current level, '
             'buffering response and returning None' )
         responseBuffer.response = response
         return None

def populateModel( response, modelInfo, responseBuffer, prevModel=None,
                   readNext=False ):
   '''
   This function returns a instance of modelInfo.cliModelClass. It populates
   the fields in the class, based on response dictionary. The genrators
   for the child nodes are also created.

   response - The python dictionary returned by a call to readAmiDgetResponse()
            Python dictionary is of the form { level:x , mioAttrId:y ,
            data:{class attributes}}. The data dictionary is further used to
            populate the fields.
   modelInfo - An instance of ModelContext, which has all the data
               required for instantiating a object for the model
   responseBuffer - Buffer used to cache ami response
   prevModel / readNext - Are used in cases the model has 'overrideHierarchy'
            attribute set, indicating it will handle the messages. prevModel
            provides the context, readNext indicates if more messages are expected.
   '''

   traceHeader = 'populateModel( model={} )'.format( modelInfo.modelClassName() )
   t8( traceHeader )

   # Handle NULL vtable case
   if not 'data' in response:
      t7( traceHeader, 'Response is a null vtable' )
      data = {}
   else:
      t7( traceHeader, 'Using response data', response )
      data = response[ 'data' ]

   t5( 'Populating model', modelInfo.modelClassName(), 'with', response )

   cliModelClass = modelInfo.cliModelClass
   attrs = modelInfo.modelAttrs

   # Create an instance of the class
   currentCliModel = None
   if not readNext:
      t7( traceHeader, 'Creating new model' )
      currentCliModel = cliModelClass()
   else:
      t7( traceHeader, 'Skipping creation, readNext is True' )

   # If the model is a GeneractorDict in its parent, get the key
   key = None
   if  modelInfo.isDictionary:
      if prevModel:
         key = prevModel.getKey( data )
      else:
         key = currentCliModel.getKey( data )
      t7( traceHeader, 'Model is in a dict, key is', key )
      # must be explicit to avoid asserting 0
      CapiAssert( key is not None, 'A key must be returned in getKey() for %s'
                  % type( currentCliModel ).__name__ if currentCliModel else '' )

   # Handle overrideHierarchy:
   # the readNext and prevModel inputs are specifically for this case.
   # readNext indicates, that the model expects more data
   # prevModel information is required to get the correct context
   if modelInfo.overrideHierarchy:
      t7( traceHeader, 'Currently overriding hierarchy' )
      if readNext:
         CapiAssert( prevModel is not None,
                     'Model should exist to add next list item' )
         t7( traceHeader,
             'readNext is True, calling overrideHierarchy with prevModel',
             type( prevModel ).__name__ )
         ( data, readNext ) = prevModel.overrideHierarchy( data )

         if not readNext:
            t7( traceHeader, 'Completed overriding previous model, '
                'applying data with a new model' )
            currentCliModel = cliModelClass()
            ( data, readNext ) = currentCliModel.overrideHierarchy( data )
         else:
            t7( traceHeader, 'prevModel requires more data' )
            currentCliModel = prevModel
      else:
         t7( traceHeader,
             'readNext is False, calling overrideHierarchy with currentCliModel' )
         ( data, readNext ) = currentCliModel.overrideHierarchy( data )

      t7( traceHeader, 'overrideHierarchy returned readNext:', readNext )
   elif modelInfo.processData:
      t7( traceHeader, 'Calling the model\'s processData()' )
      data = currentCliModel.processData( data )

   if not modelInfo.overrideHierarchy:
      t7( traceHeader, 'Creating generators for all generator children' )
      for child in modelInfo.childGenModel:
         modelList = createGenerator( child, responseBuffer )
         setattr( currentCliModel, child.parentAttrName, modelList )

      for child in modelInfo.childItrModel:
         if isinstance( child.parentAttrCollType, List ):
            modelList = data.get( child.parentAttrName, [] )
         else:
            modelList = data.get( child.parentAttrName, {} )
         setattr( currentCliModel, child.parentAttrName, modelList )
   else:
      t7( traceHeader, 'Skipping generator creation, model is '
          'overriding hierarchy' )

   t7( traceHeader, 'Copying over attributes in whatever is left of data' )
   for fieldName in attrs:
      # A simple CliModel attribute will assume the attribute will be in the
      # data dictionary
      if data and fieldName in data:
         value = data[ fieldName ]
         if isinstance( getattr( cliModelClass, fieldName ), IpGenericAddress ):
            # Handle any address with possible link-local data
            scoped = value.split( '%' )
            value = scoped[ 0 ]
         setattr( currentCliModel, fieldName, value )

   # Check if submodels can be resolved (auto-populated)
   if modelInfo.childSubmodel \
      and not modelInfo.overrideHierarchy:
      t7( traceHeader, 'Model has children that are Submodels, '
          'populate them recursively' )
      for child in modelInfo.childSubmodel:
         if child.modelAccumulate and child.mioAttrId:
            t7( traceHeader, 'Child model accumulate is True, populating submodel',
                child.modelClassName() )
            response = getResponseForLevel( child.level, child.mioAttrId,
                                            responseBuffer )
            if response:
               t7( traceHeader, 'Response found, accumulating response for '
                   'submodel:', child.modelClassName() )
               _, submodel = accumulateResponse( response, child, responseBuffer )
               setattr( currentCliModel, child.parentAttrName, submodel )

   return key, currentCliModel, readNext


def populateFullModel( response, modelCtxs, responseBuffer, cleanup=True ):
   t8( 'populateFullModel()' )
   _, model = accumulateResponse( response, modelCtxs, responseBuffer )

   # If there are no child nodes the sockets etc wouldn't have been cleaned
   # up, so clean it up now. If there are child nodes the sockets would have
   # been cleaned up as we keep reading till we get a empty response.
   if not modelCtxs.childItrModel:
      t7( 'populateFullModel() No more child nodes, clean up socket' )
      cleanupRibCapi()
   return model

def accumulateResponse( response, modelCtxs, responseBuffer ):
   '''
   response - The first response read from ribd
   modelCtxs - The dictionary of ModelContexts which contains the information
   about model hierarchy
   responseBuffer - Buffer used to cache ami response

   If the models are using regular collection CAPI types and the values for the
   model hierarchy instead of generators, then all responses from ribd will be
   read and the full response will be returned.
   '''
   traceHeader = 'accumulateResponse( model={} )' \
         .format( modelCtxs.modelClassName() )
   t8( traceHeader )

   topKey, topModel, readNext = populateModel( response, modelCtxs, responseBuffer )

   # Handle overrideHierarchy case, there is no need to check for child
   # models if overrideHierarchy is set.
   if readNext:
      t7( traceHeader, 'Model is overriding hierarchy' )
      while readNext:
         response = getResponseForLevel( modelCtxs.level, modelCtxs.mioAttrId,
                                         responseBuffer,
                                         modelCtxs.overrideHierarchy )
         if not response:
            t7( traceHeader, 'No more responses, returning model' )
            return topKey, topModel

         t7( traceHeader, 'Since it is overriding, push the next response into '
             'the model' )
         topKey, topModel, readNext = populateModel( response, modelCtxs,
                                                     responseBuffer, topModel,
                                                     readNext )
      t7( traceHeader, 'Override hierarchy complete, returning model' )
      return topKey, topModel

   t7( traceHeader, 'Populating the next level of models' )
   # Finished populating the model, now check if we have any child models
   for childCtx in modelCtxs.childItrModel:
      traceSubheader = '{} child={}'.format( traceHeader, childCtx.modelClassName() )
      t7( traceSubheader, 'Populating child model' )
      while True:
         t7( traceHeader, 'Getting next response' )
         response = getResponseForLevel( childCtx.level, childCtx.mioAttrId,
                                         responseBuffer )

         if response is None:
            t7( traceHeader, 'No more responses, breaking' )
            break

         t7( traceHeader, 'Recursive accumulateResponse()' )
         key, submodel = accumulateResponse( response, childCtx, responseBuffer )

         # Grab the correct model
         # We now have the child model, we need to set the correct
         # attribute ( list / dictionary ) in the parent object
         t7( traceHeader, 'Adding child model into parent' )
         parentAttrName = childCtx.parentAttrName
         modelCollection = getattr( topModel, parentAttrName )

         if isinstance( childCtx.parentAttrCollType, Dict ):
            modelCollection[ key ] = submodel
         else:
            modelCollection.append( submodel )
         setattr( topModel, parentAttrName, modelCollection )

   t7( traceHeader, 'Completed' )
   return topKey, topModel

# pylint: disable-next=useless-return
def showRibDamiCommand( printTo, command, args=None, clientName='Rib Agent',
                        l3Config=None ):
   t8( 'showRibDamiCommand() with printTo:', printTo, 'command:', command,
       'args:', args, 'clientName:', clientName, 'l3Config:', l3Config )
   agentCtx = RIB_AGENT_CTX
   igpAgentToCtx = { "ISIS" : ISIS_AGENT_CTX,
                     "OSPF" : OSPF_AGENT_CTX,
                     "OSPF3": OSPF3_AGENT_CTX }
   if clientName in igpAgentToCtx:
      assert l3Config, "l3Config is needed to distinguish between " \
         "ribd/multi-agent world"
      if l3Config.protocolAgentModel == "multi-agent":
         agentCtx = igpAgentToCtx[ clientName ]

   t6( 'handleDamiCommand with command:', command, 'args:', args,
       'printTo:', printTo, 'agentCtx:', agentCtx )

   try:
      handleDamiCommand( command, args, printTo, agentCtx )
   except ( SocketException, AmiRequestException, CBindingException ) as e:
      tException( 'showRibDamiCommand()', e )
      raise
   except AmiInputParamsException as e:
      tException( 'showRibDamiCommand()', e )
      raise InvalidInputError( ": " + str( e ) )
   except AmiServerConnectException as e:
      tException( 'showRibDamiCommand()', e )
      raise InformationalShowCmdError( clientName + ' inactive' )
   except AmiResponseException as e:
      tException( 'showRibDamiCommand()', e )
      raise InformationalShowCmdError( str( e ) )
   except KeyboardInterruptException as e:
      tException( 'showRibDamiCommand()', e )
      raise KeyboardInterrupt

   return None

def timestampToLocalTimeMsec( timeInSeconds ):
   utcTime = datetime.fromtimestamp( timeInSeconds )
   localTime = utcTime.strftime( '%H:%M:%S.%f' )
   return localTime[:-3]

def timestampToLocalTime( timeInSeconds ):
   localTime = time.strftime( '%H:%M:%S', time.localtime( ( timeInSeconds ) ) )
   return localTime

