#!/usr/bin/env python3
# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=no-member

from __future__ import absolute_import, division, print_function
import Tac, os, sys, errno
import subprocess
import CmdExtension
from Arnet.NsLib import DEFAULT_NS, runMaybeInNetNs
from IpLibConsts import DEFAULT_VRF
# pylint: disable-msg=W0611
from ProtoEnums import protoDict, enumDict

#-------------------------------------------------------------------------------
# For very simple show commands that can safely use cliribd output,  just spawn
# cliribd.  When cliribd is run in this way, it doesn't try to page its output,
# so we don't have to try any kind of fancy shenanigans.
# If ribd is not yet running, this will fail.  We don't want to raise an error, 
# so we simply print a message saying it is inactive.  Customers, however, 
# won't care whether Ribd is inactive --- they care about the "product" from
# which they issued the command (e.g. BGP).  Pass in a client string, used
# when ribd isn't running, to print: "<client name> inactive"
#
# If output=True, we allow cliribd to output to stdout.  Otherwise,
# we gather the output as a string, and return it.  However, if replaceKernelIntfs
# is True, we gather the output as a string and perform some regexp substitutions
# to transform kernel interface names to EOS Cli interface names.
#-------------------------------------------------------------------------------
from DeviceNameLib import kernelIntfFilter

# pylint: disable-next=inconsistent-return-statements
def cliRibdShowCommand( mode, cmd, clientName="Rib Agent", output=True,
                        replaceKernelIntfs=False, outputFilter=None,
                        skipHeaderLines=0, cport=None, vrf=None,
                        l3Config=None ):
   cliCmdExt = CmdExtension.getCmdExtender()
   # The output of cliribd command is optionally passed through
   # a filter function <outputFilter>, line by line. This is 
   # supposed to be used to replace ribd-specific output with  
   # corresponding EOS-specific output, such as interface names, 
   # interface indices, vifs, etc.
   filters = [ ]
   if replaceKernelIntfs:
      filters.append( kernelIntfFilter )
   if outputFilter:
      filters.append( outputFilter )

   port = None
   multiAgent = None
   igpAgents = [ "ISIS", "OSPF", "OSPF3", "RIP" ]
   if clientName in igpAgents:
      assert l3Config, "l3Config is needed to distinguish between " \
         "ribd/multi-agent world"
      multiAgent = l3Config.protocolAgentModel == "multi-agent"

   if multiAgent:
      cli = 'cli' + clientName.capitalize()
   else:
      cli = 'cliribd'
      port = os.environ.get( 'AMI_SERVER_PORT' )
   args = [ cli, '-q' ]
   if port is not None:
      args += [ '-p', port ]
   elif cport is not None:
      port = cport
      args += [ '-p', port ]
   try:
      if not filters or not output:
         # keep simple case simple, just run the command and return
         if output:
            stdout = sys.stdout
         else:
            stdout = Tac.CAPTURE
         if mode:
            try:
               out = cliCmdExt.runCmd( args, mode.session, vrfName=vrf,
                                       useSudo=False, asRoot=True,
                                       input=cmd, stdin=None, stdout=stdout )
            except EnvironmentError:
               # The vrf may not exist yet.
               return '\n'
         else:
            # used in tests
            nsName = DEFAULT_NS
            if vrf and vrf is not DEFAULT_VRF:
               nsName = "ns-" + vrf
            try:
               out = runMaybeInNetNs( nsName, args, input=cmd, stdout=stdout,
                                      asRoot=True )
            except Tac.SystemCommandError:
               # The namespace or the Rib-vrf may not be ready yet.
               return '\n'

         if out:
            for f in filters:
               out = f( out )
         return out
      else:
         # Execute the command, then read output and run it 
         # through filters, line by line
         # This is partially copied from Tac.run() and partially 
         # from subprocess.communicate().
         try:
            p = cliCmdExt.subprocessPopen( args, mode.session, vrfName=vrf,
                                           stdout=subprocess.PIPE, 
                                           stderr=subprocess.PIPE,
                                           stdin=subprocess.PIPE )
         except ( EnvironmentError ) as e: # pylint: disable=unused-variable
            return '\n'
         if cmd:
            p.stdin.write( cmd )
         p.stdin.close()
         for line, out in enumerate(p.stdout):
            for f in filters:
               out = f( out )
            if not skipHeaderLines:
               sys.stdout.write( out )
            elif line > skipHeaderLines - 1:
               sys.stdout.write( out )
         errout = p.stderr.read()
         for f in filters:
            errout = f( errout )
         p.wait()
         rc = p.returncode
         assert rc is not None # rc must not be None, because p.wait() was called
         if rc:
            e = Tac.SystemCommandError(
                  # pylint: disable-next=consider-using-f-string
                  "%r: error code %d" % ( " ".join(args), rc ) )
            e.error = rc
            e.output = errout
            raise e
   except ( Tac.SystemCommandError, IOError, OSError ) as e:
      if ( isinstance( e, Tac.SystemCommandError )
              and e.error == errno.ECONNREFUSED ) or \
         ( isinstance( e, IOError ) and e.errno == errno.EPIPE ) or \
         ( isinstance( e, OSError ) and e.errno == errno.EPIPE ):
         out = clientName + " inactive"
         if output:
            print( out )
         return out
      else:
         raise

