# Copyright (c) 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
# pylint: disable=consider-using-in
# pylint: disable=unidiomatic-typecheck
# pylint: disable=too-many-nested-blocks
# pylint: disable=simplifiable-if-statement
# pylint: disable=superfluous-parens

import Tac
import re, operator
from enum import Enum
from collections import defaultdict
import Arnet, Logging
from ArnetLib import bgpFormatAsn, asnStrToNum
from ctypes import c_uint32, c_float, Union
from CliParser import safeInt
from CliToken.Community import getCommIntValue
import Assert
from functools import reduce # pylint: disable=redefined-builtin
from TypeFuture import TacLazyType

CommunityValueConst = TacLazyType( 'Acl::CommunityValue', returnValueConst=True )
CommunityValueType = Tac.Type( 'Acl::CommunityValue' )
YangMatchSetOptions = Tac.Type( 'Acl::YangMatchSetOptions' )
systemTunnelRibName = Tac.Type( "Tunnel::TunnelTable::TunnelRibNameIdMap"
                              ).systemTunnelRibName

PREFIXLIST_IMPORT_FAILED = Logging.LogHandle(
   "PREFIXLIST_IMPORT_FAILED",
   severity=Logging.logError,
   fmt="Failed to import %s prefix-list %s from source %s. "
   "Possible reason: %s. Please check source validity.",
   explanation="A prefix-list is set up with an import source URL to load "
   "its set of prefix entries. The source is either a local file or a "
   "remote URL and must be available at the time of import. The entries in "
   "the file must conform to prefix-list entry syntax.",
   recommendedAction="Check the URL and content of the file" )

PREFIXLIST_IMPORT_SUCCEEDED = Logging.LogHandle(
   "PREFIXLIST_IMPORT_SUCCEEDED",
   severity=Logging.logInfo,
   fmt="Imported %s prefix-list %s from source %s.",
   explanation="A prefix-list is set up with an import source URL to load "
   "its set of prefix entries. The source is either a local file or a "
   "remote URL and must be available at the time of import. The entries in "
   "the file must conform to prefix-list entry syntax.",
   recommendedAction="No action required" )

ASPATH_ACCESS_LIST_IMPORT_FAILED = Logging.LogHandle(
   "ASPATH_ACCESS_LIST_IMPORT_FAILED",
   severity=Logging.logError,
   fmt="Failed to import ip as-path access-list %s from source %s. "
   "Possible reason: %s. Please check source validity.",
   explanation="An as-path access-list is set up with an import source URL "
    "to load its list of as-path access-list entries. The source is either a "
    "local file or a remote URL and must be available at the time of import. "
    "The entries in the file must conform to the as-path access-list entry syntax.",
   recommendedAction="Check the URL and content of the file" )

ASPATH_ACCESS_LIST_IMPORT_SUCCEEDED = Logging.LogHandle(
   "ASPATH_ACCESS_LIST_IMPORT_SUCCEEDED",
   severity=Logging.logInfo,
   fmt="Imported ip as-path access-list %s from source %s.",
   explanation="An as-path access-list is set up with an import source URL "
    "to load its list of as-path access-list entries. The source is either a "
    "local file or a remote URL and must be available at the time of import. "
    "The entries in the file must conform to the as-path access-list entry syntax.",
   recommendedAction="No action required" )

def ipCmd( ipv6 ):
   return "ip" if not ipv6 else "ipv6"

#------------------------------------------------------------------------
# Constants
#------------------------------------------------------------------------

# 'match community instances' supported range
POLICY_COMM_INST_MIN = 0
POLICY_COMM_INST_MAX = 1024

# 'match as-path length' supported range
POLICY_ASPATH_LENGTH_MIN = 0
POLICY_ASPATH_LENGTH_MAX = 4000

# 'match isis instance' supported range
POLICY_ISIS_INST_MIN = 0
POLICY_ISIS_INST_MAX = 65535

# 'match ospf instance' supported range
POLICY_OSPF_INST_MIN = 1
POLICY_OSPF_INST_MAX = 65535

# Prefix list maximum allowed sequence number
PREFIX_SEQNO_MAX_VALUE = 0xFFFFFFFF

# Max Character Limit For Remarks
REMARK_CHARACTER_LIMIT = 100

# URL schemes supported for importing prefix-list
listUrlSchemes = { "file:", "flash:", "http:", "scp:", "https:" }

inactiveAclMsg = "! inactive as-path access-lists under current regex-mode"

def addReverseMapping( d ):
   d.update( { v: k for k, v in d.items() } )

bgpOriginEnum = {
   "any" : "bgpAny",
   "egp" : "bgpEGP",
   "igp" : "bgpIGP",
   "incomplete" : "bgpIncomplete",
   }
addReverseMapping( bgpOriginEnum )

matchPermitEnum = {
   "permit" : "permitMatch",
   "deny" : "denyMatch",
   }
addReverseMapping( matchPermitEnum )

matchAcceptEnum = {
   "accept" : "permitMatch",
   "reject" : "denyMatch",
   }
addReverseMapping( matchAcceptEnum )

metricTypeEnum = {
   "type-1" : "extMetricType1",
   "type-2" : "extMetricType2",
   }
addReverseMapping( metricTypeEnum )

isisLevelEnum = {
   "level-1" : "level1",
   "level-2" : "level2",
   "level-1-2" : "level1_2",
   "none" : "levelNone",
}
addReverseMapping( isisLevelEnum )

isisStyleEnum = {
   "wide": "wideMetric",
   "none": "styleUnset",
}
addReverseMapping( isisStyleEnum )

commValueEnum = {
   'no-export' : '4294967041',
   'no-advertise' : '4294967042',
   'local-as' : '4294967043',
   'internet' : '0',
   'GSHUT' : '4294901760',
}
addReverseMapping( commValueEnum )

extCommTypeValueMap = {
   'rtAs' : 0x0002,
   'rtAs4' : 0x0202,
   'rtIp' : 0x0102,
   'sooAs' : 0x0003,
   'sooAs4' : 0x0203,
   'sooIp' : 0x0103,
   'lbw' : 0x0004,
   'lbwAny' : 0x0014,
   'color' : 0x030b,
   'esi': 0x0602,
   'ov': 0x4300,
   'l2VpnIdAs': 0x000a,
   'l2VpnIdIp': 0x010a,
   'router-id': 0x0107,
}

colorOnlyFlags = {
   'coMask': 0b11,
   'default': 0b00,
   'matchNull': 0b01,
   'matchAny': 0b10,
   'reserved': 0b11,
}

regexModeEnum = {
   'regexModeDfa' : 'asn',
   'regexModePosix' : 'string',
}
addReverseMapping( regexModeEnum )

sourceProtocolEnumToRouteProtoMap = {
   1 : 'connected',
   2 : 'ospf',
   3 : 'ospfv3',
   4 : 'bgp',
   5 : 'static',
   6 : 'rip',
   7 : 'isis',
   8 : 'attached-host',
   9 : 'dynamic',
}

# Dictionary to map bidirectionally between the cli keywords and enum
# value names.
originAsValidityEnum = {
   'valid': 'originValid',
   'notFound': 'originNotFound',
   'invalid': 'originInvalid' }
addReverseMapping( originAsValidityEnum )

# Dictionary to map bidirectionally between the cli keywords and enum
# value names.
originAsValidityCliEnum = {
   'valid': 'originValid',
   'not-found': 'originNotFound',
   'invalid': 'originInvalid',
}
addReverseMapping( originAsValidityCliEnum )

# Dictionary to map the enum value to capi representation.
# See the Bgp::Rpki::OriginValidationState enum.
originAsValidityEnumToCapi = {
   0: 'valid',
   1: 'notFound',
   2: 'invalid',
}

# Dictionary to map the enum value to cli representation.
# See the Bgp::Rpki::OriginValidationState enum.
originAsValidityEnumToCli = {
   0: 'valid',
   1: 'not-found',
   2: 'invalid',
}

# Dictionary to map between the capi and cli representations.
originAsValidityCapiToCli = {
   'valid': 'valid',
   'invalid': 'invalid',
   'not-found': 'notFound',
}
addReverseMapping( originAsValidityCapiToCli )

# as-path min and max
AS_PATH_MIN = 1
AS_PATH_MAX = 0xFFFFFFFF
AS_PATH_MAX_LENGTH = 64
AS_PATH_MAX_LENGTH_GATED = 15
regexUnderScore = "(^|[{},:]|$| )"
ExtCommLinkBandwidthSetType = Tac.Type( 'Acl::ExtCommLinkBandwidthSetFlag' )
ExtCommLinkBandwidthSetDeleteType = \
      Tac.Type( 'Acl::ExtCommLinkBandwidthSetDeleteFlag' )
CommunityType = Tac.Type( 'Acl::CommunityType' )
CommunityType = Enum( 'CommunityType', list( CommunityType.attributes ) )

RouteMapMatchOption = Tac.Type( 'Routing::RouteMap::MatchOption' )

# AS path entry regex flags
asPathEntryFlags = defaultdict( lambda: " ", [ ( False, "#" ) ] )

# URL schemes support for importing prefix-list
listUrlLocalSchemes = [ "flash:", "file:" ]
listUrlRemoteSchemes = [ "http:", "https:", "scp:", "ftp:" ]

routeMapMedType = Tac.Type( "Routing::RouteMap::MedType" )
routeMapYangSource = Tac.Type( "Routing::RouteMap::YangSource" )
defaultMetric = Tac.Value( "Routing::RouteMap::Metric", None,
                           routeMapMedType.medNormal )
invalidAigpMetricValue = 0xFFFFFFFFFFFFFFFF
routeMapAigpMetricSetType = Tac.Type( "Routing::RouteMap::AigpMetricSetType" )
defaultAigpMetric = Tac.Value( "Routing::RouteMap::AigpMetricSet",
                               invalidAigpMetricValue,
                               routeMapAigpMetricSetType.aigpMetricSetNormal )

# Used by CliSave for batching the commands
class Output:
   def __init__( self, cmds ):
      self.cmds = cmds
   def append( self, cmd ):
      self.cmds.addCommand( cmd )
   
# Match attribute enum, keyed by enum name. Must match RouteMap.tac
MatchAttributes = {}

def addMatchAttr( name, enumValue, dataType, cmdList, minValue=0,
                  maxValue=0xFFFFFFFF, values=None, enabled=True, capiName=None,
                  peerFilterEnabled=False, hidden=False, capiValues=None ):
   MatchAttributes[name] = {
      'enum'    : enumValue,    # match enum value 
      'type' : dataType,     # int, str, enum, or set
      'command' : cmdList,      # CLI command list
      'enabled' : enabled,      # If the feature is enabled for use with route-map
      'peerfilter' : peerFilterEnabled, # If enabled for use with peer-filter
      'hidden' : hidden,
      }
   if dataType == 'int' or dataType == 'range':
      MatchAttributes[name]['min'] = minValue
      MatchAttributes[name]['max'] = maxValue
   elif dataType == 'string' and values is not None:
      MatchAttributes[name]['values'] = values
   elif dataType == 'enum':
      assert values is not None
      MatchAttributes[ name ][ 'values' ] = values.copy()
      addReverseMapping( MatchAttributes[ name ][ 'values' ] )
      # The caller may provide a different dictionary to use to convert
      # the enum to the capi representation, if they are different than the
      # text representation.
      if capiValues is not None:
         MatchAttributes[ name ][ 'capiValues' ] = capiValues.copy()
         addReverseMapping( MatchAttributes[ name ][ 'capiValues' ] )
   elif dataType == 'set':
      MatchAttributes[ name ][ 'values' ] = []
      if values is not None:
         # pylint: disable-next=unnecessary-comprehension
         MatchAttributes[ name ][ 'values' ] = [ v for v in values ]

# The match rule attribute name in the CAPI model
   MatchAttributes[name]['capiName'] = capiName

def isMatchAttrEnabled( name, routeCtx=True ):
   enabledKey = 'enabled' if routeCtx else 'peerfilter'
   return name in MatchAttributes and MatchAttributes[ name ][ enabledKey ]

def matchRuleCapiName( name ):
   return MatchAttributes[ name ][ 'capiName' ]

#------------------------------------------------------------------------
# peer-filter
#------------------------------------------------------------------------
addMatchAttr(
   'matchAsRange', 121, 'range', [ 'as-range' ],
   minValue=1, maxValue=4294967295,
   capiName='asRange', enabled=False, peerFilterEnabled=True )

#--------------------------------
# ip commands
#--------------------------------
addMatchAttr(
   'matchAccessList', 1, 'string', [ 'ip', 'address', 'access-list' ],
   capiName='accessList' )

addMatchAttr(
   'matchPrefixList', 2, 'string', [ 'ip', 'address', 'prefix-list' ],
   capiName='prefixList' )

addMatchAttr(
   'matchPrefixTree', 3, 'string', [ 'ip', 'address', 'prefix-tree' ],
   enabled=False )

addMatchAttr(
   'matchBgpNextHop', 4, 'ipaddr', [ 'ip', 'bgp', 'next-hop' ],
   enabled=False)

addMatchAttr(
   'matchGateway', 5, 'ipaddr', [ 'ip', 'gateway' ], enabled=False )

# Mcast stuff. All disabled
addMatchAttr(
   'matchMcastGroupAcl', 6, 'string',
   [ 'ip', 'multicast', 'group', 'access-list' ], enabled=False )
addMatchAttr(
   'matchMcastGroupPrefixList', 7, 'string',
   [ 'ip', 'multicast', 'group', 'prefix-list' ], enabled=False )
addMatchAttr(
   'matchMcastGroupPrefixTree', 8, 'string',
   [ 'ip', 'multicast', 'group', 'prefix-tree' ], enabled=False )
addMatchAttr(
   'matchMcastSourceAcl', 9, 'string',
   [ 'ip', 'multicast', 'source', 'access-list' ], enabled=False )
addMatchAttr(
   'matchMcastSourcePrefixList', 10, 'string',
   [ 'ip', 'multicast', 'source', 'prefix-list' ], enabled=False )
addMatchAttr(
   'matchMcastSourcePrefixTree', 11, 'string',
   [ 'ip', 'multicast', 'source', 'prefix-tree' ], enabled=False )

addMatchAttr(
   'matchNextHop', 12, 'ipaddr', [ 'ip', 'next-hop' ], capiName='nextHop' )

addMatchAttr(
   'matchRouteSource', 13, 'string',
   [ 'ip', 'route-source', 'prefix-tree' ], enabled=False )

addMatchAttr(
   'matchNextHopPrefixList', 14, 'string',
   [ 'ip', 'next-hop', 'prefix-list' ], capiName='nextHopPrefixList' )

addMatchAttr(
   'matchResolvedNextHopPrefixList', 15, 'string',
   [ 'ip', 'resolved-next-hop', 'prefix-list' ],
   capiName='resolvedNextHopPrefixList' )

#--------------------------------
# ipv6 commands
#--------------------------------

# TBD

#--------------------------------
# Other commands
#--------------------------------

addMatchAttr(
   'matchPeerAs', 100, 'int', [ 'as' ], minValue=1, maxValue=4294967295,
   capiName='asNumber' )

# Tentatively mark it as-path-regex. Not used. 
addMatchAttr(
   'matchAsPath', 101, 'string', [ 'as-path-regex' ], enabled=False )

# as-path value is an AS path list name. Same as industry standard.
addMatchAttr(
   'matchAsPathList', 102, 'string', [ 'as-path' ], capiName='asPathList' )

addMatchAttr(
   'matchCommunity', 103, 'set', [ 'community' ], enabled=True,
   capiName='community' )

addMatchAttr(
   'matchExtCommunity', 104, 'set', [ 'extcommunity' ],
   enabled=True, capiName='extCommunity' )

addMatchAttr(
   'matchDistance', 105, 'int', [ 'distance' ], minValue=0, maxValue=255,
   enabled=False )
addMatchAttr(
   'matchIsisInstance', 106, 'int', [ 'isis', 'instance' ],
   minValue=POLICY_ISIS_INST_MIN, maxValue=POLICY_ISIS_INST_MAX,
   capiName='isisInstance' )

addMatchAttr(
   'matchInterface', 107, 'string', [ 'interface' ], enabled=True,
   capiName='interface' )

addMatchAttr(
   'matchIsisLevel', 108, 'enum', [ 'isis', 'level' ],
   values={ 1 : 'level-1', 2 : 'level-2' }, capiName='isisLevel' )

addMatchAttr(
   'matchLocalPref', 109, 'int', [ 'local-preference' ], capiName='localPref' )

addMatchAttr(
   'matchMed', 110, 'int', [ 'med' ], enabled=False )

addMatchAttr(
   'matchMetric', 111, 'int', [ 'metric' ], capiName='routeMetric' )

addMatchAttr(
   'matchMetricType', 112, 'enum', [ 'metric-type' ],
   values= { 1 : 'type-1', 2 : 'type-2' }, capiName='ospfMetricType' )

# Same enum as Protocol in RouteMap.tac
addMatchAttr(
   'matchProtocol', 113, 'enum', [ 'source-protocol' ],
   values=sourceProtocolEnumToRouteProtoMap,
   capiName='protocol' )

addMatchAttr(
   'matchRibs', 114, 'string', [ 'ribs' ], enabled=False )

addMatchAttr(
   'matchTag', 115, 'int', [ 'tag' ], capiName='routeTag' )

addMatchAttr(
   'matchIpv6PrefixList', 116, 'string', [ 'ipv6', 'address', 'prefix-list' ],
   capiName='ipv6PrefixList')

addMatchAttr(
   'matchIpv6NextHop', 117, 'ip6addr', [ 'ipv6', 'next-hop' ],
   capiName='ipv6NextHop' )

addMatchAttr(
   'matchIpv6NextHopPrefixList', 118, 'string',
   [ 'ipv6', 'next-hop', 'prefix-list' ], capiName='ipv6NextHopPrefixList' )

addMatchAttr(
   'matchIpv6ResolvedNextHopPrefixList', 119, 'string',
   [ 'ipv6', 'resolved-next-hop', 'prefix-list' ],
   capiName='ipv6ResolvedNextHopPrefixList' )

addMatchAttr( 'matchRouterId', 120, 'string',
              [ 'router-id', 'prefix-list' ], capiName='routerIdPrefixList' )

# NOTE: We must skip 121 as 'matchAsRange' (above) uses it.

RouteTypeCliMap = { 
   1: 'local', 
   2: 'internal', 
   3: 'external',
   4: 'confederation-external',
   5: 'vpn',
   }

addMatchAttr( 'matchRouteType', 122, 'enum', [ 'route-type' ],
              values=RouteTypeCliMap, capiName='routeType' )

addMatchAttr( 'matchIpAddrDynamicPrefixList', 123, 'string', [ 'ip', 'address',
                                                               'dynamic',
                                                               'prefix-list' ],
              capiName='ipAddrDynamicPrefixList', hidden=True )

addMatchAttr( 'matchIpv6AddrDynamicPrefixList', 124, 'string', [ 'ipv6', 'address',
                                                                 'dynamic',
                                                                 'prefix-list' ],
              capiName='ipv6AddrDynamicPrefixList', hidden=True )

addMatchAttr( 'matchCommunityInstances', 125, 'range', [ 'community',
                                                         'instances' ],
              minValue=POLICY_COMM_INST_MIN, maxValue=POLICY_COMM_INST_MAX,
              capiName='communityInstances' )

addMatchAttr( 'matchAsPathLength', 126, 'u32range', [ 'as-path', 'length' ],
              minValue=POLICY_ASPATH_LENGTH_MIN, maxValue=POLICY_ASPATH_LENGTH_MAX,
              capiName='asPathLength' )

addMatchAttr( 'matchLargeCommunity', 127, 'set', [ 'large-community' ],
              enabled=True, capiName='largeCommunity' )

matchContribAggAttrOption = 'matchContributorAggregateAttributes'
matchContribAggAttrRoleStr = 'aggregate-role'
matchContribAggAttrContribStr = 'contributor'
matchContribAggAttrAggAttrStr = 'aggregate-attributes'
# must match attribute name in RouteMapCliModels::MatchRules class
matchContribAggAttrCapiName = 'aggregation'

class MatchCmdRenderer:
   """ Base class for cmds requiring custom rendering """
   pass # pylint: disable=unnecessary-pass

class ContribAggAttrRuleRenderer( MatchCmdRenderer ):
   @staticmethod
   def render( matchRule ):
      """ Custom render handler for matchContribAggOption match clause

      RouteMap CliPlugin infra assumes a lot of things about match clauses
      and this clause is different enough that apart from a large refactor
      it requires a custom renderer
      """
      assert matchRule.option == matchContribAggAttrOption
      retStr = matchContribAggAttrRoleStr + " " + matchContribAggAttrContribStr
      aggRouteMapStr = matchRule.strValue
      if aggRouteMapStr:
         retStr += " " + matchContribAggAttrAggAttrStr
      return retStr

# this command uses custom renderer
addMatchAttr( matchContribAggAttrOption, 128, 'string', ContribAggAttrRuleRenderer(),
              enabled=True, capiName=matchContribAggAttrCapiName,
              hidden=False )

# this command is to check origin of AS_PATH
addMatchAttr(
   'matchOriginAs', 129, 'int', [ 'origin-as' ], minValue=1,
   maxValue=4294967295, capiName='originAs' )

addMatchAttr( 'matchOriginAsValidity', 130, 'enum', [ 'origin-as validity' ],
              values=originAsValidityEnumToCli, capiName='originAsValidity',
              capiValues=originAsValidityEnumToCapi )

addMatchAttr( 'matchOspfInstance', 131, 'int', [ 'ospf', 'instance' ],
              minValue=POLICY_OSPF_INST_MIN, maxValue=POLICY_OSPF_INST_MAX,
              capiName='ospfInstance' )

#------------------------------------------------------------------------
# Set
#------------------------------------------------------------------------
SetActionInfo = {}

def addSetAction( attrName, flag, flagName, dataType, cmdList,
                  enabled=True, minValue=0, maxValue=0xFFFFFFFF, values=None,
                  default=None, capiName=None, hidden=False ):
   if default is None:
      if dataType in [ 'int', 'bool' ]:
         default =  eval( dataType )( 0 ) # pylint: disable=eval-used
   SetActionInfo[attrName] = {
      'flag'    : flag,
      'flagName': flagName,      
      'type'    : dataType,      
      'command' : cmdList,
      'enabled' : enabled,
      'default' : default,
      'hidden': hidden,
      }
   if dataType in [ 'int', 'maybeU63' ]:
      SetActionInfo[attrName]['min'] = minValue
      SetActionInfo[attrName]['max'] = maxValue
   elif dataType == 'string' and values is not None:
      SetActionInfo[attrName]['values'] = values
   # The set rule attribute name in the CAPI model
   SetActionInfo[attrName]['capiName'] = capiName

def isSetActionEnabled( name ):
   return name in SetActionInfo and SetActionInfo[ name ][ 'enabled' ]

def setRuleCapiName( name ):
   return SetActionInfo[ name ][ 'capiName' ]

addSetAction(
   'asPathPrepend', 1, 'hasAsPathPrepend', 'string', [ 'as-path', 'prepend' ],
   capiName='asPathPrepend')

addSetAction(
   'propagate', 1 << 1, 'hasPropagate', 'int', [ 'propagate' ],
   enabled=False )

addSetAction(
   'nextHop', 1 << 2, 'hasNextHop', 'ipaddr', [ 'ip', 'next-hop' ],
   default=Tac.Value( 'Arnet::IpAddr', 0 ), capiName='nextHop' )

addSetAction(
   'ipv6NextHop', 1 << 3, 'hasIpv6NextHop', 'ip6addr', [ 'ipv6', 'next-hop' ],
   enabled=True, default=Arnet.Ip6Addr("0:0:0:0:0:0:0:0"), capiName='nextHop' )

addSetAction(
   'level', 1 << 4, 'hasLevel', 'string', [ 'isis', 'level' ],
   values=[ 'level-1', 'level-2', 'level-1-2' ],
   default=isisLevelEnum[ 'none' ], capiName='isisLevel' )

addSetAction(
   'isisMetricStyle', 1 << 35, 'hasMetricStyle', 'string',
   [ 'isis', 'metric', 'style' ], values=[ 'wide' ],
   default=isisStyleEnum[ 'none' ], capiName='isisMetricStyle', hidden=True )

addSetAction(
   'localPref', 1 << 5, 'hasLocalPref', 'maybeU63', [ 'local-preference' ],
   capiName='localPref' )

addSetAction(
   'metric', 1 << 7, 'hasMetric', 'metric', [ 'metric' ],
   default=defaultMetric, capiName='routeMetric' )

addSetAction(
   'metricType', 1 << 8, 'hasMetricType', 'string', [ 'metric-type' ],
   values=[ 'type-1', 'type-2' ], default=metricTypeEnum[ 'type-1' ],
   capiName='ospfMetricType' )

addSetAction( 
   'origin', 1 << 10, 'hasOrigin', 'string', [ 'origin' ],
   values=[ 'egp', 'igp', 'incomplete' ], default=bgpOriginEnum[ 'any' ],
   capiName='origin' )

addSetAction(
   'distance', 1 << 11, 'hasDistance', 'int', [ 'distance' ],
   minValue=1, maxValue=255, capiName='distance' )

addSetAction(
   'communityAddReplace', 1 << 12, 'hasCommunityAddReplace',
   'communityAddReplace', [ 'community' ], capiName='communityAddOrReplace' )

addSetAction(
   'communityDelete', 1 << 13, 'hasCommunityDelete',
   'communityDelete', [ 'community' ], capiName='communityDelete' )

addSetAction(
   'extCommunity', 1 << 14, 'hasExtCommunity', 'extCommunity', [ 'extcommunity' ],
   capiName='extCommunity' )

addSetAction(
   'tag', 1 << 15, 'hasTag', 'maybeU63', [ 'tag' ], capiName='routeTag' )

addSetAction(
   'continueSeq', 1 << 16, 'hasContinue', 'int', [ 'continue' ] )

addSetAction(
   'weight', 1 << 17, 'hasWeight', 'maybeU63', [ 'weight' ],
   minValue=0, maxValue=65535, capiName='weight' )

addSetAction(
   'nextHopPeerAddr', 1 << 18, 'hasNextHopPeerAddr', 'bool',
   [ 'ip', 'next-hop', 'peer-address' ], capiName='nextHop' )

addSetAction(
   'ipv6NextHopPeerAddr', 1 << 19, 'hasIpv6NextHopPeerAddr', 'bool',
   [ 'ipv6', 'next-hop', 'peer-address' ], capiName='nextHop' )

addSetAction(
   'nextHopUnchanged', 1 << 20, 'hasNextHopUnchanged', 'bool',
   [ 'ip', 'next-hop', 'unchanged' ], capiName='nextHop' )

addSetAction(
   'ipv6NextHopUnchanged', 1 << 21, 'hasIpv6NextHopUnchanged', 'bool',
   [ 'ipv6', 'next-hop', 'unchanged' ], capiName='nextHop' )

addSetAction(
   'evpnNextHopUnchanged', 1 << 22, 'hasEvpnNextHopUnchanged', 'bool',
   [ 'evpn', 'next-hop', 'unchanged' ], capiName='nextHop' )

addSetAction(
   'ospfBitDn', 1 << 32, 'hasOspfBitDn', 'bool',
   [ 'ospf', 'bit', 'dn' ], capiName='ospfBitDn' )

addSetAction(
   'ospfv3BitDn', 1 << 34, 'hasOspfv3BitDn', 'bool',
   [ 'ospfv3', 'bit', 'dn' ], capiName='ospfv3BitDn' )

addSetAction(
   'asPathPrependLastAs', 1 << 25, 'hasAsPathPrependLastAs', 'int',
   [ 'as-path', 'prepend', 'last-as' ], minValue=1, maxValue=AS_PATH_MAX_LENGTH,
   default=0, capiName='asPathPrependLastAs' )

addSetAction(
   'asPathReplace', 1 << 26, 'hasAsPathReplace', 'string',
   [ 'as-path', 'match', 'all', 'replacement' ], capiName='asPathReplace' )

addSetAction(
   'segmentIndex', 1 << 27, 'hasSegmentIndex', 'maybeU63', [ 'segment-index' ],
   capiName='segmentIndex' )

addSetAction(
   'igpMetric', 1 << 28, 'hasIgpMetric', 'maybeU63',
   [ 'next-hop', 'igp-metric' ], capiName='igpMetric' )

addSetAction(
   'largeCommunity', 1 << 29, 'hasLargeCommunity',
   'largeCommunity', [ 'large-community' ],
   capiName='largeCommunity' )

addSetAction(
   'bpAsPathWeight', 1 << 30, 'hasBpAsPathWeight', 'maybeU63',
   [ 'bgp', 'bestpath', 'as-path', 'weight' ], minValue=1, maxValue=65535,
   capiName='bpAsPathWeight' )

addSetAction(
   'originAsValidity', 1 << 31, 'hasOriginAsValidity', 'string',
   [ 'origin-as validity' ],
   values=list( originAsValidityEnumToCli.values() ),
   default='originNotValidated',
   capiName='originAsValidity' )

addSetAction(
   'aigpMetric', 1 << 6, 'hasAigpMetric', 'aigpMetric', [ 'aigp-metric' ],
   default=defaultAigpMetric, capiName='aigpMetric' )

addSetAction(
   'resolutionRibProfileConfig', 1 << 33, 'hasResolutionRibProfileConfig',
   'resolutionRibProfileConfig', [ 'next-hop', 'resolution', 'ribs' ],
   default=Tac.Value( "Routing::Rib::ResolutionRibProfileConfig" ),
   capiName='resolutionRibProfileConfig' )

addSetAction(
   'communityFilter', 1 << 23, 'hasCommunityFilter', 'communityFilter',
   [ 'community' ], capiName='communityFilter' )

addSetAction(
   'extCommunityFilter', 1 << 24, 'hasExtCommunityFilter', 'extCommunityFilter',
   [ 'extcommunity' ], capiName='extCommunityFilter' )

addSetAction(
   'largeCommunityFilter', 1 << 9, 'hasLargeCommunityFilter',
   'largeCommunityFilter', [ 'large-community' ],
   capiName='largeCommunityFilter' )

def tacMetricValue( med, medType ):
   return Tac.Value( "Routing::RouteMap::Metric", med, medType )

def tacAigpMetricValue( metric, aigpMetricSetType ):
   return Tac.Value( "Routing::RouteMap::AigpMetricSet",
                     metric, aigpMetricSetType )

# Instantiate a Tac AigpMetricSet value of a cli input string
def getTacAigpMetricFromCliStr( value ):
   if isinstance( value, str ):
      if value == "igp metric":
         return tacAigpMetricValue(
            0, routeMapAigpMetricSetType.aigpMetricSetIgpMetric )
      if value == "igp next-hop-cost":
         return tacAigpMetricValue(
            0, routeMapAigpMetricSetType.aigpMetricSetIgpNexthopCost )
   else:
      return value
   assert False

class SetNextHopArgTransformer:
   """Collect in one place the logic for modifying MapEntry's "setNextHopArg" field.
The MapEntry TACC class used to have multiple *nextHop* fields which
were assigned as a result of runnig "set ... next hop ..." command.
Those various fields were collected into a single "setNextHopArg" field.
This class is intended to collect in one place the logic required to set 
the new field."""

   tag = Tac.Type( "Routing::RouteMap::SetNextHopArg::SetNextHopArgTag" )
   tacAttrToArgTag = { 'nextHop' : tag.nextHopArgAddr,
                       'ipv6NextHop' : tag.nextHopArgIpv6Addr,
                       'nextHopPeerAddr' : tag.nextHopArgPeerAddr,
                       'ipv6NextHopPeerAddr' : tag.nextHopArgIpv6PeerAddr,
                       'nextHopUnchanged' : tag.nextHopArgUnchanged,
                       'ipv6NextHopUnchanged' : tag.nextHopArgIpv6Unchanged }

   flagsMask = reduce ( operator.ior,
                        ( SetActionInfo[ a ][ 'flag' ]
                          for a in tacAttrToArgTag ) )

   @classmethod
   def tacSetNextHopArg( cls, attr, value ):
      address = "0.0.0.0"
      arg = cls.tacAttrToArgTag[ attr ]
      if attr in { 'nextHop', 'ipv6NextHop' }:
         # tests are failing because they are passing in instance of
         # Arnet::Ip[6]Addr (c.f. addSetAction in RouteMapLib.py)
         address = str( value )
      elif not value:
         arg = cls.tag.nextHopArgNone
      return Tac.Value( "Routing::RouteMap::SetNextHopArg", arg,
                        Tac.Value( "Arnet::IpGenAddr", address ) )

   @classmethod
   def tacSetNextHopArgNone( cls ):
      return Tac.Value( 'Routing::RouteMap::SetNextHopArg' )

# Make the cli value string from a Tac metric value
def getCliStrFromTacMetric( metric ):
   if metric.medType == routeMapMedType.medNormal:
      return str( metric.metricValue )
   if metric.medType == routeMapMedType.medAdditive:
      return "+" + str( metric.metricValue )
   if metric.medType == routeMapMedType.medSubtractive:
      return "-" + str( metric.metricValue )
   if metric.medType == routeMapMedType.medMultiplicative:
      return f"*{float( metric.metricValue ) / 10000:.4f}"
   if metric.medType == routeMapMedType.medIgpNexthopCost:
      return "igp-nexthop-cost"
   if metric.medType == routeMapMedType.medAddIgpNexthopCost:
      return "+igp-nexthop-cost"
   if metric.medType == routeMapMedType.medValueAddIgpNexthopCost:
      return "%s +igp-nexthop-cost" % metric.metricValue
   if metric.medType == routeMapMedType.medIgpMetric:
      return "igp-metric"
   if metric.medType == routeMapMedType.medAddIgpMetric:
      return "+igp-metric"
   if metric.medType == routeMapMedType.medValueAddIgpMetric:
      return "%s +igp-metric" % metric.metricValue
   assert False

# Make the cli value string from a Tac AIGP metric value
def getCliStrFromTacAigpMetric( metric ):
   if metric.aigpMetricSetType == routeMapAigpMetricSetType.aigpMetricSetNormal:
      return str( metric.metric )
   if metric.aigpMetricSetType == \
      routeMapAigpMetricSetType.aigpMetricSetIgpNexthopCost:
      return "igp next-hop-cost"
   if metric.aigpMetricSetType == routeMapAigpMetricSetType.aigpMetricSetIgpMetric:
      return "igp metric"
   if metric.aigpMetricSetType == routeMapAigpMetricSetType.aigpMetricSetAdditive:
      return "+" + str( metric.metric )
   if metric.aigpMetricSetType == routeMapAigpMetricSetType.aigpMetricSetSubtractive:
      return "-" + str( metric.metric )
   assert False

#------------------------------------------------------------------------
# Show command implementor
#
# If output is None, print to stdout with proper formatting.
# If output is not None, it must be an list. The show output is appended
# to the output. Every sequence is appended as one list of itself.
# So the output will be a list of lists. The list is sorted by sequence
# number within a map, and by map name, e.g.:
#
# [ [ "route-map abc permit 10", ... ], [ "route-map abc deny 11" ],
#   [ "route-map xyz permit 1", ... ], [ "route-map xyz deny 2" ] ... ]
#
#------------------------------------------------------------------------
class Printer:
   def __init__( self, config ):
      self.config_ = config

   def printAll( self, output=None ):
      for name in sorted( self.config_.routeMap ):
         self.printRouteMap( self.config_.routeMap[name], output )
         
   def printRouteMap( self, routeMap, output=None ):
      
      if isinstance( routeMap, str ):
         if routeMap in self.config_.routeMap:
            routeMap = self.config_.routeMap[ routeMap ]
         else:
            return
         
      for seqno in sorted( routeMap.mapEntry ):

         entry = routeMap.mapEntry[ seqno ]
         permit = matchPermitEnum[ entry.permit ]
         if entry.statementNameFromCli:
            buf = "route-map %s statement %s %s %s" % ( routeMap.name,
                  entry.statementName, permit, seqno )
         else:
            buf = "route-map %s %s %s" % ( routeMap.name, permit, seqno )

         if output is not None:
            outputSeq = [ buf ]
         else:
            print( buf )

         if output is None:
            printRouteMapEntryAttributes( entry )
         else:
            printRouteMapEntryAttributes( entry, output=outputSeq )
            output.append( outputSeq )

def formatLargeCommStr( largeComm, asdotConfigured ):
   # Convert 4 byte ASN in LargeComm string to Asdot notation or
   # Asplain notation depending on @asdotConfigured
   lcSplit = largeComm.split( ':' )
   asn = asnStrToNum( lcSplit[ 0 ] )
   asn = bgpFormatAsn( asn, asdotConfigured )
   return asn + ":" + lcSplit[ 1 ] + ":" + lcSplit[ 2 ]

def formatRtStr( rt, asdotConfigured ):
   # Convert 4 byte ASN in RT string to Asdot notation or
   # Asplain notation depending on @asdotConfigured
   rtSplit = rt.split( ':' )
   if len( rtSplit[ 0 ].split( '.' ) ) == 4:
      # admin field is an ip-address
      return rt
   else:
      asn = asnStrToNum( rtSplit[ 0 ] )
      asn = bgpFormatAsn( asn, asdotConfigured )
      return asn + ":" + rtSplit[ 1 ]

def extCommType( value ):
   extCommValue = ( value.part1 << 32 ) | value.part2
   return ( ( extCommValue >> 48 ) & ~0x4000 )

def getExtCommTypeValue( subtype ):
   extCommTypeValue = extCommTypeValueMap[ subtype ] << 48
   return CommunityValueConst( 0,
                               extCommTypeValue >> 32,
                               extCommTypeValue & 0xFFFFFFFF )

def commValueToPrint( commValue,
                      commType=CommunityType.communityTypeStandard,
                      entry=None,
                      lbwDisplay=None,
                      lbwRange=False,
                      asdotConfigured=False,
                      withTypeStr=True ):
   '''
   @commValue       CommunityValueType object
   @commType        CommunityType of the commValue
   @entry           RouteMap::Routing::MapEntry object
   @lbwDisplay      lbw display value
   @lbwRange        Boolean flag adds "lbw range " prefix to extended community
   @asdotConfigured Boolean flag sets a way how ASN will be printed:
                    False(default) aka asplain: 65538:3:4
                    True aka asdot:             1.2:3:4
   @withTypeStr     Boolean flag sets a format of string representation:
                    True(default): "0:0",      "rt 1:2 soo 3:4"
                    False:         "internet", "1:2 3:4"
   @return String representation of a community value
   '''
   assert type( commValue ) == CommunityValueType
   assert type( commType ) == CommunityType
   value = getNumberFromCommValue( commValue )
   if commType == CommunityType.communityTypeLarge:
      #  0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
      # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
      # |                       Global Administrator                    |
      # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
      # |                         Local Data Part 1                     |
      # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
      # |                         Local Data Part 2                     |
      # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
      return '%s:%u:%u' % ( bgpFormatAsn( commValue.part0, asdotConfigured ),
                            commValue.part1, commValue.part2 )
   elif commType == CommunityType.communityTypeExtended:
      returnExt = ''
      extType = extCommType( commValue )
      if extType in [ extCommTypeValueMap[ commType ]
                      for commType in ( 'rtAs', 'sooAs', 'l2VpnIdAs' ) ]:
         if withTypeStr:
            # Don't need this yet
            assert extType != extCommTypeValueMap[ 'l2VpnIdAs' ]
            if extType == extCommTypeValueMap[ 'rtAs' ]:
               returnExt += 'rt '
            else:
               returnExt += 'soo '

         # Two-Octet AS Specific Extended Community
         #
         # 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
         # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
         # | 0x00 or 0x40  |   Sub-Type    |    Global Administrator       |
         # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
         # |                     Local Administrator                       |
         # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
         #
         # Encoded as 64-bit big-endian integer without swapping
         # Represent as: <ASN>:<nn>
         returnExt += '%u:%u' % ( ( value >> 32 ) & 0xFFFF,
                                  ( value & 0xFFFFFFFF ) )

      elif extType == extCommTypeValueMap[ 'rtAs4' ] or \
            extType == extCommTypeValueMap[ 'sooAs4' ]:
         # if ASN fits with 2-octets but 4-octet encoding requested
         asn = ( value >> 16 ) & 0xFFFFFFFF
         # long-int qualifier required if 4-octet enc and ASN fits in 2-octets
         l_tag = ""
         if ( asn <= 65535 ):
            l_tag = 'L'
         if withTypeStr:
            if extType == extCommTypeValueMap[ 'rtAs4' ]:
               returnExt += 'rt '
            else:
               returnExt += 'soo '

         # Four-Octet AS Specific Extended Community
         #
         # 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
         # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
         # | 0x02 or 0x42  |   Sub-Type    |    Global Administrator       :
         # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
         # : Global Administrator (cont.)  |    Local Administrator        |
         # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
         #
         # Encoded as 64-bit big-endian integer without swapping
         # Represent as: <ASN>:<nn>
         returnExt += '%s%s:%u' % ( bgpFormatAsn( asn, asdotConfigured ),
                                    l_tag,
                                  ( value & 0xFFFF ) )
      elif extType == extCommTypeValueMap[ 'lbw' ]:
         asn = ( value >> 32 ) & 0xFFFF
         returnExt += 'lbw '
         if entry is not None and entry.extCommLinkBandwidthSetFlag != \
                ExtCommLinkBandwidthSetType.extCommLinkBandwidthSetDefault:
            if entry.extCommLinkBandwidthSetFlag == \
                   ExtCommLinkBandwidthSetType.extCommLinkBandwidthSetAggregate:
               returnExt += 'aggregate'
            elif entry.extCommLinkBandwidthSetFlag == \
                   ExtCommLinkBandwidthSetType.extCommLinkBandwidthSetAggregateRef:
               returnExt += 'aggregate %s' % ( entry.linkBandwidthRefDisplay )
            elif entry.extCommLinkBandwidthSetFlag == \
                   ExtCommLinkBandwidthSetType.extCommLinkBandwidthSetDivideEqual:
               returnExt += 'divide equal'
            elif entry.extCommLinkBandwidthSetFlag == \
                   ExtCommLinkBandwidthSetType.extCommLinkBandwidthSetDivideRatio:
               returnExt += 'divide ratio'
         elif entry is not None and entry.extCommLinkBandwidthSetDeleteFlag == \
               ExtCommLinkBandwidthSetDeleteType.\
               extCommLinkBandwidthSetDeleteAsn:
            returnExt += 'asn %s' % ( entry.linkBandwidthDeleteAsnDisplay )
         elif entry is not None and entry.extCommLinkBandwidthSetDeleteFlag == \
               ExtCommLinkBandwidthSetDeleteType.extCommLinkBandwidthSetDeleteAny:
            # This is so that we don't have extra space between lbw and 
            # delete keywords in CliSave output in case of 
            # "set extcommunity lbw delete"
            returnExt = returnExt.strip()
         else:
            assert entry or lbwDisplay
            if asn:
               returnExt += '%u:%s' \
                   % ( asn, entry.lbwDisplay if lbwDisplay is None else \
                          lbwDisplay[ 0 ] )
            else:
               returnExt += '%s' \
                   % ( entry.lbwDisplay if lbwDisplay is None else \
                          lbwDisplay[ 0 ] )
      elif extType == extCommTypeValueMap[ 'lbwAny' ]:
         returnExt += 'lbw any'
      elif extType == 36:
         asn = ( value >> 32 ) & 0xFFFF
         if lbwRange:
            returnExt += '%u:%s' \
                  % ( asn, entry.lbwDisplay[ 1 ] if lbwDisplay is None or \
                     lbwDisplay[ 1 ] is None else lbwDisplay[ 1 ].split()[ 1 ] )
         else:
            returnExt += 'lbw range '
            returnExt += '%u:%s' \
                  % ( asn, entry.lbwDisplay[ 0 ] if lbwDisplay is None else \
                     lbwDisplay[ 0 ].split()[ 1 ] )
      elif extType == extCommTypeValueMap[ 'color' ]:
         # Color Extended Community
         #  0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
         # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
         # | 0x03          | 0x0b          |C O|           Reserved        |
         # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
         # |                          Color Value                          |
         # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
         coBits = ( ( value >> 46 ) & colorOnlyFlags[ 'coMask' ] )
         colorOnlyStr = ''
         if coBits & colorOnlyFlags[ 'matchNull' ]:
            colorOnlyStr += ' color-only endpoint-match null'
         elif coBits & colorOnlyFlags[ 'matchAny' ]:
            colorOnlyStr += ' color-only endpoint-match any'
         returnExt += 'color %u%s' % ( ( value & 0xFFFFFFFF ), colorOnlyStr )
      elif extType == extCommTypeValueMap[ 'esi' ]:
         # ES-Import extended community
         # 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
         # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
         # | Type=0x06     | Sub-Type=0x02 |          ES-Import            |
         # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
         # |                     ES-Import Cont'd                          |
         # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
         word0 = ( value >> 32 ) & 0xFFFF
         word1 = ( value >> 16 ) & 0xFFFF
         word2 = value & 0xFFFF
         ethAddr = Tac.Value( "Arnet::EthAddr", word0, word1, word2 )
         returnExt += 'evpn es-import rt ' + ethAddr.stringValue
      elif extType == extCommTypeValueMap[ 'ov' ]:
         # Origin AS validity extended community
         # 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
         # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
         # |       0x43    |      0x00     |             Reserved          |
         # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
         # |                    Reserved                   |validationstate|
         # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
         validationState = value & 0xFF
         returnExt += 'ov %d' % validationState
      elif ( extType == extCommTypeValueMap[ 'router-id' ] ):
         #  Ospf Router ID Trans IPv4 address extended community:
         #  0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
         #  +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
         #  |       0x01    |      0x07     |         Router ID             |
         #  +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
         #  |          Router ID            |                               |
         #  +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
         routerId = [ str( ( value >> 40 ) & 0xFF ),
                      str( ( value >> 32 ) & 0xFF ),
                      str( ( value >> 24 ) & 0xFF ),
                      str( ( value >> 16 ) & 0xFF ) ]
         returnExt += 'ospf router-id ' + '.'.join( routerId )
      else:
         assert extType in [ extCommTypeValueMap[ commType ]
                             for commType in ( 'rtIp', 'sooIp', 'l2VpnIdIp' ) ]
         if withTypeStr:
            # Don't need this yet
            assert extType != extCommTypeValueMap[ 'l2VpnIdIp' ]
            if extType == extCommTypeValueMap[ 'rtIp' ]:
               returnExt += 'rt '
            else:
               returnExt += 'soo '

         # IPv4 Address Specific Extended Community
         #
         # 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
         # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
         # | 0x01 or 0x41  |   Sub-Type    |    Global Administrator       |
         # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
         # | Global Administrator (cont.)  |    Local Administrator        |
         # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
         #
         # Encoded as 64-bit big-endian integer without swapping
         # Represent as: <ip>:<nn>
         returnExt += '%u.%u.%u.%u:%u' % ( ( value >> 40 ) & 0xFF,
                                           ( value >> 32 ) & 0xFF,
                                           ( value >> 24 ) & 0xFF,
                                           ( value >> 16 ) & 0xFF,
                                           ( value & 0xFFFF ) )
      return returnExt
   else:
      if withTypeStr and str( value ) in commValueEnum:
         return commValueEnum[ str( value ) ]
      high = ( value >> 16 ) & 0xFFFF
      low = value & 0xFFFF
      return '%u:%u' % ( high, low )

# Get the continue show ouput
def continueSeqShowOutput( seqNo ):
   if seqNo == 0:
      return "Continue: next sequence"
   else:
      return "Continue: sequence %d" % seqNo

def renderRange( start, end, rangeMin ):
   '''
   This method is used in CliPlugin/RouteMapCliModels.py also
   to render 'show route-map' output.
   @param start: start of range from Sysdb
   @param end: end of range from Sysdb
   @param rangeMin: Minimum allowed
   @return:
   '''
   if start == end:
      return "= %d" % ( start )
   if start == rangeMin:
      return "<= %d" % ( end )
   return ">= %d and <= %d" % ( start, end )

# print one map sequence. If output is None, print formatted to stdout with
# some descriptive text. Otherwise add cli commands to the output list.
def printRouteMapEntryAttributes( entry, output=None, asdotConfigured=False,
                                  showHidden=False ):

   def doFormatRange( attr, rule, op ):
      if attr[ 'capiName' ] == 'communityInstances':
         return renderRange( rule.intValue, rule.int2Value,
                             POLICY_COMM_INST_MIN )
      elif attr[ 'capiName' ] == 'asPathLength':
         return renderRange( rule.u32Range.low, rule.u32Range.high,
                             POLICY_ASPATH_LENGTH_MIN )
      else:
         return str( rule.intValue ) + "-" + str( rule.int2Value ) if \
                 rule.int2Value > rule.intValue else ""

   def doFormatInt( attr, rule, op ):
      if op == "matchPeerAs":
         return bgpFormatAsn( rule.intValue, asdotConfigured )
      else:
         return str( rule.intValue )

   formatFor = {}
   formatFor[ 'string' ] = lambda attr, rule, op: rule.strValue
   formatFor[ 'ipaddr' ] = lambda attr, rule, op: str( rule.ipAddrValue )
   formatFor[ 'ip6addr' ] = lambda attr, rule, op: rule.ip6AddrValue.stringValue
   formatFor[ 'enum' ] = lambda attr, rule, op: attr[ 'values' ][ rule.intValue ]
   formatFor[ 'range' ] = doFormatRange
   formatFor[ 'u32range' ] = doFormatRange
   formatFor[ 'int' ] = doFormatInt
   formatFor[ 'set' ] = lambda attr, rule, \
                        op: " ".join( rule.commListSet.commListSet )

   if output is None:
      print( "  Description:" )

   for descKey in entry.description:
      text = "description %s" % entry.description[ descKey ]
      if output is not None:
         if isinstance( output, list ):
            output.append( text )
         elif getattr( output, "addCommand" ):
            output.addCommand( text )
         else:
            assert False
      else:
         print( "    " + text )

   if output is None:
      print( "  Match clauses:" )

   for op in entry.matchRule:
      # Skip cli save `match tag <NUM>` when `match tag set <NAME>` is configured
      # because the CLIs `match tag <NUM>` and `match tag set <NAME>` can't coexist.
      if op == 'matchTag' and entry.matchTagSetName:
         continue

      rule = entry.matchRule[ op ]
      attr = MatchAttributes[ op ]
      text = "match "

      if rule.invert:
         text += "invert-result "

      # render the provided command string OR call the render function
      if isinstance( attr[ 'command' ], MatchCmdRenderer ):
         text += attr[ 'command' ].render( rule )
      else:
         text += " ".join( attr[ 'command' ] )

      if rule.orResults:
         text += " or-results"

      text += " " + formatFor[ attr[ 'type' ] ]( attr, rule, op )

      if rule.exact:
         text += " exact-match"

      if output is not None:
         if isinstance( output, list ):
            output.append( text )
         elif getattr( output, "addCommand" ):
            output.addCommand( text )
         else:
            assert False
      else:
         print( "    " + text )

   if entry.matchTagSetName:
      text = f'match tag set {entry.matchTagSetName}'
      if output is not None:
         if isinstance( output, list ):
            output.append( text )
         elif getattr( output, "addCommand" ):
            output.addCommand( text )
         else:
            assert False
      else:
         print( "    " + text )

   if output is None:
      print( "  SubRouteMap:" )

   if entry.subRouteMap.name:
      subInvert = "invert-result " if entry.subRouteMap.invert else ""
      subRouteMapCmd = "sub-route-map %s%s" % ( subInvert, entry.subRouteMap.name )
      if output is None:
         print( "    " + subRouteMapCmd )
      else:
         if isinstance( output, list ):
            output.append( subRouteMapCmd )
         elif getattr( output, "addCommand" ):
            output.addCommand( subRouteMapCmd )
         else:
            assert False

   if entry.setFlags.value & SetActionInfo['continueSeq']['flag']:
      value = entry.continueSeq
      if output is None:
         print( "  " + continueSeqShowOutput( value ) )
      else:
         buf = 'continue '
         if value > 0: 
            buf += str( value )
         if isinstance( output, list ):
            output.append( buf )
         elif getattr( output, "addCommand" ):
            output.addCommand( buf )
         else:
            assert False

   if output is None:
      print( "  Set clauses:" )

   for attr in sorted( SetActionInfo ):
      valueList = None
      if entry.setFlags.value & SetActionInfo[attr]['flag']:
         value = getattr( entry, attr )
         # Some special cases for enum values
         if attr == 'metricType':
            value = metricTypeEnum[ value ]
         elif attr == 'level':
            value = isisLevelEnum[ value ]
         elif attr == 'isisMetricStyle':
            value = isisStyleEnum[ value ]
         elif attr == 'origin':
            value = bgpOriginEnum[ value ]
         elif attr == 'tag':
            # auto tag is represented max MaybeU63 value from Ark/MaybeTypes.tac
            if value == 0x7fffffffffffffff:
               value = 'auto'
         elif attr == 'asPathPrepend':
            values = []
            for k in entry.asPathPrepend:
               if entry.asPathPrepend[k] == 0:
                  # 'auto' keyword is represented by 0 in the asPathPrepend
                  # collection
                  values.append( 'auto' )
               else:
                  values.append( bgpFormatAsn( entry.asPathPrepend[k], \
                                 asdotConfigured ) )
            if entry.asPathPrependRepeat > 1:
               values.extend( [ 'repeat', str( entry.asPathPrependRepeat ) ] )
            value = " ".join( values )
         elif attr == 'asPathReplace':
            values = []
            for k in entry.asPathReplace:
               if entry.asPathReplace[k] == 0:
                  # 'auto' keyword is represented by 0 in the asPathReplace
                  # collection
                  values.append( 'auto' )
               elif entry.asPathReplace[ k ] == AS_PATH_MAX:
                  # 'none' keyword is represented by AS_PATH_MAX in the asPathReplace
                  # collection
                  values.append( 'none' )
               else:
                  values.append( bgpFormatAsn( entry.asPathReplace[ k ],
                                               asdotConfigured ) )
            if entry.asPathReplaceRepeat > 1:
               values.extend( [ 'repeat', str( entry.asPathReplaceRepeat ) ] )
            value = " ".join( values )
         elif attr == 'med':
            continue # don't show med. it's not settable in cli

         elif attr in ( 'communityAddReplace', 'communityDelete',
                        'communityFilter' ):
            valueStr = ''

            if value is not None:
               if value.flag != 'none':
                  if value.useCommunityList:
                     valueStr += 'community-list '
                     valueStr += ' '.join( value.communityListNameSet ) + ' '
                  else:
                     for comm in value.community:
                        valueStr += commValueToPrint( comm ) + ' '

               if value.flag == 'commSetAdditive':
                  valueStr += 'additive'
               elif value.flag == 'commSetDelete':
                  valueStr += 'delete'
               elif value.flag == 'commSetFilter':
                  valueStr += 'filter '
                  if value.commType.allCommTypes:
                     valueStr += 'all'
               elif value.flag == 'commSetNone':
                  valueStr += 'none'

            value = valueStr.rstrip()

         elif attr in ( 'extCommunity', 'extCommunityFilter' ):
            if not value.useCommunityList:
               if value.flag != 'commSetNone' or entry.extCommLinkBandwidthSetFlag \
                  is not ExtCommLinkBandwidthSetType.extCommLinkBandwidthSetDefault:
                  # As discussed in http://bug/197554#c3, we split the extcommunity
                  # attribute into lbw-values and non-lbw-values when printing.
                  # Consider the sequence of inputs: 'set extcommunity rt 1:1', 'set
                  # extcommunity soo 2:2', then 'set extcommunity lbw 1G'. The
                  # generated config should be printed as 'set extcommunity rt 1:1
                  # soo 2:2' on one line, then 'set extcommunity lbw 1G' on another.
                  valueList, lbwPrint, nonLbwPrint, colorPrint = [], '', '', ''
                  flagPrint = ( " additive" if value.flag == 'commSetAdditive' else
                                " delete" if value.flag == 'commSetDelete' else
                                "" )
                  if entry.extCommLinkBandwidthSetFlag is not \
                         ExtCommLinkBandwidthSetType.extCommLinkBandwidthSetDefault:
                     lbwComm = getExtCommTypeValue( 'lbw' )
                     lbwPrint = commValueToPrint( lbwComm,
                                       commType=CommunityType.communityTypeExtended,
                                       entry=entry )
                  for commVal in value.community:
                     valType = extCommType( commVal )
                     valPrint = commValueToPrint( commVal,
                                       commType=CommunityType.communityTypeExtended,
                                       entry=entry,
                                       asdotConfigured=asdotConfigured )
                     if valType == extCommTypeValueMap[ 'lbw' ]:
                        lbwPrint = valPrint
                     elif valType == extCommTypeValueMap[ 'color' ]:
                        colorPrint += valPrint + ' '
                     else:
                        nonLbwPrint += valPrint + ' '
                  if lbwPrint:
                     valueList.append( lbwPrint + flagPrint )
                  if nonLbwPrint:
                     valueList.append( nonLbwPrint.rstrip() + flagPrint )
                  if colorPrint:
                     valueList.append( colorPrint.rstrip() + flagPrint )
               elif value.flag == 'commSetNone':
                  value = 'none'
            else:
               valueStr = ''

               if value is not None:
                  if value.flag != 'none':
                     valueStr += 'extcommunity-list '
                     valueStr += ' '.join( value.communityListNameSet ) + ' '

                  if value.flag == 'commSetAdditive':
                     valueStr += 'additive'
                  elif value.flag == 'commSetDelete':
                     valueStr += 'delete'
                  elif value.flag == 'commSetFilter':
                     valueStr += 'filter '
                     if value.commType.allCommTypes:
                        valueStr += 'all'
                     if value.commType.routeTarget:
                        valueStr += 'rt '
                     if value.commType.siteOfOrigin:
                        valueStr += 'soo '
                     if value.commType.linkBandwidth:
                        valueStr += 'lbw '
                     if value.commType.color:
                        valueStr += 'color '
                  elif value.flag == 'commSetNone':
                     valueStr += 'none'

               value = valueStr.rstrip()
         elif attr in ( 'largeCommunity', 'largeCommunityFilter' ):
            valueStr = ''

            if value is not None:
               if value.flag != 'none':
                  if value.useCommunityList:
                     valueStr += 'large-community-list '
                     valueStr += ' '.join( value.communityListNameSet ) + ' '
                  else:
                     for comm in value.community:
                        valueStr += commValueToPrint( comm,
                                          commType=CommunityType.communityTypeLarge,
                                          entry=entry,
                                          asdotConfigured=asdotConfigured ) + ' '

               if value.flag == 'commSetAdditive':
                  valueStr += 'additive'
               elif value.flag == 'commSetDelete':
                  valueStr += 'delete'
               elif value.flag == 'commSetFilter':
                  valueStr += 'filter '
                  if value.commType.allCommTypes:
                     valueStr += 'all'
               elif value.flag == 'commSetNone':
                  valueStr += 'none'

            value = valueStr.rstrip()

         elif attr == 'ipv6NextHop':
            value = value.stringValue
         elif attr == 'continueSeq':
            continue # The continue is printed before other sets
         elif attr == 'metric':
            yangSource = value.yangSource
            value = getCliStrFromTacMetric( value )
            if showHidden:
               if yangSource == routeMapYangSource.yangSourceBgpActions:
                  value = f'{value} source openconfig bgp-actions'
               elif yangSource == routeMapYangSource.yangSourceOspfActions:
                  value = f'{value} source openconfig ospf-actions'
               elif yangSource == routeMapYangSource.yangSourceIsisActions:
                  value = f'{value} source openconfig isis-actions'
         elif attr == 'aigpMetric':
            value = getCliStrFromTacAigpMetric( value )
         elif attr == 'originAsValidity':
            value = originAsValidityCliEnum[ value ]
         elif attr == 'resolutionRibProfileConfig':
            value = value.stringValue()
         attrList = ( 'nextHopPeerAddr', 'ipv6NextHopPeerAddr',
                      'nextHopUnchanged', 'ipv6NextHopUnchanged',
                      'evpnNextHopUnchanged', 'ospfBitDn', 'ospfv3BitDn' )
         valueList = [ value ] if valueList is None else valueList
         for value in valueList:
            if attr in attrList:
               buf = "    set %s" % (
                     " ".join( SetActionInfo[attr]['command'] ) )
            else:
               buf = "    set %s %s" % (
                     " ".join( SetActionInfo[attr]['command'] ), value )
            if output is not None:
               if isinstance( output, list ):
                  output.append( buf.lstrip() )
               elif getattr( output, "addCommand" ):
                  output.addCommand( buf.lstrip() )
               else:
                  assert False
            else:
               print( buf )

def printPeerFilter( pFilter, output=None, asdotConfigured=False ):
   if not pFilter:
      return
   for descKey in pFilter.description:
      text = 'description %s' % pFilter.description[ descKey ]
      if output is not None:
         if isinstance( output, list ):
            output.append( text )
         elif getattr( output, 'addCommand', None ):
            output.addCommand( text )
         elif getattr( output, 'write', None ):
            output.write( text + '\n' )
         else:
            assert False
      else:
         print( '   ' + text )

   for seqno in sorted( pFilter.mapEntry ):
      entry = pFilter.mapEntry[ seqno ]

      text = str( seqno ) + ' match '
      connector = ''
      for op in entry.matchRule:
         rule = entry.matchRule[ op ]
         attr = MatchAttributes[ op ]
         text += connector + ' '.join( attr[ 'command' ] )
         if attr[ 'type' ] == 'string':
            text += ' ' + rule.strValue
         elif attr[ 'type' ] == 'int':
            text += ' ' + bgpFormatAsn( rule.intValue, asdotConfigured )
         elif attr[ 'type' ] == 'ipaddr':
            text += ' ' + str( rule.ipAddrValue )
         elif attr[ 'type' ] == 'ip6addr':
            text += ' ' +  rule.ip6AddrValue.stringValue
         elif attr[ 'type' ] == 'enum':
            text += ' ' + attr[ 'values' ][ rule.intValue ]
         elif attr[ 'type' ] == 'range':
            # The matcher for the asRange only accepts values that are both asdot
            # or both asplain. Therefore, if the start value is below the asdot
            # lower bound, force the rendering of both start and end
            # into asplain format (as this would be how the user inputted
            # the range).
            asdotConfigured &= ( rule.intValue > 65535 )
            text += ' ' + bgpFormatAsn( rule.intValue, asdotConfigured )
            if rule.int2Value > rule.intValue:
               text += '-' + bgpFormatAsn( rule.int2Value, asdotConfigured )
         # Since we only support a single match statement, as-range, we haven't
         # had to figure out whether/how to support multipe matches in single
         # command.  This is a placeholder:
         connector = 'and '

      text += ' result ' +  matchAcceptEnum[ entry.permit ]

      if output is not None:
         if isinstance( output, list ):
            output.append( text )
         elif getattr( output, 'addCommand', None ):
            output.addCommand( text )
         elif getattr( output, 'write', None ):
            output.write( text + '\n' )
         else:
            assert False
      else:
         print( '   ' + text )

def isAsdotConfigured( asnConfig=None ):
   if asnConfig is None:
      return False

   if asnConfig.asnNotation == \
         Tac.Type( 'Routing::AsnNotation' ).asnNotationAsdot:
      return True
   else:
      return False

def convertBracketExpression( brkStr ):
   if r'\]' in brkStr:
      if brkStr[ 1 ] == '^':
         brkStr = brkStr[ 0:2 ] + ']' + brkStr[ 2 : ]
      else:
         brkStr = brkStr[ 0 ] + ']' +  brkStr[ 1 : ]
      brkStr = brkStr.replace( r'\]', '' )
   if r'\-' in brkStr:
      brkStr = brkStr.replace( r'\-', '' )
      brkStr = brkStr[ 0:-1 ] + '-' + brkStr[ -1 ]
   return brkStr

# regex used to find a underscore that is not in brackets and is not escaped
undScrRegex = r"^_|[^[//]_|^[^[]+?[^//]_|(\[\]?[^\]]+?\]|\[\]\]|\[\])(_|[^[]+?_)"

def convertRegexIndustryStandard( regex ):
   brackets = re.findall( r"[^\\]\[.*?[^\\]\]|^\[.*?[^\\]\]", regex )
   for bracket in brackets:
      if bracket[ 0 ] != '[':
         bracket = bracket[ 1: ]
      regex = regex.replace( bracket, convertBracketExpression( bracket ) )
   returnStr = ''
   index = 0
   while( index < len( regex ) ):
      result = re.search( undScrRegex, regex[ index : ] )
      if ( result ):
         if result.end() == 1:
            returnStr += regexUnderScore
         else:
            returnStr += regex[ index : index + result.end() - 1 ] + regexUnderScore
         index += result.end()
      else:
         break
   if ( index < len( regex ) ):
      returnStr += regex[ index : ]
   return returnStr

def getCommValue( value ):
   if type( value ) is CommunityValueType:
      return value
   return CommunityValueConst( 0, 0, getCommIntValue( value ) )

class LinkBwUnion( Union ):
   _fields_ = [
      ( "bw_u", c_uint32 ),
      ( "bw_f", c_float ),
   ]

def getExtCommValue( value ):
   if type( value ) is CommunityValueType:
      return value
   if isinstance( value, int ):
      return CommunityValueConst( 0, value >> 32, value & 0xFFFFFFFF )

   valueType = str( value ).split()
   if valueType[ 0 ] == 'color':
      # Color Extended Community
      #  0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
      # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
      # | 0x03          | 0x0b          |C O|           Reserved        |
      # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
      # |                          Color Value                          |
      # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
      extValue0 = ( extCommTypeValueMap[ 'color' ] << 16 )
      # We pass the value in the CLI as color-value:CO
      valueSplit = valueType[ 1 ].split( ':' )
      # color-value is just a U32
      extValue1 = int( valueSplit[ 0 ] )
      # CO will be either '00', '01', or '10'. '11' is reserved for future use
      # and should NOT be used. If we receive it, we must treat it as '00'.
      co = int( valueSplit[ 1 ], 2 )
      Assert.assertNotEqual( co, colorOnlyFlags[ 'reserved' ] )
      return CommunityValueConst( 0, extValue0 | ( co << 14 ), extValue1 )

   if valueType[ 0 ] == 'esi':
      # ES-Import extended community
      # 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
      # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
      # | Type=0x06     | Sub-Type=0x02 |          ES-Import            |
      # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
      # |                     ES-Import Cont'd                          |
      # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
      extValue0 = ( extCommTypeValueMap[ 'esi' ] << 16 )
      esiVal = Arnet.EthAddr( valueType[ 1 ] )
      extValue0 |= esiVal.word0
      extValue1 = esiVal.word1
      extValue1 <<= 16
      extValue1 |= esiVal.word2
      return CommunityValueConst( 0, extValue0, extValue1 )

   if valueType[ 0 ] == 'ov':
      # Origin AS validity extended community
      # 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
      # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
      # |       0x43    |      0x00     |             Reserved          |
      # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
      # |                    Reserved                   |validationstate|
      # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
      extValue0 = ( extCommTypeValueMap[ 'ov' ] << 16 )
      extValue1 = int( valueType[ 1 ] )
      return CommunityValueConst( 0, extValue0, extValue1 )

   if ( valueType[ 0 ] == 'router-id' ):
      #  Ospf Router ID Trans IPv4 address extended community:
      #  0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
      #  +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
      #  |       0x01    |      0x07     |         Router ID             |
      #  +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
      #  |          Router ID            |                               |
      #  +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
      extValue0 = ( extCommTypeValueMap[ 'router-id' ] << 16 )
      routerId = valueType[ 1 ].split( '.' )
      extValue0 |= ( int( routerId[ 0 ] ) << 8 )
      extValue0 |= int( routerId[ 1 ] )
      extValue1 = ( int( routerId[ 2 ] ) << 24 )
      extValue1 |= ( int( routerId[ 3 ] ) << 16 )
      return CommunityValueConst( 0, extValue0, extValue1 )

   if valueType[ 0 ] != 'lbw':
      valueSplit = valueType[ 1 ].split( ':' )
      # ip address:nn(16 bit) case
      periods = sum( c == "." for c in valueSplit[ 0 ] )
      if periods == 3:
         # IPv4 Address Specific Extended Community
         # 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
         # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
         # | 0x01 or 0x41  |   Sub-Type    |    Global Administrator       |
         # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
         # | Global Administrator (cont.)  |    Local Administrator        |
         # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
         extValue0 = ( extCommTypeValueMap[ '%sIp' % valueType[ 0 ] ] << 16 )
         # quick hack to get this working, look for function that does this
         ipAddrValue = 0
         for word in valueSplit[ 0 ].split('.'):
            ipAddrValue = ipAddrValue << 8
            ipAddrValue = ipAddrValue | int ( word )
         extValue0 |= ( ( ipAddrValue >> 16 ) & 0x0000ffff )
         extValue1 = ( ( ( ipAddrValue << 16 ) & 0xffff0000 ) |
                       ( int ( valueSplit[ 1 ] ) & 0x0000ffff ) )
      else:
         isLong = False
         asnSuffix = valueSplit[ 0 ][ -1: ]
         if ( asnSuffix.upper() == 'L' ):
            valueSplit[ 0 ] = valueSplit[ 0 ][ :-1 ]
            isLong = True
         asn = asnStrToNum( valueSplit[ 0 ], minValue=0 )
         admin = int( valueSplit[ 1 ] )
         if not isLong and asn <= 0x0000ffff:
            # Two-Octet AS Specific Extended Community
            # AS(16 bit):nn(32 bit) case
            # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
            # | 0x00 or 0x40  |   Sub-Type    |    Global Administrator       |
            # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
            # |                     Local Administrator                       |
            # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
            extValue0 = ( extCommTypeValueMap[ '%sAs' % valueType[ 0 ] ] << 16 )
            extValue0 |= ( asn & 0x0000ffff )
            extValue1 = admin & 0xffffffff
         else:
            # Four-Octet AS Specific Extended Community
            # AS(32 bit):nn(16 bit) case
            # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
            # | 0x02 or 0x42  |   Sub-Type    |    Global Administrator       :
            # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
            # : Global Administrator (cont.)  |    Local Administrator        |
            # +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
            extValue0 = ( extCommTypeValueMap[ '%sAs4' % valueType[ 0 ] ] << 16 )
            extValue0 |= ( ( asn & 0xffff0000 ) >> 16 )
            extValue1 = ( ( ( asn  & 0x0000ffff ) << 16 ) |
                          ( admin & 0x0000ffff ) )

   else:
      # AS:floating point number case
      extValue0 = ( extCommTypeValueMap[ 'lbw' ] << 16 )
      if len( valueType ) == 1 or \
            valueType[ 1 ] in ( 'aggregate', 'divide' ):
         extValue0 |= 0x40000000
         # extcomm value for divide config or simple aggregate case
         if len( valueType ) <= 2 or valueType[1] == 'divide':
            return CommunityValueConst( 0, extValue0, 0 )
         elif len( valueType ) == 3:
            valueSplit = valueType
            valueSplit.remove( "lbw" )
      elif valueType[ 1 ] == 'any':
         return CommunityValueConst( 0, 0x00140000, 0 )
      elif valueType[ 1 ] == 'range':
         extValue0 |= 0x00200000
         valueSplit = valueType[ 2 ].split( ':' )
         # This adds AS number
         extValue0 |= int( valueSplit[ 0 ] ) & 0x0000ffff
      else:
         if ':' in valueType[ 1 ]:
            valueSplit = valueType[ 1 ].split( ':' )
            # This adds AS number
            extValue0 |= int( valueSplit[ 0 ] ) & 0x0000ffff
         else:
            if valueType[ 1 ] == 'asn':
               # dummy lbw stuffed here
               valueSplit = [ valueType[ 2 ], '0' ]
               # This adds AS number
               extValue0 |= int( valueSplit[ 0 ] ) & 0x0000ffff
            else:
               # dummy ASN stuffed here
               valueSplit = [ '0', valueType[ 1 ] ]

      # Mark the community as non-transitive
      extValue0 |= 0x40000000
      lbw = LinkBwUnion()
      bwre = re.compile( r'(\d+|\d+\.\d+)(K|M|G)?$' )
      m = bwre.match( valueSplit[1] )
      multiplier = { 'K': 1000.0, 'M': 1.0e+6, 'G': 1.0e+9 }
      if m.groups()[1] and m.groups()[1] in multiplier:
         # pylint: disable-msg=W0201
         lbw.bw_f = float( m.groups()[0] ) * multiplier[m.groups()[1]]
      else:
         # pylint: disable-msg=W0201
         # pylint: disable-next=attribute-defined-outside-init
         lbw.bw_f = float( m.groups()[0] )
      extValue1 = lbw.bw_u
   return CommunityValueConst( 0, extValue0, extValue1 )

def getLbwCommValue( value ):
   lbw = LinkBwUnion()
   bwre = re.compile( r'(\d+|\d+\.\d+)(K|M|G)?$' )
   m = bwre.match( value )
   multiplier = { 'K': 1000.0, 'M': 1.0e+6, 'G': 1.0e+9 }
   if m.groups()[1] and m.groups()[1] in multiplier:
      # pylint: disable-msg=W0201
      lbw.bw_f = float( m.groups()[0] ) * multiplier[m.groups()[1]]
   else:
      # pylint: disable-msg=W0201
      # pylint: disable-next=attribute-defined-outside-init
      lbw.bw_f = float( m.groups()[0] )
   extValue = lbw.bw_f
   return extValue

def getLargeCommValue( value ):
   if type( value ) is CommunityValueType:
      return value
   if isinstance( value, int ):
      asn = value >> 64
      local1 = ( value >> 32 ) & 0xFFFFFFFF
      local2 = value & 0xFFFFFFFF
   else:
      valueSplit = str( value ).split( ':' )
      asnSuffix = valueSplit[ 0 ][ -1 : ]
      if asnSuffix.upper() == 'L':
         valueSplit[ 0 ] = valueSplit[ 0 ][ : -1 ]
      asn = asnStrToNum( valueSplit[ 0 ], minValue=0 )
      local1 = safeInt( valueSplit[ 1 ] )
      local2 = safeInt( valueSplit[ 2 ] )
      assert( ( asn is not None ) and
              ( local1 is not None ) and
              ( local2 is not None ) )
   return CommunityValueConst( asn, local1, local2 )

def parseCommunityList( comms, commType ):
   if type( comms ) is not list:
      return None
   if commType == CommunityType.communityTypeStandard:
      getValue = getCommValue
   elif commType == CommunityType.communityTypeExtended:
      getValue = getExtCommValue
   elif commType == CommunityType.communityTypeLarge:
      getValue = getLargeCommValue
   else:
      assert False, f"Unsupported community type: {commType}"
   return [ getValue( comm ) for comm in comms ]

def serializeCommunityList( commList, commType, **kwargs ):
   commValList = parseCommunityList( commList, commType )
   return " ".join(
      [ commValueToPrint( comm, commType, **kwargs ) for comm in commValList ] )

def getNumberFromCommValue( comm ):
   if type( comm ) is CommunityValueType:
      return ( comm.part0 << 64 ) | ( comm.part1 << 32 ) | comm.part2
   else:
      return comm

# Set match rule value field
def setMatchRuleValue( rule, value ):
   valueType = MatchAttributes[ rule.option ][ 'type' ]
   if valueType == 'int':
      rule.intValue = int( value )
   elif valueType == 'string':
      rule.strValue = value
   elif valueType == 'enum':
      rule.intValue = MatchAttributes[ rule.option ][ 'values' ][ value ]
   elif valueType == 'ipaddr':
      rule.ipAddrValue = value
   elif valueType == 'ip6addr':
      rule.ip6AddrValue = value
   elif valueType == 'range':
      rule.intValue = value.start
      rule.int2Value = value.end if value.end else value.start
   elif valueType == 'u32range':
      rule.u32Range = Tac.Value( "Routing::RouteMap::U32Range",
                                 value.start, value.end )
   elif valueType == 'set':
      tmpCommListSet = Tac.Value( "Routing::RouteMap::CommListSet" )
      for commList in value:
         tmpCommListSet.commListSet[ commList ] = True
      rule.commListSet = tmpCommListSet
   else:
      assert False

def getExtCommLbwDisplayValue( value ):
   valueType = value.split()
   if ':' not in valueType[1]:
      return valueType[1]
   valueSplit = valueType[1].split( ':' )
   return valueSplit[1]

# capi med type string to med type enum string
capiMedTypes = {
   "metric" : routeMapMedType.medNormal,
   "additive" : routeMapMedType.medAdditive,
   "subtractive" : routeMapMedType.medSubtractive,
   "multiplicative" : routeMapMedType.medMultiplicative,
   "igp-nexthop-cost" : routeMapMedType.medIgpNexthopCost,
   "add-igp-nexthop-cost" : routeMapMedType.medAddIgpNexthopCost,
   "value-add-igp-nexthop-cost" : routeMapMedType.medValueAddIgpNexthopCost,
   "igp-metric" : routeMapMedType.medIgpMetric,
   "add-igp-metric" : routeMapMedType.medAddIgpMetric,
   "value-add-igp-metric" : routeMapMedType.medValueAddIgpMetric
   }

# capi aigp metric type string to aigp metric type enum
capiAigpMetricTypes = {
   "metric": routeMapAigpMetricSetType.aigpMetricSetNormal,
   "igp-next-hop-cost": routeMapAigpMetricSetType.aigpMetricSetIgpNexthopCost,
   "igp-metric": routeMapAigpMetricSetType.aigpMetricSetIgpMetric,
   "additive": routeMapAigpMetricSetType.aigpMetricSetAdditive,
   "subtractive": routeMapAigpMetricSetType.aigpMetricSetSubtractive
   }

duplicateOperation = Tac.Type( "Acl::UrlImportDuplicateOperation" )

def duplicateOperationToCliStr( operation ):
   if operation is None or operation == duplicateOperation.duplicateError:
      return None
   if operation == duplicateOperation.duplicateIgnore:
      return "duplicate ignore"
   if operation == duplicateOperation.duplicateOverride:
      return "duplicate override"
   assert False, "Invalid operation: " + operation

def duplicateCliToOperation( cliStr ):
   if not cliStr or cliStr == "":
      return duplicateOperation.duplicateError
   if "ignore" in cliStr:
      return duplicateOperation.duplicateIgnore
   if "override" in cliStr:
      return duplicateOperation.duplicateOverride
   assert False, "Invalid cli: " + cliStr

def duplicateOperationToStr( operation ):
   if operation is None or operation == duplicateOperation.duplicateError:
      return "duplicateError"
   if operation == duplicateOperation.duplicateIgnore:
      return "duplicateIgnore"
   if operation == duplicateOperation.duplicateOverride:
      return "duplicateOverride"
   assert False, "Invalid operation: " + operation

def getErrorMessageCannotDeleteBecauseUsedBy( constructType, constructName, deps ):
   """ @brief Constructs the error message in case the user attempts to delete a
              policy-construct which has been currently using by a route-map.
       @param contructType  A CAPI-string for the policy-construct.
       @param constructName The identifier name of the policy-construct.
       @param [in] deps   A set of dependencies. That is, a set of all route-map
                          entries which use the construct.
                          Deps must be iterable. Each item is a pair (tuple size 2):
                            - first is the name of the route-map;
                            - second is the seqno of the route-map.
   """
   return '{} "{}" cannot be deleted because it is currently being used' \
      ' by the following route maps: "{}"'.format(
         constructType,
         constructName,
         ', '.join( [ f'{d[ 0 ]}-{d[ 1 ]}' for d in sorted( deps ) ] ) )
