#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import re
import Tac

class RouteTrack():
   def __init__( self, qtDir=None ):
      self.qtDir = qtDir or os.environ[ 'QUICKTRACEDIR' ]
      self.fileSuffix = '-RouteTrack.qt'
      self.lineRe = re.compile(
         r'(?P<qtTimestamp>\d+-\d+-\d+ \d+:\d+:\d+\.\d+) 0 \+\d+ "'
         r'ns=(?P<nsSinceBoot>\d+) '
         r'epoch=(?P<epoch>\d+) '
         r'id=(?P<eventId>\d+|\*) '
         r'(?P<rest>.+)"$' )
      self.epochRe = re.compile( r'First message for epoch, VRF '
                                 r'(?P<vrfName>\S+) '
                                 r'\((?P<vrfId>\d+)\) '
                                 r'(?P<prefix>.+)' )
      self.eventRe = re.compile( r'(?P<context>\S+) (?P<description>.+)' )

   def getMergedLogForEpoch( self, epochToMatch ):
      names = os.listdir( self.qtDir )
      rtFiles = [ f for f in names if f.endswith( self.fileSuffix ) ]
      fileLogs = [ self.readFile( f ) for f in rtFiles
                   if f.endswith( self.fileSuffix ) ]
      return self.buildMergedLogForEpoch( epochToMatch, fileLogs )

   def readFile( self, filename ):
      processName = filename.rstrip( self.fileSuffix )
      qt = Tac.run( [ 'qttail', '-c', f'{self.qtDir}/{filename}' ],
                    stdout=Tac.CAPTURE ).split( '\n' )
      log = []
      lastEpoch = None
      for line in qt:
         m = self.lineRe.match( line )
         if not m:
            continue
         epoch = int( m[ 'epoch' ] )
         eventId = m[ 'eventId' ]
         rest = m[ 'rest' ]
         if eventId == '*':
            m2 = self.epochRe.match( rest )
            lastEpoch = ( epoch, m2[ 'vrfName' ],
                          int( m2[ 'vrfId' ] ),
                          m2[ 'prefix' ] )
         else:
            description = ''
            context = ''
            m2 = self.eventRe.match( rest )
            if m2:
               description = m2[ 'description' ]
               context = m2[ 'context' ]
            log.append( { 'nsSinceBoot' : int( m[ 'nsSinceBoot' ] ),
                          'timestamp' : m[ 'qtTimestamp' ],
                          'description' : description,
                          'agent' : processName,
                          'epoch' : epoch,
                          'eventId' : int( eventId ),
                          'context' : context } )
      return ( processName, lastEpoch, log )

   def buildMergedLogForEpoch( self, epochToMatch, fileLogs ):
      mergedLog = []
      wrappedLogs = set()
      for processName, lastEpoch, log in fileLogs:
         if not lastEpoch:
            wrappedLogs.add( processName )
         for event in log:
            if epochToMatch and epochToMatch != event[ 'epoch' ]:
               continue
            mergedLog.append( event )
      events = sorted( mergedLog, key=lambda e: e[ 'nsSinceBoot' ] )
      ret = { 'events' : events }
      if wrappedLogs:
         ret[ 'wrappedLogs' ] = sorted( wrappedLogs )
      if epochToMatch:
         ret[ 'matchedEpoch' ] = epochToMatch
      return ret

