# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# Module contains utility methods for accessing Intf related information from sysdb

import os
import Tac

@Tac.memoize
def mgmtPrefix():
   # MGMT_INTF_PREFIX allows a test to override the prefix that
   # identifies a management interface
   return os.environ.get( "MGMT_INTF_PREFIX", "Management" )

def isManagement( intfName ):
   return intfName.startswith( mgmtPrefix() )

def isLoopback( intfName ):
   return intfName.startswith( 'Loopback' )

def isInternal( intfName ):
   return intfName.startswith( 'Internal' )

def isFabric( intfName ):
   return intfName.startswith( 'Fabric' )

#--------------------------------------------------------------------
# Returns if the interface supports virtual routing or not
#--------------------------------------------------------------------
def intfSupportsVirtualRouter( intfName ):
   return not isManagement( intfName ) and intfName.startswith( ( "Vlan", "Test" ) )

#-------------------------------------------------------------------
# Helper Methods
#-------------------------------------------------------------------
#
# Returns True if the interface is configured as routed interface.
# If includeEligible is set then it returns True for intfs which are capable of
# routing. intfStatusDir points to interface/status/all directory.

def _routedIntf( intfName, intfStatusDir, includeEligible ):
   # Previously, if includeEligible was False then only non-switchport interfaces
   # were returned. This was determined by looking for 'Bridged' forwarding model
   # input for those interfaces. In the new forwarding model design, this introduced
   # a dependency on Ebra, which was not acceptable. So it was decided to just ignore
   # this parameter and always return all interfaces (except Vxlan).
   if 'Vxlan' in intfName:
      return False
   elif intfName.startswith( ( 'WatchIn', 'WatchOut', 'WatchPassthrough',
         'MuxIn', 'MuxOut',
         'Upstream', 'Downstream' ) ):
      # The MetaMux, MetaWatch, Upstream/Downstream  interfaces are a type of
      # EthPhyIntf without routing capabilities and thus should be filtered out
      # by this function.
      # Ideally, we should be filtering out interfaces based on the forwardingModel
      # which is `intfForwardingModelDataLink` for MetaWatch. Unfortunately, the
      # intfStatus is not created up until the MetaWatch agent starts while the CLI
      # allows creating config for these interfaces. Then, this function returns True
      # when executing `show running-config all` when it should not for the MetaWatch
      # interfaces.
      # Because of this we are filtering directly based on the name of of the intfs
      # until there is other better mechanism to achieve this.
      # See BUG611278 for details.
      return False
   return True

#
# Returns list of interface names which are configured for routing.
# If includeEligible is set to True then the names of all interfaces capable of
# routing are returned. Else only currently configured interfaces are returned.
# If excludeLpMgmt is set then loopback and mgmt interfaces are excluded.
#
def _allRoutingIntfNames( requireMounts, includeEligible=False,
                          excludeLpMgmt=False ):
   intfConfigDir = requireMounts[ 'interface/config/all' ]
   intfStatusDir = requireMounts[ 'interface/status/all' ]
   intfNames = []
   for intfName in intfConfigDir.intfConfig:
      if isInternal( intfName ):
         continue

      if excludeLpMgmt and ( isLoopback( intfName ) or isManagement( intfName ) ):
         continue

      # Fabric interfaces now support CLI, but not any routing configurations
      if isFabric( intfName ):
         continue

      if _routedIntf( intfName, intfStatusDir, includeEligible ):
         intfNames.append( intfName )

   return intfNames

#-----------------------------------------------------------------------------------
# Returns the list of interface names which support IP address configuration.
# If 'includeEligible' is set to True then the names for all interfaces capable
# of configuring IP address are returned. Else only currently configured
# L3 interface names are returned.
#-------------------------------------------------------------------------------
def allIpIntfNames( requireMounts, includeEligible=False ):
   return _allRoutingIntfNames( requireMounts, includeEligible )

#----------------------------------------------------------------------------
# Returns the list of interface names which support routing protocols.
# If 'includeEligible' is set to True then the names for all interfaces capable
# of configuring IP address are returned. Else only currently configured
# L3 interface names are returned.
#
# This method is currently used by CliSavePlugin, so only interfaces which
# support routing protocol Cli's are returned.
# It will be better to check interface's capability using an attribute in sysdb.
# But nothing exists for now, so filtering out loopback and mgmt using names.
#----------------------------------------------------------------------------
def allRoutingProtocolIntfNames( requireMounts, includeEligible=False ):
   return _allRoutingIntfNames( requireMounts, includeEligible, excludeLpMgmt=True )
