# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

'''
Functions for generating an rsyslog template string.
Start with the "start" function, use the other functions to
add more properties/constants, and finally finish with "end".
'''

def start( name ):
   ''' Generate rsyslog template header '''
   return 'template( name="%s" type="list" ) {\n' % name

def priority():
   return constant( "<" ) + generalProperty( "pri" ) + constant( ">" )

def timestamp( dateFormat='rfc3164' ):
   return '   property( name="timestamp" dateFormat="%s" )\n' % dateFormat

def hostname():
   return '   constant( value=" #HOSTNAME_PLACEHOLDER# " )\n'

def syslogVersion( rfc5424 ):
   return '   constant( value="1 " )\n' if rfc5424 else ''

# Send NILVALUE for PROCID, MSGID and STRUCTURED-DATA
def syslogPidMsgId( rfc5424 ):
   # TMPL.message( addSpace=True ) adds a space after last '-'
   return '   constant( value=" - - -" )\n' if rfc5424 else ''

def syslogTag( fromPosition=None, toPosition=None ):
   conf = '   property( name="syslogtag"'
   if fromPosition:
      conf += ' position.from="%s"' % fromPosition
   if toPosition:
      conf += ' position.to="%s"' % toPosition
   return conf + ' )\n'

def message( addSpace=False, dropLastLF=False ):
   conf = '   property( name="msg"'
   if addSpace:
      conf += ' spifno1stsp="on"'
   if dropLastLF:
      conf += ' droplastlf="on"'
   return conf + ' )\n'

def msgRegex( expr, submatch, regexType='ERE', noMatchMode="BLANK",
              lowercase=False ):
   conf = ( '   property( name="msg" regex.type="%s" regex.submatch="%d"'
          + ' regex.nomatchmode="%s" regex.expression="%s"' )
   if lowercase:
      conf += ' caseConversion="lower"'
   return conf % ( regexType, submatch, noMatchMode, expr ) + ' )\n'

def msgField( field, delim=":" ):
   delimAscii = ord( delim )
   conf = '   property( name="msg" field.number="%d" field.delimiter="%d" )\n'
   return conf % ( field, delimAscii )

def generalProperty( name ):
   ''' An rsyslog template property with no attributes '''
   return '   property( name="%s" )\n' % name

def constant( value ):
   if value:
      return '   constant( value="%s" )\n' % value
   return ''

def end():
   ''' Finalize rsyslog template '''
   return '}\n'
