# Copyright (c) 2006-2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Pci
import os

# Defines the register offsets of the Scd version 1

PciId = Pci.Id( vendor=0x3475, device=0x0001 )
PciSubsystemId = Pci.Id( vendor=0x3475, device=0x0001 )
PciRevisionId = 0x01
PciClassCode = 0x088000

def isLegacySystem():
   """Check if this system does not support moduleType register."""
   return os.system(
      "grep -E -q 'platform=(raven|crow|magpie|mendocino)' /proc/cmdline" ) == 0

class deviceInfo:
   def __init__( self, dev ):
      self.device = dev
      self.pciAddress = None
      self.modTypeRegVal = None
      self.modType = None
      self.scdId = None
      if dev is not None:
         self.pciAddress = str( dev.address() )
         scd = dev.sysfsPath( "resource0" )
         if isLegacySystem():
            # Legacy systems don't have module type register, so override
            # the modType to fixed and give default scdId 0.
            # All legacy systems are fixed and have one scd.
            self.modType = 0
            self.scdId = 0
         elif os.path.isfile( scd ):
            resource = Pci.MmapResource( scd, readOnly=True )
            try:
               self.modTypeRegVal = resource.read32( 0x150 )
               # pylint: disable-next=consider-using-in
               if ( self.modTypeRegVal != 0xffffffff and 
                  self.modTypeRegVal != 0xdeadface ):
                  self.modType = self.modTypeRegVal & 0xff
                  self.scdId = ( self.modTypeRegVal >> 8 ) & 0xff
            except ValueError:
               # A ValueError may occur from read 32 if, for example,
               # resource size is smaller than the register value
               # to read (in this case, 0x150)
               pass
            resource.unmap()

def scdPciDevice( pciAddress=None, moduleType=None, scdId=None ):
   if pciAddress and ( moduleType is not None or scdId is not None ):
      raise ValueError( "pciAddress cannot be accompanied by another argument" ) 
   eligible = [ dev for dev in map( deviceInfo, Pci.allDevicesById( PciId ) )
         if scdAttributeMatch( dev, pciAddress, moduleType, scdId ) ]
   if not eligible:
      return None
   # Sort by following metric:
   # Prefer moduleType 0 or 1 if moduleType is not passed in. 
   # Notice if moduleType argument is passed in, all eligible scds have the same
   # moduleType and this preference is meaningless.
   # Prefer lower scdId. scdId None (technically lower than 0) is sent to the back.
   # If multiple scds of the same scdId, prefer one with higher pciAddress.
   eligible.sort( key=lambda dev: ( dev.modType in [ 0, 1 ], dev.scdId is not None,
      -dev.scdId if dev.scdId else 0, dev.pciAddress ), reverse=True )
   return eligible[0].device

def scdAttributeMatch( devInfo, pciAddress=None, moduleType=None, scdId=None ):
   if ( ( pciAddress is not None and devInfo.pciAddress !=
         str( Pci.Address( pciAddress ) ) ) or
        ( scdId is not None and devInfo.scdId != scdId ) or
        ( moduleType is not None and devInfo.modType != moduleType)
      ):
      return False
   return True

def scdPciOffset():
   dev = scdPciDevice()
   if dev is None:
      return None
   return 0x0

def scdPciResourceFile( pciAddress=None, moduleType=None, scdId=None ):
   dev = scdPciDevice( pciAddress, moduleType, scdId )
   if dev is None:
      return None
   return dev.sysfsPath( "resource0" )

def scdPciResource():
   dev = scdPciDevice()
   if dev is None:
      return None
   return dev.resource( 0 )

class InterruptBlock:
   maskSetOffset = 0x00
   maskClearOffset = 0x10
   statusOffset = 0x20

class ResetGpo:
   setOffset = 0x00
   clearOffset = 0x10

class SmbusAccelBlock:
   requestLowOffset = 0x00
   requestHighOffset = 0x10
   controlStatusOffset = 0x20
   responseOffset = 0x30

# ROM Registers
Rom_FpgaVersion_Addr = 0x00
Rom_FdlAccessMethod_Addr = 0x04
Rom_SlotidAddr_Addr = 0x08
Rom_SlotidMask_Addr = 0x0c
Rom_SmbusReqLowAddr_Addr = 0x10
Rom_SmbusReqHighAddr_Addr = 0x14
Rom_SmbusControlStatusAddr_Addr = 0x18
Rom_SmbusResponseAddr_Addr = 0x1c
Rom_BusSelect_Addr = 0x20
Rom_DeviceId_Addr = 0x24

Crc_Error_Test = 0x140
Cet_Crc_Test_Fpga = 1
Cet_Force_Crc_Error = 2
