# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from enum import IntEnum
import sys

import Pci
import ScdRegisters
from ScdRegisters import PciId

class ScdModuleType( IntEnum ):
   FIXED_SYSTEM = 0
   SUPERVISOR = 1
   FABRIC = 2
   LINECARD = 3
   SWITCHCARD = 4
   UNKNOWN = 5

moduleTypeMap = {
         ScdModuleType.FIXED_SYSTEM: 'Fixed-System', 
         ScdModuleType.SUPERVISOR: 'Supervisor', 
         ScdModuleType.FABRIC: 'Fabric', 
         ScdModuleType.LINECARD: 'Linecard', 
         ScdModuleType.SWITCHCARD: 'Switchcard',
         ScdModuleType.UNKNOWN: 'Unknown'}

noScdError = "No matching scd device found on this machine"
listFormat = '{:<15}{:<6}{:<15}'

def binary( value, numBytes=4 ):
   """String rep of an integer in binary, similar to 'hex'.
   Turns 0xc7 into 1100 0111.
   The optional bytes parameter indicates how many bytes wide the value is."""
   return "".join( ( ( value>>n &1 ) and "1" or "0" ) +
         ( ( n % 4 == 0 ) and " " or "" )
                    for n in range( numBytes*8-1,-1,-1 ) )

def scdErrorAndExit( message ):
   # pylint: disable-next=consider-using-f-string
   sys.stderr.write( "Error: %s\n" % message )
   sys.exit( 1 )

def getMatchingScds( pciAddress, scdType, scdId):
   return [ dev for dev in
         map( ScdRegisters.deviceInfo, Pci.allDevicesById( PciId ) )
         if dev.device is not None and
            ScdRegisters.scdAttributeMatch( dev,
                                            pciAddress=pciAddress,
                                            moduleType=scdType,
                                            scdId=scdId ) ]


def scdList( pciAddress, scdType, scdId ):
   """Helper for list command"""
   devices = getMatchingScds( pciAddress, scdType, scdId)

   if not devices:
      scdErrorAndExit( noScdError )

   result = [ listFormat.format( "pciAddress", "scdId", "moduleType" ) ]
   for dev in devices:
      if dev.modType > ScdModuleType.UNKNOWN:
         dev.modType = ScdModuleType.UNKNOWN

      modTypeStr = moduleTypeMap [ dev.modType ]
      result.append( listFormat.format( dev.pciAddress, 
                                        dev.scdId, modTypeStr ) )
   print( '\n'.join( result ) )

def scdIO( method, pciAddress, scdType, scdId, regAddress, data=None ):
   """Helper for read/write command"""
   scd = ScdRegisters.scdPciResourceFile( pciAddress=pciAddress,
                                          scdId=scdId, moduleType=scdType )
   if not scd:
      scdErrorAndExit( noScdError )

   val = None
   offset = ScdRegisters.scdPciOffset()
   if method == 'write':
      val = Pci.MmapResource( scd ).write32( offset + regAddress, data )
   elif method == 'read':
      val = Pci.MmapResource( scd, readOnly=True ).read32( offset + regAddress )
      print( hex( regAddress ), hex( val ), "==", binary( val, 4 ) )
   else:
      scdErrorAndExit( "unrecognized argument: " + method )

   return val
