#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
from functools import total_ordering
import _Tac
import Tac
import Tracing
import os
import threading

t3 = Tracing.trace3
t9 = Tracing.trace9

SharedMemEmType = Tac.Type( 'TacSharedMem::EntityManager' )

def entityManager( sysname=None, local=None, sysdbEm=None ):
   if sysdbEm:
      # It should supply *either* a sysdbEm object *or* a sysname/local pair.
      assert sysname is None and local is None

      # check if the object has 'cEm_' attribute, this means that we are dealing with
      # an EntityManager python wrapper, and what we really want is the underlying
      # cEntityManager
      cEm = getattr( sysdbEm, 'cEm_', None )
      if not cEm:
         cEm = sysdbEm
      assert hasattr( type( cEm ), 'tacType' ) and \
         type( cEm ).tacType.fullTypeName == 'Sysdb::EntityManager'
      sysname = cEm.sysname
      local = cEm.isLocalEm

   assert sysname is not None and local is not None
   return SharedMemEmType.getEntityManager( sysname, local )

def cohabMultithreadMounts():
   return SharedMemEmType.cohabMultithreadMounts()

def cohabMultithreadMountsIs( cmm ):
   return SharedMemEmType.cohabMultithreadMountsIs( cmm )

@total_ordering
class EntityProxy( object ):
   ''' Wraps a shared memory entity, mounting that entity upon proxy access. '''
   def __init__( self, shmemEm, entityPath, entityType, entityInfo ):
      self.__dict__[ 'shmemEm_' ] = shmemEm
      self.__dict__[ 'entityPath_' ] = entityPath
      self.__dict__[ 'entityType_' ] = entityType
      self.__dict__[ 'entityInfo_' ] = entityInfo
      self.__dict__[ 'entity_' ] = None

   def __getattr__( self, attr ):
      self.mount_()
      return getattr( self.entity_, attr )

   def __eq__( self, other ):
      self.mount_()
      return self.entity_ == other

   def __lt__( self, other ):
      self.mount_()
      return self.entity_ < other

   def __hash__( self ):
      self.mount_()
      return hash( self.entity_ )

   def __repr__( self ):
      return '<%s for %s>' % ( 'SharedMem.EntityProxy',
                               '%r' % self.entity_ if self.entity_ else
                               '%s at %s' % ( self.entityType_,
                                              self.entityPath_ ) )

   def __bool__( self ):
      self.mount_()
      return bool( self.entity_ )

   __nonzero__ = __bool__ # Python 2

   def __tac_object__( self ):
      self.mount_()
      return self.entity_

   def mount_( self ):
      if not self.entity_:
         # Perform the real mount
         t3( 'SharedMem.EntityProxy.mount_(): mounting', self.entityPath_ )
         self.__dict__[ 'entity_' ] = self.shmemEm_.doMount( self.entityPath_,
                                                             self.entityType_,
                                                             self.entityInfo_ )
         # Delete the EntityManager reference
         self.__dict__[ 'shmemEm_' ] = None

@total_ordering
class AutoUnmountEntityProxy( object ):
   ''' Wraps a shared memory entity, mounting that entity upon proxy access.
       After a period of time, it unmounts the entity if possible'''
   def __init__( self, shmemEm, entityPath, entityType, entityInfo, minRefs=1,
                 timeout=600 ):
      self.shmemEm_ = shmemEm
      self.entityPath_ = entityPath
      self.entityType_ = entityType
      self.entityInfo_ = entityInfo
      self.minRefs_ = minRefs
      self.entity_ = None

      # Whenever an entity is mounted, we'll have a timer running for
      # the specified timeout (default 10min). When the timer fires,
      # we attempt to unmount the entity if it has not been accessed
      # at all during that timeout window. If it has been accessed, we
      # reset the timer and try again next next callback.
      self.accessed_ = False
      self.timeout_ = timeout
      self.unmountRetryTimeout_ = self.timeout_
      self.timer_ = None
      self.lock_ = threading.Lock()

   def force( self ):
      t9( 'AutoUnmountEntityProxy.force()', self.entityPath_ )
      self.mount_()
      ent = self.entity_
      assert ent, 'force should never return a None entity'
      return ent

   def __getattr__( self, attr ):
      self.mount_()
      return getattr( self.entity_, attr )

   def __eq__( self, other ):
      self.mount_()
      return self.entity_ == other

   def __lt__( self, other ):
      self.mount_()
      return self.entity_ < other

   def __hash__( self ):
      self.mount_()
      return hash( self.entity_ )

   def __repr__( self ):
      return '<%s for %s>' % ( 'SharedMem.AutoUnmountEntityProxy',
                               '%r' % self.entity_ if self.entity_ else
                               '%s at %s' % ( self.entityType_,
                                              self.entityPath_ ) )

   def __bool__( self ):
      # This gets invoked as part of the timer callback mechanism - so
      # don't set the accessed flag here or else we'll never be able
      # to unmount. This should be harmless - if client code checks
      # the truthiness of the proxy and then we immediately unmount
      # after - so what? Any subsequent access by client code will
      # just trigger a remount
      self.mount_( False )
      return bool( self.entity_ )

   __nonzero__ = __bool__ # Python 2

   def __tac_object__( self ):
      self.mount_()
      return self.entity_

   def mount_( self, setAccessed=True ):

      t9( 'AutoUnmountEntityProxy.mount_() setAccessed:%s accessed:%s '
          'entity:%s path:%s' %
          ( setAccessed, self.accessed_, self.entity_, self.entityPath_ ) )

      if self.accessed_:
         # Green light path where the entity has been accessed within
         # the last timeout window

         # For sure we're not going to be unmounted, so skip the
         # lock. Even if another thread handles the timer right now -
         # we've got an entire extra timeout window before this
         # becomes a problem.
         assert self.entity_, "Accessed was set but there's no entity"
         return

      # Alright, we potentially have a race against another thread
      # performing an unmount. Grab the lock
      with self.lock_:
         t9( 'AutoUnmountEntityProxy.mount_() after lock grab - accessed:%s '
             'entity:%s path:%s' %
          ( self.accessed_, self.entity_, self.entityPath_ ) )

         if self.entity_:
            if setAccessed:
               self.accessed_ = setAccessed
         else:
            # Perform the real mount
            t3( 'AutoUnmountEntityProxy.mount_(): mounting', self.entityPath_ )
            mg = self.shmemEm_.getMountGroup()
            mg.notifySync = True
            self.entity_ = mg.doMount( self.entityPath_,
                                       self.entityType_,
                                       self.entityInfo_ )
            mg.doClose()

            # If we hold the ActivityLock, we can't run the waitFor or
            # else we'll block indefinitely. This shouldn't really
            # happen in practice, but if it does let's just return the
            # empty (pre-sync) entity instead of deadlocking.
            lockOwner = _Tac.activityLockOwner()
            if lockOwner != threading.current_thread().ident:
               sleep = os.environ.get( 'ShmemProxyTest' ) is None
               Tac.waitFor( lambda: mg.stable, sleep=sleep,
                            description='Proxy MountGroup to be stable' )

            now = Tac.now()
            self.timer_ = Tac.ClockNotifiee( self._timerCallback,
                                             now + self.timeout_ )
            self.unmountRetryTimeout_ = self.timeout_
            t9( 'AutoUnmountEntityProxy.mount_() timer set with now:', now,
                'timeout_:', self.timeout_, 'timeMin:', self.timer_.timeMin )

   def _timerCallback( self ):
      with self.lock_:
         t9( 'AutoUnmountEntityProxy._timerCallback(): accessed:',
             self.accessed_, 'path:', self.entityPath_, 'now:', Tac.now(),
             'timeMin:', self.timer_.timeMin )

         assert self.entity_, 'timer callback invoked with no entity'

         if self.accessed_:
            # The entity was accessed during the last timeout window,
            # so we're not going to unmount it. Reset the accessed
            # flags and start a new timeout window
            self.accessed_ = False
            now = Tac.now()
            self.timer_.timeMin = now + self.timeout_
            self.unmountRetryTimeout_ = self.timeout_
            t9( 'AutoUnmountEntityProxy._timerCallback() reset with now:',
                now, 'timeout_:', self.timeout_, 'timeMin:', self.timer_.timeMin )
         else:
            # Nobody used this thing for an entire timeout window,
            # unmount it if nobody else is referencing it.
            t9( 'AutoUnmountEntityProxy._timerCallback(): attempting to unmount',
                self.entityPath_ )
            self.entity_ = None
            unmounted = self.shmemEm_.doUnmountIfUnused( self.entityPath_,
                                                         self.minRefs_ )
            if unmounted:
               self.timer_ = None
               t9( 'AutoUnmountEntityProxy._timerCallback(): clearing state, '
                   'unmounted:', unmounted, 'path:', self.entityPath_ )
            else:
               # We were unable to unmount since someone is pinning a
               # reference. Let's just keep the entity in our proxy
               # for now and retry later
               self.entity_ = self.shmemEm_.getTacEntity( self.entityPath_ )
               now = Tac.now()
               secondsPerDay = 86400
               self.unmountRetryTimeout_ = min( 2 * self.unmountRetryTimeout_,
                                                secondsPerDay )
               self.timer_.timeMin = now + self.unmountRetryTimeout_
               t9( 'AutoUnmountEntityProxy._timerCallback(): failed to unmount, '
                   'timer reset with now:', now, 'timeMin:', self.timer_.timeMin,
                   'retryTimeout:', self.unmountRetryTimeout_ )
