#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from argparse import ArgumentParser, ArgumentDefaultsHelpFormatter
from TableOutput import createTable
import json
import Tac

copyAttributes = ( 'size', 'memoryAllocationOverhead' )
diffAttributes = ( 'currentAllocations', )

def read( filename ):
   with open( filename ) as f:
      return json.load( f )[ 'types' ]

def merge( a, b ):
   a.update( b )
   return a

def subtract( a, b ):
   if b is None:
      return { attr: a[ attr ] for attr in copyAttributes + diffAttributes }
   if a is None:
      return merge( { attr: 0 - b[ attr ] for attr in diffAttributes },
                    { attr: b[ attr ] for attr in copyAttributes } )
   return merge( { attr: a[ attr ] - b[ attr ] for attr in diffAttributes },
                 { attr: b[ attr ] for attr in copyAttributes } )

def memoryUse( t ):
   if t is None:
      return 0
   return ( ( t[ 'size' ] + t[ 'memoryAllocationOverhead' ] ) *
            t[ 'currentAllocations' ] )

def addMemoryUse( t ):
   t.update( { 'memoryUse': memoryUse( t ) } )
   return t

def compare( fileA, fileB, tableWidth, limit, skipNoDiff ):
   a = read( fileA )
   b = read( fileB )
   allTypes = set( a.keys() ) | set( b.keys() )
   diff = { typeName: addMemoryUse( subtract( a.get( typeName ),
                                              b.get( typeName ) ) )
            for typeName in allTypes }
   sortedTypes = sorted( diff.items(), reverse=True,
                         key=lambda t:t[ 1 ][ 'memoryUse' ] )
   # Prepare sorted data for display
   hdr = ( ( 'type', 'l' ),
           ( 'size +', 'r', ( 'overhead', ) ),
           ( 'object count', 'c',
             ( 'file1', 'file2', 'delta' ) ),
           ( 'memory size', 'c',
             ( 'file1', 'file2', 'delta' ) ) )
   maybeTruncateTypeName = \
      lambda typeName: typeName[ : 70 ] if tableWidth != 0 else typeName

   table = createTable( hdr, tableWidth=tableWidth if tableWidth else None )
   rows = [ [ maybeTruncateTypeName( typeName ),
              # pylint: disable-next=consider-using-f-string
              '%d + %2d' % ( typeInfo[ 'size' ],
                             typeInfo[ 'memoryAllocationOverhead' ] ),
              a.get( typeName, {} ).get( 'currentAllocations', 0 ),
              b.get( typeName, {} ).get( 'currentAllocations', 0 ),
              typeInfo[ 'currentAllocations' ],
              memoryUse( a.get( typeName ) ),
              memoryUse( b.get( typeName ) ),
              memoryUse( typeInfo ) ]
            for typeName, typeInfo in sortedTypes
            if not skipNoDiff or typeInfo[ 'currentAllocations' ] != 0 ]

   if limit:
      rows = rows[ : limit ]
   for row in rows:
      table.newRow( *row )
   print( 'file1:', fileA )
   print( 'file2:', fileB )
   print( table.output() )

def main():
   parser = ArgumentParser( 'Diff two ShowAllocs json files, obtained through '
                            '"dut# show agent sysdb memory allocations all | json | '
                            'tee file:/tmp/sysdb.json"',
                            formatter_class=ArgumentDefaultsHelpFormatter )
   parser.add_argument( 'inputFiles', metavar='filename', type=str, nargs=2,
                        help='files to read' )
   # --tableWidth=0 is overloaded to mean no truncation of columns
   parser.add_argument( '--tableWidth', type=int, default=200,
                        help='Specify 0 for unlimited width.' )
   parser.add_argument( '--limit', type=int, default=0 )
   parser.add_argument( '--skipNoDiff', type=bool, default=True )
   args = parser.parse_args()
   compare( *args.inputFiles, tableWidth=args.tableWidth, limit=args.limit,
            skipNoDiff=args.skipNoDiff )

if __name__ == '__main__':
   main()
