# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

REALLOCATIONS = "Reallocations remaining"
WEAR_LIFE = "Wear life remaining"
HEALTH = "Health"

STATUS = [ "FAILED", "OK" ]

def smartHealthStatus( status ):
   """Returns a dict of dicts like so:
   {
      "drive": {
                  "Wear life remaining": 0.8,
                  "Reallocations": 1.0,
                  "Health": "OK",
               },
      "usb1":  {
                  "Health": "OK",
                },
      ...
   }
   """

   healths = {}

   if not status.connectedDevices:
      return healths

   connectedDisks = status.connectedDevices.disk

   if status.smartData:
      deviceData = status.smartData.deviceData
      for serial, device in deviceData.items():
         results = { HEALTH: STATUS[ int( device.healthStatusOk ) ] }

         lifetime = device.lifetimeRemaining
         if lifetime.present:
            results[ WEAR_LIFE ] = lifetime.value.standard / 100

         reallocations = device.reallocatedSectorCount
         if reallocations.present:
            results[ REALLOCATIONS ] = reallocations.value.standard / 100

         mount = connectedDisks[ serial ].fsName()
         healths[ mount ] = results

   if status.nvmeData:
      nvmeDeviceData = status.nvmeData.deviceData
      for serial, device in nvmeDeviceData.items():
         mount = connectedDisks[ serial ].fsName()
         if mount not in healths:
            healths[ mount ] = {}
         healths[ mount ][ WEAR_LIFE ] = device.lifetimeRemaining.value / 100
         healths[ mount ][ REALLOCATIONS ] = device.availableSpare.value / 100

   return healths
