# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
import Cell

__defaultTraceHandle__ = Tracing.Handle( 'TapaggSnmp' )
t0 = Tracing.trace0

# So what's going on here? Currently the tapagg policy status
# is instantiated in a platform specific directory, 'sand'. We
# would like to avoid bringing in that PD dependency - the only
# other platform that supports tapagg today apparently is 7130 and
# that also seems to use 'sand'. Regardless, instead of bringing
# that dependency directly here, we setup a Tac::Dir reactor and
# use the notification of the status showing up inside it to trigger
# our state machine kick off.
class TapaggSnmpStatusDirReactor( Tac.Notifiee ):
   notifierTypeName = 'Tac::Dir'

   def __init__( self, path, callback ):
      self.callback = callback
      Tac.Notifiee.__init__( self, path )
      for key in self.notifier_:
         self.handleStatus( key )

   @Tac.handler( 'entityPtr' )
   def handleStatus( self, key ):
      self.callback( self.notifier_.get( key ) )

class TapaggSnmpPlugin:
   smControl = Tac.newInstance( "Arx::SmControl" )

   # This callback will trigger our core SNMP SM that will
   # populate the MIB tables. But right now we make this explicit
   # assumption and assert that there's only one status entity that
   # shows up. I am not sure when this will be broken but I would like
   # to confirm that contract.
   # Deleting the SM when the status entity goes away allows us to do
   # some testing around the MIB table clearing (say, we want to get rid
   # of the SM when tapagg is disabled)
   def tapaggPolicyStatusCallback( self, policyStatus ):
      if policyStatus:
         if self.tapaggPolicySm:
            assert False, "Unhandled case: multiple tapagg status entities"
         t0( 'Starting the policy table state machine' )
         self.tapaggPolicySm = Tac.newInstance( 'TapaggSnmp::TapaggPolicyTableSm',
               self.tapaggPolicyConfig.pmapType, policyStatus, self.mib,
               self.smControl )
      else:
         t0( 'Stopping the policy table state machine and clearing mib entries' )
         self.tapaggPolicySm = None
         self.mib.tapaggPolicyTable.clear()
         self.mib.tapaggPolicyClassTable.clear()

   def __init__( self, ctx ):
      self.ctx = ctx
      em = ctx.entityManager()
      sysdb = em.root()

      snmpRoot = sysdb.parent[ 'snmp' ]
      mibDir = snmpRoot.mkdir( 'tapaggSnmp' )
      self.mib = mibDir.newEntity( 'TapaggSnmp::TapaggMib', 'tapaggSnmpMib' )

      mg = em.mountGroup()
      self.tapaggPolicyConfig = mg.mount( 'tapagg/pmapconfig',
         'TapAgg::PmapConfig', 'r' )

      cellId = Cell.cellId()
      dirPath = f'cell/{str( cellId )}/tapagg/pmapstatus'
      self.tapaggPolicyStatusDir = mg.mount( dirPath, 'Tac::Dir', 'ri' )
      self.tapaggPolicyStatusReactor = None
      self.tapaggPolicySm = None

      def _finishMounts():
         self.tapaggPolicyStatusReactor = TapaggSnmpStatusDirReactor(
               self.tapaggPolicyStatusDir,
               self.tapaggPolicyStatusCallback )

      mg.close( lambda: None )
      ctx.callbackIs( _finishMounts )


def Plugin( ctx ):
   TapaggSnmpPlugin( ctx )
