# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import Tac
import Tracing
import Logging

traceHandle = Tracing.Handle( 'MgmtSecuritySslMonitorExpiry' )
error = traceHandle.trace0
warn = traceHandle.trace1
info = traceHandle.trace2
trace = traceHandle.trace3
debug = traceHandle.trace4

ErrorType = Tac.Type( "Mgmt::Security::Ssl::ErrorType" )

MGMTSECURITY_SSL_CERT_CLOSE_TO_EXPIRY = Logging.LogHandle(
   "MGMTSECURITY_SSL_CERT_CLOSE_TO_EXPIRY",
   severity=Logging.logWarning,
   fmt=( "SSL profile '%s' - %s '%s' will expire in %d %s." ),
   explanation=( "The SSL profile uses a certificate "
                 "that will expire in the near future. "
                 "This warning was logged due to configurations made "
                 "in 'management security' mode, using "
                 "'ssl monitor expiry ...' commands." ),
   recommendedAction=( "The SSL certificate should be renewed "
                       "as soon as possible. "
                       "Check 'show management security ssl profile' to "
                       "list these warning for all certificates "
                       "in all profiles." )
)

SECS_IN_HOUR = 3600
SECS_IN_DAY = 24 * SECS_IN_HOUR

def _getMonitoredTimeUntilExpiry( monitorConfig, now, na ):
   """
   Finds the matching configured interval of time (and associated repeat interval),
   in regards to certificate's expiration date.
   For example, we can take the following monitor expiry configuration:
   [ 90 days until expiry log every 10 days ],
   [ 30 days until expiry log every 7 days ],
   [ 10 days until expiry log every 1 day ].
   If certificate will expire in 70 days, it will return (90 * 3600, 10 * 3600).
   If certificate will expire in 4 days, it will return (10 * 3600, 7 * 3600).
   If certificate will expire in 200 days, it will return (0, 0).

   Parameters:
      monitorConfig: Configured monitor expiry entity
      now (float): Current time
      na (float): Certificate expiration time ("Not after")
   Returns:
      timeUntilExpiryAndRepeatInterval (int, int): If found, returns matching time 
      until expiry and repeat interval (both in seconds). Otherwise returns (0, 0).
   """

   assert ( now <= na ) # pylint: disable=superfluous-parens

   certTimeUntilExpiry = na - now

   timeUntilExpiry = min( [ time for time in monitorConfig if
                            time >= certTimeUntilExpiry ], default=0 )

   repeatInterval = 0
   if timeUntilExpiry:
      repeatInterval = monitorConfig[ timeUntilExpiry ].repeatInterval

   return timeUntilExpiry, repeatInterval

def _getNextMonitoredTimeUntilExpiry( monitorConfig, currentTimeUntilExpiry=
                                      Tac.endOfTime ):
   """
   Similar to _getMonitoredTimeUntilExpiry, the only difference being that it finds
   the next configured time until expiry, based on the current one.
   For example, we can take the following monitor expiry configuration:
   [90 days until expiry log every 10 days],
   [30 days until expiry log every 7 days],
   [10 days until expiry log every 1 day].
   If currentTimeUntilExpiry is 90 * 3600, it will return
   (30 * 3600, 7 * 3600).
   If currentTimeUntilExpiry is 30 * 3600, it will return
   (10 * 3600, 1 * 3600).
   If currentTimeUntilExpiry is Tac.endOfTime, it will return 
   (90 * 3600, 10 * 3600).
   If currentTimeUntilExpiry is 10 * 3600 or no monitor expiry configuration
   was made, it will return (0, 0),

   Parameters:
      monitorConfig: Configured monitor expiry entity
      currentTimeUntilExpiry (int): Current configured interval of time used
      for the certificate
   Returns:
      timeUntilExpiry (int): If found, returns matching time 
      until expiry. Otherwise returns 0.
   """

   return max( [ time for time in monitorConfig if time < currentTimeUntilExpiry ],
               default=0 )

def getDisplayableTimeAndUnit( seconds ):
   """
   This function converts seconds until expiry to either hours or days, in order to
   be displayable to the end user.
   When there are more than 24 hours until expiry, it rounds the returned value to 1
   day. Otherwise, it rounds the returned value to 1 hour. It never returns an 
   amount of time less than 1.

   Parameters:
      seconds (int): Number of seconds until certificate expiration
   Returns:
      timeAndUnit (int, str): Amount of time until expiration and the time unit
      as string, either "day(s)" or "hour(s)". 
   """

   timeUntilExpiry = seconds * 1. / 3600

   if timeUntilExpiry < 24:
      timeUnit = "hours"
   else:
      timeUnit = "days"
      timeUntilExpiry = timeUntilExpiry / 24

   timeUntilExpiry = int( round( timeUntilExpiry ) )
   if timeUntilExpiry <= 1:
      timeUnit = timeUnit[ : -1 ]
      timeUntilExpiry = 1
   return timeUntilExpiry, timeUnit

def getLastLogTime( profileStatus, cert ):
   """
   Gets the last time when a warning was logged for the certificate.

   Parameters:
      profileStatus: Profile status entity
      cert (str): Certificate name
   Returns:
      lastLogTime (float): Last time when a warning was logged for the certificate
   """
   if cert not in profileStatus.monitorExpiryLastLogTime:
      return 0

   return profileStatus.monitorExpiryLastLogTime[ cert ]

def validateCertificateExpiry( monitorConfig, now, na ):
   """
   Validates if the certificate is in the range of any configured timeUntilExpiry 
   and returns certCloseToExpiry if so. Otherwise, returns noError.

   Parameters:
      monitorConfig: Configured monitor expiry entity
      now (float): Current time
      na (float): Certificate expiration time ("Not after")
   Returns:
      errorType (ErrorType): Either certCloseToExpiry or noError.
   """
   timeUntilExpiry, _ = _getMonitoredTimeUntilExpiry( monitorConfig, now, na )

   if not timeUntilExpiry:
      return ErrorType.noError

   return ErrorType.certCloseToExpiry

def logWarnings( monitorConfig, profileName, monitorExpiryDict, warningList ):
   """
   Logs certificate expiry warnings to syslog.

   Parameters:
      monitorConfig: Configured monitor expiry entity
      profileName (str): Name of the current profile
      monitorExpiryDict (dict): Dictionary storing last log times for 
      all certificates in the current profile
      warningList (dict): List of warnings for the current profile.
   Returns:
      None
   """

   now = Tac.utcNow()

   for profileError in warningList:
      if profileError.errorType != ErrorType.certCloseToExpiry:
         continue

      certName = profileError.errorAttrValue
      assert ( certName in monitorExpiryDict ) # pylint: disable=superfluous-parens

      na = float( profileError.errorTypeExtra )
      _, repeatInterval = _getMonitoredTimeUntilExpiry( monitorConfig, now, na )
      timeSinceLastMonitorWarning = now - monitorExpiryDict[ certName ]

      displayableTime, timeUnit = getDisplayableTimeAndUnit( na - now )

      if ( timeSinceLastMonitorWarning >= repeatInterval ) or \
         ( displayableTime == 1 and
           ( timeUnit == "day" and timeSinceLastMonitorWarning >= SECS_IN_DAY ) or
           ( timeUnit == "hour" and timeSinceLastMonitorWarning >= SECS_IN_HOUR ) ):
         Logging.log( MGMTSECURITY_SSL_CERT_CLOSE_TO_EXPIRY, profileName,
                  profileError.errorAttr, profileError.errorAttrValue,
                  displayableTime, timeUnit )
         monitorExpiryDict[ certName ] = now

def getEarliestTimeCheck( monitorConfig, monitorExpiryDict, certName, now, na ):
   """
   Computes the next moment of time when the certificate expiry feature should
   check if a warning should be logged or not, based on configuration.

   Parameters:
      monitorConfig: Configured monitor expiry entity
      monitorExpiryDict (dict): Dictionary storing last log times for all 
      certificates in the current profile
      certName (str): Name of the current certificate
      now (float): Current time
      na (float): Certificate expiration time ("Not after")
   Returns:
      earliestTimeCheck (float): Next moment in time when a certificate expiry
      check should be made.
   """

   assert ( now <= na ) # pylint: disable=superfluous-parens
   assert ( certName in monitorExpiryDict ) # pylint: disable=superfluous-parens

   nextTimeUntilExpiry = _getNextMonitoredTimeUntilExpiry( monitorConfig )
   if not nextTimeUntilExpiry:
      # no configured ssl monitor expiry
      return Tac.endOfTime

   earliestTimeCheck = Tac.endOfTime

   timeUntilExpiry, repeatInterval = _getMonitoredTimeUntilExpiry(
      monitorConfig, now, na )
   if not timeUntilExpiry:
      earliestTimeCheck = na - nextTimeUntilExpiry
   else:
      expiryWarningLoggedAtTime = monitorExpiryDict[ certName ]
      if expiryWarningLoggedAtTime:
         earliestTimeCheck = expiryWarningLoggedAtTime + repeatInterval

      nextTimeUntilExpiry = _getNextMonitoredTimeUntilExpiry( monitorConfig,
                                                              timeUntilExpiry )
      if nextTimeUntilExpiry:
         earliestTimeCheck = min( earliestTimeCheck,
                                  na - nextTimeUntilExpiry )

   if not ( now < earliestTimeCheck < na ): # pylint: disable=superfluous-parens
      earliestTimeCheck = Tac.endOfTime

   # Schedule a last monitor check at 24 hours before expiry
   if now < na - SECS_IN_DAY:
      earliestTimeCheck = min( earliestTimeCheck, na - SECS_IN_DAY )

   # Schedule a last monitor check at 1 hour before expiry
   if now < na - SECS_IN_HOUR:
      earliestTimeCheck = min( earliestTimeCheck, na - SECS_IN_HOUR )

   return earliestTimeCheck

def updateProfileStatus( profileStatus, monitorExpiryDict ):
   """
   Updates profileStatus with the latest data from monitoring expiry dictionary,
   containing last logged times for all certificates of the current profile.

   Parameters:
      profileStatus: Profile status entity
      monitorExpiryDict (dict): Dictionary storing last log times for all 
      certificates in the current profile
   Returns:
      None
   """
   profileStatus.monitorExpiryLastLogTime.clear()
   for crt in monitorExpiryDict:
      profileStatus.monitorExpiryLastLogTime[ crt ] = monitorExpiryDict[ crt ]
