# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=superfluous-parens

from __future__ import absolute_import, division, print_function
import Tac
import Tracing
import pyinotify
import glob
import os
import SslCertKey
import SslMonitorExpiry
from Toggles import MgmtSecurityToggleLib
import ManagedSubprocess
import re
import shutil
import hashlib
import errno
import Logging
from TypeFuture import TacLazyType
from MgmtSecurityLib import getKeyInfoFromAutoCertConfiguredSslProfile

SECURITY_SSL_DIFFIE_HELLMAN_PARAMETERS_INIT_FAILED = Logging.LogHandle(
              "SECURITY_SSL_DIFFIE_HELLMAN_PARAMETERS_INIT_FAILED",
              severity=Logging.logError,
              fmt=( "Initializing SSL Diffie-Hellman parameters failed. Try "
                    "'reset ssl diffie-hellman parameters' to recover" ),
              explanation=( "Attempt to initialize SSL Diffie-Hellman "
                            "parameters has failed." ),
              recommendedAction=Logging.CALL_SUPPORT_IF_PERSISTS )

SECURITY_SSL_DIFFIE_HELLMAN_PARAMETERS_RESET_FAILED = Logging.LogHandle(
              "SECURITY_SSL_DIFFIE_HELLMAN_PARAMETERS_RESET_FAILED",
              severity=Logging.logError,
              fmt=( "Resetting SSL Diffie-Hellman parameters failed. "
                    "Existing parameters will be used" ),
              explanation=( "Attempt to reset SSL Diffie-Hellman parameters "
                            "has failed. Existing Diffie-Hellman parameters "
                            "will be used." ),
              recommendedAction=Logging.CALL_SUPPORT_IF_PERSISTS )

SECURITY_SSL_PROFILE_INVALID = Logging.LogHandle(
              "SECURITY_SSL_PROFILE_INVALID",
              severity=Logging.logError,
              fmt=( "SSL profile '%s' is invalid. "
                    "Check 'show management security ssl profile %s' for details" ),
              explanation=( "The SSL profile is invalid. Possible causes "
                            "are missing certificate, missing key, mismatch "
                            "of certificate with the key, expired or not yet "
                            "valid certificate and non root certificate in the "
                            "trusted list." ),
              recommendedAction=( "Check 'show management security ssl profile' "
                                  "command and fix the errors reported under "
                                  "'Error' column." ) )

SECURITY_SSL_PROFILE_VALID = Logging.LogHandle(
              "SECURITY_SSL_PROFILE_VALID",
              severity=Logging.logInfo,
              fmt="SSL profile '%s' is valid",
              explanation="The SSL profile is valid.",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

traceHandle = Tracing.Handle( 'MgmtSecuritySslReactor' )
error = traceHandle.trace0
warn  = traceHandle.trace1
info  = traceHandle.trace2
trace = traceHandle.trace3
debug = traceHandle.trace4

CertKeyPair = Tac.Type( "Mgmt::Security::Ssl::CertKeyPair" )
CertificateInfo = Tac.Type( "Mgmt::Security::Ssl::CertificateInfo" )
OcspSettings = Tac.Type( "Mgmt::Security::Ssl::OcspSettings" )
ProfileState = Tac.Type( "Mgmt::Security::Ssl::ProfileState" )
ErrorType = Tac.Type( "Mgmt::Security::Ssl::ErrorType" )
ErrorAttr = Tac.Type( "Mgmt::Security::Ssl::ErrorAttr" )
ProfileError = Tac.Type( "Mgmt::Security::Ssl::ProfileError" )
NamedDhparams = Tac.Type( "Mgmt::Security::Ssl::NamedDhparams" )
KeyLocation = TacLazyType( "Mgmt::Security::Ssl::KeyLocation" )
CertLocation = TacLazyType( "Mgmt::Security::Ssl::CertLocation" )

dhParamSize = str( 2048 )

class FileEventHandler( pyinotify.ProcessEvent ):
   def __init__ ( self, fileReactor ):
      trace( "FileEventHandler.__init__ start" )
      self.fileReactor_ = fileReactor 
      pyinotify.ProcessEvent.__init__( self )
      trace( "FileEventHandler.__init__ end" )
       
   def process_IN_MOVED_TO( self, event ):
      self.fileReactor_.handleFile( event.wd, event.name )
       
   # MOVED_FROM happnes in two contexts
   # 1. When copying file using tempfile, rename machanism
   # 2. When using rename command
   # In both cases we send delete event of the file. If its
   # case 1, then nothing happens as none of the ssl profiles
   # will be using tempfile.
   def process_IN_MOVED_FROM( self, event ):
      self.fileReactor_.handleFileDelete( event.wd, event.name )

   def process_IN_DELETE( self, event ):
      self.fileReactor_.handleFileDelete( event.wd, event.name )

        
class InotifyReactor( Tac.Notifiee, pyinotify.Notifier ):
   notifierTypeName = 'Tac::FileDescriptor'

   def __init__( self, watch_manager, default_proc_fun=None, read_freq=0,
                threshold=0, timeout=None ):
      
      pyinotify.Notifier.__init__( self, watch_manager, default_proc_fun, 
                                   read_freq, threshold, timeout )
      fileDesc_ = Tac.newInstance( 'Tac::FileDescriptor', "ssl" )
      fileDesc_.descriptor = watch_manager.get_fd()
      fileDesc_.nonBlocking = True
      Tac.Notifiee.__init__( self, fileDesc_ )
     
   @Tac.handler( 'readableCount' )
   def handleReadableCount( self ):
      self.read_events()
      self.process_events()

class FileReactor( object ): # pylint: disable=useless-object-inheritance
   def __init__( self, constants, sslReactor ):
      trace( "FileReactor.__init__ start" )
      self.sslReactor_ = sslReactor
      self.constants_ = constants
      wm = pyinotify.WatchManager()
      # pylint: disable-msg=E1101
      mask =  pyinotify.IN_MOVED_FROM | pyinotify.IN_MOVED_TO | pyinotify.IN_DELETE
      self.inotifyReactor = InotifyReactor( wm, FileEventHandler( self ) ) 
      wd = wm.add_watch( self.constants_.certsDirPath(), mask )
      self.certsWd_ = wd[ self.constants_.certsDirPath() ] 
      wd = wm.add_watch( self.constants_.keysDirPath(), mask )
      self.keysWd_ = wd[ self.constants_.keysDirPath() ]
      wd = wm.add_watch( self.constants_.autoCertsDirPath(), mask )
      self.autoCertsWd_ = wd[ self.constants_.autoCertsDirPath() ]
      wd = wm.add_watch( self.constants_.autoKeysDirPath(), mask )
      self.autoKeysWd_ = wd[ self.constants_.autoKeysDirPath() ]
      wd = wm.add_watch( self.constants_.rotationBaseDirPath(),
                         pyinotify.IN_DELETE | pyinotify.IN_MOVED_FROM )
      self.rotDirsWd_ = wd[ self.constants_.rotationBaseDirPath() ]
      debug( "Certs WatchDescriptor: ", self.certsWd_ )
      debug( "Keys WatchDescriptor: ", self.keysWd_ )
      debug( "Auto certs WatchDescriptor: ", self.autoCertsWd_ )
      debug( "Auto keys WatchDescriptor: ", self.autoKeysWd_ )
      debug( "Rotation dirs WatchDescriptor: ", self.rotDirsWd_ )
      trace( "FileReactor.__init__ end" )

   def handleFile( self, wd, fileName ):
      trace( "FileReactor.handleFile start for:", fileName, "WD:", wd )
      if wd == self.certsWd_:
         debug( "Certificate added/modified" )
         self.sslReactor_.handleCertOrCrl( fileName )
      elif wd == self.autoCertsWd_:
         debug( "Auto certificate added/modified" )
         self.sslReactor_.handleCertOrCrl( fileName, CertLocation.autoCerts )
      elif wd == self.autoKeysWd_:
         debug( "Auto key added/modified" )
         self.sslReactor_.handleKey( fileName, KeyLocation.autoKeys )
      else:
         debug( "Key added/modified" )
         self.sslReactor_.handleKey( fileName )
      trace( "FileReactor.handleFile end for: ", fileName, "WD:", wd )

   def handleFileDelete( self, wd, fileName ):
      trace( "FileReactor.handleFileDelete start for:", fileName, "WD:", wd )
      if wd == self.certsWd_:
         debug( "Certificate deleted" )
         self.sslReactor_.handleCertOrCrl( fileName )
      elif wd == self.keysWd_:
         debug( "Key deleted" )
         self.sslReactor_.handleKey( fileName )
      elif wd == self.autoCertsWd_:
         debug( "Auto certificate deleted" )
         self.sslReactor_.handleCertOrCrl( fileName, CertLocation.autoCerts )
      elif wd == self.autoKeysWd_:
         debug( "Auto key deleted" )
         self.sslReactor_.handleKey( fileName, KeyLocation.autoKeys )
      else:
         debug( "Rotation dir deleted" )
         self.sslReactor_.handleRotationDir( fileName )
      trace( "FileReactor.handleFileDelete end for:", fileName, "WD:", wd )

class ExecRequestReactor( Tac.Notifiee ):
   notifierTypeName = 'Mgmt::Security::Ssl::ExecRequest'
   def __init__( self, execRequest, sslReactor ):
      trace( "ExecRequestReactor.__init__ start" )
      self.execRequest_ = execRequest
      self.sslReactor_ = sslReactor
      Tac.Notifiee.__init__( self, self.execRequest_ ) 
      trace( "ExecRequestReactor.__init__ end" )
   
   @Tac.handler( 'dhparamsResetRequest' )
   def handleDhparamsResetRequest( self ):
      trace( "ExecRequestReactor.handleDhparamsResetRequest start" )
      self.sslReactor_.handleDhparamsResetRequest()
      trace( "ExecRequestReactor.handleDhparamsResetRequest end" )

class NetStatusReactor( Tac.Notifiee ):
   # pkgdeps: rpm NetConfig
   notifierTypeName = 'System::NetStatus'

   def __init__( self, netStatus, profileConfig, sslReactor ):
      trace( "NetStatusReactor.__init__ start" )
      self.profileConfig_ = profileConfig
      self.sslReactor_ = sslReactor
      self.netStatus_ = netStatus
      Tac.Notifiee.__init__( self, self.netStatus_ )

   @Tac.handler( 'hostname' )
   def handleHostname( self ):
      trace( "NetStatusReactor.handleHostname start" )
      for profileName in self.profileConfig_:
         self.sslReactor_.handleProfileChange( profileName )
      trace( "NetStatusReactor.handleHostname end" )

class ProfileConfigReactor( Tac.Notifiee ):
   notifierTypeName = 'Mgmt::Security::Ssl::ProfileConfig'
   def __init__ ( self, profileConfig, sslReactor ):
      trace( "ProfileConfigReactor.__init__ start for:", profileConfig.name )
      self.profileConfig_ = profileConfig
      self.sslReactor_ = sslReactor
      self.dateCheck = Tac.ClockNotifiee( self.handleDateCheck,
                                          timeMin=Tac.endOfTime )
      Tac.Notifiee.__init__( self, self.profileConfig_ )
      if ( self.profileConfig_.certKeyPair.certFile or 
           self.profileConfig_.trustedCert or
           self.profileConfig_.chainedCert or
           self.profileConfig_.crl ):
         trace( "ProfileConfigReactor.__init__ non empty profileConfig" )
         self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.__init__ end for:", profileConfig.name )
   
   @Tac.handler( 'tlsVersion' )
   def handleTlsVersion( self ):
      trace( "ProfileConfigReactor.handleTlsVersion start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleTlsVersion end" )

   @Tac.handler( 'fipsMode' )
   def handleFipsMode( self ):
      trace( "ProfileConfigReactor.handleFipsMode start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleFipsMode end" )

   @Tac.handler( 'dhparam' )
   def handleDhparams( self ):
      trace( "ProfileConfigReactor.handleDhparams start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleDhparams end" )

   @Tac.handler( 'cipherSuite' )
   def handleCipherSuite( self ):
      trace( "ProfileConfigReactor.handleCipherSuite start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleCipherSuite end" )

   @Tac.handler( 'cipherSuiteV1_3' )
   def handleCipherSuiteV1_3( self ):
      trace( "ProfileConfigReactor.handleCipherSuiteV1_3 start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleCipherSuiteV1_3 end" )

   @Tac.handler( 'certKeyPair' )
   def handleCertKeyPair( self ):
      trace( "ProfileConfigReactor.handleCertKeyPair start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleCertKeyPair end" )
   
   @Tac.handler( 'trustedCert' )
   def handleTrustedCert( self, certName ):
      trace( "ProfileConfigReactor.handleTrustedCert start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleTrustedCert end" )

   @Tac.handler( 'chainedCert' )
   def handleChainedCert( self, certName ):
      trace( "ProfileConfigReactor.handleChainedCert start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleChainedCert end" )

   @Tac.handler( 'crl' )
   def handleCrl( self, crlName ):
      trace( "ProfileConfigReactor.handleCrl start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleCrl end" )

   @Tac.handler( 'ocspProfileName' )
   def handleOcspProfileName( self ):
      trace( "ProfileConfigReactor.handleOcspProfileName start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleOcspProfileName end" )

   @Tac.handler( 'verifyExtendedParameters' )
   def handleVerifyExtendedParameters( self ):
      trace( "ProfileConfigReactor.handleVerifyExtendedParameters start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleVerifyExtendedParameters end" )

   @Tac.handler( 'verifyChainHasRootCA' )
   def handleVerifyChainHasRootCA( self ):
      trace( "ProfileConfigReactor.handleVerifyChainHasRootCA start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleVerifyChainHasRootCA end" )

   @Tac.handler( 'verifyBasicConstraintTrust' )
   def handleVerifyBasicConstraintTrust( self ):
      trace( "ProfileConfigReactor.handleVerifyBasicConstraintTrust start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleVerifyBasicConstraintTrust end" )

   @Tac.handler( 'verifyBasicConstraintChain' )
   def handleVerifyBasicConstraintChain( self ):
      trace( "ProfileConfigReactor.handleVerifyBasicConstraintChain start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleVerifyBasicConstraintChain end" )

   @Tac.handler( 'verifyHostnameMatch' )
   def handleVerifyHostnameMatch( self ):
      trace( "ProfileConfigReactor.handleVerifyHostnameMatch start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleVerifyHostnameMatch end" )

   @Tac.handler( 'verifyExpiryDateEndCert' )
   def handleVerifyExpiryDateEndCert( self ):
      trace( "ProfileConfigReactor.handleVerifyExpiryDateEndCert start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleVerifyExpiryDateEndCert end" )

   @Tac.handler( 'verifyExpiryDateTrustCert' )
   def handleVerifyExpiryDateTrustCert( self ):
      trace( "ProfileConfigReactor.handleVerifyExpiryDateTrustCert start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleVerifyExpiryDateTrustCert end" )

   @Tac.handler( 'verifyExpiryDateCrl' )
   def handleVerifyExpiryDateCrl( self ):
      trace( "ProfileConfigReactor.handleVerifyExpiryDateCrl start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleVerifyExpiryDateCrl end" )

   @Tac.handler( 'verifyPeerHostnameInSan' )
   def handleVerifyPeerHostnameInSan( self ):
      trace( "ProfileConfigReactor.verifyPeerHostnameInSan start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.verifyPeerHostnameInSan end" )

   @Tac.handler( 'oidCheck' )
   def handleVerifyOidCheck( self, oid ):
      trace( "ProfileConfigReactor.oidCheck start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.oidCheck end" )

   @Tac.handler( 'verifyPeerHostnameInCommonName' )
   def handleVerifyPeerHostnameInCommonName( self ):
      trace( "ProfileConfigReactor.verifyPeerHostnameInCommonName start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.verifyPeerHostnameInCommonName end" )

   @Tac.handler( 'verifyTrustHostnameFqdn' )
   def handleVerifyTrustHostnameFqdn( self ):
      trace( "ProfileConfigReactor.verifyTrustHostnameFqdn start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.verifyTrustHostnameFqdn end" )

   def handleDateCheck( self ):
      trace( "ProfileConfigReactor.handleDateCheck start" )
      self.sslReactor_.handleProfileChange( self.profileConfig_.name )
      trace( "ProfileConfigReactor.handleDateCheck end" )

class OcspProfileConfigReactor( Tac.Notifiee ):
   notifierTypeName = 'Mgmt::Security::Ssl::OcspProfileConfig'

   def __init__( self, ocspProfileConfig, profileConfig, sslReactor ):
      trace( "OcspProfileConfigReactor.__init__ start" )
      self.profileConfig_ = profileConfig
      self.sslReactor_ = sslReactor
      Tac.Notifiee.__init__( self, ocspProfileConfig )
      self.handleChange()
      trace( "OcspProfileConfigReactor.__init__ end" )

   def handleChange( self ):
      for profileName, profile in self.profileConfig_.items():
         if self.notifier_.name == profile.ocspProfileName:
            self.sslReactor_.handleProfileChange( profileName )

   @Tac.handler( 'certRequirement' )
   def handleCertRequirement( self ):
      trace( "OcspProfileConfigReactor.handleCertRequirement start" )
      self.handleChange()
      trace( "OcspProfileConfigReactor.handleCertRequirement end" )

   @Tac.handler( 'nonce' )
   def handleNonce( self ):
      trace( "OcspProfileConfigReactor.handleNonce start" )
      self.handleChange()
      trace( "OcspProfileConfigReactor.handleNonce end" )

   @Tac.handler( 'timeout' )
   def handleTimeout( self ):
      trace( "OcspProfileConfigReactor.handleTimeout start" )
      self.handleChange()
      trace( "OcspProfileConfigReactor.handleTimeout end" )

   @Tac.handler( 'url' )
   def handleUrl( self ):
      trace( "OcspProfileConfigReactor.handleUrl start" )
      self.handleChange()
      trace( "OcspProfileConfigReactor.handleUrl end" )

   @Tac.handler( 'vrfName' )
   def handleVrfName( self ):
      trace( "OcspProfileConfigReactor.handleVrfName start" )
      self.handleChange()
      trace( "OcspProfileConfigReactor.handleVrfName end" )

   def close( self ):
      trace( "OcspProfileConfigReactor.close start" )
      self.handleChange()
      Tac.Notifiee.close( self )
      trace( "OcspProfileConfigReactor.close end" )

class VrfStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Ip::VrfStatusLocal'

   def __init__( self, vrfStatus, profileConfig, ocspProfileConfig, sslReactor ):
      trace( "VrfStatusReactor.__init__ start" )
      self.profileConfig_ = profileConfig
      self.ocspProfileConfig_ = ocspProfileConfig
      self.sslReactor_ = sslReactor
      Tac.Notifiee.__init__( self, vrfStatus )
      self.handleChange()
      trace( "VrfStatusReactor.__init__ end" )

   def handleChange( self ):
      for ocspProfile in self.ocspProfileConfig_.values():
         if ocspProfile.vrfName == self.notifier_.vrfName:
            for profileName, profile in self.profileConfig_.items():
               if ocspProfile.name == profile.ocspProfileName:
                  self.sslReactor_.handleProfileChange( profileName )

   def close( self ):
      trace( "VrfStatusReactor.close start" )
      self.handleChange()
      Tac.Notifiee.close( self )
      trace( "VrfStatusReactor.close end" )

class ConfigReactor ( Tac.Notifiee ):
   notifierTypeName = 'Mgmt::Security::Ssl::Config'
   def __init__( self, config, status, sslReactor ):
      trace( "ConfigReactor.__init__ start" )
      self.config_ = config
      self.sslReactor_ = sslReactor
      self.status_ = status
      Tac.Notifiee.__init__( self, self.config_ )
      for profileName in self.config_.profileConfig:
         self.handleProfileConfig( profileName )
      for profileName in status.profileStatus:
         if profileName not in self.config_.profileConfig:
            self.handleProfileConfig( profileName )
      trace( "ConfigReactor.__init__ end" )
   
   @Tac.handler( 'profileConfig' )
   def handleProfileConfig( self, profileName ):
      trace( "ConfigReactor.handleProfileConfig start for: ", profileName )
      if profileName in self.config_.profileConfig:
         info( "profile added:", profileName )
         self.sslReactor_.handleProfileAdd( profileName )
      else:
         info( "profile deleted:", profileName )
         self.sslReactor_.handleProfileDelete( profileName )
      trace( "ConfigReactor.handleProfileConfig end for: ", profileName )

   @Tac.handler( 'monitorConfig' )
   def handleMonitorConfig( self, monitorTimeUntilExpiry ):
      trace( "ConfigReactor.handleMonitorConfig start for: ",
            monitorTimeUntilExpiry )
      if monitorTimeUntilExpiry in self.config_.monitorConfig:
         info( "monitor config added" )
      else:
         info( "monitor config removed" )
      for profileName in self.config_.profileConfig:
         self.sslReactor_.handleProfileChange( profileName )
      trace( "ConfigReactor.handleMonitorConfig end for: ",
            monitorTimeUntilExpiry )

class ClockStatusReactor ( Tac.Notifiee ):
   notifierTypeName = 'Mgmt::Security::ClockStatus'

   def __init__( self, clockStatus, profileConfig, sslReactor ):
      trace( "ClockStatusReactor.__init__ start" )
      self.profileConfig_ = profileConfig
      self.sslReactor_ = sslReactor
      self.status_ = sslReactor.status_
      self.clockStatus_ = clockStatus
      self.handleClockSetAtTime()
      Tac.Notifiee.__init__( self, self.clockStatus_ )
      trace( "ClockStatusReactor.__init__ end" )

   @Tac.handler( 'clockSetAtTime' )
   def handleClockSetAtTime( self ):
      trace( "ClockStatusReactor.handleClockSetAtTime start" )
      if self.clockStatus_.clockSetAtTime > self.status_.clockSetAtTimeProcessed:
         for profileName in self.profileConfig_:
            self.sslReactor_.handleProfileChange( profileName )
         self.status_.clockSetAtTimeProcessed = self.clockStatus_.clockSetAtTime
      trace( "ClockStatusReactor.handleClockSetAtTime end" )

class Dhparams ( object ): # pylint: disable=useless-object-inheritance
   def __init__( self, constants, sslReactor ):
      trace( "Dhparams.__init__ start" )
      self.constants_ = constants
      self.sslReactor_ = sslReactor
      self.tmpDhParam_ = self.constants_.baseDir + "dhparam.tmp"
      self.dhParamSubProcess_ = None
      self.dhparamsPoller_ = None
      trace( "Dhparams.__init__ end" )
   
   def initialized( self ):
      return os.path.isfile( self.constants_.dhParamPath() )

   def returnSignalingSubprocess( self, success ):
      """
      Return a signalling subprocess for the dhParam generation
      that will exit or succeed instantly for edge cases.
      """

      # pylint: disable-next=consider-using-f-string
      subprocCmd = "/bin/%s" % str( success ).lower()
      subProc = ManagedSubprocess.Popen( [ subprocCmd ],
                                         # pylint: disable-next=consider-using-with
                                         stdout=open( '/dev/null', 'w' ),
                                         stderr=ManagedSubprocess.STDOUT )
      return subProc
   
   def reset( self, retries=3 ):
      trace( "Dhparams.reset start" )

      try:
         os.remove( self.tmpDhParam_ )
      except OSError:
         pass

      debug( "Generating dhparams in tmp file:", self.tmpDhParam_ )
      # pylint: disable-next=consider-using-f-string
      debug( "Generating dhparam of size %s bits" % dhParamSize )
      if os.path.isdir( self.tmpDhParam_ ):
         # Fail quickly if there is a folder here. Used to speed up breadth
         # tests
         self.dhParamSubProcess_ = self.returnSignalingSubprocess( success=False )
      else:
         self.dhParamSubProcess_ = ManagedSubprocess.Popen(
            [ "/usr/bin/openssl", "--fips", "dhparam", "-outform", "PEM",
              "-out", self.tmpDhParam_, dhParamSize ],
            stdout=open( '/dev/null', 'w' ), # pylint: disable=consider-using-with
            stderr=ManagedSubprocess.STDOUT )

      def waitForSubProc():
         returnCode = self.dhParamSubProcess_.wait( block=False )
         if returnCode is None:
            trace( "subproc not yet finished" )
            return False
         else:
            trace( "subproc finished with code:", returnCode )
            return True

      self.dhparamsPoller_ = Tac.Poller(
            waitForSubProc,
            handler=lambda ignored: self._resetDone( retries ),
            timeoutHandler=lambda: self._resetDone( retries, timedOut=True ),
            timeout=900, description="scheduled dhparams reset" )

   def _resetDone( self, retries, timedOut=False ):
      err = None
      if timedOut :
         trace( "Generation of dhparams taking more than 15 mins" )
         self.dhParamSubProcess_.kill()
         err = "Timed out while generating dhparams"
      elif not self._validateDhparams():
         error( "Error generating dhparams" )
         err = "Error generating dhparams"
   
      if not err:
         os.rename( self.tmpDhParam_, self.constants_.dhParamPath() )
      elif retries:
         retries = retries - 1
         trace( 'Retrying generating dhparams' )
         self.reset( retries=retries )
         return
      else:
         trace( 'Retries exhausted for generating dhparams' )
                

      self.dhParamSubProcess_ = None
      self.dhparamsPoller_ = None
      self.sslReactor_.handleDhparamsResetResult( err )

   def _validateDhparams( self ):
      trace( "Dhparams._validateDhparams start" )
      if not os.path.isfile( self.tmpDhParam_ ):
         warn( "File", self.tmpDhParam_, "does not exist" )
         return False
      output = Tac.run( [ '/usr/bin/openssl', '--fips', 'dhparam',
                          '-in', self.tmpDhParam_, '-check', '-noout' ],
                        asRoot=True, stdout=Tac.CAPTURE,
                        stderr=Tac.CAPTURE, ignoreReturnCode=True )
      debug( '_validateDhparams ', output )
      return re.search( '^.* ok.$', output, re.MULTILINE )

class SslReactor( object ): # pylint: disable=useless-object-inheritance
   def __init__( self, config, status, execRequest, netStatus, clockStatus,
                 allVrfStatusLocal, allAutoCertProfileStatus ):
      trace( "SslReactor.__init__ start" )
      self.config_ = config
      self.status_ = status
      self.caCerts = {}   
      self.execRequest_ = execRequest
      self.netStatus_ = netStatus
      self.clockStatus_ = clockStatus
      self.allVrfStatusLocal = allVrfStatusLocal
      self.allAutoCertProfileStatus_ = allAutoCertProfileStatus
      # Backlog of ssl profile to be validated
      self.backlog = set()
      self.backlogTaskTimer = Tac.ClockNotifiee( self.handleBacklog,
                                                 timeMin=Tac.endOfTime )
      # Process SSL profiles for 5 seconds in a cycle.
      # one large profile can take up to 15 seconds to process and
      # if we have multiple large profiles, we can process both in a single cycle,
      # which can cross 30 seconds.
      self.maxTimePerCycle = 5
      self.constants_ = Tac.Type( "Mgmt::Security::Ssl::Constants" )
      
      SslCertKey.createSslDirs()
      
      self.fileReactor = FileReactor( self.constants_, self )
      self.configReactor = ConfigReactor( self.config_, self.status_, self )
      self.profileConfigReactor_ = Tac.collectionChangeReactor(
                                       self.config_.profileConfig,
                                       ProfileConfigReactor,
                                       reactorArgs=( self, ) )

      self.ocspProfileConfigReactor_ = Tac.collectionChangeReactor(
         self.config_.ocspProfileConfig,
         OcspProfileConfigReactor,
         reactorArgs=( self.config_.profileConfig, self ) )
      if MgmtSecurityToggleLib.toggleOcspProfileVrfEnabled():
         self.vrfStatusReactor_ = Tac.collectionChangeReactor(
            self.allVrfStatusLocal.vrf,
            VrfStatusReactor,
            reactorArgs=( self.config_.profileConfig,
                          self.config_.ocspProfileConfig,
                          self ) )
      self.netStatusReactor_ = NetStatusReactor( self.netStatus_,
                                                 self.config_.profileConfig,
                                                 self )
      self.clockStatusReactor_ = ClockStatusReactor( self.clockStatus_,
                                                     self.config_.profileConfig,
                                                     self )
      self.execRequestReactor_ = ExecRequestReactor( self.execRequest_, self )
      self.dhparams_ = Dhparams( self.constants_, self )
      if not self.dhparams_.initialized():
         info( "Generating initial dhparams" )
         self.handleDhparamsResetRequest( init=True )
      trace( "Reactor.__init__ end" )
      
   def _isProfileUsingCertOrCrl( self, profileName, certName,
                                 certLocation=CertLocation.certs ):
      profileConfig = self.config_.profileConfig[ profileName ]
      # pylint: disable-next=simplifiable-if-statement
      if ( ( profileConfig.certKeyPair.certFile == certName and
             profileConfig.certKeyPair.certLocation == certLocation ) or
           certName in profileConfig.trustedCert or
           certName in profileConfig.chainedCert or
           certName in profileConfig.crl ):
         return True
      else:
         return False

   def _checkCertOrCrlInProfile( self, certName, certLocation=CertLocation.certs ):
      for profileName in self.config_.profileConfig:
         if self._isProfileUsingCertOrCrl( profileName, certName, certLocation ):
            debug( "Profile", profileName, "is using cert", certName, "in directory",
                    certLocation )
            self.handleProfileChange( profileName )

   def _checkKeyInProfile( self, keyName, keyLocation=KeyLocation.keys ):
      for profileName in self.config_.profileConfig:
         profileConfig = self.config_.profileConfig[ profileName ]
         if ( profileConfig.certKeyPair and
              profileConfig.certKeyPair.certLocation == CertLocation.autoCerts ):
            keyInformation = getKeyInfoFromAutoCertConfiguredSslProfile(
               profileConfig, self.allAutoCertProfileStatus_ )
            if keyInformation:
               keyFile, keyFileLocation = keyInformation
               if ( keyFile in keyName and keyFileLocation == keyLocation ):
                  debug( "Profile", profileName, "is using key", keyFile,
                         "in directory", keyLocation )
                  self.handleProfileChange( profileName )
         elif ( profileConfig.certKeyPair and
                profileConfig.certKeyPair.keyFile == keyName and
                keyLocation == KeyLocation.keys ):
            debug( "Profile", profileName, "is using key", keyName, "in directory",
                   keyLocation )
            self.handleProfileChange( profileName )

   def _readFile( self, filePath ):
      try:
         with open( filePath , 'r' ) as fp:
            return fp.read()
      except IOError as e:
         if e.errno != errno.ENOENT: # pylint: disable=no-else-raise
            error( "Cannot open file", filePath,
                   "errno", e.errno  )
            raise
         else:
            return ""
   
   def handleCertOrCrl( self, certName, certLocation=CertLocation.certs ):
      trace( "SslReactor.handleCertOrCrl start for:", certName )
      self._checkCertOrCrlInProfile( certName, certLocation )
      trace( "SslReactor.handleCertOrCrl end for:", certName )
   
   def handleKey( self, keyName, keyLocation=KeyLocation.keys ):
      trace( "SslReactor.handleKey start for:", keyName )
      self._checkKeyInProfile( keyName, keyLocation )
      trace( "SslReactor.handleKey end for:", keyName )

   def handleRotationDir( self, rotationDir ):
      trace( "SslReactor.handleRotationDir start for:", rotationDir )
      match = re.match( r'[^.]+\.(.*)\.commit', rotationDir )
      if match and match.group( 1 ) in self.config_.profileConfig:
         self.handleProfileChange( match.group( 1 ) )
      trace( "SslReactor.handleRotationDir end for:", rotationDir )

   def _same( self, buf1, buf2 ):
      # pylint: disable-msg=E1101
      d1 = hashlib.md5( buf1.encode( "utf-8" ) ).hexdigest()
      # pylint: disable-msg=E1101      
      d2 = hashlib.md5( buf2.encode( "utf-8" ) ).hexdigest()
      trace( "buf1 hash:", d1 )
      trace( "buf2 hash:", d2 )
      return d1 == d2

   def _getCertKeyStr( self, keyData, profileConfig, certDict ):
      certFileList = ( [ profileConfig.certKeyPair.certFile ] +
                   list( profileConfig.chainedCert ) )
      certListData = []
      for cert in certFileList:
         certListData += list( certDict[ cert ].values() )
      return '\n'.join( [ keyData ] + certListData )
   
   def _getTrustedCertsStr( self, profileConfig, certDict ):
      certsDataList = []
      for cert in profileConfig.trustedCert:
         # keep the certificates in the same order as they were in the source file
         # (sort key is line number)
         for _, certData in sorted( certDict[ cert ].items() ):
            certsDataList.append( certData )
      return '\n'.join( certsDataList )

   def _getCrlsStr( self, profileConfig, crlDict ):
      crlsDataList = []
      for crl in profileConfig.crl:
         crlsDataList += list( crlDict[ crl ].values() )
      return '\n'.join( crlsDataList )

   def _isServerCert( self, profileConfig, certDict ):
      cert = profileConfig.certKeyPair.certFile
      if cert:
         certData = next( iter( certDict[ cert ].values() ) )
         return SslCertKey.isServerCert( certData )
      else:
         return False
   
   def _isClientCert( self, profileConfig, certDict ):
      cert = profileConfig.certKeyPair.certFile
      if cert:
         certData = next( iter( certDict[ cert ].values() ) )
         return SslCertKey.isClientCert( certData )
      else:
         return False

   def _getCertInfo( self, profileConfig, certDict ):
      cert = profileConfig.certKeyPair.certFile
      if not cert:
         return CertificateInfo( '', 0, 0 )
      certData = next( iter( certDict[ cert ].values() ) )
      return SslCertKey.getCertInfo( certData )

   def _getOcspSettings( self, ocspProfileName ):
      ocspProfileConfig = self.config_.ocspProfileConfig.get( ocspProfileName )
      if not ocspProfileConfig:
         return None

      ocspSettings = OcspSettings( ocspProfileConfig.name )
      ocspSettings.certRequirement = ocspProfileConfig.certRequirement
      ocspSettings.nonce = ocspProfileConfig.nonce
      ocspSettings.timeout = ocspProfileConfig.timeout
      ocspSettings.url = ocspProfileConfig.url
      ocspSettings.vrfName = ocspProfileConfig.vrfName
      return ocspSettings

   def isCommitInProgress( self, profileName ):
      return glob.glob( self.constants_.rotationBaseDirPath() +
                        # pylint: disable-next=consider-using-f-string
                        '*.{}.commit'.format( profileName ) )

   def _checkCertificates( self, profileConfig, certDict, keyData ):
      cert = profileConfig.certKeyPair.certFile
      key = profileConfig.certKeyPair.keyFile
      errList = []
      warningList = []
      update = False

      def _checkCertKeyPair():
         if cert not in certDict:
            error( "Certificate does not exist:", cert )
            errList.append( ProfileError( ErrorAttr.certificate,
                                          cert,
                                          ErrorType.notExist ) )
         else:
            if len( certDict[ cert ] ) != 1:
               errType = ErrorType.fileMultiplePEMs
               error( "Certificate has multiple PEM encoded certificates: ", cert )
               errList.append( ProfileError( ErrorAttr.certificate,
                                             cert,
                                             errType ) )
            else:
               certData = next( iter( certDict[ cert ].values() ) )
               errType, warnings = SslCertKey.validateCertificateData( certData,
                  validateExtended=profileConfig.verifyExtendedParameters,
                  validateCa=False,
                  validateExpiryDate=True,
                  treatCertExpiredAsWarning=\
                     ( not profileConfig.verifyExpiryDateEndCert ),
                  isTrust=False,
                  validateHostname=True,
                  treatHostnameMismatchAsWarning=\
                     ( not profileConfig.verifyHostnameMatch ),
                  validateExpiryMonitoringConfig=self.config_.monitorConfig
               )

               if errType != ErrorType.noError:
                  error( "Certificate", cert, "is", errType )
                  errList.append( ProfileError( ErrorAttr.certificate,
                                                cert,
                                                errType ) )
               for warningType, warningTypeExtra in warnings:
                  warn( "Certificate", cert, "is", warningType )
                  p = ProfileError( ErrorAttr.certificate,
                                    cert,
                                    warningType )
                  if warningTypeExtra:
                     p.errorTypeExtra = warningTypeExtra
                  warningList.append( p )
         if not keyData:
            error( "Key does not exist:", key )
            errList.append( ProfileError( ErrorAttr.key,
                                          key,
                                          ErrorType.notExist ) )
         # Either certificate or key does not exist.
         # no need to validate. Return from here
         if cert not in certDict or not keyData:
            return
         certData = next( iter( certDict[ cert ].values() ) )
         if not SslCertKey.isCertificateMatchesKey( certData, keyData ):
            error( "Certificate does not match with key" )
            errList.append( ProfileError( ErrorAttr.certificate,
                                          cert,
                                          ErrorType.notMatchingCertKey ) )

      def _checkChainedCert():
         if ( not cert or cert not in certDict or
              not SslCertKey.verifyCertChain( cert,
                                              profileConfig.chainedCert,
                                              certDict,
                                              profileConfig.verifyChainHasRootCA ) ):
            errList.append( ProfileError( ErrorAttr.profile,
                                          profileConfig.name,
                                          ErrorType.certChainNotValid ))
         # pylint: disable-msg=R1702
         # Too many nested blocks
         for c in profileConfig.chainedCert:
            if c != profileConfig.certKeyPair.certFile:
               if c not in certDict:
                  error( "Chained certificate", c, "does not exist" )
                  errList.append( ProfileError( ErrorAttr.chainedCertificate,
                                                c,
                                                ErrorType.notExist ) )
               else:
                  for lineno, certData in certDict[ c ].items():
                     errType, warnings = SslCertKey.validateCertificateData(
                        certData,
                        validateExtended=False,
                        validateCa=profileConfig.verifyBasicConstraintChain,
                        validateExpiryDate=True,
                        isTrust=False,
                        validateExpiryMonitoringConfig=self.config_.monitorConfig
                     )

                     if len( certDict[ c ] ) > 1:
                        errValue = "in '" + c + "' at line " + str( lineno ) + \
                           " with CN: " + SslCertKey.getCommonName( certData )
                     else:
                        errValue = c

                     if errType != ErrorType.noError:
                        error( "Chained certificate", errValue, "is", errType )
                        errList.append(
                           ProfileError( ErrorAttr.chainedCertificate,
                                          errValue,
                                          errType ) )
                     for warningType, warningTypeExtra in warnings:
                        warn( "Chained certificate", errValue, "is", warningType )
                        p = ProfileError( ErrorAttr.chainedCertificate,
                                          errValue,
                                          warningType )
                        if warningTypeExtra:
                           p.errorTypeExtra = warningTypeExtra
                        warningList.append( p )


      if not cert and not profileConfig.chainedCert:
         return ( [], False, [] )

      if cert:
         _checkCertKeyPair()

      if profileConfig.chainedCert or profileConfig.verifyChainHasRootCA:
         _checkChainedCert()

      if not len( errList ): # pylint: disable=use-implicit-booleaness-not-len
         oldData = self._readFile( self.constants_.certKeyPath(
                                      profileConfig.name ) )
         newData = self._getCertKeyStr( keyData, profileConfig, certDict )
         trace( "Comparing old certkey file with new one" )
         update = not self._same( oldData, newData )
      return ( errList, update, warningList )

   def _checkTrustedCert( self, profileConfig, certDict, certKeyErrList, crlDict ):
      errList = []
      warningList = []
      update = False
      # pylint: disable-msg=R1702
      # Too many nested blocks
      for cert in profileConfig.trustedCert:
         if cert != profileConfig.certKeyPair.certFile:
            if cert not in certDict:
               error( "Trusted certificate", cert, "does not exist" )
               errList.append( ProfileError( 
                                          ErrorAttr.trustedCertificate,
                                          cert,
                                          ErrorType.notExist ) )
            else:
               for lineno, certData in certDict[ cert ].items():
                  errType, warnings = SslCertKey.validateCertificateData(
                     certData,
                     validateExtended=False,
                     validateCa=profileConfig.verifyBasicConstraintTrust,
                     validateExpiryDate=True,
                     # The system-supplied list of CA certs might have some certs
                     # that are expired. Treat these cases as warnings by default
                     # so that the SSL profile can be 'valid'
                     treatCertExpiredAsWarning=\
                        ( not profileConfig.verifyExpiryDateTrustCert or \
                          cert == self.constants_.system ),
                     isTrust=True,
                     validateFqdn=profileConfig.verifyTrustHostnameFqdn,
                     validateExpiryMonitoringConfig=self.config_.monitorConfig
                  )

                  if len( certDict[ cert ] ) > 1:
                     errValue = "in '" + cert + "' at line " + str( lineno ) + \
                        " with CN: " + SslCertKey.getCommonName( certData )
                  else:
                     errValue = cert

                  if errType != ErrorType.noError:
                     error( "Trusted certificate", errValue, "is", errType )
                     errList.append( ProfileError( ErrorAttr.trustedCertificate,
                                                   errValue,
                                                   errType ) )
                  for warningType, warningTypeExtra in warnings:
                     warn( "Trusted certificate", errValue, "is", warningType )
                     p = ProfileError( ErrorAttr.trustedCertificate,
                                       errValue,
                                       warningType )
                     if warningTypeExtra:
                        p.errorTypeExtra = warningTypeExtra
                     warningList.append( p )
         
      if not self._isEmpty( profileConfig ):
         errType =  SslCertKey.validateTrustedChain( profileConfig, 
                                                     certDict, crlDict )
         if errType != ErrorType.noError:
            error( "Profile", profileConfig.name, "is", errType )
            errList.append( ProfileError( ErrorAttr.profile,
                                          profileConfig.name,
                                          errType ) )
               
      if not certKeyErrList and not errList:
         oldData = self._readFile( self.constants_.trustedCertsPath( 
                                                   profileConfig.name ) )
         newData = self._getTrustedCertsStr( profileConfig, certDict )
         trace( "Comparing old trusted certs file with new one" )            
         update = not self._same( oldData, newData )
         for cert in profileConfig.trustedCert:
            for certData in certDict[ cert ].values():
               self.caCerts.setdefault(
                  profileConfig.name, [] ).append( certData )
 
      return ( errList, update, warningList )

   def _checkCrl( self, profileConfig, crlDict, prevErrList ):
      errList = []
      warningList = []
      update = False
      for crl in profileConfig.crl:
         if crl not in crlDict:
            error( "CRL", crl, "does not exist" )
            errList.append( ProfileError( ErrorAttr.crl,
                                          crl,
                                          ErrorType.notExist ) )
         else:
            for lineno, crlData in crlDict[ crl ].items():
               errType, warnings = SslCertKey.validateCertificateData( crlData,
                  validateExtended=profileConfig.verifyExtendedParameters,
                  validateCa=False,
                  validateExpiryDate=True,
                  treatCertExpiredAsWarning=( not profileConfig.verifyExpiryDateCrl )
               )

               if errType == ErrorType.noError:
                  errType = SslCertKey.validateCrlCa( crlData,
                                          self.caCerts,
                                          profileConfig.name,
                                          profileConfig.verifyExtendedParameters )

               errValue = ""
               if errType != ErrorType.noError or warnings:
                  if len( crlDict[ crl ] ) > 1:
                     errValue = "in '" + crl + "' at line " + str( lineno )
                  else:
                     errValue = crl

               if errType != ErrorType.noError:
                  error( "CRL", crl, "is", errType )
                  errList.append( ProfileError( ErrorAttr.crl,
                                                errValue,
                                                errType ) )
               for warningType, warningTypeExtra in warnings:
                  warn( "CRL", crl, "is", warningType )
                  p = ProfileError( ErrorAttr.crl,
                                    errValue,
                                    warningType )
                  if warningTypeExtra:
                     p.errorTypeExtra = warningTypeExtra
                  warningList.append( p )

      # pylint: disable-next=use-implicit-booleaness-not-len
      if not len( prevErrList ) and not len( errList ):
         oldData = self._readFile( self.constants_.crlsPath( 
                                                   profileConfig.name ) )
         newData = self._getCrlsStr( profileConfig, crlDict )
         trace( "Comparing old CRL file with new one" )
         update = not self._same( oldData, newData )

      return ( errList, update, warningList )

   def _checkOcspProfile( self, profileConfig, profileStatus ):
      errList = []
      warningList = []
      update = False

      ocspProfileName = profileConfig.ocspProfileName
      if ocspProfileName:
         ocspProfile = self.config_.ocspProfileConfig.get( ocspProfileName )
         if ocspProfile:
            if ( MgmtSecurityToggleLib.toggleOcspProfileVrfEnabled()
                 and ocspProfile.vrfName
                 and ocspProfile.vrfName != "default"
                 and ocspProfile.vrfName not in self.allVrfStatusLocal.vrf ):
               errList.append( ProfileError( ErrorAttr.ocsp,
                                             ocspProfileName,
                                             ErrorType.vrfDoesNotExist ) )
            update = ( not profileStatus.ocspSettings
                       or profileStatus.ocspSettings.certRequirement !=
                       ocspProfile.certRequirement
                       or profileStatus.ocspSettings.nonce != ocspProfile.nonce
                       or profileStatus.ocspSettings.timeout != ocspProfile.timeout
                       or profileStatus.ocspSettings.url != ocspProfile.url
                       or profileStatus.ocspSettings.vrfName != ocspProfile.vrfName )
         else:
            error( f"OCSP {ocspProfileName} does not exist" )
            errList.append( ProfileError( ErrorAttr.ocsp,
                                          ocspProfileName,
                                          ErrorType.notExist ) )
      else:
         update = bool( profileStatus.ocspSettings )

      return ( errList, update, warningList )

   def _catTrustedCert( self, profileConfig, certDict ):
      trustedCertsFile = self.constants_.trustedCertsPath( profileConfig.name )
      tmpFile = trustedCertsFile + ".tmp"
      with open( tmpFile, 'w' ) as tmpFp:
         trustedCertsStr = self._getTrustedCertsStr( profileConfig, certDict )
         tmpFp.write( trustedCertsStr )
      os.rename( tmpFile, trustedCertsFile )

   def _deleteTrustedCertsHashFiles( self, profileConfig ):
      oldHashDirPath = self.constants_.opensslHashProfilePath(
         profileConfig.name ) + '/*.[0-999]*'
      trace( "_deleteTrustedCertsHashFiles oldHashFileList ", oldHashDirPath )
      # remove existing .0 hashes
      for h in glob.glob( oldHashDirPath ):
         trace( "_deleteTrustedCertsHashFile removed ", h )
         os.remove( h )

   def _createTrustedCertsHashFiles( self, profileConfig, certDict, tempDir ):
      certsHashDict = { }
      for cert in profileConfig.trustedCert:
         certsDataList = list( certDict[ cert ].values() )
         for certData in certsDataList:
            filePath, certHash = SslCertKey.generateTrustedCertHash( certData,
                                                                     tempDir )
            certFilePath = tempDir + "/" + certHash + "."

            if certHash not in certsHashDict:
               certFilePath = certFilePath + "0"
               certsHashDict[ certHash ] = 1
            else:
               certFilePath = certFilePath + str( certsHashDict[ certHash ] )
               certsHashDict[ certHash ] += 1

            os.rename( filePath, certFilePath )
            debug( "_createTrustedHashFiles in certHashList: ", certFilePath )

   def _catCrls( self, profileConfig, crlDict ):
      crlFile = self.constants_.crlsPath( profileConfig.name )
      tmpFile = crlFile + ".tmp"
      with open( tmpFile, 'w' ) as tmpFp:
         crlsStr = self._getCrlsStr( profileConfig, crlDict )
         tmpFp.write( crlsStr )
      os.rename( tmpFile, crlFile )
   
   def _deleteCrlHashFiles( self, profileConfig ):
      oldHashDirPath = self.constants_.opensslHashProfilePath(
         profileConfig.name ) + '/*.r*'
      error( "_deleteCrlHashFiles oldHashFileList ", oldHashDirPath )
      # remove existing .r0 hashes
      for h in glob.glob( oldHashDirPath ):
         error( "_deleteCrlHashFiles removed HashFile ", h )
         os.remove( h )

   def _createCrlHashFiles( self, profileConfig, crlDict, tempDir ):
      crlHashDict = { }
      for crl in profileConfig.crl:
         crlsDataList = list( crlDict[ crl ].values() )
         for crlData in crlsDataList:
            filePath, crlHash = SslCertKey.generateCrlHash( crlData, tempDir )
            crlFilePath = tempDir + "/" + crlHash + ".r"
            if crlHash not in crlHashDict:
               crlFilePath = crlFilePath + "0"
               crlHashDict[ crlHash ] = 1
            else:
               crlFilePath = crlFilePath + str( crlHashDict[ crlHash ] )
               crlHashDict[ crlHash ] += 1

            os.rename( filePath, crlFilePath )
            debug( "_createCrlHashFiles  in crlHashList: ", crlFilePath )

   def _catCertKey( self, profileConfig, certDict, key ):
      certKeyFile = self.constants_.certKeyPath( profileConfig.name )
      tmpFile = certKeyFile + ".tmp"
      oldmask = os.umask( 0o006 )
      try:
         with open( tmpFile, 'w' ) as tmpFp:
            certKeyStr = self._getCertKeyStr( key, profileConfig, certDict )
            tmpFp.write( certKeyStr )
         os.rename( tmpFile, certKeyFile )
      except: # pylint: disable=try-except-raise
         raise
      finally:
         os.umask( oldmask )

   def handleProfileAdd( self, profileName ):
      trace( "SslReactor.handleProfileAdd start for: ", profileName )
      if not os.path.isdir( self.constants_.profileDirPath( profileName ) ):
         SslCertKey.dirCreate( self.constants_.profileDirPath( profileName ) )

      if not os.path.isdir( self.constants_.opensslHash0ProfileDirPath(
         profileName ) ):
         SslCertKey.dirCreate( self.constants_.opensslHash0ProfileDirPath(
            profileName ) )
      if not os.path.isdir( self.constants_.opensslHash1ProfileDirPath(
         profileName ) ):
         SslCertKey.dirCreate( self.constants_.opensslHash1ProfileDirPath(
            profileName ) )

      if profileName in self.status_.profileStatus:
         if not os.path.isdir( self.constants_.profileDirPath( profileName ) ):
            SslCertKey.dirCreate( self.constants_.profileDirPath( profileName ) )
         debug( "ProfileStatus already exists for:", profileName )
      else:
         debug( "Adding new ProfileStatus for:", profileName )
         profileStatus = self.status_.newProfileStatus( profileName )
         profileStatus.error.enq( ProfileError( ErrorAttr.profile,
                                                profileName,
                                                ErrorType.noProfileData ) )
      trace( "SslReactor.handleProfileAdd end for: ", profileName )
      
   def _isEmpty( self, profileConfig ):
      return ( not profileConfig.certKeyPair.certFile and
               not profileConfig.trustedCert )

   def _getCertDict( self, profileConfig, profileStatus ):
      certDict = { }
      monitorExpiryDict = { }
      if profileConfig.certKeyPair.certFile:
         cert = profileConfig.certKeyPair.certFile
         certLocation = profileConfig.certKeyPair.certLocation
         certs = SslCertKey.extractCerts( cert, certLocation=certLocation )
         for certData, lineno in certs:
            if cert not in certDict:
               certDict[ cert ] = { }
               monitorExpiryDict[ cert ] = \
                  SslMonitorExpiry.getLastLogTime( profileStatus, cert )
            certDict[ cert ][ lineno ] = certData
      
      for cert in ( list( profileConfig.trustedCert ) +
                    list( profileConfig.chainedCert ) ):
         certs = SslCertKey.extractCerts( cert )
         if len( certs ) > 0:
            for certData, lineno in certs:
               if SslCertKey.hasCertificate( certData ):
                  if cert not in certDict:
                     certDict[ cert ] = { }
                     monitorExpiryDict[ cert ] = \
                        SslMonitorExpiry.getLastLogTime( profileStatus, cert )
                  certDict[ cert ][ lineno ] = certData
      return certDict, monitorExpiryDict

   def _getCrlDict( self, profileConfig ):
      crlDict = { }
      for crl in list( profileConfig.crl ):
         crls = SslCertKey.extractCrls( crl )
         if len( crls ) > 0:
            for crlData, lineno in crls:
               if SslCertKey.hasCrl( crlData ):
                  if crl not in crlDict:
                     crlDict[ crl ] = { }
                  crlDict[ crl ][ lineno ] = crlData
      return crlDict
   
   def _getKey( self, profileConfig ):
      key = ""
      if profileConfig.certKeyPair and profileConfig.certKeyPair.certFile:
         # If ssl profile is configured with auto cert, the corresponding key
         # coud be in /keys or /autokeys directory, otherwise key is in /keys
         # directory.
         if profileConfig.certKeyPair.certLocation == CertLocation.autoCerts:
            # Get the corresponding auto cert profile status
            keyInformation = getKeyInfoFromAutoCertConfiguredSslProfile(
               profileConfig, self.allAutoCertProfileStatus_ )
            if keyInformation:
               keyFile, keyLocation = keyInformation
               if keyLocation == KeyLocation.autoKeys:
                  key = self._readFile( self.constants_.autoKeyPath( keyFile ) )
               else:
                  key = self._readFile( self.constants_.keyPath( keyFile ) )
         else:
            key = self._readFile( self.constants_.keyPath(
               profileConfig.certKeyPair.keyFile ) )
      return key

   def _getEarliestTimeCheck( self, certDict, crlDict, monitorExpiryDict ):
      now = int( Tac.utcNow() )
      trace( "UTC now is", now )

      minimum = Tac.endOfTime

      def _getEarliestTimeCheckFor( cDict, isCrl=False ):
         nonlocal minimum, now
         for pemFile, pemValue in list( cDict.items() ):
            for lineno, pemData in pemValue.items():
               ( nb, na ) = SslCertKey.getCertificateOrCrlDates( pemData )
               trace( "PEM in ", pemFile, "at line ", lineno, " nb, na:", nb, na )
               if ( nb >= now ):
                  minimum = min( minimum, nb )
               elif ( na >= now ):
                  earliestMonitorExpiry = Tac.endOfTime
                  if not isCrl:
                     earliestMonitorExpiry = SslMonitorExpiry.getEarliestTimeCheck(
                        self.config_.monitorConfig, monitorExpiryDict, pemFile,
                        now, na )
                  minimum = min( minimum, na, earliestMonitorExpiry )

      _getEarliestTimeCheckFor( certDict )
      _getEarliestTimeCheckFor( crlDict, isCrl=True )

      assert ( minimum >= now )
      return ( minimum - now )

   def _isEmptyProfileStatus( self, profileStatus ):
      if profileStatus.state != ProfileState.invalid:
         return False
      if len( profileStatus.error.values() ) > 1:
         return False
      err = next( iter( profileStatus.error.values() ) )
      if err.errorType != ErrorType.noProfileData:
         return False
      return True

   def scheduleBacklogTask( self ):
      trace( "Schedule Backlog Task" )
      self.backlogTaskTimer.timeMin = Tac.now()

   def handleProfileChange( self, profileName ):
      trace( "SslReactor.handleProfileChange start for:", profileName )
      self.backlog.add( profileName )
      trace( "Current backlog: ", self.backlog )
      self.scheduleBacklogTask()
      trace( "SslReactor.handleProfileChange end for:", profileName )

   def handleBacklog( self ):
      startTime = Tac.now()
      trace( "SslReactor.handleBacklog start:", startTime,
             ", max time:", self.maxTimePerCycle )
      while self.backlog:
         profileName = self.backlog.pop()
         if profileName not in self.config_.profileConfig:
            trace( "Profile deleted. Nothing to do:", profileName )
            continue

         trace( "Process profile:", profileName )
         self.processProfileChange( profileName )

         timeSpent = Tac.now() - startTime
         trace( "Time spent is:", timeSpent )

         if timeSpent >= self.maxTimePerCycle:
            trace( "Max time reached in this cycle" )
            break

      if self.backlog:
         trace( "Pending backlog: ", self.backlog )
         self.scheduleBacklogTask()

      trace( "SslReactor.handleBacklog end" )

   def processProfileChange( self, profileName ):
      if self.isCommitInProgress( profileName ):
         trace( "Commit in progress for:", profileName )
         return

      trace( "SslReactor.processProfileChange start for:", profileName )
      profileConfig = self.config_.profileConfig[ profileName ]
      profileStatus = self.status_.profileStatus[ profileName ]
      errList = []
      warningList = []
      certDict, monitorExpiryDict = self._getCertDict( profileConfig,
                                                       profileStatus )
      crlDict = self._getCrlDict( profileConfig )
      key = self._getKey( profileConfig )
      oldState = profileStatus.state
      oldEmptyProfile = self._isEmptyProfileStatus( profileStatus )

      trace( "certDict keys are", list( certDict ) )
      trace( "crlDict keys are", list( crlDict ) )

      trace( "Create date cache" )
      certDictValues = []
      crlDictValues = []
      for cert in certDict: # pylint: disable=consider-using-dict-items
         certDictValues += list( certDict[ cert ].values() )
      for crl in crlDict: # pylint: disable=consider-using-dict-items
         crlDictValues += list( crlDict[ crl ].values() )
      SslCertKey.createProfileCache( list( certDictValues ) +
                                     list( crlDictValues ) )

      profileConfigReactor = self.profileConfigReactor_.reactors_[ profileName ]

      errCk, updateCk, warningCk = self._checkCertificates( profileConfig,
                                                            certDict, key )
      errList.extend( errCk )
      warningList.extend( warningCk )
      errTc, updateTc, warningTc = self._checkTrustedCert( profileConfig, certDict,
                                                errList, crlDict )
      errList.extend( errTc )
      warningList.extend( warningTc )

      ( errCrl, updateCrl, warningCrl ) = self._checkCrl( profileConfig, crlDict,
                                                                        errList )
      errList.extend( errCrl )
      warningList.extend( warningCrl )

      ( errOcsp, updateOcsp, warningOcsp ) = self._checkOcspProfile(
         profileConfig, profileStatus )
      errList.extend( errOcsp )
      warningList.extend( warningOcsp )

      SslMonitorExpiry.logWarnings( self.config_.monitorConfig, profileName,
                                    monitorExpiryDict, warningList )

      seconds = self._getEarliestTimeCheck( certDict, crlDict, monitorExpiryDict )
      if seconds is not None:
         trace( "Earliest time check is after", seconds, "seconds" )
         profileConfigReactor.dateCheck.timeMin = Tac.now() + seconds
      else:
         trace( "Resetting time check timer to end of time" )
         profileConfigReactor.dateCheck.timeMin = Tac.endOfTime

      if not errList:
         if self._isEmpty( profileConfig ):
            errList.append( ProfileError( ErrorAttr.profile,
                                          profileName,
                                          ErrorType.noProfileData ))
            newState = ProfileState.invalid
         else:
            newState = ProfileState.valid
      else:
         newState = ProfileState.invalid

      def updateRequired():
         if ( profileStatus.state != newState ):
            return True
         if ( newState == ProfileState.invalid ):
            for e in errList:
               if e not in profileStatus.error.values():
                  return True
            if len( errList ) != len( profileStatus.error.values() ):
               return True
         if ( newState == ProfileState.valid ):
            if profileConfig.certKeyPair.certFile:
               newCertKeyPath = self.constants_.certKeyPath( profileConfig.name )
            else:
               newCertKeyPath = ""
            if len( profileConfig.trustedCert ) > 0:
               newTrustedCertsPath = self.constants_.trustedCertsPath( 
                                                            profileConfig.name )
            else:
               newTrustedCertsPath = ""
            if len( profileConfig.crl ) > 0:
               newCrlsPath = self.constants_.crlsPath( profileConfig.name )
            else:
               newCrlsPath = ""
            if ( profileStatus.certKeyPath !=  newCertKeyPath or
                 profileStatus.trustedCertsPath != newTrustedCertsPath or
                 profileStatus.crlsPath != newCrlsPath or
                 profileStatus.tlsVersion != profileConfig.tlsVersion or
                 profileStatus.fipsMode != profileConfig.fipsMode or
                 profileStatus.dhparam != profileConfig.dhparam or
                 profileStatus.cipherSuite != profileConfig.cipherSuite or
                 profileStatus.cipherSuiteV1_3 != profileConfig.cipherSuiteV1_3 or
                 updateCk or updateTc or updateCrl or updateOcsp or
                 profileStatus.verifyExtendedParameters != \
                       profileConfig.verifyExtendedParameters or
                 profileStatus.verifyPeerHostnameInSan !=
                    profileConfig.verifyPeerHostnameInSan or
                 profileStatus.verifyPeerHostnameInCommonName !=
                 profileConfig.verifyPeerHostnameInCommonName or
                  profileStatus.oidCheck.items() !=
                  profileConfig.oidCheck.items() ):
               return True
            if len( errList ) != len( profileStatus.error.values() ):
               return True
            if len( warningList ) != len( profileStatus.warning.values() ):
               return True
         return False

      if ( updateRequired() ):
         profileStatus.state = ProfileState.updating
         if ( newState == ProfileState.invalid ):
            trace( "Updating profile status to invalid" )
            profileStatus.error.clear()
            profileStatus.warning.clear()
            profileStatus.tlsVersion = self.constants_.allTlsVersion
            profileStatus.fipsMode = False
            profileStatus.dhparam = NamedDhparams.generated
            profileStatus.cipherSuite = self.constants_.defaultCipherSuite()
            profileStatus.cipherSuiteV1_3 = self.constants_.defaultCipherSuiteV1_3()
            profileStatus.certKeyPath = ""
            profileStatus.certKeyPair = CertKeyPair()
            profileStatus.hasChainedCerts = False
            profileStatus.chainedCert.clear()
            profileStatus.trustedCertsPath = ""
            profileStatus.trustedCert.clear()
            profileStatus.crlsPath = ""
            profileStatus.crl.clear()
            profileStatus.ocspSettings = None
            profileStatus.certInfo = CertificateInfo( "", 0, 0 )
            profileStatus.verifyExtendedParameters = False
            profileStatus.isServerCert = True
            profileStatus.isClientCert = True
            profileStatus.verifyPeerHostnameInCommonName = False
            profileStatus.oidCheck.clear()
            profileStatus.verifyPeerHostnameInSan = False
            profileStatus.monitorExpiryLastLogTime.clear()
            for e in errList:
               profileStatus.error.enq( e )
         else:
            trace( "Updating profile status to valid" )
            profileStatus.error.clear()
            profileStatus.warning.clear()
            profileStatus.hasChainedCerts = len( profileConfig.chainedCert ) > 0

            # sync config and status certs
            for crt in profileStatus.chainedCert:
               if crt not in profileConfig.chainedCert:
                  del profileStatus.chainedCert[ crt ]
            for crt in profileConfig.chainedCert:
               profileStatus.chainedCert[ crt ] = 1
            for crt in profileStatus.trustedCert:
               if crt not in profileConfig.trustedCert:
                  del profileStatus.trustedCert[ crt ]
            for crt in profileConfig.trustedCert:
               profileStatus.trustedCert[ crt ] = 1
            for crt in profileStatus.crl:
               if crt not in profileConfig.crl:
                  del profileStatus.crl[ crt ]
            for crt in profileConfig.crl:
               profileStatus.crl[ crt ] = 1

            profileStatus.tlsVersion = profileConfig.tlsVersion
            profileStatus.fipsMode = profileConfig.fipsMode
            profileStatus.dhparam = profileConfig.dhparam
            profileStatus.cipherSuite = profileConfig.cipherSuite
            profileStatus.cipherSuiteV1_3 = profileConfig.cipherSuiteV1_3
            if profileConfig.certKeyPair.certFile:
               self._catCertKey( profileConfig, certDict, key )
               profileStatus.certKeyPath = self.constants_.certKeyPath(
                                                            profileConfig.name )
               if profileConfig.certKeyPair.certLocation == CertLocation.autoCerts:
                  keyInformation = getKeyInfoFromAutoCertConfiguredSslProfile(
                     profileConfig, self.allAutoCertProfileStatus_ )
                  if keyInformation:
                     keyFile, keyLocation = keyInformation
                     certKeyPair = CertKeyPair( profileConfig.certKeyPair.certFile,
                                                keyFile )
                     certKeyPair.keyLocation = keyLocation
                  else:
                     certKeyPair = CertKeyPair( profileConfig.certKeyPair.certFile,
                                                profileConfig.certKeyPair.keyFile )
               else:
                  certKeyPair = CertKeyPair( profileConfig.certKeyPair.certFile,
                                             profileConfig.certKeyPair.keyFile )
               certKeyPair.certLocation = profileConfig.certKeyPair.certLocation
               profileStatus.certKeyPair = certKeyPair
            else:
               profileStatus.certKeyPath = ""
               profileStatus.certKeyPair = CertKeyPair()

            oldDir = ''
            if profileStatus.opensslHashPath:
               oldDir = os.readlink( profileStatus.opensslHashPath )
            currDir = self._cleanDirForNewHash( profileConfig, oldDir )

            if len( profileConfig.trustedCert ) > 0:
               self._catTrustedCert( profileConfig, certDict )
               self._createTrustedCertsHashFiles( profileConfig,
                                                  certDict, currDir )
               profileStatus.trustedCertsPath = self.constants_.trustedCertsPath(
                                                   profileConfig.name )
            else:
               self._deleteTrustedCertsHashFiles( profileConfig )
               profileStatus.trustedCertsPath = ""
            if len( profileConfig.crl ) > 0:
               self._catCrls( profileConfig, crlDict )
               self._createCrlHashFiles( profileConfig, crlDict, currDir )
               profileStatus.crlsPath = self.constants_.crlsPath(
                                            profileConfig.name )
            else:
               profileStatus.crlsPath = ""
               self._deleteCrlHashFiles( profileConfig )
            if profileConfig.ocspProfileName:
               profileStatus.ocspSettings = self._getOcspSettings(
                  profileConfig.ocspProfileName )
            else:
               profileStatus.ocspSettings = None

            if len( profileConfig.crl ) > 0 or len(
                                profileConfig.trustedCert ) > 0:
               SslCertKey.renameDirAtomically( currDir,
                self.constants_.opensslHashProfilePath( profileConfig.name ),
                      self.constants_.profileDirPath( profileConfig.name ) )
               profileStatus.opensslHashPath = \
                        self.constants_.opensslHashProfilePath(
                                             profileConfig.name )
            else:
               profileStatus.opensslHashPath = ""

            for oid in list( profileStatus.oidCheck.keys() ):
               if oid not in profileConfig.oidCheck:
                  del profileStatus.oidCheck[ oid ]
            for oid, val in list( profileConfig.oidCheck.items() ):
               profileStatus.oidCheck[ oid ] = val

            profileStatus.certInfo = self._getCertInfo( profileConfig,
                                                        certDict )
            profileStatus.isServerCert = self._isServerCert( profileConfig,
                                                             certDict )
            profileStatus.isClientCert = self._isClientCert( profileConfig,
                                                             certDict )
            profileStatus.verifyExtendedParameters = \
                  profileConfig.verifyExtendedParameters
            profileStatus.verifyPeerHostnameInCommonName = \
               profileConfig.verifyPeerHostnameInCommonName
            profileStatus.verifyPeerHostnameInSan = \
               profileConfig.verifyPeerHostnameInSan

         for warning in warningList:
            trace( "Add warnings to profile status" )
            profileStatus.warning.enq( warning )

         profileStatus.state = newState

      # Log the state change messages
      if oldState != profileStatus.state:
         if profileStatus.state == ProfileState.invalid:
            if not self._isEmptyProfileStatus( profileStatus ):
               # pylint: disable-msg=E0602
               Logging.log( SECURITY_SSL_PROFILE_INVALID,
                            profileName, profileName )
         else:
            trace( "Logging invalid->valid state change" )
            # pylint: disable-msg=E0602
            Logging.log( SECURITY_SSL_PROFILE_VALID, profileName )
      else:
         if oldEmptyProfile:
            # pylint: disable-msg=E0602
            Logging.log( SECURITY_SSL_PROFILE_INVALID,
                         profileName, profileName )

      for err in profileStatus.error.values():
         debug( "Error: ", err.errorAttr, err.errorAttrValue, 
                 err.errorType )         

      SslMonitorExpiry.updateProfileStatus( profileStatus,
                                            monitorExpiryDict )

      trace( "Clear profile cache" )
      SslCertKey.clearProfileCache()

      trace( "SslReactor.processProfileChange end for: ", profileName )

   def handleProfileDelete( self, profileName ):
      trace( "SslReactor.handleSslProfileDelete start for: ", profileName )
      if hasattr( self, 'profileConfigReactor_' ):
         debug( "On delete config reactors:", 
                self.profileConfigReactor_.reactors_ )
         profileConfigReactor = self.profileConfigReactor_.reactors_[ profileName ]
         profileConfigReactor.dateCheck.timeMin = Tac.endOfTime
      if profileName in self.status_.profileStatus:
         debug( "Deleting ProfileStatus for:", profileName )
         shutil.rmtree( self.constants_.profileDirPath( profileName ), 
                        ignore_errors=True )
         del self.status_.profileStatus[ profileName ]
         self.caCerts.pop( profileName, None )
      trace( "SslReactor.handleSslProfileDelete end for: ", profileName )
      
   def handleDhparamsResetRequest( self, init=False ):
      trace( "SslReactor.handleDhparamsResetRequest start" )
      if self.status_.dhparamsResetInProgress:
         trace( "Dhparams reset is already in progress" )
         return
      if ( init or self.execRequest_.dhparamsResetRequest > 
           self.status_.dhparamsResetProcessed ):
         self.status_.dhparamsResetInProgress = True
         self.dhparams_.reset()
      else:
         trace( "Ignoring reset request" )
      trace( "SslReactor.handleDhparamsResetRequest end" )
      
   def handleDhparamsResetResult( self, err=None ):
      trace( "SslReactor.handleDhparamsResetResult start:", err )
      if not err:
         self.status_.dhparamsResetProcessed = Tac.now()   
         self.status_.dhparamsLastResetFailed = False   
      else:
         error( "Dhparams reset error:", err )
         self.status_.dhparamsLastResetFailed = True         
         if not self.dhparams_.initialized():
            # pylint: disable-msg=E0602
            Logging.log( SECURITY_SSL_DIFFIE_HELLMAN_PARAMETERS_INIT_FAILED )
         else:
            # pylint: disable-msg=E0602
            Logging.log( SECURITY_SSL_DIFFIE_HELLMAN_PARAMETERS_RESET_FAILED )
      self.status_.dhparamsResetInProgress = False
      trace( "SslReactor.handleDhparamsResetResult end:", err )

   def _cleanDirForNewHash( self, profileConfig, oldDir ):
      if oldDir == self.constants_.opensslHash0ProfileDirPath(
            profileConfig.name ):
         currDir = self.constants_.opensslHash1ProfileDirPath(
               profileConfig.name )
      else:
         currDir = self.constants_.opensslHash0ProfileDirPath(
               profileConfig.name )
      for f in os.listdir( currDir ):
         os.remove( os.path.join( currDir, f ) )

      return currDir

