#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
import weakref
import os

traceHandle = Tracing.Handle( 'IpsecSswan' )

t0 = traceHandle.trace0

class IkeDaemonConfigHandler( Tac.Notifiee ):
   ''' Reacts to fips logging changes from mgmtSecurity CLI via 
       Ipsec '''
   notifierTypeName = 'Ipsec::DaemonConfig'

   def __init__( self, parent, daemonConfig ):
      t0( "Starting IkeDaemonConfigHandler" )
      self.parent = weakref.proxy( parent )
      self.daemonConfig = daemonConfig
      Tac.Notifiee.__init__( self, daemonConfig )

   @Tac.handler( 'fipsLogging' )
   def handleFipsLogging( self ):
      t0( "Updating strongswan ike daemon fips mode logging" )
      self.parent.handleFips()

class IkeDaemonFipsHandler( Tac.Notifiee ):
   ''' Reacts to fips changes from the CLI '''
   notifierTypeName = 'Ipsec::Ike::Config'
   opensslPluginFipsContents = '''
openssl {{
    fips_mode = 1
    load = yes
    fips_mode_logging = {FipsModeLog}
}} '''
   opensslPluginNonFipsContents = '''
openssl {
    load = yes
} '''

   def getOpensslPluginConfFile( self ):
      confFile = os.environ.get( 'STRONGSWAN_CHARON_OPENSSL_CONF', None)
      return confFile if confFile else \
         "/etc/strongswan/strongswan.d/charon/openssl.conf" 

   def __init__( self, parent, ikeConfig, ikeDaemonStatus,
                ikeDaemonConfig ):
      t0( "Starting IkeDaemonFipsHandler" ) 
      self.parent = weakref.proxy( parent )
      self.ikeConfig = ikeConfig
      self.ikeDaemonStatus = ikeDaemonStatus
      self.ikeDaemonConfig = ikeDaemonConfig
      Tac.Notifiee.__init__( self, ikeConfig )
      self.idcHandler = IkeDaemonConfigHandler( self, ikeDaemonConfig )
      self.updateFipsConfig( self.ikeConfig.fipsRestrictions )
      self.fipsShadow = None

   def onSwitchover( self ):
      # parent should set this before calling onSwitchover
      assert self.ikeDaemonStatus

      fips = self.ikeConfig.fipsRestrictions
      if self.fipsShadow != fips:
         self.updateFipsConfig( fips )
      else:
         self.maybeUpdateFipsStatus( fips )

   def updateFipsConfig( self, fips, fipsLogging=0 ):
      # pylint: disable-next=consider-using-f-string
      t0( "Updating Ike daemon for %s mode" % "FIPS" if fips else "non-FIPs" )
      with open( self.getOpensslPluginConfFile(), "w" ) as f:
         if fips:
            f.write( self.opensslPluginFipsContents.format(
                FipsModeLog=fipsLogging ) )
         else:
            f.write( self.opensslPluginNonFipsContents )

      self.maybeUpdateFipsStatus( fips )
      self.fipsShadow = fips

   def maybeUpdateFipsStatus( self, fips ):
      # when we are the standby sup, ikeDaemonStatus is None
      if self.ikeDaemonStatus:
         self.ikeDaemonStatus.fipsMode = fips

   @Tac.handler( 'fipsRestrictions' )
   def handleFips( self ):
      t0( "Updating strongswan ike daemon for fips mode" )
      # we will let Ipsec agent worry about restarting us
      # so no need to respond.
      # TBD Investigate  but lets handle ourselves too. 
      self.parent.stopService()
      fips = self.ikeConfig.fipsRestrictions
      fipsLogging = int( self.ikeDaemonConfig.fipsLogging )
      self.updateFipsConfig( fips, fipsLogging )
      self.parent.startService()

