# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Cell
from StageGraphUtil import getPlatformType
import StageSysdbHelper
import Tac
import Tracing
from Toggles.StageMgrToggleLib import toggleMacsecSsuEnabled, \
   toggleStrataIgmpSnoopingAsuEnabled
import Swag

traceHandle = Tracing.Handle( 'StageGraphs' )
t0 = traceHandle.trace0

def registerAsuShutdown( entityManager ):
   sh = StageSysdbHelper.StageHelper( entityManager, 'shutdown' )
   sh.applicationIs( "Asu Shutdown" )

   platform = getPlatformType()
   t0( "registerAsuShutdown platform:", platform )

   # Since ElectionMgr is launched through EosInit Script, we can't rely on agent
   # config to determine if it is running. As a result, setting completeNotRunnable
   # to True will always mark stage as complete automatically. This is undesired
   # on a modular system where ElectionMgr is actually running. As ElectionMgr is
   # not runnable on all the systems, we create different graph here to workaround
   # this.
   if Cell.electionMgrSupported():
      sh.registerStageDependency( "DisableSwitchover", [] )
      # SCD_DEFAULT_PEER_DOWN_TIMEOUT is 30s, this means it's possible to take up to
      # 30 seconds for redundancy state to change in an extreme case. Hence, double
      # the timeout value to 60s to be safe.
      sh.registerStage( "ElectionMgr", "DisableSwitchover", timeout=60 )
      sh.registerStageDependency( "MlagShutdown", [ "DisableSwitchover" ] )
   else:
      sh.registerStageDependency( "MlagShutdown", [] )
   sh.registerStage( "Mlag", "MlagShutdown", timeout=90, completeNotRunnable=True )

   # BUG985044: Disable ErrdisableIntfsConfig in SWAG mode as it never completes.
   if Swag.swagMode():
      predecessor = "MlagShutdown"
   else:
      predecessor = "ErrdisableIntfsConfig"
      sh.registerStageDependency( "ErrdisableIntfsConfig", [ "MlagShutdown" ] )
      sh.registerStage( "Ebra", "ErrdisableIntfsConfig", timeout=180 )

   sh.registerStageDependency( "NetworkAgentShutdown", [ predecessor ] )
   sh.registerStage( "Arp", "NetworkAgentShutdown", timeout=30 )
   sh.registerStage( "Bmp", "NetworkAgentShutdown", timeout=30,
                     completeNotRunnable=True )
   sh.registerStage( "Fhrp", "NetworkAgentShutdown", timeout=30,
                     completeNotRunnable=True )
   sh.registerStage( "Ipv6RouterAdvt", "NetworkAgentShutdown", timeout=30,
                     completeNotRunnable=True )
   sh.registerStage( "Rib", "NetworkAgentShutdown", timeout=30,
                     completeNotRunnable=True )
   sh.registerStage( "Ospf", "NetworkAgentShutdown", timeout=30,
                     completeNotRunnable=True )
   sh.registerStage( "Ospf3", "NetworkAgentShutdown", timeout=30,
                     completeNotRunnable=True )
   sh.registerStage( "Isis", "NetworkAgentShutdown", timeout=30,
                     completeNotRunnable=True )
   sh.registerStage( "Bfd", "NetworkAgentShutdown", timeout=30, critical=False )
   sh.registerStage( "Bgp", "NetworkAgentShutdown", timeout=30,
                     completeNotRunnable=True )
   sh.registerStage( "Pim", "NetworkAgentShutdown", timeout=30,
                     completeNotRunnable=True )
   sh.registerStage( "PimBsr", "NetworkAgentShutdown", timeout=30,
                     completeNotRunnable=True )
   sh.registerStage( "Stp", "NetworkAgentShutdown", timeout=30,
                     completeNotRunnable=True )

   if toggleStrataIgmpSnoopingAsuEnabled():
      sh.registerStage( "IgmpSnooping", "NetworkAgentShutdown", timeout=30,
                        completeNotRunnable=True )

   if toggleMacsecSsuEnabled():
      # Delay the shutdown stage for Macsec. MacsecShutdown which will start only
      # after NetworkAgentShutdown is completed.
      sh.registerStageDependency( "MacsecShutdown", [ "NetworkAgentShutdown" ] )
      sh.registerStage( "Macsec", "MacsecShutdown", timeout=120,
                        completeNotRunnable=True )

   sh.registerStageDependency( "PlatformBfdShutdown", [ "NetworkAgentShutdown" ] )
   sh.registerStage( "StrataBfd", "PlatformBfdShutdown", timeout=30,
                     completeNotRunnable=True )
   sh.registerStage( "SandBfd", "PlatformBfdShutdown", timeout=30,
                     completeNotRunnable=True )

   lacpTxAgentDep = "NetworkAgentShutdown"
   sh.registerStageDependency( "LacpPacketsConfig", [ lacpTxAgentDep ] )
   sh.registerStage( "LacpTxAgent", "LacpPacketsConfig", timeout=30 )

   # BUG985629: Disable broken stages when in SWAG mode
   if Swag.swagMode():
      return

   sh.registerStageDependency( "VxlanArpShutdown", [ "NetworkAgentShutdown" ] )

   if platform == "strata":
      if toggleMacsecSsuEnabled():
         # Macsec will need this agent for handling MKPDU. Stopping the agent only
         # after MacsecShutdown stage is completed
         sh.registerStageDependency( "DmaDriverShutdown", [ "LacpPacketsConfig",
                                                         "MacsecShutdown" ] )
      else:
         sh.registerStageDependency( "DmaDriverShutdown", [ "LacpPacketsConfig" ] )
      sh.registerStage( "StrataL2", "DmaDriverShutdown", timeout=30 )

      if not Swag.swagMode():
         # BUG974858: StrataCounters in swagMode does not support stage management
         sh.registerStageDependency( "CountersShutdown", [] )
         sh.registerStage( "StrataCounters", "CountersShutdown", timeout=60,
                           completeNotRunnable=True )
