# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
# pylint: disable=superfluous-parens

# Important note: if you expect to use an object returned by one of these
# routines you must have created a StageMgr() early enough to allow its
# mounts to have completed before you call any other method.
# Further, each time you specify a new agent, you must allow the agent config
# to be mounted, too.

# Trace level: 
#   0 - critical event, error
#   1 - significant events, warning, abnormaly
#   2 - debug, function name, checkpoints
#   3 - debug, internal states
#   4 - debug, misc
#   5 - debug, trivial

import Tac, Cell, Tracing, EntityManager
import weakref
from StageMgr import defaultStageInstance

__defaultTraceHandle__ = Tracing.Handle( 'StageHelper' )

t0 = Tracing.trace0
t1 = Tracing.trace1
t2 = Tracing.trace2
t3 = Tracing.trace3
t4 = Tracing.trace4
t5 = Tracing.trace5

def eventConfigCompleteIs( eventConfig,
                           stageInstance=defaultStageInstance,
                           complete=True ):
   eventConfig.complete[ stageInstance ] = complete

def eventConfigComplete( eventConfig,
                         stageInstance=defaultStageInstance ):
   if stageInstance in eventConfig.complete:
      return eventConfig.complete[ stageInstance ]
   return False

def progressIs( progressDir, stageInstance=defaultStageInstance ):
   return progressDir.newProgress( stageInstance )

def progressDel( progressDir, stageInstance=defaultStageInstance ):
   if stageInstance in progressDir.progress:
      del progressDir.progress[ stageInstance ]

def stageInProgress( progressDir,
                     stageName,
                     stageInstance=defaultStageInstance ):
   if not stageInstance in progressDir.progress:
      return False
   return stageName in progressDir.progress[ stageInstance ].stage

def stageInProgressIs( progressDir,
                       stageName,
                       stageInstance=defaultStageInstance ):
   progress = progressDir.newProgress( stageInstance )
   progress.stage[ stageName ] = True

def stageInProgressDel( progressDir,
                        stageName,
                        stageInstance=defaultStageInstance ):
   if stageInstance not in progressDir.progress:
      return
   progress = progressDir.progress[ stageInstance ]
   del progress.stage[ stageName ]

def allStagesInProgress( progressDir, stageInstance=defaultStageInstance ):
   if not stageInstance in progressDir.progress:
      return []
   return list( progressDir.progress[ stageInstance ].stage.items() )

def stageProgressionComplete( stageStatus,
                              stageInstance=defaultStageInstance, 
                              completionStatus=None ):
   if completionStatus :
      if not stageInstance in completionStatus.status:
         return False
      return completionStatus.status[ stageInstance ].complete
   else:
      if not stageInstance in stageStatus.instStatus:
         return False
      return stageStatus.instStatus[ stageInstance ].complete

def stageProgressionCompleteIs( stageStatus,
                                stageInstance=defaultStageInstance,
                                completionStatus=None,
                                complete=True ):
   if completionStatus:
      if not stageInstance in completionStatus.status:
         completionStatus.status.newMember( stageInstance )
      completionStatus.status[ stageInstance ].complete = complete
   else: 
      if not stageInstance in stageStatus.instStatus:
         stageStatus.instStatus.newMember( stageInstance )
      stageStatus.instStatus[ stageInstance ].complete = complete

def activeStageList( stageStatus, stageInstance=defaultStageInstance ):
   if stageInstance not in stageStatus.instStatus:
      return None
   return list( stageStatus.instStatus[ stageInstance ].activeStage.keys() )

def activeStage( stageStatus, stageName, stageInstance=defaultStageInstance ):
   if stageInstance not in stageStatus.instStatus:
      return False
   return stageName in stageStatus.instStatus[ stageInstance ].activeStage

def checkSwitchoverComplete( switchoverStatus, switchoverStageName, 
                             checkAgentStatusComplete=None ):
   stage = ( switchoverStatus and 
             switchoverStageName in switchoverStatus.stage and
             switchoverStatus.stage[ switchoverStageName ] )
   if not stage:
      return False

   for agentName, event in stage.event.items():
      if agentName == 'Sysdb':
         if not eventConfigComplete( event ):
            return False
      else:
         assert checkAgentStatusComplete is not None, \
            ( 'Cannot validate complete for '
              'agent: {} stage: {}'.format( agentName, event ) )
         if not checkAgentStatusComplete( agentName, switchoverStageName ):
            return False
   return True

class ProgressSm( Tac.Notifiee ):
   """A quick and dirty wrapper to simplify handling of switchover stages.
   Interface is fine --- it captures a common case --- but this should 
   be rewritten pretty simply in tacc.  If we need to wait for 
   locallyReadOnly to be false, then we should pass in a filtered 
   redundancyStatus rather than the raw redundancyStatus"""
   notifierTypeName = 'Stage::ProgressDir'

   def __init__( self, progressDir, eventCfg, handler, completes=False ):
      t2( 'ProgressSm:__init__' )
      self.progressDir_ = progressDir
      self.eventCfg_ = eventCfg
      self.handler_ = handler
      self.completes_ = completes
      self.finished_ = False
      self.stageHandler_ = None
      Tac.Notifiee.__init__( self, progressDir )
      for key in self.progressDir_.progress:
         self.handleProgress( key=key )

   class StageHandler( Tac.Notifiee ):
      notifierTypeName = 'Stage::Progress'

      def __init__( self, progress, eventCfg, handler, completes=False ):
         t2( 'SwitchoverHandler:__init__' )
         self.eventCfg_ = eventCfg
         self.handler_ = handler
         self.completes_ = completes
         self.finished_ = False
         Tac.Notifiee.__init__( self, progress )
         for key in progress.stage:
            self.handleStage( key=key )

      @Tac.handler( 'stage' )
      def handleStage( self, key=None ):
         t2( 'handleStage', key )
         if not key:
            return
         progress = self.notifier_
         if not self.eventCfg_:
            if key in progress.stage:
               self.handler_()
            return
         if self.finished_:
            t4( "Already executed handler for", self.eventCfg_.agent, 
                self.eventCfg_.stage, "returning" )
            return
         elif self.eventCfg_.complete:
            t4( "Event already completed" )
            return
         if self.eventCfg_.stage in progress.stage:
            t0( " handling event for", self.eventCfg_.agent, self.eventCfg_.stage )
            self.handler_()
            self.finished_ = True
            if self.completes_:
               self.eventCfg_.complete = True
         else:
            t3( "Expected stage", self.eventCfg_.stage, "not yet active" )

   @Tac.handler( 'progress' )
   def handleProgress( self, key=None ):
      t2( 'handleProgress', key )
      if not key:
         return
      if key in self.progressDir_.progress:
         self.stageHandler_ = self.StageHandler( self.progressDir_.progress[ key ],
                                                 self.eventCfg_,
                                                 self.handler_,
                                                 self.completes_ )
      elif self.stageHandler_:
         del self.stageHandler_
         self.stageHandler_ = None

class StageHelper:
   def __init__( self, 
                 entityManager,
                 stageClass,
                 stageClassDirPath=None,
                 stageInputClassDirPath=None,
                 stageGlobalConfigPath=None,
                 # if you know you will define events in a set of agents,
                 # pass their names here, so the agent configs can be mounted
                 # at instantiation.
                 agent=(),
                 # If you've already mounted the following entities,
                 # StageMgr() doesn't need to mount them itself.  Or,
                 # if you want to override the corresponding config, status,
                 # or agentDir that live in Sysdb
                 status=None,
                 config=None,
                 agentDir=None,
                 progressDir=None,
                 log=None):
      t3( 'StageMgr.__init__' )
      self.em_ = entityManager
      if stageClassDirPath:
         # Both paths should be set together.  Otherwise, this seems to be a mistake.
         assert stageInputClassDirPath
      self.stageClassPath_ = getStageClassPath( stageClass, stageClassDirPath )
      self.stageInputClassPath_ = getStageInputClassPath( stageClass, 
                                                          stageInputClassDirPath )
      self.config_ = None
      self.status_ = None
      self.progressDir_ = None
      self.agentConfigDir_ = None
      self.log_ = None
      self.mounted_ = False
      self.mountedAgent_ = set()
      self.agentMountHandlers_ = {}
      self.eventMountHandlers_ = {}
      self.agentConfigDir_ = None
      self._mountAgentState( agent=agent, status=status, config=config,
                             agentDir=agentDir, progressDir=progressDir, log=log )

      self.eventReactors_ = []

   def _deferCreateEvent( self, agent, stage ):
      t3( '_deferCreateEvent', agent, stage )
      if( agent in self.mountedAgent_ ):
         # do it immediately
         self._createEvent( agent, stage )
      else:
         agentHandlers = self.agentMountHandlers_.setdefault( agent, [] )
         agentHandlers.append( stage )

   def _createEventHandler( self, agent, stage, handler ):
      t0( '_createEventHandler', agent, stage )
      if(( agent in self.mountedAgent_ ) and
         ( stage in self.agentConfigDir_[ agent ].event )):
         # do it immediately
         handler( self.agentConfigDir_[ agent ].event )
      else:
         agentEventHandlers = self.eventMountHandlers_.setdefault( agent, {} )
         eventHandlers = agentEventHandlers.setdefault( stage, [] )
         eventHandlers.append( handler )

   def _agentIsMounted( self, agent, mounted ):
      t0( '_agentIsMounted', agent, mounted )
      if( mounted and not ( agent in self.mountedAgent_ )):
         self.mountedAgent_.add( agent )
         if agent in self.agentMountHandlers_:
            for event in self.agentMountHandlers_[ agent ]:
               self._createEvent( agent, event )
            del self.agentMountHandlers_[ agent ]

   def _createEvent( self, agent, stage ):
      t0( '_createEvent', agent, stage )
      # assumes that agentConfig is already mounted under agentConfigDir
      agentCfg = self.agentConfigDir_[ agent ]
      eventCfg = agentCfg.event.get( stage )
      if not eventCfg:
         t5( "Creating event stage", stage, "for agent", agent )
         eventCfg = agentCfg.newEvent( stage, agent )
         if( agent in self.eventMountHandlers_ ):
            agentEvents = self.eventMountHandlers_[ agent ]
            if stage in agentEvents:
               eventHandlers = agentEvents[ stage ]
               for eventHandler in eventHandlers:
                  eventHandler( eventCfg )
               del agentEvents[ stage ]

   def _mountAgentState( self, agent=None, status=None, config=None,
                         agentDir=None, progressDir=None, log=None ):
      t3( "mountAgentState", agent or "" )
      # Try to avoid unnecessary mounts.  Check if we already exist.
      # Check if we can just use newEntity.  Check if mount is in progress.
      # Finally, do we need to mount from scratch...
      if self.mounted_:
         agentsMounted = True
         assert self.agentConfigDir_
         if agent:
            for a in agent:
               if not a in self.agentConfigDir_:
                  agentsMounted = False
                  break
         if agentsMounted:
            return
      statusPath = self.stageClassPath_ + '/status'
      configPath = self.stageClassPath_ + '/config/config'
      agentDirPath = self.stageInputClassPath_
      progressPath = self.stageClassPath_ + '/progress'
      logPath = self.stageClassPath_ + '/log'
      
      try:
         em = self.em_
         self.agentConfigDir_ = ( agentDir or
                                  em.entity( agentDirPath, "Tac::Dir", "wi" ) )
         self.config_ = ( config or
                          em.entity( configPath, "Stage::Config", "r" ) )
         self.status_ = ( status or
                          em.entity( statusPath, "Stage::Status", "r" ) )
         self.progressDir_ = ( progressDir or 
                          em.entity( progressPath, "Stage::ProgressDir", "r" ) )
         self.log_ = ( log or 
                          em.entity( logPath, "Stage::Log", "r" ) )
                                               
         if agent:
            msg = "agent must be a list/tuple of agent names"
            # pylint: disable-next=consider-merging-isinstance
            iterable = isinstance( agent, list ) or isinstance( agent, tuple )
            assert iterable, msg 
            for agentName in agent:
               assert isinstance( agentName, str ), msg
            for agentName in agent:
               agentPath = agentDirPath + "/" + agentName 
               agentConfig = em.entity( agentPath, # pylint: disable-msg=W0612
                                        "Stage::AgentConfig", "wcf" )
               if agentConfig:
                  self._agentIsMounted( agentName, True )
         self.mounted_ = True
         return # no need to mount anything
      except Exception as e: # pylint: disable-msg=W0703
         t3( "Exception:", e, " explicitly mounting stage state." )
         # We're going to need to mount _something_
         mg = self.em_.mountGroup()
         self.agentConfigDir_ = ( agentDir or
                                  mg.mount( agentDirPath, "Tac::Dir", "wfi"  ) )
         self.config_ = ( config or
                          mg.mount( configPath, "Stage::Config", "r"  ) )
         self.status_ = ( status or 
                          mg.mount( statusPath, "Stage::Status", "r"  ) )
         if agent:
            for agentName in agent:
               agentPath = agentDirPath + "/" + agentName 
               mg.mount( agentPath, "Stage::AgentConfig", "wcf"  )
         self.progressDir_ = ( progressDir or 
                            mg.mount( progressPath, "Stage::ProgressDir", "r" ) )

         def finishUp():
            if agent:
               for agentName in agent:
                  self._agentIsMounted( agentName, True )
            self.mounted_ = True
         if( Tac.activityManager.inExecTime.isZero and 
             ( self.em_.isLocalEm() or not mg.cMg_.parentMg )):
            mg.close( blocking=True )
            finishUp()
         else:
            mg.close( finishUp, blocking=False )

   def eventConfig( self, agent, stage, finish=None, create=True,
                    readOnly=False, reactOnly=False ):
      t3( 'eventConfig', agent, stage )
      if( finish and not create ):
         msg = """Cannot specify finish and not create.  If this is really
what you intend, then use 'reactOnly' arg"""
         assert reactOnly, msg
      assert  not ( readOnly and create ), "can't specify readOnly and create"
      if( agent in self.mountedAgent_ ):
         t5( "Agent", agent, "already mounted" )
         # pass. No need to mount agent. Go straight to config for event
      elif( EntityManager.isEntityFuture( self.agentConfigDir_ ) ):
         msg = """cannot call switchover methods until the mounts in 
                  Switchover.Switchover() have completed their mg.close()."""
         assert False, msg
      elif( not agent in self.agentConfigDir_ ):
         t4( "Agent", agent, "does not yet exist." )
         acType = "Stage::AgentConfig"
         if create:
            self.agentConfigDir_.newEntity( acType, agent )
            self._agentIsMounted( agent, True )
         else:
            if finish:
               self._createEventHandler( agent, stage, finish )
            return None
      elif( not self.agentConfigDir_[ agent ] ):
         # exists but not mounted
         t4( "Agent", agent, "exists, but not yet fully mounted." )
         mg = self.em_.mountGroup()
         agentDirName = self.stageInputClassPath_ + '/' + agent
         mode = "wfc" if ( not readOnly ) else "r"
         mg.mount( agentDirName, "Stage::AgentConfig", mode )
         if finish:
            self._createEventHandler( agent, stage, finish )
         if create:
            self._deferCreateEvent( agent, stage )
         # prefer blocking close if possible
         blocking = ( Tac.activityManager.inExecTime.isZero and
                      ( self.em_.isLocalEm() or not mg.cMg_.parentMg))
         def finishUp():
            self._agentIsMounted( agent, True )
         mg.close( finishUp, blocking=blocking )
         if blocking:
            return self.agentConfigDir_[ agent ].event.get( stage )
         else:
            return None
      agentCfg = self.agentConfigDir_[ agent ]
      eventCfg = agentCfg.event.get( stage )
      if create and not eventCfg:
         self._createEvent( agent, stage )
         eventCfg = agentCfg.event[ stage ]
      if finish and eventCfg:
         finish( eventCfg )
      return( eventCfg )

   def findEventConfig( self, agent, stage ):
      return self.eventConfig( agent, stage, create=False, readOnly=True )

   def onEventConfig( self, agent, stage, handler ):
      self.eventConfig( agent, stage, finish=handler )

   def agentMounted( self, agent ):
      return ( agent in self.mountedAgent_ )
   
   def stageIs( self, agent, stage, dependencies=None, force=False ):
      if not dependencies:
         dependencies = []
      t1( "registering stage", stage, "by agent", agent, "after",
          dependencies )
      def addDependencies( cfg ):
         stages = set()
         if not force:
            for agentConfig in self.agentConfigDir_.values():
               stages.update( agentConfig.event )
         for dependency in dependencies:
            if not force:
               assert dependency in stages
            cfg.dependency[ dependency ] = True
      cfg = self.eventConfig( agent, stage, addDependencies )
      return cfg

   def registerStage( self, agent, stage, dependencies=None, force=False,
                      timeout=None, enableAutoComplete=False,
                      autoCompleteTimeout=None, complete=None,
                      completeNotRunnable=False,
                      instanceName=defaultStageInstance ):
      cfg = self.stageIs( agent, stage, dependencies, force=force )
      cfg.completeNotRunnable = completeNotRunnable
      if timeout is not None:
         cfg.timeout = timeout
      if enableAutoComplete:
         if autoCompleteTimeout is not None:
            cfg.autoCompleteTimeout = autoCompleteTimeout
         cfg.timeout = cfg.autoCompleteTimeout + 5
         cfg.enableAutoComplete = True
      if complete is not None:
         eventConfigCompleteIs( cfg,
                                stageInstance=instanceName,
                                complete=complete )
      return cfg
      
   def unregisterStage( self, agent, stage ):
      t3( "unregistering stage", stage, "by agent", agent )
      eventCfg = self.eventConfig( agent, stage, create=False )
      if eventCfg:
         agentCfg =  self.agentConfigDir_.get( agent )
         if agentCfg and ( stage in agentCfg.event ):
            del agentCfg.event[ stage ]
      # if there is a handler for this event, try to remove the last 
      # reference to it.  Won't work unless used same helper object in
      # same process to register it in the first place.
      self.unregisterEventHandler( agent, stage )

   def registerEventHandler( self, agent, stage, function ):
      """Create a reactor that will call function when STAGE is 
      active, and will mark event as complete when done.  It goes without 
      saying that this routine can only be used if function is able to avoid 
      returning until all of the relevant operations are completed.  As this 
      is called from a reactor, all of the work must be synchronous.
      If some of the work needs to be done in reactors, and may involve
      asynchrony, consider registerEventReactor"""
      t2( 'registerEventHandler' )
      eventCfg = self.eventConfig( agent, stage )
      assert( eventCfg )
      self.eventReactors_.append(
         ProgressSm( self.progressDir_, eventCfg, function,
                     completes=True ) )

   def unregisterEventHandler( self, agent, stage ):
      t2( 'unregisterEventHandler' )
      for reactor in self.eventReactors_:
         if ( reactor.agent_ == agent ) and ( reactor.stage_ == stage ):
            self.eventReactors_.remove( reactor )
            return
      
   def registerEventReactor( self, agent, stage, function ):
      """Create a reactor that will call function when STAGE is the
      active switchoverStage, but will NOT mark event as complete when
      the function returns.  There may still be more work to be done.
      Use this instead of registerEventHandler if some of the work needs 
      to be done in reactors, and may involve asynchrony.  It is your 
      responsibility to call eventCompleted when the event is done."""
      t2( 'registerSwitchoverEventReactor' )
      eventCfg = self.eventConfig( agent, stage )
      assert( eventCfg )
      self.eventReactors_.append(
         ProgressSm( self.progressDir_, eventCfg, function,
                     completes=False ) )

   def eventCompleted( self, agent, stage, instanceName=defaultStageInstance ):
      t2( 'EventCompleted', agent, stage )
      event = self.eventConfig( agent, stage )
      eventConfigCompleteIs( event, stageInstance=instanceName )

   def registerStageDependency( self, stage, dependencies, force=False,
                                instanceName=defaultStageInstance ):
      """Register a dependency between stages without actually performing
      an event in the dependent stage.  Dependencies must already
      exist (unless force=True)"""
      t2( 'registerStageDependency' )
      identifier = 1
      def makeDummyName():
         return "__dummyInternal" + str( identifier ) + "__"
      agent = makeDummyName()
      while agent in self.agentConfigDir_:
         agentOkAsDummy = True
         if self.agentConfigDir_.get( agent ):
            agentConfig = self.agentConfigDir_[ agent ]
            for event in agentConfig.event.values():
               if not eventConfigComplete( event, stageInstance=instanceName ):
                  agentOkAsDummy = False
                  break # Stop checking events because we can't use this name
         if agentOkAsDummy:
            break # Use this name
         identifier = identifier + 1
         agent = makeDummyName()
         
      cfg = self.registerStage( agent, stage, dependencies, force=force )
      event = self.eventConfig( agent, stage )
      event.critical = False
      # If event.complete, will never be added to pendingEvents in 
      # SwitchoverSm
      eventConfigCompleteIs( event, stageInstance=instanceName )
      return cfg

   def registerEvent( self, agent, stage, dependencies, function,
                      force=False, completes=False, 
                      sysdbIndependent=False ):
      """Define an event in stage STAGE, list dependencies between stages, and
      create reactor that will call function when stage is active.  See 
      registerSwitchoverEventHandler for restrictions on FUNCTION"""
      t0( 'registerSwitchoverEvent' )
      cfg = self.stageIs( agent, stage, dependencies, force=force )
      eventCfg = self.eventConfig( agent, stage )
      self.eventReactors_.append( 
         ProgressSm( self.progressDir_, eventCfg, function, 
                     completes=completes ) )
      return cfg

   def applicationIs( self, application ):
      """ Override the default application name, which was set to the stageClass."""
      self.config_.application = application

   def stageTimeoutIsFatalIs( self, isFatal=True ):
      """ Set whether critical stages time outs will invoke fatal error request """
      self.config_.stageTimeoutIsFatal = isFatal

   def continueOnTimeoutIs( self, continueOn=False ):
      """ Set whether to continue on timeout, ignoring critial and fatal """
      self.config_.continueOnTimeout = continueOn
      
   def globalStageCompletionTimeoutIs( self, timeout=0 ):
      """ Set how long global stage progression will timeout """
      self.config_.globalStageCompletionTimeout = timeout      

   def globalStageCompletionTimeout( self ):
      """ Get how long global stage progression will timeout """
      return self.config_.globalStageCompletionTimeout      
      
   def resetModeIs( self, mode ):
      """ Set reset mode - whether to reset local or reset all """
      self.config_.resetMode = mode      
      
   def reloadCauseDescIs( self, desc ):
      """ Set reload cause description code """
      self.config_.reloadCauseDesc = desc

class PyStagesHelper:
   def __init__( self, stageClass, stageStatus, progressDir,
                 agentConfig, agent, completionStatus=None, agentStatus=None ):
      self.cPyStagesHelper_ = Tac.newInstance( \
         "AgentBase::PyStagesHelper", stageClass, stageStatus,
         progressDir, agentConfig )
      if completionStatus:
         self.cPyStagesHelper_.setInstanceCompletionStatusDir( completionStatus )
      if agentStatus:
         self.cPyStagesHelper_.agentStatus = agentStatus
      weakSelf = weakref.proxy( agent )
      self.cPyStagesHelper_.pythonObj = id( weakSelf )

   def cStagesHelper( self ):
      return self.cPyStagesHelper_

   def waitForStage( self, stage ):
      self.cPyStagesHelper_.waitForStage( stage )

   def waitForStageInstance( self, stage, instanceName ):
      self.cPyStagesHelper_.waitForStageInstance( stage, instanceName )

   def doStageComplete( self, stage ):
      self.cPyStagesHelper_.doStageComplete( stage )

   def doStageCompleteForInstance( self, stage, instanceName ):
      self.cPyStagesHelper_.doStageCompleteForInstance( stage, instanceName )

   def waitForStageProgressionCompleteInstance( self, instanceName ):
      self.cPyStagesHelper_.waitForStageProgressionCompleteInstance( instanceName )

   def waitForStageProgressionComplete( self ):
      self.cPyStagesHelper_.waitForStageProgressionComplete()

   def stageInProgress( self, stage ):
      return self.cPyStagesHelper_.stageInProgress( stage )

   def stageInProgressForInstance( self, stage, instanceName ):
      return self.cPyStagesHelper_.stageInProgressForInstance( stage, instanceName )

def getStageClassDirPath( path=None ):
   if path is None:
      path = "cell/%d/stage" % Cell.cellId()
   return path

def getStageClassPath( stageClass, path=None ):
   return getStageClassDirPath( path ) + '/' + stageClass

def getStageInputClassPath( stageClass, path=None ):
   if path is None:
      path = "cell/%d/stageInput" % Cell.cellId()
   return path + '/' + stageClass 

def getStageGlobalConfigPath( path=None ):
   if path is None:
      path = "stage/stageGlobalConfig"
   return path

# to be called in sysdb plugin
def registerStageGlobalConfig( entityManager, stageGlobalConfigPath=None ):
   stageGlobalConfigPath = getStageGlobalConfigPath( stageGlobalConfigPath )
   entityManager.registerConfigMount( stageGlobalConfigPath,
                                      'Stage::StageGlobalConfig' )

# to be called in sysdb plugin
def registerStageClass( entityManager, stageClass, stageClassDirPath=None,
                        stageInputClassDirPath=None, useDefaultInstance=True ):
   if stageClassDirPath:
      # Both paths should be set together.  Otherwise, this seems to be a mistake.
      assert stageInputClassDirPath
   stageClassPath = getStageClassPath( stageClass, stageClassDirPath )
   stageClassDirPath = getStageClassDirPath( stageClassDirPath )
   stageInputClassPath = getStageInputClassPath( stageClass, stageInputClassDirPath )

   entityManager.register( stageClassDirPath, 'Tac::Dir' )
   entityManager.register( stageClassPath, 'Tac::Dir' )
   entityManager.register( stageClassPath + '/config', 'Tac::Dir' )
   config = entityManager.register( stageClassPath + '/config/config',
                                    'Stage::Config')
   config.stageClass = stageClass
   config.application = stageClass
   agentDir = entityManager.register( stageInputClassPath, 'Tac::Dir' )
   config.agentDir = agentDir
   entityManager.register( stageClassPath + '/status',  'Stage::Status')
   entityManager.register( stageClassPath + '/completionstatus', 
                                          'Stage::CompletionStatusDir')
   entityManager.register( stageClassPath + '/log', 'Stage::Log' )
   progressDir = entityManager.register( stageClassPath + '/progress', 
                                         'Stage::ProgressDir')
   if useDefaultInstance:
      progressDir.progress.newMember( defaultStageInstance )

# Register StageMgr StageGraphs. The PyClient object passed in must be from Sysdb
# agent.
def registerStageGraphsInSysdb( sysdbRoot ):
   pyClientObj = sysdbRoot.pyClient()
   cmd = '''
from Sysdb import globalEntityManager
from StageGraphs import registerStageGraphs
entMan = globalEntityManager
assert entMan
registerStageGraphs( entMan )
'''
   pyClientObj.execute( cmd )

