# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import time

class StopWatch:
   '''Simple class for measuring block execution times

   Calls supplied log function with starting/finishing/failing messages.
   Can optionally raise `TimeLimitError` if time limit is exceeded.

   Use as a context manager, access result in the `value` property.

   Usage example:
   ```python
   with StopWatch( 'destroying system', log=print ):
      os.system( 'rm -rf /' )
   ```
   Possible output:
   ```
   started destroying system
   finished destroying system in 0.0s
   ```
   '''

   class UsageError( Exception ):
      '''StopWatch usage error'''

   class TimeLimitError( Exception ):
      '''StopWatch time limit error'''

      TEMPLATE = 'exceeded time limit {description} with {time}'

   TIME_TEMPLATE_DEFAULT = '{:.1f}s'  # e.g.: '2.7s'
   LIMITLESS_STR_TEMPLATE = '{value}'  # e.g.: '2.7s'
   LIMITED_STR_TEMPLATE = '{value} out of {limit}'  # e.g.: '2.7s out of 3.1s'

   STARTED_TEMPLATE = 'started {description}'
   FINISHED_TEMPLATE = 'finished {description} in {time}'
   FAILED_TEMPLATE = 'failed {description} after {time}'

   def __init__( self, description, log, *, timeSource=time.perf_counter,
                 timeTemplate=TIME_TEMPLATE_DEFAULT, timeLimit=None ):
      '''Initializes new StopWatch object

      Arguments:
         `description`: action description for trace and exception messages,
                        in the -ing form (e.g. 'making widgets');
         `log`: function to be called with status messages as a sole argument
         `timeSource`: nullary function returning monotonic time,
                       defaults to `time.perf_counter`;
         `timeTemplate`: `format` template for formatting result returned by
                         timeSource, default is `TIME_TEMPLATE_DEFAULT`;
         `timeLimit`: if specified, `TimeLimitError` will be raised upon successful
                      completion if it took longer than `timeLimit`;
      '''
      self.description = description
      self.log = log
      self.timeSource = timeSource
      self.timeTemplate = timeTemplate
      self.timeLimit = timeLimit
      self.started = None
      self.finished = None

   @property
   def value( self ):
      '''Property with current (if running) or total (if stopped) execution time

      Units of time are those returned by `timeSource` constructor argument,
      seconds by default.

      Cannot be called if stopwatch has never been started,
      otherwise `UsageError` is raised.'''
      if self.started is None:
         raise self.UsageError( 'never started' )
      return ( self.finished or self.timeSource() ) - self.started

   @property
   def running( self ):
      '''Property showing if stopwatch is running'''
      return self.started is not None and self.finished is None

   def start( self ):
      '''Resets and starts stopwatch

      Cannot be called if already running, otherwise `UsageError` is raised.'''
      if self.running:
         raise self.UsageError( 'already running' )
      self.started = self.timeSource()
      self.finished = None

   def stop( self ):
      '''Stops stopwatch

      Can only be called when running, otherwise `UsageError` is raised.'''
      if not self.running:
         raise self.UsageError( 'not running' )
      self.finished = self.timeSource()

   def __str__( self ):
      '''Returns current stopwatch value as well as time limit (if any) as string

      Either `LIMITLESS_STR_TEMPLATE` or `LIMITED_STR_TEMPLATE` is used depending
      on presense of time limit. Format of each time is controlled by `timeTemplate`
      constructor argument.'''

      value = 0 if self.started is None else self.value
      valueString = self.timeTemplate.format( value )
      if self.timeLimit is None:
         return self.LIMITLESS_STR_TEMPLATE.format( value=valueString )
      else:
         limitString = self.timeTemplate.format( self.timeLimit )
         return self.LIMITLESS_STR_TEMPLATE.format( value=valueString,
                                                    limit=limitString )

   def __enter__( self ):
      '''Resets and starts stopwatch upon entering the context

      Produces `STARTED_TEMPLATE` trace message.'''
      self.log( self.STARTED_TEMPLATE.format( description=self.description ) )
      self.start()
      return self

   def __exit__( self, exception_type, exception_value, exception_traceback ):
      '''Stops stopwatch upon exiting the context

      Produces `FAILED_TEMPLATE` trace message if active exception is present,
      `FINISHED_TEMPLATE` otherwise.

      Raises `TimeLimitError` exception upon successful completion if it took longer
      than time limite specified in the constructor. Exception message is defined by
      `TimeLimitError.TEMPLATE`.'''
      self.stop()
      if exception_type is not None:
         self.log( self.FAILED_TEMPLATE.format( description=self.description,
                                                time=self ) )
      else:
         self.log( self.FINISHED_TEMPLATE.format( description=self.description,
                                                  time=self ) )
         if self.timeLimit is not None and self.value > self.timeLimit:
            message = self.TimeLimitError.TEMPLATE.format(
                  description=self.description, time=self )
            raise self.TimeLimitError( message )
