# Copyright (c) 2008-2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, SuperServer, os, weakref, Cell, Tracing
import EpochConsts

t0 = Tracing.trace0
t1 = Tracing.trace1

class HostnameReactor( Tac.Notifiee ):
   """ Notifiees the BannerConfigNotifiee of changes to the hostname,
   so that the login banner and motd can be rewritten """

   notifierTypeName = "System::NetStatus"

   def __init__( self, netStatus, bannerConfigNotifiee ):
      self.bannerConfigNotifiee_ = weakref.proxy( bannerConfigNotifiee )
      Tac.Notifiee.__init__( self, netStatus )
      # NOTE - no need to call handleHostname explicitly here, as I know that
      # sync will already get called after the BannerConfigNotifiee is
      # constructed

   @Tac.handler( 'hostname' )
   def handleHostname( self ):
      self.bannerConfigNotifiee_.sync()

class BannerConfigReactor( Tac.Notifiee ):

   notifierTypeName = 'System::BannerConfig'

   def __init__( self, bannerConfig, bannerConfigDirReactor ):
      self.bannerConfigDirReactor_ = weakref.proxy( bannerConfigDirReactor )
      Tac.Notifiee.__init__( self, bannerConfig )
      # Since both handleLoginBanner() and handleMotd() do the same thing
      # just call one of them
      self.handleLoginBanner()

   @Tac.handler( 'loginBanner' )
   def handleLoginBanner( self ):
      self.bannerConfigDirReactor_.sync()

   @Tac.handler( 'motd' )
   def handleMotd( self ):
      self.bannerConfigDirReactor_.sync()

   def close( self ):
      self.bannerConfigDirReactor_.sync()
      Tac.Notifiee.close( self )

class BannerConfigDirReactor( SuperServer.GenericService ):
   """ Reactor to the 'sys/banner/config' directory that contains the login banner
   and motd display configuration. """

   notifierTypeName = 'Tac::Dir'
   BannerConfigType = Tac.Type( "System::BannerConfig" )

   def __init__( self, serviceName, bannerConfig, extraBannerConfig, netStatus,
                 swiVariant ):
      # Link /etc/issue.net to /etc/issue. This will ensure that the console
      # and telnet login banners are the same
      if ( os.path.exists( "/etc/issue.net" ) and
           not os.path.realpath( "/etc/issue.net" ) == "/etc/issue" ):
         os.unlink( "/etc/issue.net" )
      # Make sure an /etc/issue file exists, otherwise the symlink
      # will fail
      if not os.path.exists( "/etc/issue" ):
         with open( "/etc/issue", "w" ):
            pass
      if not os.path.exists( "/etc/issue.net" ):
         os.symlink( "/etc/issue", "/etc/issue.net" )

      # Collection of reactors for our banner configs
      self.bannerConfigReactor_ = {}
      self.extra_ = extraBannerConfig
      
      # Create a reactor to the system hostname, so that the login banner
      # and motd can appropriately replace the special $(hostname) token
      self.netStatus_ = netStatus
      self.hostnameReactor_ = HostnameReactor( netStatus, self )

      self.swiVariant = swiVariant

      SuperServer.GenericService.__init__( self, serviceName, bannerConfig )
      for k in self.notifier_.entityPtr:
         self.handleEntityPtr( k )
     # Required for setting a timer to retry on writeConfigFile failure
      # This has to be an instance variable because otherwise, the Tac.ClockNotifiee
      # will go out of scope, causing the notifee to go away, as described by the 
      # comments in Tac.ClockNotifiee
      self.activity_ = None
      self.activityInterval_ = 60

   @Tac.handler( 'entityPtr' )
   def handleEntityPtr( self, key ):
      t0( 'handleEntityPtr', key, type( self.notifier_.entityPtr ) )
      def reactorFilter( k ):
         # We are only interested in entries with a valid entity type
         entity = self.notifier_.entityPtr.get( k )
         # pylint: disable-next=unidiomatic-typecheck
         return type( entity ) == self.BannerConfigType
      Tac.handleCollectionChange( BannerConfigReactor, key,
                                  self.bannerConfigReactor_,
                                  self.notifier_.entityPtr,
                                  reactorArgs=( self, ),
                                  reactorFilter=reactorFilter )

   def sync( self ):
      # Update /etc/issue and /etc/motd

      def _formatMessage( msgs ):
         msg = '\n'.join( msgs )
         msg = msg.replace( "$(hostname)", self.netStatus_.hostname )
         return msg

      # Create the login banner and motd from all of the configs.
      loginBanners = []
      motds = []

      # Add the international disclaimer banner if needed
      SwiVariant = Tac.Type( "HwEpoch::SwiVariant" )
      if self.swiVariant == SwiVariant.international:
         loginBanners.append( EpochConsts.InternationalDisclaimer )

      for bannerConfig in sorted( list( self.notifier_.entityPtr.values() ) +
                                  list( self.extra_.entityPtr.values() ) ):
         # pylint: disable-next=unidiomatic-typecheck
         if type( bannerConfig ) != self.BannerConfigType:
            # The mount is in progress, ignore
            t0( bannerConfig, "mount is in progress" )
            continue
         t1( "bannerConfig", bannerConfig )
         if bannerConfig.loginBanner:
            loginBanners.append( bannerConfig.loginBanner )
         if bannerConfig.motd:
            motds.append( bannerConfig.motd )

      if not ( self.writeConfigFile( "/etc/issue",
                                     _formatMessage( loginBanners ),
                                     saveBackupConfig=False,
                                     writeHeader=False ) and \
               self.writeConfigFile( "/etc/motd",
                                     _formatMessage( motds ), 
                                     saveBackupConfig=False,
                                     writeHeader=False ) ):
         # Schedules a sync() in a minute
         if self.activity_:
            self.activity_.timeMin = min( self.activity_.timeMin,
                                          Tac.now() + self.activityInterval_ )
         else:
            self.activity_ = Tac.ClockNotifiee()
            self.activity_.handler = self.sync
            self.activity_.timeMin = Tac.now() + self.activityInterval_

   def warm( self ):
      # This is unnecessary as nobody is calling in here.
      return True

class BannerManager( SuperServer.SuperServerAgent ):
   """ This agent reacts to the 'system/banner/config' object and configures
   the login banner and motd appropriately. """
   def __init__( self, entityManager ):
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      mg = entityManager.mountGroup()
      bannerConfigDir = mg.mount( 'sys/banner/config', 'Tac::Dir', 'ri' )
      bannerConfigCellDir = mg.mount( 
         Cell.path( 'sys/banner/config' ), 'Tac::Dir', 'ri' )
      netStatus = mg.mount( Cell.path( 'sys/net/status' ), 
                            'System::NetStatus', 'r' )
      hwEpochStatus = mg.mount( 'hwEpoch/status', 'HwEpoch::Status', 'r' )
      self.notifiee_ = None
      self.notifieeCell_ = None
      def _finished():
         self.notifiee_ = BannerConfigDirReactor( "BannerManager",
                                                  bannerConfigDir,
                                                  bannerConfigCellDir,
                                                  netStatus,
                                                  hwEpochStatus.swiVariant )
         self.notifieeCell_ = BannerConfigDirReactor( 
            # pylint: disable-next=consider-using-f-string
            "BannerManager-Cell%d" % Cell.cellId(), 
            bannerConfigCellDir,
            bannerConfigDir,
            netStatus,
            hwEpochStatus.swiVariant )
      mg.close( _finished )

def Plugin( ctx ):
   ctx.registerService( BannerManager( ctx.entityManager ) )
