# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ContainerMgrConfigSm import ContainerMgrConfigReactor
from ContainerMgrContainerSm import ContainerConfigReactor
from ContainerMgrImageLoadSm import ImageLoadNotifiee
from ContainerMgrRegistrySm import RegistryReactor
import QuickTrace
import SuperServer
import Tac
import Tracing

traceHandle = Tracing.Handle( "ContainerMgr" )

t0 = traceHandle.trace0 # important function calls
t1 = traceHandle.trace1 # error/exception
t2 = traceHandle.trace2 # other function calls

qv = QuickTrace.Var
qt0 = QuickTrace.trace0 # Important function calls
qt1 = QuickTrace.trace1 # error/exception
qt2 = QuickTrace.trace2 # other function calls

class ContainerMgr( SuperServer.SuperServerAgent ):
   def __init__( self, entityManager ):
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      mg = entityManager.mountGroup()
      self.containerMgrConfig = mg.mount( 'containerMgr/config',
                                          'ContainerMgr::ContainerMgrConfig', 'r' )

      # TODO: Remove after ContainerMgr is decoupled from SuperServer
      self.containerMgrSuperServerConfig = \
            mg.mount( 'containerMgr/superServerConfig',
                      'ContainerMgr::ContainerMgrSuperServerConfig', 'r' )
      self.containerMgrSuperServerStatus = \
            mg.mount( 'containerMgr/superServerStatus',
                      'ContainerMgr::ContainerMgrSuperServerStatus', 'w' )

      self.kernelRoutingOverrideStatus = mg.mount( 'l3/intf/kernelRoutingOverride',
                                       'Routing::KernelRoutingOverrideStatus', 'w' )
      self.containerConfigDir = mg.mount( 'containerMgr/container/config',
                                          'ContainerMgr::ContainerConfigDir', 'r' )
      self.containerStatusDir = mg.mount( 'containerMgr/container/status',
                                          'ContainerMgr::ContainerStatusDir', 'w' )
      self.profileConfigDir = mg.mount(
            'containerMgr/container/profileConfig',
            'ContainerMgr::ContainerProfileConfigDir', 'r' )
      self.consolidatedContainerConfigDir = \
            mg.mount( 'containerMgr/container/consolidatedConfig',
                      'ContainerMgr::ContainerConfigDir', 'w' )
      self.imageLoadConfigDir = mg.mount( 'containerMgr/image/config',
                                          'ContainerMgr::ImageLoadConfigDir', 'r' )
      self.imageLoadStatusDir = mg.mount( 'containerMgr/image/status',
                                          'ContainerMgr::ImageLoadStatusDir', 'w' )
      self.registryConfigDir = mg.mount( 'containerMgr/registry/config',
                                         'ContainerMgr::RegistryConfigDir', 'r' )
      self.registryStatusDir = mg.mount( 'containerMgr/registry/status',
                                         'ContainerMgr::RegistryStatusDir', 'w' )
      self.profileDirSm = None
      self.notifiee_ = None
      self.registryNotifiee_ = None
      self.containerNotifee_ = None
      self.imageLoadNotifiee_ = None

      def _finished():
         if not self.active():
            return
         self.createReactors()
      mg.close( _finished )

   def daemonStarted( self ):
      assert self.notifiee_ is not None
      return self.notifiee_.daemonStarted

   def handleContainersConfig( self ):
      for containerName in self.containerConfigDir.container:
         reactor = self.containerNotifee_.reactor( containerName )
         reactor.handleContainerConfig()

      # Remove stale containers from status
      staleContainers = [ containerName
                          for containerName in self.containerStatusDir.container
                          if containerName not in self.containerConfigDir.container ]
      if staleContainers and not self.daemonStarted():
         t0( "handleContainersConfig stale container removal skipped" )
         qt0( "handleContainersConfig stale container removal skipped" )
         return

      for containerName in staleContainers:
         ContainerConfigReactor.stopContainer( containerName )
         ContainerConfigReactor.removeContainer( containerName, force=True )
         del self.containerStatusDir.container[ containerName ]

   def getInsecureServers( self ):
      """ Return a string of quoted insecure registry names separated by commas"""
      insecureServers = [
            '"' + r.params.serverName + '"'
            for r in sorted( self.registryConfigDir.registry.values() )
            if r.params.insecure and r.params.serverName
      ]
      return ','.join( insecureServers )

   def logoutStaleRegistries( self ):
      for serverName in self.registryStatusDir.secureServer:
         serverInfo = self.registryStatusDir.secureServer[ serverName ]
         registryName = serverInfo.registryName
         registryConfig = self.registryConfigDir.registry.get( registryName )

         configRemoved = ( registryConfig is None )
         registryConfigStale = (
               ( registryConfig is not None ) and
               ( registryConfig.params != serverInfo.params ) )
         logoutNeeded = configRemoved or registryConfigStale
         if logoutNeeded:
            RegistryReactor.logout( self.registryStatusDir, serverName )

   def handleInsecureRegistries( self ):
      insecureServers = self.getInsecureServers()
      if self.registryStatusDir.insecureServers != insecureServers:
         self.registryStatusDir.insecureServers = insecureServers
         self.notifiee_.sync()

   def handleRegistryUpdate( self ):
      """The nested SM, registryNotifiee, handles logging in to secure registries.
      However we still need to log out of secure registries,
      and sync config file when insecure registries are added / removed.
      This is done better outside with a walk over the entire list rather than in the
      registry specific nested SMs.
      """
      t2( "ContainerMgr.handleRegistryUpdate called." )
      qt2( "ContainerMgr.handleRegistryUpdate called." )
      if not self.daemonStarted():
         t2( "ContainerMgr.handleRegistryUpdate: skip" )
         qt2( "ContainerMgr.handleRegistryUpdate: skip" )
         return
      self.logoutStaleRegistries()
      self.handleInsecureRegistries()

   def handleRegistryConfig( self ):
      for registryName in self.registryConfigDir.registry:
         reactor = self.registryNotifiee_.reactor( registryName )
         reactor.handleRegistryConfig()
      self.handleRegistryUpdate()

      # Remove stale registries from status
      staleRegistries = [ registryName
                          for registryName in self.registryStatusDir.registry
                          if registryName not in self.registryConfigDir.registry ]
      for registryName in staleRegistries:
         del self.registryStatusDir.registry[ registryName ]

   def cleanupContainers( self ):
      for containerName in self.containerStatusDir.container:
         reactor = self.containerNotifee_.reactor( containerName )
         reactor.cleanup()

   def cleanupRegistries( self ):
      for registryName in self.registryStatusDir.registry:
         reactor = self.registryNotifiee_.reactor( registryName )
         reactor.cleanup()

   def handleImages( self ):
      assert self.imageLoadNotifiee_ is not None
      self.imageLoadNotifiee_.handleInitialized()

   def cleanupImages( self ):
      assert self.imageLoadNotifiee_ is not None
      self.imageLoadNotifiee_.cleanup()

   def createReactors( self ):
      t0( "ContainerMgr.createReactors called." )
      qt0( "ContainerMgr.createReactors called." )
      self.notifiee_ = ContainerMgrConfigReactor( self.containerMgrConfig,
                                                  self.containerMgrSuperServerConfig,
                                                  self.containerMgrSuperServerStatus,
                                                  self.kernelRoutingOverrideStatus,
                                                  self.registryConfigDir,
                                                  self )
      self.registryNotifiee_ = Tac.collectionChangeReactor(
                  self.registryConfigDir.registry, RegistryReactor,
                  reactorArgs=( self, ) )
      self.profileDirSm = Tac.newInstance(
            "ContainerMgr::ContainerProfileDirSm",
            self.profileConfigDir,
            self.containerConfigDir,
            self.consolidatedContainerConfigDir )
      self.imageLoadNotifiee_ = ImageLoadNotifiee( self.imageLoadConfigDir,
                                                   self.imageLoadStatusDir,
                                                   self )
      self.containerNotifee_ = Tac.collectionChangeReactor(
            self.consolidatedContainerConfigDir.container,
            ContainerConfigReactor,
            reactorArgs=( self, ) )

   def applyInitialConfig( self ):
      t0( "ContainerMgr.applyInitialConfig called." )
      qt0( "ContainerMgr.applyInitialConfig called." )
      self.handleImages()
      self.handleRegistryConfig()
      self.handleContainersConfig()
      t0( "ContainerMgr.applyInitialConfig finished." )
      qt0( "ContainerMgr.applyInitialConfig finished." )

   def cleanup( self ):
      t0( "ContainerMgr.cleanup called." )
      qt0( "ContainerMgr.cleanup called." )
      self.cleanupContainers()
      self.cleanupImages()
      self.cleanupRegistries()

   def onSwitchover( self, protocol ):
      t0( "ContainerMgr.onSwitchover called." )
      qt0( "ContainerMgr.onSwitchover called." )
      self.registryStatusDir.registry.clear()
      self.registryStatusDir.secureServer.clear()
      self.registryStatusDir.insecureServers = ""
      self.containerStatusDir.container.clear()
      self.createReactors()

def Plugin( ctx ):
   ctx.registerService( ContainerMgr( ctx.entityManager ) )
