# Copyright (c) 2008-2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, SuperServer, os, Cell
import CEosHelper

class CoredumpConfigReactor( Tac.Notifiee ):

   notifierTypeName = 'System::CoredumpControlConfig'

   def __init__( self, coredumpConfig, coredumpStatus ):
      Tac.Notifiee.__init__( self, coredumpConfig )
      self.status_ = coredumpStatus
      self.config_ = coredumpConfig
      self.ssd_ = False
      if os.path.exists("/var/core") and os.path.exists("/var/log/agents"):
         dev_major_ssd = os.major(os.stat("/var/core").st_dev)
         dev_major_tmp = os.major(os.stat("/tmp").st_dev)
         if dev_major_ssd > 0 and dev_major_ssd != dev_major_tmp:
            self.ssd_ = True
      self.handleCoredump()

   @Tac.handler( 'mode' )
   def handleCoredump( self ):
      # Execute this code only on a real DUT
      # pylint: disable-next=consider-using-with
      if not "SWI" in open( "/proc/cmdline" ).read():
         return
      coredumpConfig = self.config_
      if coredumpConfig.mode == "compressed":
         Tac.run( [ '/sbin/sysctl', '-w', \
                     "kernel.core_pattern=|/sbin/core_dumper %p %t %e" ],
                     asRoot=True, stdout=Tac.DISCARD, stderr=Tac.DISCARD )
         self.status_.mode = "compressed"
      elif coredumpConfig.mode == "defercompress":
         Tac.run( [ '/sbin/sysctl', '-w', \
                  "kernel.core_pattern=/var/core/core.%p.%t.%e" ],
                   asRoot=True, stdout=Tac.DISCARD, stderr=Tac.DISCARD )
         self.status_.mode = "compressedDeferred"
      elif coredumpConfig.mode == "disabled":
         Tac.run( [ '/sbin/sysctl', '-w', \
                  "kernel.core_pattern=/dev/null" ],
                   asRoot=True, stdout=Tac.DISCARD, stderr=Tac.DISCARD )
         self.status_.mode = "disabled"

   def close( self ):
      Tac.Notifiee.close( self )

class Coredump( SuperServer.SuperServerAgent ):
   # This agent reacts to the 'sys/coredump/config' 
   # We can add newer coredump related attributes as needed in the future
   def __init__( self, entityManager ):
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      mg = entityManager.mountGroup()
      # We only would read the state here, not sure if this *has* to be
      # immediate read.
      coredumpConfig = mg.mount( 'sys/coredump/config', \
                                 'System::CoredumpControlConfig', 'ri' )
      coredumpCellStatus = mg.mount( 
         Cell.path( 'sys/coredump/status' ), \
                    'System::CoredumpControlStatus', 'wf' )

      def _finished():
         # pylint: disable-next=attribute-defined-outside-init
         self.notifiee_ = CoredumpConfigReactor( coredumpConfig,
                                                 coredumpCellStatus) 
      mg.close( _finished )

def Plugin( ctx ):
   # The host manages the core dumps in cEOS
   if not CEosHelper.isCeos():
      ctx.registerService( Coredump( ctx.entityManager ) )
