#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pkgdeps: rpm kea-dhcp
# pkgdeps: rpm kea-dhcp-hooks

from itertools import chain

import SuperServer
import Tac
import DhcpServerReactor
import json
import QuickTrace
import os
import errno
import ArPyUtils
from EosDhcpServerLib import clientClassConfigTypes
from EosDhcpServerLib import DhcpIntfStatusMessages as DISM
from EosDhcpServerLib import Dhcp6DirectRelayMessages as D6DRM
from EosDhcpServerLib import hasLinkLocalAddr
from EosDhcpServerLib import formatKeaRemoteIdHexTest
from EosDhcpServerLib import formatKeaRemoteIdStrTest
from EosDhcpServerLib import overrideLockfileDir
from EosDhcpServerLib import tmpKeaConfFile
from EosDhcpServerLib import tmpToCheckKeaConfFile
from EosDhcpServerLib import keaLeasePath
from EosDhcpServerLib import keaConfigPath
from EosDhcpServerLib import keaControlConfigPath
from EosDhcpServerLib import keaServiceName
from EosDhcpServerLib import keaPidPath
from EosDhcpServerLib import keaControlSock
from EosDhcpServerLib import runKeaDhcpCmd
from EosDhcpServerLib import configReloadCmdData
from EosDhcpServerLib import clearLeaseCmdData
from EosDhcpServerLib import KeaDhcpCommandError
from EosDhcpServerLib import defaultVendorId
from EosDhcpServerLib import getSubOptionData
from EosDhcpServerLib import vendorSubOptionType
from EosDhcpServerLib import OptionType
from EosDhcpServerLib import optionToHex
from EosDhcpServerLib import maxCheckConfigInternalRecursionLevel
from EosDhcpServerLib import incrementArnetIpAddr
from EosDhcpServerLib import decrementArnetIpAddr
from EosDhcpServerLib import filterDuplicateOptions
from EosDhcpServerLib import fqdnCodeToKeaName
from EosDhcpServerLib import optionTypeToKeaType
from EosDhcpServerLib import subOptionTypeToKeaType
from SysConstants.if_h import IFF_RUNNING
import re
import glob
from IpLibConsts import DEFAULT_VRF
import SharedMem
import Tracing
import weakref
import Arnet
from collections import defaultdict

__defaultTraceHandle__ = Tracing.Handle( "DhcpServer" )
t0 = __defaultTraceHandle__.trace0
t1 = __defaultTraceHandle__.trace1
t2 = __defaultTraceHandle__.trace2
t3 = __defaultTraceHandle__.trace3
t4 = __defaultTraceHandle__.trace4
t5 = __defaultTraceHandle__.trace5

qv = QuickTrace.Var
qt0 = QuickTrace.trace0
qt1 = QuickTrace.trace1
qt3 = QuickTrace.trace3
qt4 = QuickTrace.trace4

addrFamilyEnum = Tac.Type( "Arnet::AddressFamily" )

disabledReasonsEnum = Tac.Type( 'DhcpServer::DisabledReasonsEnum' )

globalClientClassTag = 'global'

def prependNamePrefix( prefix, name ):
   return prefix + '_' + name

def configBtest():
   return "DHCPSERVER_CONFIG_BREADTH_TEST" in os.environ

def configStest():
   return 'DHCPSERVER_CONFIG_STEST' in os.environ

def getSubnetFailureReason( reason, isPreExistingFailure=False ):
   '''
   We have several different subnet error messages that can be put into 2 categories:
   - config related errors (starts with 'subnet configuration failed'):
   we will use `configPattern` regex
   - other errors (e.g. trying to add the same subnet with different subnet-id):
   we will use `alreadyExistPattern`

   An example error message is:
   Error encountered: subnet configuration failed: a pool of type V4, with the
   following address range: 192.168.1.110-192.168.1.254 overlaps with an existing
   pool in the subnet: 192.168.1.0/24 to which it is being added.

   The match regex would be:
   match.group('pool') : 192.168.1.110-192.168.1.254
   match.group('overlap') : overlaps with an existing pool
   match.group('nomatch') : None
   match.group('configPattern') : overlaps with an existing pool
   match.group('alreadyExistPattern') : None

   @Output
      ranges, disabledReasonEnum: "192.168.1.110-192.168.1.254", overlappingRanges
   '''
   heading = '(?P<heading>address range: )'
   poolPattern = r'(?P<pool>[0-9a-f:\-\.]+) '
   overlapPattern = '(?P<overlap>overlaps with an existing pool)'
   notMatchPattern = '(?P<notmatch>does not match)'
   configPattern = '(?P<configPattern>({}|{}))'.format(
      overlapPattern, notMatchPattern )
   preExistingSubnetPattern = '(?P<preExisting> already exists)'

   pattern = heading + poolPattern + configPattern
   pattern += ( '|' + preExistingSubnetPattern ) if isPreExistingFailure else ''

   match = re.search( pattern, reason )
   if not match:
      disabledReasonEnum = disabledReasonsEnum.unknown
      return 'Unknown failure', disabledReasonEnum

   disabledReasonEnum = None
   if match.group( 'configPattern' ):
      if match.group( 'overlap' ):
         disabledReasonEnum = disabledReasonsEnum.overlappingRanges

      elif match.group( 'notmatch' ):
         disabledReasonEnum = disabledReasonsEnum.invalidRanges

      ranges = match.group( 'pool' )
      return ranges, disabledReasonEnum

   # We will never hit preExistingSubnet error
   disabledReasonEnum = disabledReasonsEnum.unknown
   return 'Unknown failure', disabledReasonEnum

class ClientClassTranscriberBase:
   def __init__( self, dhcpServerVrfService, clientClassConfig, namePrefix,
                 subnetOrPool=False ):
      t0( 'Generating config for', clientClassConfig.key )
      self.dhcpServerVrfService = dhcpServerVrfService
      self.clientClassKeaConf = {}
      self.clientClassOptionsKeaConf = {}
      self.isInactive = False
      self.keaOptionsDefConfig = []
      self.optionCodeSet = dhcpServerVrfService.optionCodeSet
      self.opt43DefaultIdConfigured = dhcpServerVrfService.opt43DefaultIdConfigured
      self.l2MatchConfigured = False
      keaMatchConfig = self.createMatchString(
            clientClassConfig.matchCriteriaModeConfig )
      # The optionsDef generated below needs to be added to the global
      # 'option-def' block and hence needs to be saved.
      keaOptionsDefConfig = self.createOptionDef( clientClassConfig )
      keaOptionsDataConfig = self.createOptionData( clientClassConfig )

      # Check if client class is inactive and if so do not add the client class,
      # including its option definitions to the kea-conf.
      if self.clientClassInactive( clientClassConfig, keaMatchConfig,
                                   keaOptionsDataConfig ):
         t0( 'Client class is inactive', clientClassConfig.key )
         self.isInactive = True
         return

      if keaMatchConfig:
         self.clientClassKeaConf[ 'test' ] = keaMatchConfig
      self.keaOptionsDefConfig = keaOptionsDefConfig
      clientClassConf = self.clientClassKeaConf
      optionsConf = self.clientClassOptionsKeaConf
      clientClassConf[ 'name' ] = prependNamePrefix( namePrefix,
                                                     clientClassConfig.key )

      if subnetOrPool:
         optionsConf[ 'name' ] = clientClassConf[ 'name' ] + '_options'
         optionsConf[ 'test' ] = "member(\'{}\')".format(
                                             clientClassConf[ 'name' ] )
         optionsConf[ 'only-if-required' ] = True
         self.addLeaseTimeBase( optionsConf, clientClassConfig.leaseTime )
         if keaOptionsDataConfig:
            optionsConf[ 'option-data' ] = keaOptionsDataConfig
      else:
         self.addLeaseTimeBase( clientClassConf, clientClassConfig.leaseTime )
         if keaOptionsDataConfig:
            clientClassConf[ 'option-data' ] = keaOptionsDataConfig

   def clientClassInactive( self, clientClassConfig, keaMatchConfig,
                            keaOptionsConfig ):
      noMatchCriteria = not keaMatchConfig
      noAssignments = not keaOptionsConfig
      if clientClassConfigTypes[ type( clientClassConfig ) ] == 'range':
         noAssignedIp = ( clientClassConfig.assignedIp ==
                          clientClassConfig.ipAddrDefault )
         noAssignments = noAssignments and noAssignedIp
      return noMatchCriteria or noAssignments

   def addLeaseTimeBase( self, keaConf, leaseTime ):
      leaseTime = int( leaseTime )
      if leaseTime:
         keaConf[ 'valid-lifetime' ] = leaseTime
         self.maybeAddPreferredLifetime( keaConf, leaseTime )

   def maybeAddPreferredLifetime( self, keaConf, leaseTime ):
      raise NotImplementedError

   def addAfSpecificOptions( self, optionsConfig ):
      raise NotImplementedError

   def addAfMatchCriteria( self, testsList, matchCriteria ):
      raise NotImplementedError

   def createL2InfoTestStr( self, l2Info ):
      raise NotImplementedError

   def clientClassConfig( self, clientClassConfig ):
      return

   def createMatchStringBase( self, matchCriteriaBase ):
      '''
      Creates the kea config test string of a client class for all match criteria,
      i.e., l2Info, hostMacAddress, vendorId, and af specifics match criteria, found
      in matchCriteriaBase.
      '''
      testsList = []
      for l2Info in matchCriteriaBase.l2Info:
         testsList.append( self.createL2InfoTestStr( l2Info ) )
      self.l2MatchConfigured = ( self.l2MatchConfigured or
                                 len( matchCriteriaBase.l2Info ) > 0 )
      self.addAfMatchCriteria( testsList, matchCriteriaBase )
      op = ' or ' if matchCriteriaBase.matchAny else ' and '
      return op.join( testsList )

   def createMatchString( self, matchCriteria ):
      '''
      Creates the kea config test string of a client class by calling
      createMatchStringBase on matchCriteria and on each subMatchCriteria.
      '''
      if not matchCriteria:
         return ''
      outerMatchCriteriaStr = self.createMatchStringBase( matchCriteria )
      matchCriteriaList = [ outerMatchCriteriaStr ] if outerMatchCriteriaStr else []
      for subMatchCriteria in matchCriteria.subMatchCriteria.values():
         nestedMatchCriteriaStr = self.createMatchStringBase( subMatchCriteria )
         matchCriteriaList.append( '(' + nestedMatchCriteriaStr + ')' )
      op = ' or ' if matchCriteria.matchAny else ' and '
      return op.join( matchCriteriaList )

   def createOptionDef( self, optionsConfig ):
      optionsDef = []
      if optionsConfig.privateOptionConfig:
         privateOptionConfig = optionsConfig.privateOptionConfig
         # add all private-option definitions
         privateOptions = chain( privateOptionConfig.stringPrivateOption.values(),
                                 privateOptionConfig.ipAddrPrivateOption.values() )
         for privateOption in privateOptions:
            if privateOption.key.code not in self.optionCodeSet:
               self.optionCodeSet.add( privateOption.key.code )
               t0( 'Adding a private-option def' )
               optionsDef.append( {
                  'name': f'privateOption{privateOption.key.code}',
                  'code': privateOption.key.code,
                  'type': 'binary'
               } )
      if optionsConfig.arbitraryOptionConfig:
         arbitraryOptionConfig = optionsConfig.arbitraryOptionConfig
         arbitraryOptions = chain(
            arbitraryOptionConfig.stringArbitraryOption.values(),
            arbitraryOptionConfig.fqdnArbitraryOption.values(),
            arbitraryOptionConfig.hexArbitraryOption.values(),
            arbitraryOptionConfig.ipAddrArbitraryOption.values() )
         uniqueOptions = filterDuplicateOptions( arbitraryOptions, optionsDef,
                                                 self.opt43DefaultIdConfigured )
         for option in uniqueOptions:
            # Add arbitrary option definition for options in private-option range
            # so kea interpret them correctly. See AID10486
            if ( ( option.key.code > 223 or option.key.code == 43 ) and
                 option.key.code not in self.optionCodeSet ):
               t0( 'Adding an arbitrary-option def',
                   option.key.code )
               self.optionCodeSet.add( option.key.code )
               optionType = 'binary'
               if option.type == OptionType.optionFqdn:
                  optionType = 'fqdn'
               optionsDef.append( {
                  'code': option.key.code,
                  'type': optionType,
                  'name': f'arbitraryOption{option.key.code}'
               } )

      return optionsDef

   def createOptionData( self, optionsConfig ):
      options = []
      if optionsConfig.privateOptionConfig:
         privateOptionConfig = optionsConfig.privateOptionConfig
         # add all client class private-option data
         privateOptions = chain( privateOptionConfig.stringPrivateOption.values(),
                                 privateOptionConfig.ipAddrPrivateOption.values() )
         for privateOption in privateOptions:
            t0( 'Adding a privateOption' )
            privateOptionData = optionToHex( optionType=privateOption.type,
                                             optionData=privateOption.data,
                                             optionDataOnly=True )
            options.append( {
               'name': f'privateOption{privateOption.key.code}',
               'code': privateOption.key.code,
               'data': privateOptionData,
               'always-send': privateOption.alwaysSend
            } )
      afSpecificOptions = self.addAfSpecificOptions( optionsConfig )
      options.extend( afSpecificOptions )
      if optionsConfig.arbitraryOptionConfig:
         arbitraryOptionConfig = optionsConfig.arbitraryOptionConfig
         arbitraryOptions = chain(
            arbitraryOptionConfig.stringArbitraryOption.values(),
            arbitraryOptionConfig.fqdnArbitraryOption.values(),
            arbitraryOptionConfig.hexArbitraryOption.values(),
            arbitraryOptionConfig.ipAddrArbitraryOption.values() )
         uniqueOptions = filterDuplicateOptions( arbitraryOptions, options,
                                                 self.opt43DefaultIdConfigured,
                                             self.dhcpServerVrfService.ipVersion_ )
         t0( ( len( arbitraryOptionConfig.stringArbitraryOption ) +
               len( arbitraryOptionConfig.fqdnArbitraryOption ) +
               len( arbitraryOptionConfig.hexArbitraryOption ) +
               len( arbitraryOptionConfig.ipAddrArbitraryOption ) ),
               'client class arbitrary options before filtering duplicates' )
         t0( len( uniqueOptions ), 'unique client class arbitrary options' )
         for arbitraryOption in uniqueOptions:
            t0( 'Adding a client class arbitrary option', arbitraryOption.key.code )
            arbitraryOptionData = optionToHex( optionType=arbitraryOption.type,
                                               optionData=arbitraryOption.data,
                                               optionDataOnly=True )
            option = {
               'code': arbitraryOption.key.code,
               'data': arbitraryOptionData,
            }
            if arbitraryOption.type != OptionType.optionFqdn:
               option.update( { "csv-format": False } )
            if arbitraryOption.alwaysSend:
               option[ 'always-send' ] = arbitraryOption.alwaysSend
            if option[ 'code' ] > 223 or option[ 'code' ] == 43:
               optionName = f'arbitraryOption{arbitraryOption.key.code}'
               option[ 'name' ] = optionName
            options.append( option )
      return options

class ClientClassTranscriberIpv4( ClientClassTranscriberBase ):
   vendorIdTestFmt = "(option[60].hex == 0x{})"
   hostMacTestFmt = "(pkt4.mac == 0x{})"
   circuitIdHexTestFmt = "(relay4[1].hex == 0x{})"
   circuitIdStrTestFmt = "(relay4[1].hex == '{}')"
   remoteIdHexTestFmt = "(relay4[2].hex == 0x{})"
   remoteIdStrTestFmt = "(relay4[2].hex == '{}')"
   l2IntfTestFmt = "(substring(relay4[1].hex,2,all) == '{}:{}')"
   switchMacTestFmt = "(substring(relay4[2].hex,2,all) == 0x{})"

   def __init__( self, dhcpServerVrfService, clientClassConfig, namePrefix,
                 subnetOrPool=False ):
      ClientClassTranscriberBase.__init__( self, dhcpServerVrfService,
                                           clientClassConfig, namePrefix,
                                           subnetOrPool )

   def createL2InfoTestStr( self, l2Info ):
      l2InfoTests = []
      if l2Info.intf and l2Info.vlanId != 0:
         intfName = str( l2Info.intf ).strip( "'" )
         l2InfoTests.append( self.l2IntfTestFmt.format( intfName, l2Info.vlanId ) )
      if l2Info.mac != '00:00:00:00:00:00':
         macHex = l2Info.mac.replace( ':', '' )
         l2InfoTests.append( self.switchMacTestFmt.format( macHex ) )
      l2InfoTestStr = ' and '.join( l2InfoTests )
      if len( l2InfoTests ) > 1:
         # If there are 2 tests here, surround them in parenthesis
         l2InfoTestStr = '(' + l2InfoTestStr + ')'
      return l2InfoTestStr

   def addAfMatchCriteria( self, testsList, matchCriteria ):
      '''
      Creates the kea config test string of a client class for each IPv4 specific
      match criteria, i.e., circuitIdRemoteHexOrSr.
      '''
      for vendorId in matchCriteria.vendorId:
         testsList.append(
            self.vendorIdTestFmt.format( vendorId.encode( "utf-8" ).hex() ) )
      for mac in matchCriteria.hostMacAddress:
         testsList.append( self.hostMacTestFmt.format( mac.replace( ':', '' ) ) )
      for circIdRemId in matchCriteria.circuitIdRemoteIdHexOrStr:
         circIdRemIdTests = []
         if circIdRemId.circuitIdHex:
            testStr = self.circuitIdHexTestFmt.format( circIdRemId.circuitIdHex )
            circIdRemIdTests.append( testStr )
         elif circIdRemId.circuitIdStr:
            testStr = self.circuitIdStrTestFmt.format( circIdRemId.circuitIdStr )
            circIdRemIdTests.append( testStr )
         if circIdRemId.remoteIdHex:
            testStr = self.remoteIdHexTestFmt.format( circIdRemId.remoteIdHex )
            circIdRemIdTests.append( testStr )
         elif circIdRemId.remoteIdStr:
            testStr = self.remoteIdStrTestFmt.format( circIdRemId.remoteIdStr )
            circIdRemIdTests.append( testStr )
         circIdRemIdTestStr = ' and '.join( circIdRemIdTests )
         if len( circIdRemIdTests ) > 1:
            # If there are 2 tests here, surround them in parenthesis
            circIdRemIdTestStr = '(' + circIdRemIdTestStr + ')'
         testsList.append( circIdRemIdTestStr )

   def maybeAddPreferredLifetime( self, keaConf, leaseTime ):
      pass

   def addAfSpecificOptions( self, optionsConfig ):
      options = []

      # Options 3 (routers/default-gateway), 6 (domain-name-servers) and
      # 15 (domain-name) will always be returned (i.e. even if they aren't
      # explicitly requested by the client)
      if optionsConfig.defaultGateway != '0.0.0.0':
         options.append( {
            'name': 'routers',
            'data': optionsConfig.defaultGateway,
            'always-send': True } )
      if optionsConfig.dnsServers:
         options.append( {
            'name': 'domain-name-servers',
            'data': ', '.join( optionsConfig.dnsServers.values() ),
            'always-send': True } )
      if optionsConfig.domainName:
         options.append( {
            'name': 'domain-name',
            'data': optionsConfig.domainName,
            'always-send': True } )
      if optionsConfig.tftpBootFileName:
         bootFileName = optionsConfig.tftpBootFileName.replace( ",", r"\," )
         options.append( {
            'name': 'boot-file-name',
            'data': bootFileName } )
      if optionsConfig.tftpServerOption66:
         options.append( {
            'name': 'tftp-server-name',
            'data': optionsConfig.tftpServerOption66 } )
      if optionsConfig.tftpServerOption150:
         options.append( {
            'name': 'tftp-server-address',
            'data': ','.join( optionsConfig.tftpServerOption150.values() ) } )
      return options

class ClientClassTranscriberIpv6( ClientClassTranscriberBase ):
   # Ethernet starts at byte #20 in the remote-ID. Add 4 because of the enterprise
   # ID.
   l2IntfTestFmt = "(substring(option[37].hex,24,all) == '{}:{}')"
   # Layer 2 information may come from a relay agent. "relay6[-1]" uses the relay
   # packet closest to the client.
   relayL2IntfTestFmt = "(substring(relay6[-1].option[37].hex,24,all) == '{}:{}')"
   # MAC address is found in bytes 4-10 in the remote-ID. Add 4 because of the
   # enterprise ID.
   switchMacTestFmt = "(substring(option[37].hex,8,6) == 0x{})"
   relaySwitchMacTestFmt = "(substring(relay6[-1].option[37].hex,8,6) == 0x{})"
   # Verify that the enterprise ID is Arista's enterprise ID
   entIdTestStr = '(substring(option[37].hex,0,4) == 0x00007571)'
   relayEntIdTestStr = '(substring(relay6[-1].option[37].hex,0,4) == 0x00007571)'

   andFmt = "({} and {})"
   orFmt = "({} or {})"

   vendorClassTestFmt = "(vendor-class[{}].data[{}] == 0x{})"
   duidTestFmt = "option[1].hex == 0x{}"
   hostMacTestFmt = "(member('KNOWN') and member('mac_{}'))"

   def __init__( self, dhcpServerVrfService, clientClassConfig, namePrefix,
                 subnetOrPool=False ):
      ClientClassTranscriberBase.__init__( self, dhcpServerVrfService,
                                           clientClassConfig, namePrefix,
                                           subnetOrPool )

   def createL2InfoTestStr( self, l2Info ):
      '''
      Option 37 is split into two sections, an enterprise ID and a remote-ID. The
      enterprise ID takes up 4 bytes. For Arista switches, the remote-id is currently
      encoded as:
      | ----------------------------------- |
      | 0               8               15  |
      |    duid type       hardware type    |
      | 16              24              31  |
      |     MAC byte 1      MAC byte 2      |
      | 32              40              47  |
      |     MAC byte 3      MAC byte 4      |
      | 48              56              63  |
      |     MAC byte 5      MAC byte 6      |
      | 64              72              79  |
      |      00000000        00000000       |
      | 80              88              95  |
      |      00000000        00000000       |
      | 96              104             111 |
      |      00000000        00000000       |
      | 112             120             127 |
      |      00000000        00000000       |
      | 128             136             143 |
      |            EthernetX:Y              |
      |                ...                  |
      | ----------------------------------- |
      '''
      l2InfoTests = []
      l2InfoTestsRelay = []
      if l2Info.intf and l2Info.vlanId != 0:
         intfName = str( l2Info.intf ).strip( "'" )
         l2InfoTests.append( self.l2IntfTestFmt.format( intfName, l2Info.vlanId ) )
         l2InfoTestsRelay.append( self.relayL2IntfTestFmt.format( intfName,
                                                                  l2Info.vlanId ) )
      if l2Info.mac != '00:00:00:00:00:00':
         macHex = l2Info.mac.replace( ':', '' )
         l2InfoTests.append( self.switchMacTestFmt.format( macHex ) )
         l2InfoTestsRelay.append( self.relaySwitchMacTestFmt.format( macHex ) )
      l2InfoTestStr = ' and '.join( l2InfoTests )
      l2InfoTestStrRelay = ' and '.join( l2InfoTestsRelay )
      if len( l2InfoTests ) > 1:
         # If there are 2 tests here, surround them in parenthesis
         l2InfoTestStr = '(' + l2InfoTestStr + ')'
         l2InfoTestStrRelay = '(' + l2InfoTestStrRelay + ')'
      l2InfoTestStr = self.andFmt.format( self.entIdTestStr, l2InfoTestStr )
      l2InfoTestStrRelay = self.andFmt.format( self.relayEntIdTestStr,
                                               l2InfoTestStrRelay )
      return self.orFmt.format( l2InfoTestStr, l2InfoTestStrRelay )

   def addAfMatchCriteria( self, testsList, matchCriteria ):
      '''
      Creates the kea config test string of a client class for each IPv6 specific
      match criteria.
      '''
      for vendorClass in matchCriteria.vendorClass:
         vendorClassDataList = []
         entId = '*' if vendorClass.anyEnterpriseId else vendorClass.enterpriseId
         # Use enumerate here because index of Sysdb VendorClass.data collection
         # starts at 1 when we want to start at 0
         for i, data in enumerate( vendorClass.data.values() ):
            vendorClassDataList.append( self.vendorClassTestFmt.format(
               entId, i, data.encode( "utf-8" ).hex() ) )
         if len( vendorClassDataList ) > 1:
            vendorClassTest = '(' + ' and '.join( vendorClassDataList ) + ')'
         else:
            vendorClassTest = vendorClassDataList[ 0 ]
         testsList.append( vendorClassTest )
      for mac in matchCriteria.hostMacAddress:
         testsList.append( self.hostMacTestFmt.format( mac ) )
         macClientClass = { 'name': f'mac_{mac}' }
         self.dhcpServerVrfService.macClientClassesDict[ mac ] = macClientClass
      for duid in matchCriteria.duid:
         testsList.append( self.duidTestFmt.format( duid ) )
      for remId in matchCriteria.remoteIdHexOrStr:
         if remId.hex:
            remIdTestStr = formatKeaRemoteIdHexTest( remId.hex )
         elif remId.str:
            remIdTestStr = formatKeaRemoteIdStrTest( remId.str )
         else:
            # This case should not be possible. remoteIdHexOrStr should contain
            # either a hex or str value.
            continue
         testsList.append( remIdTestStr )

   def maybeAddPreferredLifetime( self, keaConf, leaseTime ):
      keaConf[ 'preferred-lifetime' ] = leaseTime

   def addAfSpecificOptions( self, optionsConfig ):
      options = []
      if optionsConfig.dnsServers:
         dnsServers = [ dnsServer.stringValue
                        for dnsServer in optionsConfig.dnsServers.values() ]
         options.append( {
            'name': 'dns-servers',
            'data': ', '.join( dnsServers ) } )
      if optionsConfig.domainName:
         options.append( {
            'name': 'domain-search',
            'data': optionsConfig.domainName } )
      if optionsConfig.tftpBootFileName:
         bootFileName = optionsConfig.tftpBootFileName.replace( ",", r"\," )
         options.append( {
            'name': 'bootfile-url',
            'data': bootFileName } )
      return options

# reservations mac-address
def reservationsMacAddrFailureReason( reason ):
   '''
   Example Error Message:
   Error encountered: failed to add new host using the HW address
   'd6:b4:20:d8:78:94 and DUID '(null)' to the IPv4 subnet id '123' for the address
   1.0.0.12: There's already a reservation for this address

   @Output
     duplicateReservedIp (str), disabledReasonEnum : "1.0.0.12", duplicateReservedIp
   '''
   # match "1.0.0.12"
   ipv4AddrPattern = r'(?P<ipv4Addr>\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3})'
   errorPattern = f'for the address {ipv4AddrPattern}'

   errorRe = re.compile( errorPattern )
   match = errorRe.search( reason )
   disabledReasonEnum = disabledReasonsEnum.duplicateReservedIp
   if not match:
      t0( 'Unknown failure for mac-address reservations' )
      disabledReasonEnum = disabledReasonsEnum.unknown
      return 'Unknown failure', disabledReasonEnum

   duplicateReservedIp = match.group( 'ipv4Addr' )
   return duplicateReservedIp, disabledReasonEnum

def reservationsMacAddrFailureReason6( reason ):
   '''
   Example Error Message:
   Error encountered: failed to add address reservation for host using the HW address
   '00:0a:00:0b:00:0c and DUID '(null)' to the IPv6 subnet id '1' for address/prefix
   1::2: There's already reservation for this address/prefix

   @Output
     duplicateReservedIp (str), disabledReasonEnum : "1::2", duplicateReservedIp
   '''
   ipv6AddrPattern = r'(?P<ipv6Addr>[0-9a-fA-F:.]+)'
   errorPattern = f'for address/prefix {ipv6AddrPattern}:'

   errorRe = re.compile( errorPattern )
   match = errorRe.search( reason )
   disabledReasonEnum = disabledReasonsEnum.duplicateReservedIp
   if not match:
      t0( 'Unknown failure for mac-address reservations' )
      disabledReasonEnum = disabledReasonsEnum.unknown
      return 'Unknown failure', disabledReasonEnum

   duplicateReservedIp = match.group( 'ipv6Addr' )
   return duplicateReservedIp, disabledReasonEnum

def configErrorSubnetIdRe():
   # Example: "to the IPv4 subnet id '123'"
   # match "'123'"
   lookBehind = "(?<=subnet id )"
   subnetIdPattern = fr'{lookBehind}(?P<subnetId>\'\d+\')'
   return re.compile( subnetIdPattern )

def configErrorSubnetRe():
   lookahead = '(?= to which it is being added| already exists)'
   reason = f'(?P<subnet>[0-9a-f.:/]+){lookahead}'
   return re.compile( reason )

def kernelDeviceName( intfStatusAll, intf ):
   if intf in intfStatusAll.intfStatus:
      return intfStatusAll.intfStatus[ intf ].deviceName
   return ""

class DhcpServerVrfService( SuperServer.LinuxService ):
   notifierTypeName = '*'

   def __init__( self, config, status, vrf, intfStatus, ipVersion ):
      self.ipVersion_ = ipVersion
      self.vrf_ = vrf
      self.config_ = config
      self.status_ = status
      self.intfStatusAll_ = intfStatus
      # intfCfg_ contains the config for kea
      self.intfCfg_ = set()
      # activeInterfaces is a list of tuples ( intfName, ip addresses )
      self.activeInterfaces_ = []
      # optionCodeSet is a set containing all option codes in option-def
      self.optionCodeSet = set()
      # store "vendor-option ipv4 default" to filter arbitrary options
      self.opt43DefaultIdConfigured = None
      # subnetsWithAssignedIps is a set of all subnets with assigned IPs. This is
      # used to associate pools with client classes that will exclude the assigned
      # IPs' client classes.
      self.subnetsWithAssignedIps = set()
      # macClientClassesDict is a dictionary of MACs to empty client classes with
      # just a name, "mac" + a MAC address. Global reservations will be set up to
      # reserve these client classes for the corresponding MAC addresses and these
      # client classes will be subsequently used by other DHCPv6 client classes to
      # match on MAC addresses.
      self.macClientClassesDict = {}
      self.serviceName_ = keaServiceName.format( version=ipVersion,
                                                 vrf=vrf.value )
      self.leasePath_ = keaLeasePath.format( version=ipVersion, vrf=vrf.value )
      self.configPath_ = keaConfigPath.format( version=ipVersion, vrf=vrf.value )
      self.configCtrlPath_ = keaControlConfigPath.format( version=ipVersion,
                                                          vrf=vrf.value )
      self.pidPath_ = keaPidPath.format( version=ipVersion, vrf=vrf.value )
      self.controlSock_ = keaControlSock.format( version=ipVersion,
                                                 vrf=vrf.value )
      # Tmp file, used to save "ideal" config from the user
      self.tmpKeaConfFile_ = tmpKeaConfFile.format( version=ipVersion,
                                                    vrf=vrf.value )
      # Tmp file, used to check malformed configs if any
      self.tmpToCheckKeaConfFile_ = tmpToCheckKeaConfFile.format( version=ipVersion,
                                                                  vrf=vrf.value )
      self.hasGlobalL2MatchCriteria_ = False
      self.hasRangeL2MatchCriteria_ = False
      self.hasSubnetL2MatchCriteria_ = False
      SuperServer.LinuxService.__init__( self, self.serviceName_, 'keactrl',
                                         self.config_, self.configPath_,
                                         configFileHeaderEnabled=False )
      self.writeConfigFile( self.configCtrlPath_, self.controlConfConfig(),
                            updateInPlace=True )

   @property
   def l2MatchConfigured_( self ):
      return ( self.hasGlobalL2MatchCriteria_ or self.hasSubnetL2MatchCriteria_ or
               self.hasRangeL2MatchCriteria_ )

   def isAddrZero( self, ip ):
      raise NotImplementedError

   def controlConfConfig( self ):
      config = ''
      config += '# Location of Kea configuration files.\n'
      config += 'kea_dhcp{ver}_config_file={path}\n'.format(
         ver=self.ipVersion_, path=self.configPath_ )

      config += '# Location of Kea binaries.\n'
      config += 'dhcp4_srv=/usr/sbin/kea-dhcp4\n'
      config += 'dhcp6_srv=/usr/sbin/kea-dhcp6\n'
      config += 'dhcp_ddns_srv=/usr/sbin/kea-dhcp-ddns\n'
      config += 'ctrl_agent_srv=/usr/sbin/kea-ctrl-agent\n'
      config += 'kea_ctrl_agent_config_file=/etc/kea/kea-ctrl-agent.conf\n'
      config += 'kea_dhcp_ddns_config_file=/etc/kea/kea-dhcp-ddns.conf\n'

      if self.ipVersion_ == "4":
         config += 'dhcp4=yes\n'
         config += 'dhcp6=no\n'
         config += 'kea_dhcp6_config_file=/etc/kea/kea-dhcp6.conf\n'
      else:
         assert self.ipVersion_ == "6"
         config += 'dhcp4=no\n'
         config += 'dhcp6=yes\n'
         config += 'kea_dhcp4_config_file=/etc/kea/kea-dhcp4.conf\n'

      config += 'dhcp_ddns=no\n'
      config += 'ctrl_agent=no\n'
      config += 'kea_verbose=no\n'
      return config

   def serviceCmd( self, cmd ):
      return [ 'keactrl', cmd, '-s', f'dhcp{self.ipVersion_}',
               '-c', self.configCtrlPath_ ]

   def checkServiceCmdOutput( self, cmd, output ):
      statusStr = f"DHCPv{self.ipVersion_} server: active"
      # The calling code assumes Tac.SystemCommandError will be thrown if the process
      # isn't started.
      if output and not any( statusStr in line for line in output ):
         raise Tac.SystemCommandError( output )
      return True

   def _runKeaCtrlCmd( self, cmd ):
      traceStatement = 'Kea cmd:'
      qt0( traceStatement, qv( cmd ), qv( self.ipVersion_ ) )
      traceStatement += f' {cmd} {self.ipVersion_}'
      t3( traceStatement )
      timeout = self.serviceCmdTimeout( cmd )
      nscmd = []
      # stop cmd doesn't need to run in non-default vrf (because it doesn't need
      # to be aware of the interfaces that are in the non-default vrf). Running
      # it in the non-default vrf can also cause a crash when this is called
      # after the vrf is removed.
      if self.vrf_.value != DEFAULT_VRF and cmd != 'stop':
         nscmd = [ "ip", "netns", "exec", self.vrf_.nsName() ]
      cmd = nscmd + self.serviceCmd( cmd )
      timeout = self.serviceCmdTimeout( ' '.join( cmd ) )
      # Its a real shame that we are using the env variable to set this dir. I tried
      # changing the local status dir in the build to be /var, but that failed
      # miserably because of how kea is structured.
      env = { "KEA_LOCKFILE_DIR": overrideLockfileDir }
      try:
         os.makedirs( overrideLockfileDir )
      except OSError as e:
         if e.errno != errno.EEXIST:
            raise

      env.update( os.environ )
      oldmask = os.umask( 0 )
      Tac.run( cmd, stdout=Tac.DISCARD, stderr=Tac.DISCARD, timeout=timeout,
               asRoot=True, env=env )
      oldmask = os.umask( oldmask )

   def getPid( self ):
      try:
         with open( self.pidPath_ ) as f:
            keaPid = int( f.read() )
            t0( "kea PID:", keaPid, self.ipVersion_ )
            qt0( "kea PID:", qv( keaPid ), qv( self.ipVersion_ ) )
            return keaPid
      except OSError as e:
         if e.errno != errno.ENOENT:
            raise e
      except ValueError:
         pass
      qt0( "no current kea PID", qv( self.ipVersion_ ) )
      return -1

   def started( self ):
      pid = self.getPid()
      if pid <= 0:
         return False
      # check if pid is running
      try:
         t0( "Killing kea PID:", pid, self.ipVersion_ )
         qt0( "Killing kea PID:", qv( pid ), qv( self.ipVersion_ ) )
         os.kill( pid, 0 )
      except OSError as e:
         if e.errno == errno.ESRCH:
            return False
         elif e.errno == errno.EPERM:
            return True
         else:
            raise
      else:
         return True

   def _globalPrivateOptionsDef( self, stringPrivateOptions, ipAddrPrivateOptions ):
      optionsDef = []

      # add all private-option definitions
      for privateOption in chain( stringPrivateOptions.values(),
                                  ipAddrPrivateOptions.values() ):
         if privateOption.key.code not in self.optionCodeSet:
            t0( 'Adding a private-option def' )
            optionsDef.append( {
               'name': f'privateOption{privateOption.key.code}',
               'code': privateOption.key.code,
               'type': 'binary'
            } )
            self.optionCodeSet.add( privateOption.key.code )

      return optionsDef

   def _globalArbitraryOptionsDef( self, optionList ):
      optionsDef = []
      spaceStr = "kea-dhcp4" if self.ipVersion_ == '4' else "kea-dhcp6"
      # Add arbitrary option definition for options in private-option range
      # so kea interpret them correctly. See AID10486
      for arbitraryOption in optionList:
         if ( ( arbitraryOption.key.code > 223 or
                ( self.ipVersion_ == '4' and arbitraryOption.key.code == 43 ) ) and
              arbitraryOption.key.code not in self.optionCodeSet ):
            t0( 'Adding an arbitrary-option def' )
            optionsDef.append( {
               'code': arbitraryOption.key.code,
               'type': 'binary',
               'name': f'arbitraryOption{arbitraryOption.key.code}'
            } )
            self.optionCodeSet.add( arbitraryOption.key.code )
         elif ( ( arbitraryOption.type == OptionType.optionFqdn ) and
                arbitraryOption.key.code not in self.optionCodeSet ):
            t0( 'Adding an fqdn arbitrary-option def' )
            optionsDef.append( {
               'name': fqdnCodeToKeaName( str( arbitraryOption.key.code ) ),
               'code': arbitraryOption.key.code,
               'type': optionTypeToKeaType( arbitraryOption.type ),
               "space": spaceStr
            } )
            self.optionCodeSet.add( arbitraryOption.key.code )
      return optionsDef

   def _globalPrivateOptionsData( self, stringPrivateOptions,
                                  ipAddrPrivateOptions ):
      options = []

      # add all global private-option data
      for privateOption in chain( stringPrivateOptions.values(),
                                  ipAddrPrivateOptions.values() ):
         t0( 'Adding a privateOption' )
         privateOptionData = optionToHex( optionType=privateOption.type,
                                          optionData=privateOption.data,
                                          optionDataOnly=True )
         options.append( {
            'name': f'privateOption{privateOption.key.code}',
            'code': privateOption.key.code,
            'data': privateOptionData,
            'always-send': privateOption.alwaysSend
         } )
      return options

   def _globalArbitraryOptionsData( self, optionList ):
      options = []
      # add all global arbitraryOption data
      for arbitraryOption in optionList:
         t0( 'Adding an arbitraryOption', arbitraryOption.key.code )
         arbitraryOptionData = optionToHex( optionType=arbitraryOption.type,
                                            optionData=arbitraryOption.data,
                                            optionDataOnly=True )
         option = {
            'code': arbitraryOption.key.code,
            'data': arbitraryOptionData,
         }
         if arbitraryOption.type != OptionType.optionFqdn:
            option.update( { "csv-format": False } )
         if arbitraryOption.alwaysSend:
            option[ 'always-send' ] = arbitraryOption.alwaysSend
         if option[ 'code' ] > 223 or ( self.ipVersion_ == '4' and
                                        option[ 'code' ] == 43 ):
            option[ 'name' ] = f'arbitraryOption{arbitraryOption.key.code}'
         options.append( option )
      return options

   def conf( self ):
      '''Return the content to write to DHCP server service's kea-dhcp4.conf file'''
      t0( 'Creating kea-dhcpX.conf file content' )
      out = {}
      self.optionCodeSet = set()
      self.subnetsWithAssignedIps = set()
      self.macClientClassesDict = {}
      # If the server is disabled or unconfigured, we should
      # not attempt to create a kea config
      if ( self.config_.disabled or not self._updateDhcpServerDisabledStatus() ):
         t0( 'server is disabled' )
         out[ 'serverIsDisabled' ] = True
         return json.dumps( out, indent=4, sort_keys=True )

      self._addAfConf( out )

      # Lease file
      out[ 'lease-database' ] = {
         'type': 'memfile',
         'name': self.leasePath_
         }

      # Echo client id
      if self.ipVersion_ == '4':
         if not self.config_.echoClientIdIpv4:
            t0( 'echo-client-id is disabled' )
            out[ 'echo-client-id' ] = False

      # Hooks for lease commands
      out[ 'hooks-libraries' ] = [ {
         'library': '/usr/lib{}/kea-dhcp/hooks/libdhcp_lease_cmds.so'.format(
            '64' if ArPyUtils.arch() != 32 else "" ) }, {
         'library': '/usr/lib{}/kea-dhcp/hooks/libdhcp_bootp.so'.format(
            '64' if ArPyUtils.arch() != 32 else "" ) } ]

      out[ 'control-socket' ] = {
         'socket-type': 'unix',
         'socket-name': self.controlSock_ }

      out[ 'sanity-checks' ] = {
         'lease-checks': 'fix-del'
         }

      loggingConfig = self._loggingConfig()
      if loggingConfig:
         out[ 'loggers' ] = loggingConfig
      # Custom option definition for subnet name
      # Kea DHCP does not have an option for subnet name
      # thus we will have to create our own option data with
      # name "name". 222 is just a random unused number for type code.
      self.optionCodeSet.add( 222 )
      out[ 'option-def' ] = [ {
         'name': 'name',
         'code': 222,
         'type': 'string' } ]

      out[ 'option-data' ] = []

      # add client classes
      vendorClientClasses, defaultOptionsData = self._vendorClientClasses(
            self.config_ )
      self.opt43DefaultIdConfigured = defaultOptionsData
      globalClientClasses, globalClientClassOptionDefs = (
                                    self._globalClientClasses( self.config_ ) )

      # Process option-def after vendor options
      optionsDef = self._optionsDef()
      out[ 'option-def' ].extend( optionsDef )

      # TODO: BUG651055: Remove underscore from _inactiveSubnetClientClass and
      # output it in the CLI show command.
      ( subnetClientClasses, _inactiveSubnetClientClasses,
        subnetClientClassOptionDefs ) = self._subnetClientClasses()
      ( rangeClientClasses, inactiveRangeClientClasses,
        rangeClientClassOptionDefs ) = self._rangeClientClasses( self.config_ )
      # Note that macClientClasses need to be before rangeClientClasses and
      # globalClientClasses because it's possible for rangeClientClasses and
      # globalClientClasses to rely on macClientClasses.
      clientClasses = ( list( self.macClientClassesDict.values() ) +
                        vendorClientClasses + rangeClientClasses +
                        subnetClientClasses + globalClientClasses )
      clientClassOptionDefs = chain( globalClientClassOptionDefs,
                                     subnetClientClassOptionDefs,
                                     rangeClientClassOptionDefs )
      for optionDefs in clientClassOptionDefs:
         out[ 'option-def' ].extend( optionDefs )

      if clientClasses:
         out[ 'client-classes' ] = clientClasses

      if self.macClientClassesDict:
         out[ 'reservations' ] = self._globalMacClientClassReservationsConfig()
         out[ 'reservation-mode' ] = 'global'

      # add default vendorId option-data
      if defaultOptionsData:
         out[ 'option-data' ].extend( defaultOptionsData )

      lifetime = self._leaseTime()
      if lifetime:
         out[ 'valid-lifetime' ] = int( lifetime )
         if self.ipVersion_ == '6':
            # IPv6 DHCP requires preferred-lifetime to be no less than valid-lifetime
            out[ 'preferred-lifetime' ] = int( lifetime )

      intfConfig = list( self.intfCfg_ )
      if intfConfig:
         out[ 'interfaces-config' ] = { 'interfaces': intfConfig }

      subnetConfig, hostReservationIDs = self._subnetConfig(
                                                inactiveRangeClientClasses,
                                                _inactiveSubnetClientClasses )
      if subnetConfig:
         out[ f'subnet{self.ipVersion_}' ] = subnetConfig

      if hostReservationIDs:
         out[ 'host-reservation-identifiers' ] = hostReservationIDs

      optionsData = self._globalOptionsData( self.config_ )
      if optionsData:
         out[ 'option-data' ].extend( optionsData )

      self._modifyDataDirectory( out )

      allConfig = { f'Dhcp{self.ipVersion_}': out }

      if self.ipVersion_ == '4':
         self.status_.dhcpSnoopingV4Runnable = self.l2MatchConfigured_
      else:
         self.status_.dhcpSnoopingV6Runnable = self.l2MatchConfigured_

      # need to clear disabled messages/reasons since it will be updated by
      # checkConfig
      self._resetStaleDisabledMessages()
      self.checkConfig( allConfig )
      return json.dumps( allConfig, indent=4, sort_keys=True )

   def checkConfig( self, allConfig, level=0 ):
      # since tmpToCheckKeaConfFile will keep removing invalid subnets recursively
      # until we reach a "valid" config. at the end it will
      # be the same as the KeaConfFile, therefore save a copy beforehand
      # for troubleshooting (since this should match the dhcpserver::config)
      t0( 'Current checkConfig level:', level )
      numLines = 0
      if level == 0:
         with open( self.tmpKeaConfFile_, "w" ) as tmpFile:
            tmpFile.write( json.dumps( allConfig, indent=4 ) )

      # save working config
      with open( self.tmpToCheckKeaConfFile_, "w" ) as tmpFile:
         tmpFile.write( json.dumps( allConfig, indent=4 ) )

      if level > maxCheckConfigInternalRecursionLevel:
         t0( 'Avoiding infinite recursion. Reached maximum recursion of:', level )
         os.remove( self.tmpToCheckKeaConfFile_ )
         # removing invalid config
         allConfig.clear()
         allConfig[ 'serverIsDisabled' ] = True
         if self.ipVersion_ == "4":
            self.status_.ipv4ServerDisabled = 'Server is disabled'
         else:
            self.status_.ipv6ServerDisabled = 'Server is disabled'

      else:
         nscmd = ""
         if self.vrf_.nsName() != DEFAULT_VRF:
            nscmd = f"ip netns exec {self.vrf_.nsName()} "
         cmd = '{}kea-dhcp{} -t {}'.format(
            nscmd, self.ipVersion_, self.tmpToCheckKeaConfFile_ ).split( ' ' )

         # Count number of lines
         numLines = 0
         with open( self.tmpToCheckKeaConfFile_ ) as tmpFile:
            numLines = len( tmpFile.readlines() )
         # The motivation here is to scale the timeout based on the size of
         # config file. The timeout increases linearly (with a factor of 0.05 which
         # appears to be the average time keactrl takes to parse one line in the
         # file) with the number of lines in the file. It also decreases
         # exponentially based on the level of recursion as an optimization.
         # We also add a small floor of 100 seconds to timeout here so that we have
         # enough time to process very small Kea config files
         timeout = 100 + numLines * 0.05 * 1 / pow( 2, level )
         # Increase the timeout value for Abuild and stest
         if os.getenv( 'ABUILD' ) or configStest():
            timeout = 600
         try:
            output = Tac.run( cmd, stdout=Tac.DISCARD, stderr=Tac.CAPTURE,
                              timeout=int( timeout ), asRoot=True )
         except Tac.SystemCommandError as e:
            output = e.output
         self._checkConfigInternal( allConfig, output, level=level )

   def startService( self ):
      if configBtest():
         return

      qt0( 'start service' )
      if not self.serviceEnabled():
         self.stopService()
         return

      try:
         os.makedirs( '/var/lib/run/kea' )
      except OSError as err:
         if err.errno != errno.EEXIST:
            raise

      if self.started():
         # startService is called from _maybeRestartService but there's
         # no change in the config file. If there's already a process running
         # we just reload the config.
         qt0( 'reload config' )
         cmdData = configReloadCmdData()
         try:
            runKeaDhcpCmd( self.ipVersion_, self.vrf_.value, cmdData )
            return
         except KeaDhcpCommandError:
            # Happens after reboot when some interfaces are not up/present yet
            # or when socket is not present.
            qt0( 'Kea command error running:', qv( cmdData ), qv( self.ipVersion_ ) )
            return
         except OSError as e:
            qt0( 'socket error:', qv( os.strerror( e.errno ) ) )
         # call sync() to reschedule _maybeRestartService()
         t0( 'Start Sync' )
         self.sync()
      else:
         self._runKeaCtrlCmd( "start" )

   def stopService( self ):
      if configBtest():
         return
      pid = self.getPid()
      if pid <= 0:
         return
      qt0( 'stop service kea PID:', qv( pid ), qv( self.ipVersion_ ) )

      # wait for pid to be deleted
      try:
         self._runKeaCtrlCmd( "stop" )
         timeout = self.serviceCmdTimeout( "stop" )
         Tac.waitFor( lambda: not self.started(), description='kea process to stop',
                      timeout=timeout,
                      sleep=True )
      except ( Tac.SystemCommandError, Tac.Timeout ):
         qt0( 'Error/Timeout when trying to stop kea service' )
         return

      # delete unix domain socket
      try:
         os.unlink( self.controlSock_ )
      except OSError:
         qt0( 'failed to unlink socket file' )

   def restartService( self ):
      qt0( 'restart service' )

      # Start the service if no kea process is running
      if not self.started():
         self.startService()
         return

      cmdData = configReloadCmdData()
      try:
         qt0( 'reload config' )
         runKeaDhcpCmd( self.ipVersion_, self.vrf_.value, cmdData )
         return
      except KeaDhcpCommandError:
         # Happens during ConfigReplace when some interfaces are not
         # up/present/ready yet.
         # Also can happen when trying to do a reload and some interface flaps.
         qt0( 'Kea command error running:', qv( cmdData ), qv( self.ipVersion_ ) )
         return
      except OSError as e:
         qt0( 'socket error:', qv( os.strerror( e.errno ) ) )
      # call sync() to reschedule _maybeRestartService()
      t0( 'Restart Sync' )
      self.sync()

   def verboseLoggingPath( self ):
      if "DHCPSERVER_VERBOSE_LOGGING" in os.environ:
         return f"/tmp/kea-{self.vrf_.nsName()}-debug.log"
      if self.config_.debugLogPath:
         return self.config_.debugLogPath
      return None

   def _loggingConfig( self ):
      logPath = self.verboseLoggingPath()
      if logPath:
         return [
               { "name": f"kea-dhcp{self.ipVersion_}",
                 "severity": "DEBUG",
                 "debuglevel": 99,
                 "output_options": [
                    {
                       "output": logPath
                    }
                 ],
               }
            ]
      return [
            { "name": f"kea-dhcp{self.ipVersion_}",
              "severity": "WARN",
              "output_options": [
                 {
                    "output": "syslog",
                 }
              ]
            }
         ]

   def _addPoolsForReservedIps( self, pools, reservedIps, assignedIpToClientClass,
                                rangeStr, otherRequiredClientClasses ):
      for ip in reservedIps:
         clientClass = assignedIpToClientClass[ ip ]
         reservedPool = {}
         reservedPool[ 'pool' ] = f'{str( ip )}-{str( ip )}'
         reservedPool[ 'client-class' ] = prependNamePrefix( rangeStr,
                                                             clientClass.key )
         requiredClientClassName = prependNamePrefix( rangeStr,
                                                      clientClass.key + '_options' )
         reqList = sorted( otherRequiredClientClasses + [ requiredClientClassName ] )
         reservedPool[ 'require-client-classes' ] = reqList
         t3( 'addPoolsForReservedIps: adding', reservedPool )
         qt3( 'addPoolsForReservedIps: adding', reservedPool )
         pools.append( reservedPool )

   def _addPoolsExcludingIps( self, pools, pool, subnetRange, excludedIps ):
      '''
      @params
         pools - List of pools to add to
         pool - Dictionary with options representing a pool. Any pool that is added
                to "pools" by this function will be initially created as a copy of
                "pool".
         subnetRange - Tac Range
         excludedIps - IPs to exclude from the pool
      @description
         This function creates pools of IPs that explicitly do not include any pool
         in excludedIps, as those IPs have been reserved and should not be added to
         the general pool.
      '''
      # Remove duplicates to simplify the logic of creating ranges excluding these
      # IPs
      excludedIps = set( excludedIps )
      # Convert IPs to Arnet.Ip[6]Addr to sort them by IP value comparisons
      # instead of by string comparisons.
      v4 = self.ipVersion_ == '4'
      arnetIps = ( [ Arnet.IpAddr( i ) for i in excludedIps ] if v4 else
                   excludedIps )
      excludedIps = sorted( arnetIps )
      ArnetIpAddr = Arnet.IpAddr if v4 else Arnet.Ip6Addr
      curStart = ArnetIpAddr( subnetRange.start )
      end = ArnetIpAddr( subnetRange.end )
      t3( 'addPoolsExcludingIps: start:', curStart, ', end:', end )
      qt3( 'addPoolsExcludingIps: start:', curStart, ', end:', end )
      for ip in excludedIps:
         if ip == curStart:
            curStart = incrementArnetIpAddr( ip, v4=v4 )
            continue
         prevIp = decrementArnetIpAddr( ip, v4=v4 )
         rangeStr = f'{str( curStart )}-{str( prevIp )}'
         curPool = pool.copy()
         curPool[ 'pool' ] = rangeStr
         qt3( 'addPoolsExcludingIps: adding', curPool )
         pools.append( curPool )
         curStart = incrementArnetIpAddr( ip, v4=v4 )
      if curStart <= end:
         finalPool = pool.copy()
         rangeStr = f'{str( curStart )}-{str( end )}'
         finalPool[ 'pool' ] = rangeStr
         t3( 'addPoolsExcludingIps: adding', finalPool )
         qt3( 'addPoolsExcludingIps: adding', finalPool )
         pools.append( finalPool )

   # Subnet
   def _populateSubnetConfig( self, keaSubnetId, subnetId,
                             inactiveSubnetClientClasses,
                             inactiveRangeClientClasses, hostReservationIDs,
                             ipVersion='4' ):
      subnetConfig = {}
      subnet = self._getSubnetConfig( subnetId )
      subnetConfig[ 'subnet' ] = str( subnet.subnetId )
      subnetConfig[ 'id' ] = keaSubnetId
      subnetConfig[ 'pools' ] = self._rangeConfig(
                                    subnet, inactiveRangeClientClasses[ subnetId ] )

      subnetClientClassValues = list( subnet.clientClassConfig.values() )
      requireClientClasses = []
      self._filterSubnetClientClasses( str( subnet.subnetId ),
                                       subnetClientClassValues,
                                       requireClientClasses,
                                       inactiveSubnetClientClasses[
                                             subnet.subnetId ] )
      if requireClientClasses:
         subnetConfig[ 'require-client-classes' ] = requireClientClasses

      # valid-lifetime
      if subnet.leaseTime:
         subnetConfig[ 'valid-lifetime' ] = int( subnet.leaseTime )
         # perferred-lifetime for ipv6
         if ipVersion == '6':
            subnetConfig[ 'preferred-lifetime' ] = int( subnet.leaseTime )

      # options data
      options = self._subnetOptionsData( subnet )
      if options:
         subnetConfig[ 'option-data' ] = options

      # host reservations mac address
      if subnet.reservationsMacAddr:
         self._reservationsConfig( subnet, subnetConfig, hostReservationIDs )

      return subnetConfig

   def _filterSubnetClientClasses( self, subnetId, subnetClientClassValues,
                                 requireClientClasses,
                                 inactiveClientClasses ):
      '''
      Filters out all client classes in this subnet that are inactive, e.g., no
      match criteria or no assignments is configured.

      If a client class is assigned options, two classes are generated. The first
      class only contains the 'test' and the second option class will contain the
      assigned options and will be set to 'only-if-required' in order to limit the
      scope of the options to only the subnet the client class was defined in. All
      client classes in requireClientClasses will then be added to
      'require-client-class' of subnet.
      '''
      for clientClass in subnetClientClassValues:
         if prependNamePrefix( subnetId, clientClass.key ) in inactiveClientClasses:
            continue
         keaName = prependNamePrefix( subnetId, clientClass.key + '_options' )
         requireClientClasses.append( keaName )

   def _filterRangeClientClasses( self, rangeStr, rangeClientClassValues,
                                 requireClientClasses, excludedIps,
                                 assignedIpToClientClass,
                                 inactiveClientClasses ):
      '''
      Filters out all client classes in this range that have an assigned IP and adds
      the assigned IPs to excludedIps and assignedIpToClientClass.

      If a client class is assigned options, two classes are generated. The first
      class only contains the 'test' and the second option class will contain the
      assigned options and will be set to 'only-if-required' in order to limit the
      scope of the options to only the range the client class was defined in. The
      function also adds the corresponding options class of all client classes
      without an assigned IP to requireClientClasses. All client classes in
      requireClientClasses will then be added to 'require-client-class' of range.
      '''
      for clientClass in rangeClientClassValues:
         if prependNamePrefix( rangeStr, clientClass.key ) in inactiveClientClasses:
            continue
         if clientClass.assignedIp == clientClass.ipAddrDefault:
            keaName = prependNamePrefix( rangeStr, clientClass.key + '_options' )
            requireClientClasses.append( keaName )
         else:
            excludedIps.append( clientClass.assignedIp )
            assignedIpToClientClass[ clientClass.assignedIp ] = clientClass

   def _rangeConfig( self, subnet, inactiveRangeClientClasses ):
      pools = []
      if subnet.rangeConfig:
         for rangeConfig in subnet.rangeConfig.values():
            _range = rangeConfig.range
            rangeStr = f"{_range.start}-{_range.end}"
            pool = {}
            excludedIps = []
            assignedIpToClientClass = {}
            requireClientClasses = []
            inactiveClientClasses = inactiveRangeClientClasses[ rangeStr ]

            if rangeConfig.clientClassConfig:
               rangeClientClassValues = list(
                     rangeConfig.clientClassConfig.values() )
               self._filterRangeClientClasses( rangeStr, rangeClientClassValues,
                                               requireClientClasses, excludedIps,
                                               assignedIpToClientClass,
                                               inactiveClientClasses )

            if subnet.subnetId in self.subnetsWithAssignedIps:
               poolClientClass = rangeStr + "_pool_client_class"
               pool[ 'client-class' ] = poolClientClass

            if requireClientClasses:
               pool[ 'require-client-classes' ] = requireClientClasses

            # In order to reserve IP addresses by client classes, we remove the
            # reserved IPs from the main pool of IPs and create a pools of 1 IP
            # associated with their respective client classes for reserved IPs.

            # Create pools for reserved IPs
            self._addPoolsForReservedIps( pools, excludedIps,
                                          assignedIpToClientClass, rangeStr,
                                          requireClientClasses )

            # Create pools for IPs that are not reserved
            self._addPoolsExcludingIps( pools, pool, _range, excludedIps )
      return pools

   def _leaseTime( self ):
      raise NotImplementedError()

   def reservationsIpStr( self ):
      raise NotImplementedError

   def _getSubnetConfig( self, subnetId ):
      raise NotImplementedError()

   def _subnetConfig( self, inactiveRangeClientClasses,
                      inactiveSubnetClientClasses ):
      raise NotImplementedError()

   def _reservationsConfig( self, subnet, subnetConfig, hostReservationIDs ):
      t4( 'reservations: ', len( subnet.reservationsMacAddr ) )
      reservations = []
      for hostReservation in subnet.reservationsMacAddr.values():
         # a valid reservation must have at least 1 attribute configured
         reservation = { 'hw-address': hostReservation.hostId }
         ipAddr = hostReservation.ipAddr
         hostname = hostReservation.hostname
         if not self.isAddrZero( ipAddr ):
            ipAddrStr = str( ipAddr )
            # In IPv6, kea supports assigning multiple IPs in the same reservation.
            keaIp = ipAddrStr if self.ipVersion_ == '4' else [ ipAddrStr ]
            t4( 'reserving ipAddr', str( ipAddr ), 'for', hostReservation.hostId )
            qt4( 'reserving ipAddr', str( ipAddr ), 'for', hostReservation.hostId )
            reservation[ self.reservationsIpStr() ] = keaIp
         if hostname:
            t4( 'reserving hostname', hostname, 'for', hostReservation.hostId )
            qt4( 'reserving hostname', hostname, 'for', hostReservation.hostId )
            reservation[ 'hostname' ] = hostname
         if not self.isAddrZero( ipAddr ) or hostname:
            reservations.append( reservation )

      if reservations:
         hostReservationIDs.add( "hw-address" )
         subnetConfig[ 'reservations' ] = reservations
         # For DHCPv6, if client class matching by MAC address is configured, we will
         # use a global reservation to assign the MAC address to an internal client
         # class to match on. To do this, we need to set "reservation-mode" to
         # "global" in the global scope which disable subnet reservations. Setting
         # "reservation-mode" to "all" here overrides that.
         #
         # Note that for DHCPv4, this is currently not needed because we do not
         # currently use global reservations in DHCPv4, but we set this flag here
         # anyway for possible future compatibility.
         subnetConfig[ 'reservation-mode' ] = 'all'

   def _vendorOptionsDef( self, config ):
      return []

   def _vendorClientClasses( self, config ):
      return [], []

   def _globalClientClasses( self, config ):
      return []

   def _globalMacClientClassReservationsConfig( self ):
      t0( f'IPv{self.ipVersion_} MAC Client Classes ',
          len( self.macClientClassesDict ) )
      qt3( f'IPv{self.ipVersion_} MAC Client Classes ',
           len( self.macClientClassesDict ) )
      reservationsList = []
      for mac, macDict in self.macClientClassesDict.items():
         reservationDict = {}
         reservationDict[ "hw-address" ] = mac
         reservationDict[ "client-classes" ] = [ f'mac_{mac}' ]
         reservationDict[ "client-classes" ] = [ macDict[ "name" ] ]
         reservationsList.append( reservationDict )
      return reservationsList

   def _processRangeOrSubnetClientClasses( self, config, namePrefix,
                                           ClientClassTranscriberType,
                                           extraProcessingFunc,
                                           extraFuncArgs=None ):
      '''
      Params:
      - config: A SubnetConfig or RangeConfig that will be used to look up
           clientClassConfig to be iterated over.
      - namePrefix: Prefix to be added to the client class name in kea
      - ClientClassTranscriberType: Either ClientClassTranscriberIpv4 or
           ClientClassTranscriberTypeIpv6
      - extraProcessingFunc: Function that performs subnet or range specific actions
           for the client class.

      Returns:
      ( list of client classes in kea config syntax,
        list of inactive client class names,
        list of client class option defs )
      '''
      clientClassList = []
      inactiveClientClassList = []
      clientClassDefs = []
      for clientClassConfig in config.clientClassConfig.values():
         transcriber = ClientClassTranscriberType( self, clientClassConfig,
                                                   namePrefix,
                                                   subnetOrPool=True )
         keaName = prependNamePrefix( namePrefix, clientClassConfig.key )
         if transcriber.isInactive:
            inactiveClientClassList.append( keaName )
            continue
         clientClassList.append( transcriber.clientClassKeaConf )
         clientClassList.append( transcriber.clientClassOptionsKeaConf )
         clientClassDefs.append( transcriber.keaOptionsDefConfig )
         extraProcessingFunc( clientClassConfig, transcriber, keaName,
                              extraFuncArgs )
      return clientClassList, inactiveClientClassList, clientClassDefs

   def _processSubnetClientClass( self, _clientClassConfig, transcriber, _keaName,
                                  _args ):
      self.hasSubnetL2MatchCriteria_ = ( self.hasSubnetL2MatchCriteria_ or
                                        transcriber.l2MatchConfigured )

   def _subnetClientClasses( self ):
      '''
      Loops through all subnet client classes and returns a list of active client
      classes and a list of inactive client classes.
      '''
      config = self.config_
      t0( f'IPv{self.ipVersion_} Subnet Client Classes' )
      qt3( f'IPv{self.ipVersion_} Subnet Client Classes' )
      if self.ipVersion_ == '4':
         subnetConfigDir = config.subnetConfigIpv4
         ClientClassTranscriberType = ClientClassTranscriberIpv4
      else:
         subnetConfigDir = config.subnetConfigIpv6
         ClientClassTranscriberType = ClientClassTranscriberIpv6
      subnetClientClassList = []
      subnetClientClassDefs = []
      allInactiveClientClasses = {}
      self.hasSubnetL2MatchCriteria_ = False
      for subnet, subnetConfig in subnetConfigDir.items():
         subnetStr = str( subnet )
         ( perSubnetClientClassList, perSubnetInactiveClientClasses,
           perSubnetClientClassDefs ) = self._processRangeOrSubnetClientClasses(
                                                     subnetConfig, subnetStr,
                                                     ClientClassTranscriberType,
                                                     self._processSubnetClientClass )
         allInactiveClientClasses[ subnet ] = perSubnetInactiveClientClasses
         subnetClientClassList.extend( perSubnetClientClassList )
         subnetClientClassDefs.extend( perSubnetClientClassDefs )
      return subnetClientClassList, allInactiveClientClasses, subnetClientClassDefs

   def _processRangeClientClass( self, clientClassConfig, transcriber, keaName,
                                 excludedClasses ):
      if clientClassConfig.assignedIp != clientClassConfig.ipAddrDefault:
         excludedClasses.append( keaName )
      self.hasRangeL2MatchCriteria_ = ( self.hasRangeL2MatchCriteria_ or
                                        transcriber.l2MatchConfigured )

   def _rangeClientClasses( self, config ):
      '''
      Loops through all range client classes and returns a list of active client
      classes and a list of inactive client classes.
      '''
      t0( f'IPv{self.ipVersion_} Range Client Classes' )
      qt3( f'IPv{self.ipVersion_} Range Client Classes' )
      if self.ipVersion_ == '4':
         subnetConfigDir = config.subnetConfigIpv4
         ClientClassTranscriberType = ClientClassTranscriberIpv4
      else:
         subnetConfigDir = config.subnetConfigIpv6
         ClientClassTranscriberType = ClientClassTranscriberIpv6
      rangeClientClassList = []
      rangeClientClassDefs = []
      allInactiveClientClasses = {}
      self.hasRangeL2MatchCriteria_ = False
      for subnet, subnetConfig in subnetConfigDir.items():
         excludedClasses = []
         perSubnetInactiveRangeClientClasses = {}
         for rangeConfig in subnetConfig.rangeConfig.values():
            rangeStr = "{}-{}".format( rangeConfig.range.start,
                                       rangeConfig.range.end )
            ( perRangeClientClassList, perRangeInactiveClientClasses,
              perRangeClientClassDefs ) = self._processRangeOrSubnetClientClasses(
                                                      rangeConfig, rangeStr,
                                                      ClientClassTranscriberType,
                                                      self._processRangeClientClass,
                                                      extraFuncArgs=excludedClasses )
            rangeClientClassList.extend( perRangeClientClassList )
            rangeClientClassDefs.extend( perRangeClientClassDefs )
            perSubnetInactiveRangeClientClasses[ rangeStr ] = (
                                       perRangeInactiveClientClasses )

         if excludedClasses:
            # If there is at least one assigned IP, create client classes for every
            # range in the subnet that will exclude client classes associated with
            # the assigned IP.
            for rangeKey in subnetConfig.rangeConfig:
               rangeStr = "{}-{}".format( rangeKey.start,
                                          rangeKey.end )
               rangeClientClassList.append(
                     self._poolOrSubnetClientClass(
                        rangeStr, pool=True,
                        excludedClasses=excludedClasses ) )
            self.subnetsWithAssignedIps.add( subnet )
         allInactiveClientClasses[ subnetConfig.subnetId ] = (
               perSubnetInactiveRangeClientClasses )
      return rangeClientClassList, allInactiveClientClasses, rangeClientClassDefs

   def _optionsDef( self ):
      return []

   def globalPrivateOptions( self ):
      raise NotImplementedError

   def globalArbitraryOptions( self ):
      raise NotImplementedError

   def _globalOptionsData( self, config ):
      raise NotImplementedError()

   def _subnetOptionsData( self, config ):
      raise NotImplementedError()

   def _resetStaleDisabledMessages( self ):
      raise NotImplementedError()

   def _checkOverlappingSubnet( self, out ):
      raise NotImplementedError()

   def addDisabledReason( self, subnetPrefix, disabledReasonEnum, reason,
                          disabledMsg ):
      raise NotImplementedError()

   def configErrorRe( self ):
      raise NotImplementedError()

   def _checkConfigInternal( self, allConfig, output, level=0 ):
      raise NotImplementedError()

   def _removeSubnet( self, subnets, subnetPrefix ):
      raise NotImplementedError()

   def _containsHostReservations( self, subnets ):
      for item in subnets:
         if item.get( "reservations" ):
            return True
      return False

   def configuredInterfaces( self ):
      raise NotImplementedError()

   def handleDisable( self, disabled ):
      raise NotImplementedError()

   def handleDebugLog( self ):
      self.sync()

   def _addAfConf( self, out ):
      '''
      Overwritten by DhcpServerIpv4Service and DhcpServerIpv6Service to add v4 and v6
      specific config to the kea conf file.
      '''
      raise NotImplementedError()

   def dhcpInterfaceStatus( self ):
      raise NotImplementedError()

   def genKeaIntfCfgCmds( self, linuxIntfName, ipAddrs ):
      raise NotImplementedError()

   def getOrCreateSubnetStatus( self, subnetId ):
      raise NotImplementedError()

   def _modifyDataDirectory( self, out ):
      raise NotImplementedError()

   def _poolOrSubnetClientClass( self, namePrefix, pool=False,
                                 excludedClasses=None ):
      '''
      Subnets and pools are associated with a pool or subnet client class called
      [subnet|pool]_[1.1.1.0/24|1.1.1.1-1.1.1.2]_client_class if there is a
      requirement to exclude client classes from the subnet or pool. The
      corresponding subnet or pool will then be locked to this new client class.
      Currently, this is only done for pools when an IP in the subnet is assigned to
      a client class. This function returns a dictionary representing a client class
      for a pool or subnet that excludes all all client classes in excludedClasses.
      '''
      poolClientClass = {}
      name = namePrefix + '_{}_client_class'.format( 'pool' if pool else 'subnet' )
      poolClientClass[ 'name' ] = name
      memberStr = ''
      if excludedClasses:
         memberStr = f"not member('{excludedClasses[ 0 ]}')"
         for clientClass in excludedClasses[ 1 : ]:
            memberStr += f" and not member('{clientClass}')"
         poolClientClass[ 'test' ] = memberStr
      return poolClientClass

   def subnetGenPrefix( self, subnetPrefix ):
      """
      Cast subnet prefix to IpGenPrefix

      @Input
         subnetPrefix (Arnet::Prefix) or (Arnet::Ip6Prefix)

      @Returns
         genPrefix (Arnet::IpGenPrefix)
      """

      genPrefix = Arnet.IpGenPrefix( str( subnetPrefix ) )
      return genPrefix

   def handleActiveIntfs( self, intfCfg, activeIntfs ):
      t1( "IPv%s: Old active: %s  --  New active: %s" %
          ( self.ipVersion_, self.intfCfg_, intfCfg ) )
      if intfCfg == self.intfCfg_ and activeIntfs == self.activeInterfaces_:
         return
      self.intfCfg_ = intfCfg
      self.activeInterfaces_ = activeIntfs

      # If no more active interfaces, reset the broadcast status
      if not self.intfCfg_ and self.ipVersion_ == "6":
         for subnetId in self.status_.ipv6SubnetToId:
            self.status_.subnetBroadcastStatus[ subnetId ] = (
               D6DRM.NO_IP6_ADDRESS_MATCH )

      self.sync()

   def handleClearLeaseEntry( self, ipAddress ):
      t1( "V{}: handle clear lease {} in vrf {}".format( self.ipVersion_, ipAddress,
                                                         self.vrf_ ) )
      cmdData = clearLeaseCmdData( str( ipAddress ), ipVersion=self.ipVersion_ )
      try:
         runKeaDhcpCmd( self.ipVersion_, self.vrf_.value, cmdData )
         return
      except KeaDhcpCommandError:
         # Happens after reboot when some interfaces are not up/present yet
         # or when socket is not present
         qt0( 'Kea command error running:', qv( cmdData ), qv( self.ipVersion_ ) )
         return
      except OSError as e:
         qt0( 'socket error:', qv( os.strerror( e.errno ) ) )

   def serviceEnabled( self ):
      raise NotImplementedError()

   def _updateDhcpServerDisabledStatus( self ):
      raise NotImplementedError()

   def _hasValidSubnetConfiguration( self ):
      raise NotImplementedError()

   def _removeLeaseFile( self ):
      for lf in glob.glob( self.leasePath_ + '*' ):
         try:
            os.remove( lf )
         except OSError as e:
            # File was removed after glob.glob found it. Note in python 3 the above
            # can be made more explicit by changing OSError to FileNotFoundError
            if e.errno != errno.ENOENT:
               # If the error is not ENOENT, the file was changed into a directory.
               # Let SuperServer crash here since it is unclear why someone would try
               # to change the leases file into a directory.
               raise

class DhcpServerIpv4Service( DhcpServerVrfService ):
   notifierTypeName = 'DhcpServer::Config'

   def __init__( self, config, status, vrf, intfStatus ):
      DhcpServerVrfService.__init__( self, config, status, vrf, intfStatus, "4" )

      self.status_.ipv4IdToSubnet.clear()
      self.status_.ipv4SubnetToId.clear()
      self.status_.ipv4SubnetLastUsedId = 0
      self.status_.ipv4SpaceIdToVendorId.clear()
      self.status_.ipv4VendorIdToSpaceId.clear()
      self.status_.ipv4VendorIdSpaceLastUsedId = 0
      self.status_.lastClearIdIpv4 = 0
      self.status_.subnetStatus.clear()
      self.status_.interfaceIpv4Disabled.clear()

      self.configReactor_ = DhcpServerReactor.ServiceV4Reactor( self.config_, self )

   def isAddrZero( self, ip ):
      return ip == '0.0.0.0'

   ##############################
   # Implemeting Derived Methods
   ##############################
   def serviceProcessWarm( self ):
      ''' Check if the DHCP server service has fully restsarted '''
      return bool( self.config_.interfacesIpv4 )

   def _addAfConf( self, out ):
      '''
      Adds v4 specific config to the kea conf file
      '''
      # Authorative
      out[ 'authoritative' ] = True

      # Store option 82 data in the lease file
      out[ 'store-extended-info' ] = True

   # Options Def
   def _optionsDef( self ):
      """
      IPv4 Options Def

      @Returns
         optionsDef (list of dicts) : options-def to be added in the kea config json
      """
      optionsDef = []

      # Custom option definition for DHCP code 150 (TFTP server addresses).
      # Kea DHCP does not support code 150 because it has multiple definitions
      # https://www.iana.org/assignments/bootp-dhcp-parameters/
      #       bootp-dhcp-parameters.xhtml#options
      # We will have to define our own option 150 in compliance with RFC5859.
      optionsDef.append( {
         'name': 'tftp-server-address',
         'code': 150,
         'type': 'ipv4-address',
         'array': True } )

      if self.globalPrivateOptions():
         optionsDef.extend(
               self._globalPrivateOptionsDef(
                     self.globalPrivateOptions().stringPrivateOption,
                     self.globalPrivateOptions().ipAddrPrivateOption ) )

      optionsDef.extend( self._vendorOptionsDef( self.config_ ) )

      if self.globalArbitraryOptions():
         arbitraryOptions = chain(
            self.globalArbitraryOptions().stringArbitraryOption.values(),
            self.globalArbitraryOptions().fqdnArbitraryOption.values(),
            self.globalArbitraryOptions().hexArbitraryOption.values(),
            self.globalArbitraryOptions().ipAddrArbitraryOption.values() )
         uniqueOptions = filterDuplicateOptions( arbitraryOptions, optionsDef,
                                                 self.opt43DefaultIdConfigured )
         optionsDef.extend( self._globalArbitraryOptionsDef( uniqueOptions ) )

      return optionsDef

   # Below function adds all attributes that are common for both subnet and global
   # config. Note:- if no DNS server is configured for subnet, kea will use the
   # global DNS server configuration.
   def _commonOptionsData( self, tftpServerOption66=None, tftpServerOption150=None,
                           tftpBootFileName=None, servers=None ):
      options = []
      if tftpServerOption66:
         options.append( {
            'name': 'tftp-server-name',
            'data': tftpServerOption66 } )
      if tftpServerOption150:
         options.append( {
            'name': 'tftp-server-address',
            'data': ','.join( tftpServerOption150.values() ) } )
      if tftpBootFileName:
         bootFileName = tftpBootFileName.replace( ",", r"\," )
         options.append( {
            'name': 'boot-file-name',
            'data': bootFileName } )
      if servers:
         options.append( {
            'name': 'domain-name-servers',
            'data': ', '.join( servers.values() ) } )
      return options

   # Subnet Options data
   def _subnetOptionsData( self, config ):
      options = []
      # subnet config options
      servers = config.dnsServers
      if config.defaultGateway != '0.0.0.0':
         options.append( {
            'name': 'routers',
            'data': config.defaultGateway } )
      if config.subnetName:
         options.append( {
            'name': 'name',
            'data': config.subnetName } )
      options.extend( self._commonOptionsData( config.tftpServerOption66,
                                               config.tftpServerOption150,
                                               config.tftpBootFileName,
                                               servers ) )
      return options

   # Global Options Data
   def _globalOptionsData( self, config ):
      options = []
      # global config options
      servers = config.dnsServersIpv4
      if config.domainNameIpv4:
         options.append( { 'name': 'domain-name',
                           'data': config.domainNameIpv4 } )
      options.extend( self._commonOptionsData( config.tftpServerOption66Ipv4,
                                               config.tftpServerOption150Ipv4,
                                               config.tftpBootFileNameIpv4,
                                               servers ) )

      if self.globalPrivateOptions():
         stringPrivateOptions = self.globalPrivateOptions().stringPrivateOption
         ipAddrPrivateOptions = self.globalPrivateOptions().ipAddrPrivateOption
         options.extend( self._globalPrivateOptionsData( stringPrivateOptions,
                                                         ipAddrPrivateOptions ) )

      if self.globalArbitraryOptions():
         arbitraryOptions = chain(
            self.globalArbitraryOptions().stringArbitraryOption.values(),
            self.globalArbitraryOptions().fqdnArbitraryOption.values(),
            self.globalArbitraryOptions().hexArbitraryOption.values(),
            self.globalArbitraryOptions().ipAddrArbitraryOption.values() )
         uniqueOptions = filterDuplicateOptions( arbitraryOptions, options,
                                                 self.opt43DefaultIdConfigured )
         options.extend( self._globalArbitraryOptionsData( uniqueOptions ) )
         t0( ( len( self.globalArbitraryOptions().stringArbitraryOption ) +
               len( self.globalArbitraryOptions().fqdnArbitraryOption ) +
               len( self.globalArbitraryOptions().hexArbitraryOption ) +
               len( self.globalArbitraryOptions().ipAddrArbitraryOption ) ),
             'arbitrary options before filtering duplicates' )
         t0( len( uniqueOptions ), 'unique options' )
      return options

   def reservationsIpStr( self ):
      return 'ip-address'

   def _getSubnetConfig( self, subnetId ):
      return self.config_.subnetConfigIpv4[ subnetId ]

   def _subnetConfig( self, inactiveRangeClientClasses,
                      inactiveSubnetClientClasses ):
      output = []
      hostReservationIDs = set()
      for keaSubnetId, subnetId in self.status_.ipv4IdToSubnet.items():
         subnetConfig = self._populateSubnetConfig( keaSubnetId, subnetId,
                                                    inactiveSubnetClientClasses,
                                                    inactiveRangeClientClasses,
                                                    hostReservationIDs )

         output.append( subnetConfig )
      return output, list( hostReservationIDs )

   def _vendorOptionsDef( self, config ):
      optionsDef = []
      # add all subOption definitions
      for vendorId, spaceId in self.status_.ipv4VendorIdToSpaceId.items():
         for subOption in config.vendorOptionIpv4[ vendorId ].\
                                 subOptionConfig.values():
            keaType = subOptionTypeToKeaType( subOption.type )
            isArray = ( subOption.type != vendorSubOptionType.string and
                        len( getSubOptionData( subOption ) ) > 1 )
            optionsDef.append( {
               'name': f'subOption{subOption.code}',
               'code': subOption.code,
               'type': keaType,
               'array': isArray,
               'space': f'{spaceId}-space'
            } )

      # add default Option 43 definition if set
      if config.vendorOptionIpv4.get( defaultVendorId ):
         self.optionCodeSet.add( 43 )
         spaceId = self.status_.ipv4VendorIdToSpaceId[ defaultVendorId ]
         optionsDef.append( {
            'name': 'vendor-encapsulated-options',
            'type': 'empty',
            'code': 43,
            'encapsulate': f'{spaceId}-space'
         } )

      return optionsDef

   def _vendorClientClasses( self, config ):
      clientClasses = []
      defaultOptionsData = []
      for vendorId, spaceId in self.status_.ipv4VendorIdToSpaceId.items():
         clientClass = {
            'name': f'VENDOR_CLASS_{vendorId}',
            }

         # add all subOption options-data
         optionsData = []
         for subOption in config.vendorOptionIpv4[ vendorId ].\
                                 subOptionConfig.values():
            data = getSubOptionData( subOption )
            optionsData.append( {
               'name': f'subOption{subOption.code}',
               'code': subOption.code,
               'space': f'{spaceId}-space',
               'data': data
               } )

         # add option43 data and definition
         optionsData.append( {
            'name': 'vendor-encapsulated-options'
            } )

         optionsDef = [ {
            'name': 'vendor-encapsulated-options',
            'type': 'empty',
            'code': 43,
            'encapsulate': f'{spaceId}-space',
            } ]

         if vendorId == defaultVendorId:
            # There is no client-class for default vendorId, instead it needs
            # to be in Dhcp4 option-data
            defaultOptionsData.extend( optionsData )
         else:
            clientClass[ 'option-data' ] = optionsData
            clientClass[ 'option-def' ] = optionsDef
            clientClasses.append( clientClass )

      return clientClasses, defaultOptionsData

   def _globalClientClasses( self, config ):
      '''
      Loops through all IPv4 global client classes and returns them
      and their option-defs in a list form.
      '''
      t0( 'IPv4 client classes:', config.clientClassConfigIpv4 )
      qt3( 'IPv4 client classes:', config.clientClassConfigIpv4 )
      globalClientClassKeaConfList = []
      globalClientClassOptionDefs = []
      self.hasGlobalL2MatchCriteria_ = False
      for clientClassConfig in config.clientClassConfigIpv4.values():
         transcriber = ClientClassTranscriberIpv4( self,
                                                   clientClassConfig,
                                                   globalClientClassTag )
         if transcriber.isInactive:
            continue
         globalClientClassKeaConfList.append( transcriber.clientClassKeaConf )
         globalClientClassOptionDefs.append( transcriber.keaOptionsDefConfig )
         self.hasGlobalL2MatchCriteria_ = ( self.hasGlobalL2MatchCriteria_ or
                                            transcriber.l2MatchConfigured )

      return globalClientClassKeaConfList, globalClientClassOptionDefs

   def _resetStaleDisabledMessages( self ):
      """ clear disabled messages/reasons """
      for subnetStatus in self.status_.subnetStatus.values():
         subnetStatus.disabledReasons.clear()
         subnetStatus.disabledMessage = ''

   def _checkOverlappingSubnet( self, out ):
      """
      Update overlapping subnet error message and remove overlapping subnets

      @Input
         out (Dict) : kea config as a dict

      @Returns
         None
      """
      for subnetGenPrefix, subnetStatus in self.status_.subnetStatus.items():
         qt0( 'updating overlapped disable message' )

         # remove stale subnetStatus
         if ( not subnetStatus.overlappingSubnet and
              not subnetStatus.disabledReasons ):
            del self.status_.subnetStatus[ subnetGenPrefix ]
            continue

         subnetId = subnetGenPrefix.v4Prefix
         if not subnetStatus.disabledMessage:
            subnetStatus.disabledMessage = 'Subnet is disabled'

         # remove disabled subnet from potential config file
         subnets = out[ 'subnet4' ]
         self._removeSubnet( subnets, subnetId )

   def addDisabledReason( self, subnetPrefix, disabledReasonEnum, reason,
                          disabledMsg ):
      """
      Update subnetStatus with the appropriate disabled reason

      @Input
         subnetPrefix (Arnet::Prefix)
         disabledReasonsEnum (disabledReasons)
         reason : reason to be saved
            duplicateReservedIp (str) : "1.0.0.0"
            invalidRange (str) : "192.168.1.110-192.168.1.254"
            overlappingRange (str) : "192.168.1.110-192.168.1.254"
      @Returns
         None
      """
      subnetStatus = self.getOrCreateSubnetStatus( subnetPrefix )
      subnetStatus.disabledMessage = disabledMsg
      subnetStatus.disabledReasons.add( disabledReasonEnum )
      ipAddrFn = Arnet.IpAddr
      rangeType = "DhcpServer::Range"
      subnetTraceStr = "Subnet " + str( subnetPrefix )

      # duplicateReservedIp
      if disabledReasonEnum == disabledReasonsEnum.duplicateReservedIp:
         t0( subnetTraceStr, 'has a duplicateReservedIp:', reason )
         subnetStatus.duplicateReservedIp = reason

      # invalidRanges
      elif disabledReasonEnum == disabledReasonsEnum.invalidRanges:
         rangeStr = reason.split( "-" )
         t0( subnetTraceStr, 'has an invalid range:', rangeStr )
         start = ipAddrFn( rangeStr[ 0 ] )
         end = ipAddrFn( rangeStr[ 1 ] )
         invalidRange = Tac.Value( rangeType, start, end )
         subnetStatus.invalidRangeV4 = invalidRange

      # overlappinRanges
      elif disabledReasonEnum == disabledReasonsEnum.overlappingRanges:
         rangeStr = reason.split( "-" )
         t0( subnetTraceStr, 'has an overlapping range:', rangeStr )
         start = ipAddrFn( rangeStr[ 0 ] )
         end = ipAddrFn( rangeStr[ 1 ] )
         overlappingRange = Tac.Value( rangeType, start, end )
         subnetStatus.overlappingRangeV4 = overlappingRange

      elif disabledReasonEnum == disabledReasonsEnum.unknown:
         t0( subnetTraceStr, 'has an unknown disabled reason' )

      else:
         t0( subnetTraceStr, 'has an unknown disabled reason ENUM' )
         assert False

   def configErrorRe( self ):
      errorHeading = '(?P<heading>Error encountered: )'

      subnetConfigFailure = '(?P<subnetConfigFailure>subnet configuration failed)'
      subnetExistingFailure = '(?P<subnetExistingFailure>subnet with the prefix of)'
      serverConfigFailure = '(?P<serverConfigFailure>[a-z A-Z]+)'
      # reservations mac-address
      reservationsConfigFailure = ( '(?P<reservationsConfigFailure>failed to add new'
                                    ' host using the HW address)' )
      fails = [ subnetConfigFailure, subnetExistingFailure,
                reservationsConfigFailure, serverConfigFailure ]
      failRegex = '(?P<failRegex>{}|{}|{}|{}):?\n*'.format( *fails )

      reasonRegex = '(?P<reason> ?.*)'

      errorRegex = errorHeading + failRegex + reasonRegex

      return re.compile( errorRegex )

   def _checkConfigInternal( self, allConfig, output, level=0 ):
      # we only need to check this once
      out = allConfig[ 'Dhcp4' ]
      if output:
         t0( "Malformed config", output )
         match = self.configErrorRe().search( output )

         # Unknown error message
         if not match or match.group( 'serverConfigFailure' ):
            self.status_.ipv4ServerDisabled = 'Server is disabled'

         # reservations mac-address error message
         elif match.group( 'reservationsConfigFailure' ):
            subnetId = configErrorSubnetIdRe().search( match.group( 'reason' ) ).\
                                               group( 'subnetId' ).strip( "\'" )
            subnetPrefix = self.status_.ipv4IdToSubnet[ int( subnetId ) ]

            # get failure reason
            reason, disabledReasonEnum = reservationsMacAddrFailureReason(
                                                         match.group( 'reason' ) )
            disabledMsg = f'Subnet is disabled - {reason}'

            # save disabled reason (duplicate reservation of IPv4 address)
            self.addDisabledReason( subnetPrefix, disabledReasonEnum, reason,
                                    disabledMsg )

            # remove the subnet configuration from potential config file
            subnets = out[ 'subnet4' ]
            self._removeSubnet( subnets, subnetPrefix )

            # check if there is another subnet with host reservations
            if not self._containsHostReservations( subnets ):
               del out[ 'host-reservation-identifiers' ]

         else:
            isPreExistingFailure = match.group( 'subnetExistingFailure' ) \
                                   is not None
            # get subnetID
            subnet = configErrorSubnetRe().search( match.group( 'reason' ) ).\
                     group( 'subnet' ).strip( "\'" )
            subnetPrefix = Arnet.Prefix( subnet )

            # get failure reason
            reason, disabledReasonEnum = getSubnetFailureReason(
                                             match.group( 'reason' ),
                                             isPreExistingFailure )
            if isPreExistingFailure:
               reason = subnet
            disabledMsg = f'Subnet is disabled - {reason}'

            # save disabled reason
            self.addDisabledReason( subnetPrefix, disabledReasonEnum, reason,
                                    disabledMsg )

            # Remove that subnet configuration from potential config file
            subnets = out[ 'subnet4' ]
            self._removeSubnet( subnets, subnetPrefix )

         # since we can only catch ONE error at a time, remove the disabled subnet
         # and recursively check for another invalid subnet if any
         self.checkConfig( allConfig, level=level + 1 )
         return

      # we want to do this at the end to catch the most descriptive
      # disable messages from kea first
      self._checkOverlappingSubnet( out )

      # if there is any invalid subnet, we should NOT remove the tmpKeaConfFile
      # for troubleshooting/debugging purposes
      # Currently subnetStatus ONLY holds disabled IPv4 subnets
      if not self.status_.subnetStatus:
         t0( 'Removing the tmp file' )
         os.remove( self.tmpKeaConfFile_ )

      # only needed to gather malformed configs
      os.remove( self.tmpToCheckKeaConfFile_ )
      return

   def _removeSubnet( self, subnets, subnetPrefix ):
      for item in subnets:
         if item[ 'subnet' ] == str( subnetPrefix ):
            t0( 'Removing subnet:', subnetPrefix )
            subnets.remove( item )

   def getOrCreateSubnetStatus( self, subnetId ):
      """
      Get or create the subnet status for a given subnetId

      @Input
         subnetId (Arnet::Prefix)

      @Returns
         subnetStatus (DhcpServer::Subnet4Status)
      """

      subnetGenPref = self.subnetGenPrefix( subnetId )
      subnetStatus = self.status_.subnetStatus.get( subnetGenPref )
      if not subnetStatus:
         subnetStatus = self.status_.subnetStatus.newMember( subnetGenPref )
      return subnetStatus

   def _updateSubnetOverlap( self, subnetId, add=True ):
      """
      Update the overlapping subnet(s) based on the newly added/deleted subnetId

      @Input
         subnetId (Arnet::Prefix) : subnet to be added/deleted
         add (Bool) : True if we are adding subnetId

      @Returns
         None
      """

      qt0( 'Updating overlapping subnets for:', qv( subnetId ) )
      subnetsOverlapped = []
      # get all the subnets that overlaps subnetId
      for subnet in self.status_.ipv4SubnetToId:
         if subnetId == subnet:
            # Skip ourselves, happens during SSO
            continue
         if subnet.overlaps( subnetId ):
            qt0( 'overlapping subnet:', qv( subnet ) )
            subnetsOverlapped.append( subnet )

      subnetIdGenPref = self.subnetGenPrefix( subnetId )
      if add:
         if subnetsOverlapped:
            qt0( 'Adding overlapping subnet(s)' )
            for subnet in subnetsOverlapped:
               # set the overlapping flag to both subnets
               subnetStatus = self.getOrCreateSubnetStatus( subnetId )
               subnetGenPref = self.subnetGenPrefix( subnet )
               subnetStatus.overlappingSubnet[ subnetGenPref ] = True

               subnetStatus2 = self.getOrCreateSubnetStatus( subnet )
               subnetStatus2.overlappingSubnet[ subnetIdGenPref ] = True
      else:
         # update the status of the overlapped subnet
         qt0( 'Removing overlapping subnet(s)' )
         del self.status_.subnetStatus[ subnetIdGenPref ]

         for subnet in subnetsOverlapped:
            subnetStatus = self.getOrCreateSubnetStatus( subnet )
            del subnetStatus.overlappingSubnet[ subnetIdGenPref ]

            # remove if there are no more subnets overlapping it
            if ( not subnetStatus.overlappingSubnet and
                 not subnetStatus.disabledReasons ):
               subnetGenPref = self.subnetGenPrefix( subnet )
               del self.status_.subnetStatus[ subnetGenPref ]

   def handleSubnetAdded( self, subnetId ):
      qt0( 'adding subnet:', qv( subnetId ) )
      self._updateSubnetOverlap( subnetId )
      subnetKeaId = self.status_.ipv4SubnetToId.get( subnetId, 0 )
      if not subnetKeaId:
         self.status_.ipv4SubnetLastUsedId = self.status_.ipv4SubnetLastUsedId + 1
         subnetKeaId = self.status_.ipv4SubnetLastUsedId
      self.status_.ipv4IdToSubnet[ subnetKeaId ] = subnetId
      self.status_.ipv4SubnetToId[ subnetId ] = subnetKeaId

   def handleSubnetRemoved( self, subnetId ):
      qt0( 'removing subnet:', qv( subnetId ) )
      _id = self.status_.ipv4SubnetToId.get( subnetId )
      if _id:
         del self.status_.ipv4IdToSubnet[ _id ]
      del self.status_.ipv4SubnetToId[ subnetId ]
      self._updateSubnetOverlap( subnetId, add=False )

   def handleVendorOptionAdded( self, vendorId ):
      qt0( 'adding vendor option:', qv( vendorId ) )
      spaceId = self.status_.ipv4VendorIdToSpaceId.get( vendorId )
      if not spaceId:
         self.status_.ipv4VendorIdSpaceLastUsedId += 1
         spaceId = self.status_.ipv4VendorIdSpaceLastUsedId
      self.status_.ipv4SpaceIdToVendorId[ spaceId ] = vendorId
      self.status_.ipv4VendorIdToSpaceId[ vendorId ] = spaceId

   def handleVendorOptionRemoved( self, vendorId ):
      qt0( 'removing vendor option:', qv( vendorId ) )
      spaceId = self.status_.ipv4VendorIdToSpaceId.get( vendorId )
      if spaceId:
         del self.status_.ipv4SpaceIdToVendorId[ spaceId ]
      del self.status_.ipv4VendorIdToSpaceId[ vendorId ]

   def globalPrivateOptions( self ):
      return self.config_.globalPrivateOptionIpv4

   def globalArbitraryOptions( self ):
      return self.config_.globalArbitraryOptionIpv4

   def configuredInterfaces( self ):
      return self.config_.interfacesIpv4

   def _leaseTime( self ):
      return int( self.config_.leaseTimeIpv4 )

   def genKeaIntfCfgCmds( self, linuxIntfName, ipAddrs ):
      return [ linuxIntfName ]

   def handleDisable( self, disabled ):
      if disabled:
         self.status_.ipv4ServerDisabled = 'Server is disabled'
      elif self.serviceEnabled():
         self.status_.ipv4ServerDisabled = ''
      else:
         self.status_.ipv4ServerDisabled = self.status_.ipv4ServerDisabledDefault

   def handleClearAllLease( self, clearId, start=True ):
      t1( "V4: handle clear all lease" )
      self.stopService()
      self._removeLeaseFile()
      self.status_.lastClearIdIpv4 = clearId
      if start:
         self.startService()

   def shutdown( self ):
      t0( 'server shutting down' )
      # Stop the server and delete all leases
      self.handleClearAllLease( self.status_.lastClearIdIpv4 + 1, start=False )
      self.status_.dhcpSnoopingV4Runnable = False
      self.status_.ipv4ServerDisabled = self.status_.ipv4ServerDisabledDefault
      self.configReactor_.close()

   def dhcpInterfaceStatus( self ):
      return self.status_.interfaceIpv4Disabled

   def serviceEnabled( self ):
      ''' Check if the DHCP server service is enabled  or if the configuration is
          adequate to start the service '''
      serverDisabled = self.status_.ipv4ServerDisabled == 'Server is disabled'
      return ( bool( self.intfCfg_ ) and not self.config_.disabled
               and self.config_.dhcpServerMode and not serverDisabled
               and self._hasValidSubnetConfiguration() )

   def _updateDhcpServerDisabledStatus( self ):
      '''
      If the server was not explicitly disabled, then this new config might
      be valid, thus we can clear ipv4ServerDisabled.

      @Returns
        True if ipv4ServerDisabled is cleared
      '''
      if ( self.config_.dhcpServerMode != self.config_.dhcpServerModeDefault and
           self.config_.interfacesIpv4 and self._hasValidSubnetConfiguration() ):
         self.status_.ipv4ServerDisabled = ''
         return True
      else:
         self.status_.ipv4ServerDisabled = self.status_.ipv4ServerDisabledDefault
         return False

   def _hasValidSubnetConfiguration( self ):
      for subnetConfig in self.config_.subnetConfigIpv4.values():
         if subnetConfig.ranges or subnetConfig.rangeConfig:
            return True
      return False

   def _modifyDataDirectory( self, out ):
      pass

class DhcpServerIpv6Service( DhcpServerVrfService ):
   notifierTypeName = 'DhcpServer::Config'

   def __init__( self, config, status, vrf, intfStatus ):
      DhcpServerVrfService.__init__( self, config, status, vrf, intfStatus, "6" )

      self.status_.ipv6IdToSubnet.clear()
      self.status_.ipv6SubnetToId.clear()
      self.status_.ipv6SubnetLastUsedId = 0
      self.status_.lastClearIdIpv6 = 0
      self.status_.subnet6Status.clear()
      self.status_.interfaceIpv6Disabled.clear()
      self.status_.subnetBroadcastStatus.clear()

      self.configReactor_ = DhcpServerReactor.ServiceV6Reactor( self.config_, self )

   def isAddrZero( self, ip ):
      return ip.isZero

   def serviceProcessWarm( self ):
      ''' Check if the DHCP server service has fully restsarted '''
      return bool( self.config_.interfacesIpv6 )

   def _addAfConf( self, out ):
      '''
      Currently does nothing. If there is v6 sepcific conf in the future, they will
      be added here
      '''

   # Options Def
   def _optionsDef( self ):
      """
      IPv6 Options Def

      @Returns
         optionsDef (list of dicts) : options-def to be added in the kea config json
      """
      optionsDef = []

      # example to verify if the feature toggle is properly enabled/disabled
      optionsDef.append( {
         'name': 'testingFeatureToggleFlexibleMatching',
         'code': 3,
         'space': 'featureToggle',
         'type': 'ipv6-address',
         } )

      if self.globalPrivateOptions():
         optionsDef.extend(
               self._globalPrivateOptionsDef(
                     self.globalPrivateOptions().stringPrivateOption,
                     self.globalPrivateOptions().ipAddrPrivateOption ) )

      if self.globalArbitraryOptions():
         arbitraryOptions = chain(
            self.globalArbitraryOptions().stringArbitraryOption.values(),
            self.globalArbitraryOptions().fqdnArbitraryOption.values(),
            self.globalArbitraryOptions().hexArbitraryOption.values(),
            self.globalArbitraryOptions().ipAddrArbitraryOption.values() )
         uniqueOptions = filterDuplicateOptions( arbitraryOptions, optionsDef,
                                                 self.opt43DefaultIdConfigured,
                                                 ipVersion='6' )
         optionsDef.extend( self._globalArbitraryOptionsDef( uniqueOptions ) )

      return optionsDef

   # Below function adds attributes that are common for both subnet and global
   # config. Note:- if no DNS server is configured for subnet, kea will use the
   # global DNS server configuration.
   def _commonOptionsData( self, servers=None, tftpBootFileNameIpv6=None ):
      options = []
      if servers:
         options.append( {
            'name': 'dns-servers',
            'data': ', '.join( [ str( v ) for v in servers.values() ] ) } )
      if tftpBootFileNameIpv6:
         bootFileName = tftpBootFileNameIpv6.replace( ",", r"\," )
         options.append( {
            'name': 'bootfile-url',
            'data': bootFileName } )
      return options

   def _subnetOptionsData( self, config ):
      # subnet config options
      options = []
      servers = config.dnsServers
      if config.subnetName:
         options.append( {
            'name': 'name',
            'data': config.subnetName } )
      commonOptionsData = self._commonOptionsData(
            servers=servers, tftpBootFileNameIpv6=config.tftpBootFileName )
      options.extend( commonOptionsData )
      return options

   def _globalOptionsData( self, config ):
      # global config options
      options = []
      servers = config.dnsServersIpv6
      if config.domainNameIpv6:
         options.append( { 'name': 'domain-search',
                           'data': config.domainNameIpv6 } )
      commonOptionsData = self._commonOptionsData( servers,
                                                   config.tftpBootFileNameIpv6 )
      options.extend( commonOptionsData )
      if self.globalPrivateOptions():
         stringPrivateOptions = self.globalPrivateOptions().stringPrivateOption
         ipAddrPrivateOptions = self.globalPrivateOptions().ipAddrPrivateOption
         options.extend( self._globalPrivateOptionsData( stringPrivateOptions,
                                                         ipAddrPrivateOptions ) )

      if self.globalArbitraryOptions():
         arbitraryOptions = chain(
            self.globalArbitraryOptions().stringArbitraryOption.values(),
            self.globalArbitraryOptions().fqdnArbitraryOption.values(),
            self.globalArbitraryOptions().hexArbitraryOption.values(),
            self.globalArbitraryOptions().ipAddrArbitraryOption.values() )
         uniqueOptions = filterDuplicateOptions( arbitraryOptions, options,
                                                 self.opt43DefaultIdConfigured,
                                                 ipVersion='6' )
         options.extend( self._globalArbitraryOptionsData( uniqueOptions ) )
         t0( ( len( self.globalArbitraryOptions().stringArbitraryOption ) +
               len( self.globalArbitraryOptions().fqdnArbitraryOption ) +
               len( self.globalArbitraryOptions().hexArbitraryOption ) +
               len( self.globalArbitraryOptions().ipAddrArbitraryOption ) ),
             'arbitrary options before filtering duplicates' )
         t0( len( uniqueOptions ), 'unique options' )
      return options

   def _allIntfMatchSubnet( self, subnetId, intfsToSubnets ):
      allIntfMatches = set()
      # Only look at active ones, since they are guarenteed to have addresses
      for intf, addrkeys in self.activeInterfaces_:
         for addrkey in addrkeys:
            # If the subnet contains the ip of the interface, and the subnet mask
            # length is <= to the mask length for the interface, then this subnet
            # can reply to broadcast requests on this interface
            if ( subnetId.contains( addrkey.addr.v6Addr ) and
                 subnetId.len <= addrkey.mask ):
               intfsToSubnets[ intf ].add( subnetId )
               allIntfMatches.add( intf )

      t5( "Subnet", str( subnetId ), "Interfaces", allIntfMatches )
      return allIntfMatches

   def _globalClientClasses( self, config ):
      '''
      Loops through all IPv6 global client classes and returns them
      and their option-defs in a list form.
      '''
      t0( 'IPv6 client classes:', len( config.clientClassConfigIpv6 ) )
      qt3( 'IPv6 client classes:', len( config.clientClassConfigIpv6 ) )
      globalClientClassKeaConfList = []
      globalClientClassOptionsDef = []
      self.hasGlobalL2MatchCriteria_ = False
      for clientClassConfig in config.clientClassConfigIpv6.values():
         transcriber = ClientClassTranscriberIpv6( self,
                                                   clientClassConfig,
                                                   globalClientClassTag )
         if transcriber.isInactive:
            continue
         globalClientClassKeaConfList.append( transcriber.clientClassKeaConf )
         globalClientClassOptionsDef.append( transcriber.keaOptionsDefConfig )
         self.hasGlobalL2MatchCriteria_ = (
               self.hasGlobalL2MatchCriteria_ or transcriber.l2MatchConfigured )

      return globalClientClassKeaConfList, globalClientClassOptionsDef

   def reservationsIpStr( self ):
      return 'ip-addresses'

   def _getSubnetConfig( self, subnetId ):
      return self.config_.subnetConfigIpv6[ subnetId ]

   def _subnetConfig( self, inactiveRangeClientClasses,
                      inactiveSubnetClientClasses ):
      output = []
      intfsToSubnets = defaultdict( set )
      subnetConfigs = {}
      maybeSubnetValid = set()
      hostReservationIDs = set()
      self.status_.subnetBroadcastStatus.clear()
      for keaSubnetId, subnetId in self.status_.ipv6IdToSubnet.items():
         subnetConfig = self._populateSubnetConfig( keaSubnetId, subnetId,
                                                    inactiveSubnetClientClasses,
                                                    inactiveRangeClientClasses,
                                                    hostReservationIDs,
                                                    '6' )
         allIntfMatches = self._allIntfMatchSubnet( subnetId, intfsToSubnets )
         self.status_.subnetBroadcastStatus[ subnetId ] = D6DRM.NO_IP6_ADDRESS_MATCH

         if len( allIntfMatches ) > 1:
            t4( "Subnet", str( subnetId ), "too many interface matches" )
            self.status_.subnetBroadcastStatus[ subnetId ] = (
               D6DRM.MULTIPLE_INTERFACES_MATCH_SUBNET.format(
                  " ".join( sorted( allIntfMatches ) ) ) )
         else:
            t4( "subnet", str( subnetId ), "Might be valid" )
            maybeSubnetValid.add( subnetId )

         subnetConfigs[ subnetId ] = subnetConfig
         output.append( subnetConfig )

      # Now check that each interface only matches a single subnet. If it does,
      # update the subnets config, otherwise update the subnet broadcast status.
      # Sort the items so the same config always generates the same status messages.
      for intf, subnets in sorted( intfsToSubnets.items() ):
         if len( subnets ) == 1:
            subnetId = subnets.pop()
            if subnetId not in maybeSubnetValid:
               # More that 1 interface matched this subnet
               continue
            kniIntf = kernelDeviceName( self.intfStatusAll_, intf )
            subnetConfigs[ subnetId ][ 'interface' ] = kniIntf
            del self.status_.subnetBroadcastStatus[ subnetId ]
         else:
            msg = D6DRM.MULTIPLE_SUBNETS_MATCH_INTERFACE.format( intf )
            t4( "Subnet", str( subnets ), "too many subnets match interface", intf )
            for subnet in subnets:
               self.status_.subnetBroadcastStatus[ subnet ] = msg

      return output, list( hostReservationIDs )

   def _resetStaleDisabledMessages( self ):
      """ clear disabled messages/reasons """
      for subnet6Status in self.status_.subnet6Status.values():
         subnet6Status.disabledReasons.clear()
         subnet6Status.disabledMessage = ''

   def _checkOverlappingSubnet( self, out ):
      """
      Update overlapping subnet error message and remove overlapping subnets

      @Input
         out (Dict) : kea config as a dict

      @Returns
         None
      """
      for subnetGenPrefix, subnetStatus in self.status_.subnet6Status.items():
         qt0( 'updating overlapped disable mesage' )
         # remove stale subnetStatus
         if ( not subnetStatus.overlappingSubnet and
              not subnetStatus.disabledReasons ):
            del self.status_.subnet6Status[ subnetGenPrefix ]
            continue

         subnetId = subnetGenPrefix.v6Prefix
         if not subnetStatus.disabledMessage:
            subnetStatus.disabledMessage = 'Subnet is disabled'

         # remove disabled subnet from potential config file
         subnets = out[ 'subnet6' ]
         self._removeSubnet( subnets, subnetId )

   def addDisabledReason( self, subnetPrefix, disabledReasonEnum, reason,
                          disabledMsg ):
      """
      Update subnetStatus with the appropriate disabled reason

      @Input
         subnetPrefix (Arnet::Ip6Prefix)
         disabledReasonsEnum (disabledReasons)
         reason : reason to be saved
            duplicateReservedIp (str) : "1::1"
            invalidRange (str) : "2::1-2::3"
            overlappingRange (str) : "2::1-2::3"
         disabledMsg (str)
      @Returns
         None
      """
      subnetStatus = self.getOrCreateSubnetStatus( subnetPrefix )
      subnetStatus.disabledMessage = disabledMsg
      subnetStatus.disabledReasons.add( disabledReasonEnum )
      ipAddrFn = Arnet.Ip6Addr
      rangeType = "DhcpServer::Range6"
      subnetTraceStr = "Subnet " + str( subnetPrefix )

      # duplicateReservedIp
      if disabledReasonEnum == disabledReasonsEnum.duplicateReservedIp:
         t0( subnetTraceStr, 'has a duplicateReservedIp:', reason )
         subnetStatus.duplicateReservedIp = Arnet.Ip6Addr( reason )

      if disabledReasonEnum == disabledReasonsEnum.invalidRanges:
         rangeStr = reason.split( "-" )
         start = ipAddrFn( rangeStr[ 0 ] )
         end = ipAddrFn( rangeStr[ 1 ] )
         invalidRange = Tac.Value( rangeType, start, end )
         subnetStatus.invalidRangeV6 = invalidRange

      if disabledReasonEnum == disabledReasonsEnum.overlappingRanges:
         rangeStr = reason.split( "-" )
         start = ipAddrFn( rangeStr[ 0 ] )
         end = ipAddrFn( rangeStr[ 1 ] )
         overlappingRange = Tac.Value( rangeType, start, end )
         subnetStatus.overlappingRangeV6 = overlappingRange

   def configErrorRe( self ):
      errorHeading = '(?P<heading>Error encountered: )'

      subnetConfigFailure = '(?P<subnetConfigFailure>subnet configuration failed)'
      subnetExistingFailure = '(?P<subnetExistingFailure>subnet with the prefix of)'
      serverConfigFailure = '(?P<serverConfigFailure>[a-z A-Z]+)'
      # reservations mac-address
      reservationsConfigFailure = ( '(?P<reservationsConfigFailure>failed to add'
                                    ' address reservation for host using the HW'
                                    ' address)' )
      fails = [ subnetConfigFailure, subnetExistingFailure,
                reservationsConfigFailure, serverConfigFailure ]
      failRegex = '(?P<failRegex>{}|{}|{}|{}):?\n*'.format( *fails )

      reasonRegex = '(?P<reason> ?.*)'

      errorRegex = errorHeading + failRegex + reasonRegex

      return re.compile( errorRegex )

   def _checkConfigInternal( self, allConfig, output, level=0 ):
      # we only need to check this once
      out = allConfig[ 'Dhcp6' ]
      if output:
         t0( "Malformed config", output )
         match = self.configErrorRe().search( output )

         # Unknown error message
         if not match or match.group( 'serverConfigFailure' ):
            self.status_.ipv6ServerDisabled = 'Server is disabled'
         # reservations mac-address error message
         elif match.group( 'reservationsConfigFailure' ):
            subnetId = configErrorSubnetIdRe().search( match.group( 'reason' ) ).\
                                               group( 'subnetId' ).strip( "\'" )
            subnetPrefix = self.status_.ipv6IdToSubnet[ int( subnetId ) ]

            # get failure reason
            reason, disabledReasonEnum = reservationsMacAddrFailureReason6(
                                                         match.group( 'reason' ) )
            disabledMsg = f'Subnet is disabled - {reason}'

            # save disabled reason (duplicate reservation of IPv6 address)
            self.addDisabledReason( subnetPrefix, disabledReasonEnum, reason,
                                    disabledMsg )

            # remove the subnet configuration from potential config file
            subnets = out[ 'subnet6' ]
            self._removeSubnet( subnets, subnetPrefix )

            # check if there is another subnet with host reservations
            if not self._containsHostReservations( subnets ):
               del out[ 'host-reservation-identifiers' ]
         else:
            isPreExistingFailure = match.group( 'subnetExistingFailure' ) \
                                   is not None

            match2 = configErrorSubnetRe().search( match.group( 'reason' ) )
            if not match2:
               self.status_.ipv6ServerDisabled = 'Server is disabled'
               self.checkConfig( allConfig, level=level + 1 )
               return

            subnet = match2.group( 'subnet' ).strip( "'" )
            subnetPrefix = Arnet.Ip6Prefix( subnet )

            # get failure reason
            reason, disabledReasonEnum = getSubnetFailureReason(
                                                   match.group( 'reason' ),
                                                   isPreExistingFailure )

            if isPreExistingFailure:
               reason = subnet
            disabledMsg = f'Subnet is disabled - {reason}'

            # save the disabled reason
            self.addDisabledReason( subnetPrefix, disabledReasonEnum, reason,
                                    disabledMsg )

            # Remove that subnet configuration from potential config file
            subnets = out[ 'subnet6' ]
            self._removeSubnet( subnets, subnetPrefix )

         # since we can only catch ONE error at a time, remove the disabled subnet
         # and recursively check for another invalid subnet if any
         self.checkConfig( allConfig, level=level + 1 )
         return

      # we want to do this at the end to catch the most descriptive
      # disable messages from kea first
      self._checkOverlappingSubnet( out )

      # if there is any invalid subnet, we should NOT remove the tmpKeaConfFile
      # for troubleshooting/debugging purposes
      # Currently subnet6Status ONLY holds disabled IPv6 subnets
      if not self.status_.subnet6Status:
         t0( 'Removing the tmp file' )
         os.remove( self.tmpKeaConfFile_ )

      # only needed to gather malformed configs
      os.remove( self.tmpToCheckKeaConfFile_ )
      return

   def _removeSubnet( self, subnets, subnetPrefix ):
      for item in subnets:
         if item[ 'subnet' ] == str( subnetPrefix ):
            t0( 'Removing subnet:', subnetPrefix )
            subnets.remove( item )

   def getOrCreateSubnetStatus( self, subnetId ):
      """
      Get or create the overlapping subnets for a given subnetId

      @Input
         subnetId (Arnet::Ip6Prefix)

      @Returns
         subnetStatus (DhcpServer::Subnet6Status)
      """

      subnetGenPref = self.subnetGenPrefix( subnetId )
      subnet6Status = self.status_.subnet6Status.get( subnetGenPref )
      if not subnet6Status:
         subnet6Status = (
                   self.status_.subnet6Status.newMember( subnetGenPref ) )
      return subnet6Status

   def _updateSubnetOverlap( self, subnetId, add=True ):
      """
      Update the overlapping subnet(s) based on the newly added/deleted subnetId

      @Input
         subnetId (Arnet::Ip6Prefix) : subnet to be added/deleted
         add (Bool) : True if we are adding subnetId

      @Returns
         None
      """

      qt0( 'Updating overlapping subnets for:', qv( subnetId ) )
      subnets6Overlapped = []
      # get all the subnets that overlaps subnetId
      for subnet in self.status_.ipv6SubnetToId:
         if subnetId == subnet:
            # Skip ourselves, happens during SSO
            continue
         if subnet.overlaps( subnetId ):
            qt0( 'Overlapping subnet:', qv( subnet ) )
            subnets6Overlapped.append( subnet )

      subnetIdGenPref = self.subnetGenPrefix( subnetId )
      if add:
         if subnets6Overlapped:
            qt0( 'Adding overlapping subnet(s)' )
            for subnet in subnets6Overlapped:
               # set the overlapping flag to both subnets
               subnet6Status = self.getOrCreateSubnetStatus( subnetId )
               subnetGenPref = self.subnetGenPrefix( subnet )
               subnet6Status.overlappingSubnet[ subnetGenPref ] = True

               subnet6Status2 = self.getOrCreateSubnetStatus( subnet )
               subnet6Status2.overlappingSubnet[ subnetIdGenPref ] = True
      else:
         # update the status of the overlapped subnet
         qt0( 'Removing overlapping subnet(s)' )
         del self.status_.subnet6Status[ subnetIdGenPref ]
         for subnet in subnets6Overlapped:
            subnet6Status = self.getOrCreateSubnetStatus( subnet )
            del subnet6Status.overlappingSubnet[ subnetIdGenPref ]

            # remove if there are no more subnets overlapping it
            if ( not subnet6Status.overlappingSubnet and
                 not subnet6Status.disabledReasons ):
               subnetGenPref = self.subnetGenPrefix( subnet )
               del self.status_.subnet6Status[ subnetGenPref ]

   def handleSubnetAdded( self, subnetId ):
      qt0( 'adding subnet:', qv( subnetId ) )
      self._updateSubnetOverlap( subnetId )
      subnetKeaId = self.status_.ipv6SubnetToId.get( subnetId, 0 )
      if not subnetKeaId:
         self.status_.ipv6SubnetLastUsedId = self.status_.ipv6SubnetLastUsedId + 1
         subnetKeaId = self.status_.ipv6SubnetLastUsedId
      self.status_.ipv6IdToSubnet[ subnetKeaId ] = subnetId
      self.status_.ipv6SubnetToId[ subnetId ] = subnetKeaId
      self.status_.subnetBroadcastStatus[ subnetId ] = D6DRM.NO_IP6_ADDRESS_MATCH

   def handleSubnetRemoved( self, subnetId ):
      qt0( 'removing subnet:', qv( subnetId ) )
      _id = self.status_.ipv6SubnetToId.get( subnetId )
      if _id:
         del self.status_.ipv6IdToSubnet[ _id ]
      del self.status_.ipv6SubnetToId[ subnetId ]
      del self.status_.subnetBroadcastStatus[ subnetId ]
      self._updateSubnetOverlap( subnetId, add=False )

   def globalPrivateOptions( self ):
      return self.config_.globalPrivateOptionIpv6

   def globalArbitraryOptions( self ):
      return self.config_.globalArbitraryOptionIpv6

   def configuredInterfaces( self ):
      return self.config_.interfacesIpv6

   def _leaseTime( self ):
      return int( self.config_.leaseTimeIpv6 )

   def handleDisable( self, disabled ):
      if disabled:
         self.status_.ipv6ServerDisabled = 'Server is disabled'
      elif self.serviceEnabled():
         self.status_.ipv6ServerDisabled = ''
      else:
         self.status_.ipv6ServerDisabled = self.status_.ipv6ServerDisabledDefault

   def handleClearAllLease( self, clearId, start=True ):
      t1( "V6: handle clear all lease" )
      self.stopService()
      self._removeLeaseFile()
      self.status_.lastClearIdIpv6 = clearId
      if start:
         self.startService()

   def shutdown( self ):
      # Stop the server and delete all leases
      self.handleClearAllLease( self.status_.lastClearIdIpv6 + 1, start=False )
      self.status_.dhcpSnoopingV6Runnable = False
      self.status_.ipv6ServerDisabled = self.status_.ipv6ServerDisabledDefault
      self.configReactor_.close()

   def dhcpInterfaceStatus( self ):
      return self.status_.interfaceIpv6Disabled

   def genKeaIntfCfgCmds( self, linuxIntfName, ipAddrs ):
      cfg = set()
      for addrKey in ipAddrs:
         # Add the interface name and all ipv6 non link local addresses to kea
         # This allows kea to reply to relayed (unicast) requests
         if not addrKey.addr.isV6LinkLocal:
            cfg.add( linuxIntfName + "/" + str( addrKey.addr ) )

      return cfg

   def serviceEnabled( self ):
      ''' Check if the DHCP server service is enabled  or if the configuration is
          adequate to start the service '''
      serverDisabled = self.status_.ipv6ServerDisabled == 'Server is disabled'
      return ( bool( self.intfCfg_ ) and not self.config_.disabled
               and self.config_.dhcpServerMode and not serverDisabled
               and self._hasValidSubnetConfiguration() )

   def _updateDhcpServerDisabledStatus( self ):
      '''
      If the server was not explicitly disabled, then this new config might
      be valid, thus we can clear ipv6ServerDisabled.

      @Returns
        True if ipv6ServerDisabled is cleared
      '''
      if ( self.config_.dhcpServerMode != self.config_.dhcpServerModeDefault and
           self.config_.interfacesIpv6 and self._hasValidSubnetConfiguration() ):
         self.status_.ipv6ServerDisabled = ''
         return True
      else:
         self.status_.ipv6ServerDisabled = self.status_.ipv6ServerDisabledDefault
         return False

   def _hasValidSubnetConfiguration( self ):
      for subnetConfig in self.config_.subnetConfigIpv6.values():
         if subnetConfig.ranges or subnetConfig.rangeConfig:
            return True
      return False

   def _modifyDataDirectory( self, out ):
      # Kea 2.0.0 stores the server identifier file (v6 only) in /var/lib/kea
      # Explicitly specifying the exact location here, since kea was otherwise
      # attempting to generate it in '/var/lib/lib/kea'
      out[ 'data-directory' ] = "/var/lib/kea"

class DhcpServerVrfManager:
   '''
   Starts the ipv4 or ipv6 dhcp server for a specific vrf
   '''
   def __init__( self, vrf, config, status, kniStatus, manager,
                 dhcpRelayHelperConfig ):
      self.vrf_ = vrf
      self.config_ = config
      self.status_ = status
      self.kniStatus_ = kniStatus
      self.manager_ = manager
      self.dhcpRelayHelperConfig_ = dhcpRelayHelperConfig
      self.interfaceToKni_ = {}
      self.interfaceToAddrs_ = {}
      self.v4Service_ = DhcpServerIpv4Service(
            config, status, vrf, self.manager_.intfStatusAll() )
      self.v6Service_ = DhcpServerIpv6Service(
            config, status, vrf, self.manager_.intfStatusAll() )
      proxy = weakref.proxy( self )
      self.vrfManagerReactor_ = DhcpServerReactor.VrfManagerReactor( self.config_,
                                                                     proxy )
      self.kniReactor_ = DhcpServerReactor.KniReactor( self.kniStatus_, proxy )
      self.dhcpRelayHelperConfigReactor_ = (
         DhcpServerReactor.DhcpRelayHelperConfigReactor(
         self.dhcpRelayHelperConfig_, proxy ) )

      # resync config and status after startup
      self.handleDisable( self.config_.disabled )

   def services( self ):
      return [ self.v4Service_, self.v6Service_ ]

   def shutdown( self ):
      t2( "VrfM: shutdown" )
      self.kniReactor_.close()
      self.vrfManagerReactor_.close()
      self.dhcpRelayHelperConfigReactor_.close()
      for service in self.services():
         service.shutdown()
         service.cleanupService()

   def handleDisable( self, disabled ):
      t2( "VrfM: handleDisable" )
      for service in self.services():
         service.handleDisable( disabled )

   def handleIntfChange( self ):
      t2( "VrfM: handleIntfChange" )
      for service in self.services():
         self._handleActiveInterfaces( service, self.vrf_ )

   def _handleActiveInterfaces( self, service, vrf ):
      t2( 'handleActiveInterfaces:', qv( service ), qv( vrf ) )
      ipVersion = service.ipVersion_
      interfacesConfigured = service.configuredInterfaces()
      dhcpIntfStatus = service.dhcpInterfaceStatus()
      activeInterfaces = []
      intfCfg = set()

      if not interfacesConfigured:
         t3( "No interfaces configured for", ipVersion )
         service.handleActiveIntfs( intfCfg, activeInterfaces )
         return

      for intf in interfacesConfigured:
         intfLocal = self.manager_.intfStatusLocal_.intfStatusLocal.get( intf )
         if not intfLocal:
            t1( "Skipping ", qv( ipVersion ),
                 " (", qv( intf ), "), no intf status local" )
            dhcpIntfStatus[ intf ] = DISM.NO_INTF_STATUS_LOCAL_MSG
            continue
         if intfLocal.netNsName != vrf.nsName():
            qt1( "Skipping ", qv( ipVersion ),
                 " (", qv( intf ), "), not in vrf ", vrf )
            dhcpIntfStatus[ intf ] = DISM.NOT_IN_VRF_MSG.format( vrf.value )
            continue
         if self.dhcpRelayHelperConfig_.alwaysOn:
            t1( "Skipping ", qv( intf ), ", relay is always on" )
            dhcpIntfStatus[ intf ] = DISM.DHCP_RELAY_ALWAYS_MSG
            continue
         if intf in self.dhcpRelayHelperConfig_.intfConfig:
            t1( "Skipping ", qv( intf ),
                 ", relay is configured on this interface" )
            dhcpIntfStatus[ intf ] = DISM.DHCP_RELAY_CFG_MSG
            continue
         linuxIntfName = kernelDeviceName( self.manager_.intfStatusAll(), intf )
         if not linuxIntfName:
            t1( "Skipping ", qv( ipVersion ),
                 " (", qv( intf ), "), no intf status all" )
            dhcpIntfStatus[ intf ] = DISM.NO_INTF_STATUS_ALL_MSG
            continue
         kniIntf = self.interfaceToKni_.get( linuxIntfName )
         if not kniIntf:
            t1( "Skipping ", qv( ipVersion ), " (", qv( intf ), "), no kni" )
            dhcpIntfStatus[ intf ] = DISM.NO_KNI_MSG
            continue
         if kniIntf.flags & IFF_RUNNING != IFF_RUNNING:
            t1( "Skipping ", qv( ipVersion ), " (", qv( intf ), "), not running" )
            dhcpIntfStatus[ intf ] = DISM.NOT_UP_MSG
            continue
         ipAddrs = self._ipActiveOnInterface( kniIntf.key, ipVersion )
         if not ipAddrs:
            t1( "Skipping ", qv( ipVersion ),
                 " (", qv( intf ), "), no ip address" )
            dhcpIntfStatus[ intf ] = DISM.NO_IP_ADDRESS_MSG
            continue
         if ipVersion == 6:
            # make sure that the interface has a link-local address configured
            if not hasLinkLocalAddr( ipAddrs ):
               qt1( "Skipping ", qv( ipVersion ),
                    " (", qv( intf ), "), no link-local address configured" )
               dhcpIntfStatus[ intf ] = DISM.NO_LINK_LOCAL_ADDRESS_MSG
               continue

         keaIntfCfgCmds = service.genKeaIntfCfgCmds( linuxIntfName, ipAddrs )

         intfCfg.update( keaIntfCfgCmds )
         activeInterfaces.append( ( intf, ipAddrs ) )
         del dhcpIntfStatus[ intf ]

      service.handleActiveIntfs( intfCfg, activeInterfaces )

   def _ipActiveOnInterface( self, intfKey, ipVersion ):
      addrKeys = self.interfaceToAddrs_.get( intfKey )
      if not addrKeys:
         return []

      addrs = []

      addrFamily = (
         addrFamilyEnum.ipv4 if ipVersion == "4" else addrFamilyEnum.ipv6 )

      for addrKey in addrKeys:
         ipState = self.kniStatus_.addr.get( addrKey )
         if not ipState:
            continue
         if ipState.key.addr.af != addrFamily:
            continue
         if ipState.assigned:
            addrs.append( addrKey )
            if ipVersion == "4":
               # V4 doesn't care about all the addrs, just that one is assigned
               return addrs
      return addrs

   def handleDebugLog( self, filePath ):
      t2( "VrfM: handleDebugLog <%s>" % filePath )
      for service in self.services():
         service.handleDebugLog()

class DhcpServerManager( SuperServer.SuperServerAgent ):
   '''
   Starts the DhcpServerVrfManagers
   '''
   def __init__( self, entityManager ):
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      mg = entityManager.mountGroup()
      self.vrfManagers_ = {}
      self.configs_ = {}
      self.status_ = {}
      self.kniStatus_ = {}
      self.intfStatusLocal_ = None
      self.intfStatusLocalReactor_ = None
      self.intfStatusAll_ = None
      self.intfStatusAllReactor_ = None
      self.configReactors_ = {}
      self.vrfConfigDirReactor_ = None

      # mounting configs to react to
      self.vrfConfigDir = mg.mountPath( 'dhcpServer/vrf/config' )
      self.vrfStatusDir = mg.mountPath( 'dhcpServer/vrf/status' )
      self.dhcpRelayHelperConfig = mg.mountPath( 'ip/helper/dhcprelay/config' )
      self.shmemEm = SharedMem.entityManager( sysdbEm=self.entityManager )

      def _finished():
         t2( "Fishing mounting" )
         if self.active():
            self.onSwitchover( None )
      mg.close( _finished )

   def onSwitchover( self, protocol ):
      t2( "onSwitchover" )
      self.intfStatusLocal_ = self.intfStatusLocal()
      self.intfStatusLocalReactor_ = Tac.collectionChangeReactor(
          self.intfStatusLocal_.intfStatusLocal,
          DhcpServerReactor.IntfStatusLocalReactor,
          reactorArgs=( weakref.proxy( self ), ) )
      self.intfStatusAll_ = self.intfStatusAll()
      self.intfStatusAllReactor_ = Tac.collectionChangeReactor(
            self.intfStatusAll_.intfStatus,
            DhcpServerReactor.IntfStatusAllReactor,
            reactorArgs=( weakref.proxy( self ), ) )
      self.vrfConfigDirReactor_ = DhcpServerReactor.VrfConfigDirReactor(
         self.vrfConfigDir, self )
      self.vrfStatusDirReactor_ = DhcpServerReactor.VrfStatusDirReactor(
         self.vrfStatusDir )

   def handleIntfChange( self ):
      t2( "M: handleIntfChange" )
      for vrfM in self.vrfManagers_.values():
         vrfM.handleIntfChange()

   def handleDhcpServerMode( self, mode, vrf ):
      t2( "handleDhcpServerMode" )
      config = self.configs_[ vrf ]
      status = self.status_[ vrf ]
      kniStatus = self.kniStatus_[ vrf ]
      if mode != config.dhcpServerModeDefault:
         t2( "M: Enabling" )
         if vrf not in self.vrfManagers_:
            t2( "M: Creating" )
            self.vrfManagers_[ vrf ] = DhcpServerVrfManager(
               vrf, config, status, kniStatus, self, self.dhcpRelayHelperConfig )
      else:
         t2( "M: Disabling" )
         vrfM = self.vrfManagers_.pop( vrf, None )
         if vrfM:
            t2( "M: Deleting" )
            vrfM.shutdown()

def Plugin( ctx ):
   ctx.registerService( DhcpServerManager( ctx.entityManager ) )
