# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import Cell
import Tac
import SuperServer

class HardwareSecurityChipReactor( Tac.Notifiee ):
   notifierTypeName = "Mgmt::Security::HardwareStatus"

   def __init__( self, hardwareStatus, secStatus ):
      Tac.Notifiee.__init__( self, hardwareStatus )
      self.secStatus_ = secStatus
      self.hardwareStatus_ = hardwareStatus
      self.handleSecurityChipVersion()

   @Tac.handler( "securityChipVersion" )
   def handleSecurityChipVersion( self ):
      self.secStatus_.securityChipVersion = \
            self.hardwareStatus_.securityChipVersion

class HardwareEntropyReactor( Tac.Notifiee ):
   notifierTypeName = "Mgmt::Security::HardwareStatus"

   def __init__( self, hardwareEntropyStatus, secStatus ):
      Tac.Notifiee.__init__( self, hardwareEntropyStatus )
      self.secStatus_ = secStatus
      self.hardwareStatus_ = hardwareEntropyStatus
      self.handleStatusChange()

   def handleStatusChange( self ):
      self.secStatus_.entropySourceHardwareEnabled = \
            self.hardwareStatus_.entropySourceHardwareEnabled
      self.secStatus_.entropySourceHardwareSupported = \
            self.hardwareStatus_.entropySourceHardwareSupported
      self.secStatus_.entropyServerEnabled = \
            self.hardwareStatus_.entropyServerEnabled
      self.secStatus_.entropyQueueSizeBytes = \
            self.hardwareStatus_.entropyQueueSizeBytes

   @Tac.handler( "entropySourceHardwareEnabled" )
   def handleHardwareEnabled( self ):
      self.handleStatusChange()

   @Tac.handler( "entropySourceHardwareSupported" )
   def handleHardwareSupported( self ):
      self.handleStatusChange()

   @Tac.handler( "entropyServerEnabled" )
   def handleEntropyServerEnabled( self ):
      self.handleStatusChange()

   @Tac.handler( "entropyQueueSizeBytes" )
   def handleQueueSize( self ):
      self.handleStatusChange()

   def close( self ):
      Tac.Notifiee.close( self )

class HavegedEntropyReactor( Tac.Notifiee ):
   notifierTypeName = "Mgmt::Security::HavegedStatus"

   def __init__( self, havegedEntropyStatus, secStatus ):
      Tac.Notifiee.__init__( self, havegedEntropyStatus )
      self.secStatus_ = secStatus
      self.havegedStatus_ = havegedEntropyStatus
      self.handleHavegedEntropy()

   @Tac.handler( 'entropySourceHavegedEnabled' )
   def handleHavegedEntropy( self ):
      self.secStatus_.entropySourceHavegedEnabled = \
            self.havegedStatus_.entropySourceHavegedEnabled

   def close( self ):
      Tac.Notifiee.close( self )

class JitterEntropyReactor( Tac.Notifiee ):
   notifierTypeName = "Mgmt::Security::JitterStatus"

   def __init__( self, havegedEntropyStatus, secStatus ):
      Tac.Notifiee.__init__( self, havegedEntropyStatus )
      self.secStatus_ = secStatus
      self.havegedStatus_ = havegedEntropyStatus
      self.handleJitterEntropy()

   @Tac.handler( 'entropySourceJitterEnabled' )
   def handleJitterEntropy( self ):
      self.secStatus_.entropySourceJitterEnabled = \
            self.havegedStatus_.entropySourceJitterEnabled

   def close( self ):
      Tac.Notifiee.close( self )

class EntropySm( SuperServer.SuperServerAgent ):
   def __init__( self, entityManager ):
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      self.warm_ = False
      mg = entityManager.mountGroup()

      self.hardwareEntropyStatus = \
            mg.mount( Cell.path( 'mgmt/security/hardwareStatus' ),
                                 'Mgmt::Security::HardwareStatus', 'r' )
      self.havegedEntropyStatus = \
            mg.mount( Cell.path( 'mgmt/security/havegedStatus' ),
                                 'Mgmt::Security::HavegedStatus', 'r' )
      self.jitterEntropyStatus = \
            mg.mount( Cell.path( 'mgmt/security/jitterStatus' ),
                                 'Mgmt::Security::JitterStatus', 'r' )
      self.secStatus = mg.mount( Cell.path( 'mgmt/security/status' ),
                                            'Mgmt::Security::Status', 'w' )

      self.hardwareSecurityChipNotifiee_ = None
      self.hardwareNotifiee_ = None
      self.havegedNotifiee_ = None
      self.jitterNotifiee_ = None

      def _finished():
         self.hardwareSecurityChipNotifiee_ = HardwareSecurityChipReactor(
                                       self.hardwareEntropyStatus, self.secStatus )

         # run other SMs only if active
         if self.active():
            self.onSwitchover( None )

      mg.close( _finished )

   def onSwitchover( self, protocol ):
      self.hardwareNotifiee_ = HardwareEntropyReactor( self.hardwareEntropyStatus,
                                                       self.secStatus )
      self.havegedNotifiee_ = HavegedEntropyReactor( self.havegedEntropyStatus,
                                                     self.secStatus )
      self.jitterNotifiee_ = JitterEntropyReactor( self.jitterEntropyStatus,
                                                   self.secStatus )
      self.warm_ = True

   def warm( self ):
      if not self.active():
         return True
      return self.warm_

def Plugin( context ):
   context.registerService( EntropySm( context.entityManager ) )
