# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import PyWrappers.Haveged as haveged
import SuperServer
import Tac
import Cell
import Tracing

t0 = Tracing.trace0

class HavegedService( SuperServer.LinuxService ):
   notifierTypeName = "Mgmt::Security::Config"

   def __init__( self, mgmtConfig, havegedStatus ):
      self.config_ = mgmtConfig
      self.status_ = havegedStatus
      self.started_ = False
      SuperServer.LinuxService.__init__( self, haveged.name(),
                                         haveged.name(), self.config_,
                                         "/etc/default/haveged" )
      self.handleEntropySourceHaveged()

   def startService( self ):
      t0( "Starting haveged service" )
      if not self.started_:
         SuperServer.LinuxService.startService( self )
         self.started_ = True
         self.status_.entropySourceHavegedEnabled = True

   def stopService( self ):
      t0( "Stopping haveged service" )
      if self.started_:
         SuperServer.LinuxService.stopService( self )
         self.started_ = False
         self.status_.entropySourceHavegedEnabled = False

   def serviceEnabled( self ):
      return self.started_ and self.config_.entropySourceHaveged

   def conf( self ):
      return ""

   def serviceProcessWarm( self ):
      return self.serviceEnabled()

   @Tac.handler( 'entropySourceHaveged' )
   def handleEntropySourceHaveged( self ):
      t0( "Handling haveged service" )
      if self.config_.entropySourceHaveged:
         self.startService()
      else:
         self.stopService()

class Haveged( SuperServer.SuperServerAgent ):
   def __init__( self, entityManager ):
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      self.warm_ = False
      mg = entityManager.mountGroup()
      self.mgmtConfig = mg.mount( 'mgmt/security/config', 
                                  'Mgmt::Security::Config', 'r' )
      self.havegedStatus = mg.mount( Cell.path( 'mgmt/security/havegedStatus' ),
                                     'Mgmt::Security::HavegedStatus', 'w' )
      self.service_ = None

      def _finished():
         # run only if active
         if self.active():
            self.onSwitchover( None )

      mg.close( _finished )

   def onSwitchover( self, protocol ):
      self.service_ = HavegedService( self.mgmtConfig, self.havegedStatus )
      self.warm_ = True

   def warm( self ):
      if not self.active():
         return True
      return self.warm_

def Plugin( ctx ):
   ctx.registerService( Haveged( ctx.entityManager ) )
