# Copyright (c) 2005-2011, 2014-2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

'''
The LogMgr is responsible for pushing the logging configuration state
into the underlying system files, specifically /etc/rsyslog.conf.

It does this by reacting to the LogConfig object, writing the
configuration state into /etc/rsyslog.conf in the format understood by
syslogd, and then restarting syslogd.
'''

import Cell
import glob
import grp
import Logging
import LoggingDefs
import LoggingLib
import os
import PyWrappers.Rsyslog as rsyslog
import RsyslogTemplate as TMPL
import QuickTrace
import socket
import SuperServer
import sys
import Tac
import time
import Tracing
import weakref
import Aresolve
import IpUtils
import Arnet
import CEosHelper
import stat
import EosVersion
import MgmtSecuritySslStatusSm
from TypeFuture import TacLazyType
from BothTrace import traceXX as bt

CbStrategy = Aresolve.CallbackStrategy

# Needed to react to MatchList tac entities
# pkgdeps: library MatchList

# pylint: disable-msg=W1401
traceHandle = Tracing.Handle( "LogMgr" )
t0 = traceHandle.trace0
t1 = traceHandle.trace1
t3 = traceHandle.trace3
t5 = traceHandle.trace5 # vrf related config handling
t6 = traceHandle.trace6 # warmup-related debugging
qv = QuickTrace.Var
qt0 = QuickTrace.trace0
qt2 = QuickTrace.trace2
bt00 = lambda msg: bt( 0, 0, msg )
bt50 = lambda msg: bt( 5, 0, msg ) # DNS querier setup and changes
bt52 = lambda msg: bt( 5, 2, msg ) # Periodic DNS query results

defaultVrf = None
LOGMGR_CONSOLE_LOGGING_ERROR = None
LOGMGR_TIMESTAMP_FORMAT_ERROR = None
LOGMGR_TCP_SOURCE_INTERFACE_UNSUPPORTED = None
LOGMGR_SOURCE_ADDRESS_UNASSIGNED = None
LOGMGR_SOURCE_ADDRESS_ASSIGNED_TO_INACTIVE_INTERFACE = None
LOGMGR_TCP_SOURCE_ADDRESS_BOUND_TO_VIRTUAL_INTERFACE = None

SYS_SYSTEM_RESTARTED = None

# Period between DNS queries made by Aresolve for existing queries
DNS_RETRY_PERIOD = 120

SslConstants = TacLazyType( "Mgmt::Security::Ssl::Constants" )
ProfileState = TacLazyType( "Mgmt::Security::Ssl::ProfileState" )
SslFeature = TacLazyType( "Mgmt::Security::Ssl::SslFeature" )

def isValidIp( ip ):
   """Returns an int, 4 or 6 if ip is an IP address of that family,
   else None."""
   try:
      IpUtils.IpAddress( ip )
      return 4
   except ValueError:
      # Invalid or IPv6
      try:
         IpUtils.IpAddress( ip, addrFamily=socket.AF_INET6 )
         return 6
      except ValueError:
         return None

def managementIntf():
   """ Returns Management0 if modular, else Management1 (for fixed system,
   generic [cvx, veos, etc]"""
   return 'Management0' if Cell.cellType() == 'supervisor' else 'Management1'

LOGMGR_CONSOLE_LOGGING_ERROR = Logging.LogHandle(
   "LOGMGR_CONSOLE_LOGGING_ERROR",
   severity=Logging.logWarning,
   fmt="An error occurred initializing console logging",
   explanation="Console logging may not function properly because an error "
               "occurred while setting access permissions.",
   recommendedAction=Logging.CALL_SUPPORT_IF_PERSISTS )

LOGMGR_TIMESTAMP_FORMAT_ERROR = Logging.LogHandle(
   "LOGMGR_TIMESTAMP_FORMAT_ERROR",
   severity=Logging.logError,
   fmt="Unknown timestamp format '%s' was specified",
   explanation="Logging to may not function as expected "
               "because an unknown timestamp format was requested.  A default"
               "timestamp format will be used instead",
   recommendedAction=Logging.CALL_SUPPORT_IF_PERSISTS )

LOGMGR_TCP_SOURCE_INTERFACE_UNSUPPORTED = Logging.LogHandle(
   "LOGMGR_TCP_SOURCE_INTERFACE_UNSUPPORTED",
   severity=Logging.logWarning,
   fmt="Source-Interface spoofing requested with TCP.",
   explanation="Configuring logging source-interface with a "
               "TCP host is not supported. The source-interface "
               "option will be ignored.",
   recommendedAction="Please choose either UDP with source-interface spoofing "
                     "or TCP without source-interface spoofing." )

LOGMGR_SOURCE_ADDRESS_UNASSIGNED = Logging.LogHandle(
   "LOGMGR_SOURCE_ADDRESS_UNASSIGNED",
   severity=Logging.logWarning,
   fmt="Source-Address %s is not assigned to any interface in VRF %s.",
   explanation="Configured logging source-address is unassigned to any interface "
               "in the configured VRF. The source-address option will be ignored "
               "until it is assigned to an interface on the configured VRF.",
   recommendedAction="Please assign the configured source-address to an "
                     "interface on the configured VRF." )

LOGMGR_SOURCE_ADDRESS_ASSIGNED_TO_INACTIVE_INTERFACE = Logging.LogHandle(
   "LOGMGR_SOURCE_ADDRESS_ASSIGNED_TO_INACTIVE_INTERFACE",
   severity=Logging.logWarning,
   fmt="Source-Address %s is assigned to an inactive interface %s in VRF %s.",
   explanation="Configured logging source-address is assigned to an inactive "
               "interface on the configured VRF. TCP hosts won't received syslog "
               "from a inactive interface.",
   recommendedAction="Please assign the configured source-address to an "
                     "active interface on the configured VRF." )

LOGMGR_TCP_SOURCE_ADDRESS_BOUND_TO_VIRTUAL_INTERFACE = Logging.LogHandle(
   "LOGMGR_TCP_SOURCE_ADDRESS_BOUND_TO_VIRTUAL_INTERFACE",
   severity=Logging.logWarning,
   fmt="Source-Address %s is assigned to a virtual interface %s in VRF %s "
       "with TCP host(s).",
   explanation="Configuring logging source-address with TCP host(s) require the "
                 "interface to be a physical interface with valid Layer 2 "
                 "configurations.",
   recommendedAction="Please assign the configured source-address to a physical "
                     "interface on the configured VRF." )

SYS_SYSTEM_RESTARTED = Logging.LogHandle(
   "SYS_SYSTEM_RESTARTED",
   severity=Logging.logNotice,
   fmt="System restarted",
   explanation="The system has restarted",
   recommendedAction=Logging.NO_ACTION_REQUIRED )

SYS_SYSTEM_INFO = Logging.LogHandle(
   "SYS_SYSTEM_INFO",
   severity=Logging.logInfo,
   fmt="%s",
   explanation="[ informational log message ]",
   recommendedAction=Logging.NO_ACTION_REQUIRED )

LOGMGR_TLS_MISSING_TRUSTED_CERTIFICATE = Logging.LogHandle(
   "LOGMGR_TLS_MISSING_TRUSTED_CERTIFICATE",
   severity=Logging.logWarning,
   fmt="No trusted certificate is configured for SSL profile %s.",
   explanation="No trusted certificate is configured. TLS settings will be ignored.",
   recommendedAction="Please add valid trusted certificate to the configured "
                     "SSL profile." )

class VrfStateReactor( Tac.Notifiee ):
   notifierTypeName = 'Ip::VrfStatusLocal'

   def __init__( self, vrfStatusLocal, vrfName, configReactor ):
      Tac.Notifiee.__init__( self, vrfStatusLocal )
      self.vrfName = vrfName
      self.configReactor = configReactor
      self.handleVrfState()

   @Tac.handler( 'state' )
   def handleVrfState( self ):
      state = self.notifier_.state
      vrfName = self.notifier_.vrfName
      qt2( "VrfStateReactor:: state: VRF ", qv( vrfName ), " is ", qv( state ) )
      if vrfName in self.configReactor.loggingConfig.vrfLoggingHost:
         self.configReactor.sync()

   def close( self ):
      qt2( "VrfStateReactor:: VSL for VRF ", qv( self.vrfName ), " is removed" )
      if self.vrfName in self.configReactor.loggingConfig.vrfLoggingHost:
         self.configReactor.sync()
      Tac.Notifiee.close( self )

class SrcIntfVrfReactor:
   def __init__( self, intf, vrfName, configReactor, ipIntfStatus, ip6IntfStatus ):
      self.vrfName = vrfName
      self.intf = intf
      self.ipIntfStatus = ipIntfStatus
      self.ip6IntfStatus = ip6IntfStatus
      self.configReactor = weakref.proxy( configReactor )
      qt2( "LogMgr::LogConfig VRF ", qv( vrfName ),
           " source interface added: ", qv( intf ) )
      reactorFilter = lambda x: x == self.intf

      self.ipStatusReactor_ = Tac.collectionChangeReactor( self.ipIntfStatus,
                                  IpIntfStatusReactor, reactorFilter=reactorFilter,
                                  reactorArgs=( self.configReactor, self.vrfName, ) )
      self.ipStatusReactor_.inCollection = True
      self.ip6StatusReactor_ = Tac.collectionChangeReactor( self.ip6IntfStatus,
                                  Ip6IntfStatusReactor, reactorFilter=reactorFilter,
                                  reactorArgs=( self.configReactor, self.vrfName, ) )
      self.ip6StatusReactor_.inCollection = True

   def close( self ):
      qt2( "LogMgr::LogConfig VRF ", qv( self.vrfName ),
           " source interface removed" )
      self.ipStatusReactor_.close()
      self.ip6StatusReactor_.close()
      del self.ipStatusReactor_
      del self.ip6StatusReactor_

class IpIntfStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Ip::IpIntfStatus'

   def __init__( self, ipIntfStatus, configReactor, vrfName ):
      Tac.Notifiee.__init__( self, ipIntfStatus )
      self.vrfName = vrfName
      self.configReactor = configReactor
      self.loggingStatus = self.configReactor.loggingStatus
      self.l3StatusReactor = L3StatusReactor( self.notifier_.l3Status,
                                             self.handleVrfOrActiveAddrWithMask )
      self.handleVrfOrActiveAddrWithMask( )

   @Tac.handler( 'activeAddrWithMask' )
   def handleVrfOrActiveAddrWithMask( self ):
      activeAddrWithMask = self.notifier_.activeAddrWithMask
      qt2( "IpIntfStatusReactor:: VRF: ", qv( self.vrfName ) )
      qt2( "IpIntfStatusReactor:: addr: ", qv( activeAddrWithMask.address ) )
      srcIntfStatus = self.loggingStatus.srcIntfStatus.get( self.vrfName )
      if srcIntfStatus is not None:
         address = srcIntfStatus.activeAddrWithMask.address
         qt2( "LogMgr::LogStatus old intfAddr: ", qv( address ) )
      notifierVrf = self.notifier_.vrf
      if notifierVrf == '':
         notifierVrf = defaultVrf

      t5( "handleVrfOrActiveAddrWithMask: " + notifierVrf
          + ", " + activeAddrWithMask.address )

      if self.vrfName == notifierVrf and \
         activeAddrWithMask.address != '0.0.0.0':
         self.loggingStatus.srcIntfStatus[ self.vrfName ] = self.notifier_
         self.configReactor.sync()
      else:
         if self.vrfName in self.loggingStatus.srcIntfStatus:
            del self.loggingStatus.srcIntfStatus[ self.vrfName ]
            self.configReactor.sync()

   def close( self ):
      del self.loggingStatus.srcIntfStatus[ self.vrfName ]
      self.configReactor.sync()
      Tac.Notifiee.close( self )

class Ip6IntfStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Ip6::IntfStatus"

   def __init__( self, ip6IntfStatus, configReactor, vrfName ):
      Tac.Notifiee.__init__( self, ip6IntfStatus )
      self.vrfName = vrfName
      self.configReactor = configReactor
      self.loggingStatus = self.configReactor.loggingStatus
      self.l3StatusReactor = L3StatusReactor( self.notifier_.l3Status,
                                              self.handleVrf )
      self.handleVrf()

   def handleVrfOrAddr( self ):
      notifierVrf = self.notifier_.vrf if self.notifier_.vrf else defaultVrf

      if self.vrfName == notifierVrf and len( self.notifier_.addr ):
         self.loggingStatus.srcIntf6Status[ self.vrfName ] = self.notifier_
         self.configReactor.sync()
      elif self.vrfName in self.loggingStatus.srcIntf6Status:
         del self.loggingStatus.srcIntf6Status[ self.vrfName ]
         self.configReactor.sync()

   @Tac.handler( 'addr' )
   def handleAddr( self, key ):
      qt2( "Ip6IntfStatusReactor:: addr: ", qv( key ) )
      self.handleVrfOrAddr()

   def handleVrf( self ):
      qt2( "Ip6IntfStatusReactor:: vrf" )
      self.handleVrfOrAddr()

   def close( self ):
      del self.loggingStatus.srcIntf6Status[ self.vrfName ]
      self.configReactor.sync()
      Tac.Notifiee.close( self )

class L3StatusReactor( Tac.Notifiee ):
   notifierTypeName = "L3::Intf::Status"
   def __init__( self, l3Status, reactorFunction ):
      self.reactorFunction_ = reactorFunction
      Tac.Notifiee.__init__( self, l3Status )

   @Tac.handler( 'vrf' )
   def handleVrf( self ):
      self.reactorFunction_()

# Reactor for 'logging [VRF] source-address [IPv4 Address]' command
# Attempts to match the configured source-address to an IP interface
# Filters IpIntfStatus by the VRF configured for the source-address
class SrcIpAddrVrfReactor:
   def __init__( self, ip, vrfName, configReactor, vrfIpIntfStatus ):
      self.vrfName = vrfName
      self.ip = ip
      self.vrfIpIntfStatus = vrfIpIntfStatus
      self.configReactor = weakref.proxy( configReactor )
      self.loggingStatus = self.configReactor.loggingStatus
      qt2( "LogMgr::LogConfig VRF ", qv( vrfName ),
           " source address added: ", qv( ip ) )

      reactorFilter = lambda x: x == self.vrfName
      self.vrfIpStatusReactor_ = Tac.collectionChangeReactor( self.vrfIpIntfStatus,
                                     VrfIpIntfStatusReactor,
                                     reactorFilter=reactorFilter,
                                     reactorArgs=( self.configReactor, self.ip, ) )
      self.vrfIpStatusReactor_.inCollection = True

      if not self.loggingStatus.srcIpIntfName.get( self.vrfName ):
         Logging.log( LOGMGR_SOURCE_ADDRESS_UNASSIGNED, self.ip,
                      self.vrfName )
      # BUG795583: Also syslog if source-address IP is assigned to an inactive intf

   def close( self ):
      qt2( "LogMgr::LogConfig VRF ", qv( self.vrfName ),
           " source address removed: ", qv( self.ip ) )
      self.vrfIpStatusReactor_.close()
      del self.vrfIpStatusReactor_

# Reactor for 'logging [VRF] source-address [IPv6 Address]' command
# Attempts to match the configured source-address to an IP6 interface
# Filters Ip6IntfStatus by the VRF configured for the source-address
class SrcIp6AddrVrfReactor:
   def __init__( self, ip6, vrfName, configReactor, vrfIp6IntfStatus ):
      self.vrfName = vrfName
      self.ip6 = ip6
      self.vrfIp6IntfStatus = vrfIp6IntfStatus
      self.configReactor = weakref.proxy( configReactor )
      self.loggingStatus = self.configReactor.loggingStatus
      qt2( "LogMgr::LogConfig VRF ", qv( vrfName ),
           " source address added: ", qv( ip6 ) )

      reactorFilter = lambda x: x == self.vrfName
      self.vrfIp6StatusReactor_ = Tac.collectionChangeReactor( self.vrfIp6IntfStatus,
                                      VrfIp6IntfStatusReactor,
                                      reactorFilter=reactorFilter,
                                      reactorArgs=( self.configReactor, self.ip6, ) )
      self.vrfIp6StatusReactor_.inCollection = True

      if not self.loggingStatus.srcIp6IntfName.get( self.vrfName ):
         Logging.log( LOGMGR_SOURCE_ADDRESS_UNASSIGNED, self.ip6,
                      self.vrfName )
      # BUG795583: Also syslog if source-address Ip6 is assigned to an inactive intf

   def close( self ):
      qt2( "LogMgr::LogConfig VRF ", qv( self.vrfName ),
           " source address removed: ", qv( self.ip6 ) )
      self.vrfIp6StatusReactor_.close()
      del self.vrfIp6StatusReactor_

# Nested reactor for 'logging [VRF] source-address [IPv4 Address]' command
# Attempts to match the configured source-address to an IP interface
# Reacts to all IpIntfStatus changes in the VRF configured for the source-address
class VrfIpIntfStatusReactor:
   def __init__( self, vrfIpIntfStatus, configReactor, ip ):
      self.ip = ip
      self.vrfIpIntfStatus = vrfIpIntfStatus
      self.vrfName = vrfIpIntfStatus.vrfName
      self.configReactor = configReactor

      self.ipIntfStatusReactorForIpAddr_ = Tac.collectionChangeReactor(
                                               self.vrfIpIntfStatus.ipIntfStatus,
                                               IpIntfAddrReactor,
                                               reactorArgs=( self.configReactor,
                                                             self.ip,
                                                             self.vrfName ) )
      self.ipIntfStatusReactorForIpAddr_.inCollection = True

   def close( self ):
      self.ipIntfStatusReactorForIpAddr_.close()
      del self.ipIntfStatusReactorForIpAddr_

# Nested reactor for 'logging [VRF] source-address [IPv6 Address]' command
# Attempts to match the configured source-address to an IP6 interface
# Reacts to all Ip6IntfStatus changes in the VRF configured for the source-address
class VrfIp6IntfStatusReactor:
   def __init__( self, vrfIp6IntfStatus, configReactor, ip6 ):
      self.ip6 = ip6
      self.vrfIp6IntfStatus = vrfIp6IntfStatus
      self.vrfName = self.vrfIp6IntfStatus.vrfName
      self.configReactor = configReactor

      self.ip6IntfStatusReactorForIp6Addr_ = Tac.collectionChangeReactor(
                                                 self.vrfIp6IntfStatus.ip6IntfStatus,
                                                 Ip6IntfAddrReactor,
                                                 reactorArgs=( self.configReactor,
                                                               self.ip6,
                                                               self.vrfName ) )
      self.ip6IntfStatusReactorForIp6Addr_.inCollection = True

   def close( self ):
      self.ip6IntfStatusReactorForIp6Addr_.close()
      del self.ip6IntfStatusReactorForIp6Addr_

# Doubly nested reactor for 'logging [VRF] source-address [IPv4 Address]' command
# Attempts to match the configured source-address to an IP interface
# Reacts to all activeAddrWithMask attribute change for every IpIntfStatus in the VRF
# configured for the source-address
class IpIntfAddrReactor( Tac.Notifiee ):
   notifierTypeName = 'Ip::IpIntfStatus'

   def __init__( self, ipIntfStatus, configReactor, ip, vrfName ):
      Tac.Notifiee.__init__( self, ipIntfStatus )
      self.ip = ip
      self.vrfName = vrfName
      self.configReactor = configReactor
      self.loggingStatus = self.configReactor.loggingStatus
      self.vrfLoggingTCP = self.configReactor.vrfLoggingTCP
      self.handleActiveAddrWithMask()

   @Tac.handler( 'activeAddrWithMask' )
   def handleActiveAddrWithMask( self ):
      ipIntfStatusAddrWithMask = self.notifier_.activeAddrWithMask
      qt2( "IpIntfAddrReactor:: VRF: ", qv( self.vrfName ) )
      qt2( "IpIntfAddrReactor:: addr: ", qv( ipIntfStatusAddrWithMask ) )

      intfName = self.notifier_.intfId
      oldIntfName = self.loggingStatus.srcIpIntfName.get( self.vrfName )
      qt2( "LogMgr::LogStatus old srcIpIntfName: ", qv( oldIntfName ) )

      t5( "handleActiveAddrWithMask: " + self.vrfName +
          ", " + intfName + ", " + ipIntfStatusAddrWithMask.stringValue )

      if ipIntfStatusAddrWithMask.address != "0.0.0.0" and \
         ipIntfStatusAddrWithMask.address == self.ip:
         self.loggingStatus.srcIpIntfName[ self.vrfName ] = intfName
         vrfHasTCPHost = self.vrfLoggingTCP.get( self.vrfName )
         if vrfHasTCPHost:
            if "Loopback" in intfName:
               Logging.log( LOGMGR_TCP_SOURCE_ADDRESS_BOUND_TO_VIRTUAL_INTERFACE,
                            self.ip, intfName, self.vrfName )
         self.configReactor.sync()
      elif oldIntfName == intfName:
         del self.loggingStatus.srcIpIntfName[ self.vrfName ]
         self.configReactor.sync()

   def close( self ):
      oldIntfName = self.loggingStatus.srcIpIntfName.get( self.vrfName )
      intfName = self.notifier_.intfId

      if oldIntfName == intfName:
         del self.loggingStatus.srcIpIntfName[ self.vrfName ]
         self.configReactor.sync()

      Tac.Notifiee.close( self )

# Doubly nested reactor for 'logging [VRF] source-address [IPv6 Address]' command
# Attempts to match the configured source-address to an IP6 interface
# Reacts to all addr attribute change for every Ip6IntfStatus in the VRF
# configured for the source-address
class Ip6IntfAddrReactor( Tac.Notifiee ):
   notifierTypeName = "Ip6::IntfStatus"

   def __init__( self, ip6IntfStatus, configReactor, ip6, vrfName ):
      Tac.Notifiee.__init__( self, ip6IntfStatus )
      self.ip6 = ip6
      self.vrfName = vrfName
      self.configReactor = configReactor
      self.loggingStatus = self.configReactor.loggingStatus
      self.vrfLoggingTCP = self.configReactor.vrfLoggingTCP
      self.handleAddr( None )

   @Tac.handler( 'addr' )
   def handleAddr( self, _ ):
      intfEqualsIp6 = False
      if len( self.notifier_.addr ):
         for addrWithMask in self.notifier_.addr:
            if addrWithMask.address.stringValue == self.ip6.stringValue:
               intfEqualsIp6 = True
               break

      oldIntfName = self.loggingStatus.srcIp6IntfName.get( self.vrfName )
      qt2( "LogMgr::LogStatus old srcIp6 intfName: ", qv( oldIntfName ) )
      intfName = self.notifier_.intfId

      if intfEqualsIp6:
         self.loggingStatus.srcIp6IntfName[ self.vrfName ] = intfName
         vrfHasTCPHost = self.vrfLoggingTCP.get( self.vrfName )
         if vrfHasTCPHost:
            if "Loopback" in intfName:
               Logging.log( LOGMGR_TCP_SOURCE_ADDRESS_BOUND_TO_VIRTUAL_INTERFACE,
                            self.ip6, intfName, self.vrfName )
         self.configReactor.sync()
      elif oldIntfName == intfName:
         del self.loggingStatus.srcIp6IntfName[ self.vrfName ]
         self.configReactor.sync()

   def close( self ):
      oldIntfName = self.loggingStatus.srcIp6IntfName.get( self.vrfName )
      intfName = self.notifier_.intfId

      if oldIntfName == intfName:
         del self.loggingStatus.srcIp6IntfName[ self.vrfName ]
         self.configReactor.sync()

      Tac.Notifiee.close( self )

class VrfLoggingHostReactor( Tac.Notifiee ):
   notifierTypeName = 'LogMgr::VrfLoggingHost'

   def __init__( self, vrfLoggingHost, vrfName, configReactor ):
      Tac.Notifiee.__init__( self, vrfLoggingHost )
      self.vrfName = vrfName
      self.configReactor = weakref.proxy( configReactor )
      self.vrfLoggingTCP = self.configReactor.vrfLoggingTCP

   @Tac.handler( 'loggingHost' )
   def handleLoggingHost( self, host ):
      loggingHost = self.notifier_.loggingHost.get( host )
      protocol = None
      if loggingHost:
         protocol = loggingHost.protocol
         if protocol == "tcp":
            self.vrfLoggingTCP[ self.notifier_.vrfName ] = True

         qt2( "VrfLoggingHostReactor:: VRF: ", qv( self.vrfName ),
              " Host: ", qv( host ), " Protocol: ", qv( protocol ), " added" )
      else:
         qt2( "VrfLoggingHostReactor:: VRF: ", qv( self.vrfName ),
              " Host: ", qv( host ), "removed" )

      if protocol != "tcp":
         vrfHasTCPHost = False
         for anyHost in self.notifier_.loggingHost.values():
            if anyHost.protocol == "tcp":
               vrfHasTCPHost = True
               break

         if vrfHasTCPHost:
            for vrf, intfName in \
                self.configReactor.loggingStatus.srcIpIntfName.items():
               if "Loopback" in intfName:
                  ip = self.configReactor.loggingConfig.srcIpAddress[ vrf ]
                  Logging.log( LOGMGR_TCP_SOURCE_ADDRESS_BOUND_TO_VIRTUAL_INTERFACE,
                               ip, intfName, vrf )
            for vrf, intfName in \
                self.configReactor.loggingStatus.srcIp6IntfName.items():
               if "Lookback" in intfName:
                  ip6 = self.configReactor.loggingConfig.srcIp6Address[ vrf ]
                  Logging.log( LOGMGR_TCP_SOURCE_ADDRESS_BOUND_TO_VIRTUAL_INTERFACE,
                               ip6, intfName, vrf )

         self.vrfLoggingTCP[ self.notifier_.vrfName ] = vrfHasTCPHost

      self.configReactor.sync()

   def close( self ):
      qt2( "VrfLoggingHostReactor:: VRF: ", qv( self.vrfName ), " removed" )
      self.vrfLoggingTCP.pop( self.vrfName, None )
      self.configReactor.sync()
      Tac.Notifiee.close( self )

class MatchStringListReactor( Tac.Notifiee ):
   notifierTypeName = 'MatchList::MatchStringListConfig'

   def __init__( self, matchStringListConfig, configReactor ):
      Tac.Notifiee.__init__( self, matchStringListConfig )
      self.configReactor = weakref.proxy( configReactor )
      self.configReactor.sync()

   @Tac.handler( 'matchInfo' )
   def handleMatchInfo( self, _ ):
      self.configReactor.sync()

   def close( self ):
      self.configReactor.sync()
      Tac.Notifiee.close( self )

class TcpBadAuthSyslogConf :
   """Generate /etc/rsyslog.conf entries for dispatching BGP BADAUTH logs"""

   # most all of the following class-specific attributes are static data which
   #  are the "parts" used to programmatically assemble templates and rules

   _prefixBGP = 'BGP_BADAUTH'
   _prefixLDP = 'LDP_BADAUTH'
   _ldpPort = 646

   # "_templates" is the list of kernel messages we are interested in, their
   # "industry-standard" counterparts, and the "kernel format" (v4 ? v6?)
   # they originally appear in. We have to take care of the following
   # idiosyncracies: a) the kernel formats the password-mimatch log differntly
   # for v4 and v6 b) the kernal has two log messages for the v6 password-
   # mismatch case; one of them appears to be an internal API error and
   # presumably should ideally not surface in logs, but we've got to cover
   # that as well in case it does, and c) the kernel does not produce v6 logs
   # for the cases where only one peer has configured a password. These aspects
   # are reflected in the structure of the _templates list, which is:
   # [NAME, (v4/v6), translated_txt, original_txt]

   _templates = [['NOTFOUND',('V4'), "No MD5 digest", "MD5 Hash not found"],
                 ['UNEXPECTED', ('V4'), "Unexpected MD5 digest",
                  "Unexpected MD5 Hash found"],
                 ['FAILED', ('V4'), "Invalid MD5 digest", "MD5 Hash failed"],
                 ['MISMATCH', ('V6'), "Invalid MD5 digest", "MD5 Hash mismatch"]]

   # The following are the regular expressions and match-numbers to be used in
   # rsyslog "property-replacer". They go into template defintions. The inner-most
   # list is: [1st_match_number, 2nd_match_number, regexp]

   _addrParser = { 'V4': [ 0, 1, r'\(([^,]*)' ], 'V6': [ 1, 2, r'\[([^]]*)' ] }
   _portParser = { 'V4': [ 0, 1, ', ([0-9]*)' ], 'V6': [ 0, 1, r'\]\:([0-9]*)' ] }

   # "_discriminator" is a string that helps us distinguish between kernel's v4-
   # and v6-format messages. It goes into 'selector' part of rsyslog rule

   _discriminator = {'V4' : ', ', 'V6' : ']:'}

   # A "template" for generating templates, and one for generating rules. When
   # format specifiers in these strings are replaced with actual values, we'll
   # have fully-formed rsyslog templates and rules

   _genericTBgp = \
         """$template %s,\"%s%s #HOSTNAME_PLACEHOLDER# %sBgp: \\%%TCP-6-BADAUTH: \
%s from %%msg:R,ERE,1,BLANK,%d:%s--end%%\
(%%msg:R,ERE,1,BLANK,%d:%s--end%%) \
to %%msg:R,ERE,1,BLANK,%d:%s--end%%\
(%%msg:R,ERE,1,BLANK,%d:%s--end%%)\\n\"
"""

   _genericTLdp = \
         """$template %s,\"%s%s #HOSTNAME_PLACEHOLDER# %sLdp: \\%%TCP-6-BADAUTH: \
%s from %%msg:R,ERE,1,BLANK,%d:%s--end%%\
(%%msg:R,ERE,1,BLANK,%d:%s--end%%) \
to %%msg:R,ERE,1,BLANK,%d:%s--end%%\
(%%msg:R,ERE,1,BLANK,%d:%s--end%%)\\n\"
"""

   _genericRBgp = """if %s$syslogfacility-text == 'kern' and $msg contains %s \
and $msg contains %s and not ($msg contains ', %s)') then %s"""

   _genericRLdp = """if %s$syslogfacility-text == 'kern' and $msg contains %s \
and $msg contains %s and $msg contains ', %s)' then %s"""


   # "_list" is generated at run-time using the static data above.
   # It contains names of all the templates that will be written to
   # rsyslog.conf. The inner-most hash contains key-value pairs of
   # the format: {template_name : [translated_msg, original_msg]}

   _listBGP = { 'LOCAL' : {'V4': {}, 'V6': {} },
             'REMOTE' : {'V4': {}, 'V6': {} } }
   _listLDP = { 'LOCAL' : {'V4': {}, 'V6': {} },
             'REMOTE' : {'V4': {}, 'V6': {} } }
   _list = [ [ _listBGP, _genericTBgp, _prefixBGP, _genericRBgp ],
             [ _listLDP, _genericTLdp, _prefixLDP, _genericRLdp ] ]

   def __init__(self) :
      # pylint: disable-next=too-many-nested-blocks
      for protoList, _, protoPrefix, _ in self._list:
         for loc, verList in protoList.items():
            for ver in verList :
               for t in self._templates :
                  if ver in t[1] :
                     if loc == 'LOCAL' :
                        name = ( f"{protoPrefix}_{t[0]}_{ver}" )
                     else :
                        name = ( f"{protoPrefix}_{t[0]}_{ver}_{loc}" )
                     # pylint: disable-next=unnecessary-dict-index-lookup
                     protoList[loc][ver][name] = [t[2], t[3]]

   def generateTemplates(self, priStr, tsFormat, peer=False, peerTag='') :

      templates = ""
      peerStr = '_PEER' if peer else ''
      for protoList, protoTemplate, _, _ in self._list:
         for loc, verList in protoList.items():
            if loc == 'LOCAL' :
               pri = ''
            else :
               pri = priStr
            for ver, nameList in verList.items():
               for name, msgs in nameList.items():
                  templates += protoTemplate % (name + peerStr,
                                  pri, tsFormat, peerTag, msgs[0],
                                  self._addrParser[ver][0], self._addrParser[ver][2],
                                  self._portParser[ver][0], self._portParser[ver][2],
                                  self._addrParser[ver][1], self._addrParser[ver][2],
                                  self._portParser[ver][1], self._portParser[ver][2])
                  templates += '\n'
      return templates

   def generateRules(self, targets, tagFilter='', peer=False ) :

      if len(targets) == 0 :
         return ""
      rules = ""
      peerStr = '_PEER' if peer else ''
      for protoList, _, _, protoRule in self._list:
         for ver, nameList in protoList['LOCAL'].items() :
            for name, msgs in nameList.items() :
               associations = []
               for t in targets :
                  if t.startswith( '@' ) or t == ":omudpspoof:":
                     associations.append( f"{t};{name}_REMOTE{peerStr}" )
                  else :
                     associations.append( f"{t};{name}{peerStr}" )
               assocStr = '\n& '.join(associations)
               rules += protoRule % (tagFilter, ("\'%s\'" % msgs[1]),
                        ("\'%s\'" % self._discriminator[ver]), self._ldpPort,
                        assocStr)
               rules += '\n'
      return rules

class ConfigReactorReq( Tac.Notifiee ):
   notifierTypeName = "LogMgr::LogConfigReq"

   def __init__( self, loggingConfig, loggingConfigReq, loggingStatus,
                 configReactor ):
      t5( "ConfigReactorReq initialzed" )
      Tac.Notifiee.__init__( self, loggingConfigReq )
      self.loggingConfig = loggingConfig
      self.loggingConfigReq = loggingConfigReq
      self.loggingStatus = loggingStatus
      self.configReactor = configReactor

   # Handler/Reactor for "clear logging" command
   # It'll remove only the system log files (Ex: /var/log/eos /var/log/eos.*.z)
   # /var/log/messages is not removed, so logs can still be retrieved from here.
   @Tac.handler( 'clearLogRequest' )
   def handleClearLogRequest( self ):
      # If active supe issues clear log command, we do it on
      # standby too. The status is written to local entity,
      # so it is not reported to Sysdb.
      request = self.loggingConfigReq.clearLogRequest
      lastRequest = self.loggingStatus.lastClearLogRequest
      logFilename = self.loggingConfig.bufferedLogFilename
      flashLogFilename = self.loggingConfig.persistentLogFilename

      if request.timestamp <= lastRequest.timestamp:
         return
      startService = False
      if self.configReactor.serviceEnabled():
         self.configReactor.stopService()
         startService = True

      fileMap = { logFilename: "/var/log", flashLogFilename: "/mnt/flash/persist" }
      for fileName, dirPath in fileMap.items():
         # filename is a constAttr with value defined in Log.tac
         # Still play it safe by checking the path
         assert fileName.startswith( dirPath )
         # Truncate the current log file
         with open( fileName, "w" ) as logFile:
            logFile.truncate()
      # Remove logrotated files
      for filename in glob.glob( "%s.*.gz" % logFilename ):
         try:
            os.unlink( filename )
         except OSError as e:
            import errno # pylint: disable=import-outside-toplevel
            if e.errno != errno.ENOENT:
               qt0( "Error %s: Failed to remove log file %s" % \
                   ( qv( e.errno ), qv( filename ) ) )
      self.loggingStatus.lastClearLogRequest = request
      if startService:
         self.configReactor.startService()

class ConfigReactor( SuperServer.SystemdService ):
   notifierTypeName = "LogMgr::LogConfig"

   def __init__( self, loggingConfig, loggingStatus,
                 sysMgrNetStatus, ipStatus, ip6Status,
                 allVrfStatusLocal, matchListConfig,
                 sshConfig, sslConfig, sslStatus,
                 allIntfStatusDir, logMgr, emRoot ):
      assert os.path.exists( "/sbin/rsyslogd" )
      SuperServer.SystemdService.__init__( self, "LogMgr", rsyslog.serviceName(),
                                         loggingConfig, "/etc/rsyslog.conf" )

      resync = not self.writeConfigFile( "/etc/sysconfig/rsyslog",
                                         self.rsyslogSysConfig() )

      self.devLog_ = "/dev/log"
      self.devLogCtimeBeforeRestart_ = None

      self.emRoot = emRoot
      self.loggingConfig = loggingConfig
      self.loggingStatus = loggingStatus
      self.netStatus_ = sysMgrNetStatus
      self.ipStatus = ipStatus
      self.ip6Status = ip6Status
      self.allVrfStatusLocal = allVrfStatusLocal
      self.matchListConfig = matchListConfig
      self.matchListReactor_ = None
      self.sshConfig = sshConfig
      self.sslConfig = sslConfig
      self.sslStatus = sslStatus
      self.allIntfStatusDir = allIntfStatusDir
      self.sslReactor = {}
      self.logMgr = logMgr
      self.vrfLoggingHostReactor_ = {}
      self.vrfLoggingTCP = {}
      # XXX_KARTIK
      # Ideally need to sync up with what's used in openlog
      self.eosFacilityName_ = "local4"
      self.act = None
      # shortTime: how often (seconds) do we want to try resolving new/failed lookups
      # longTime: how often do we want to try reresolving succesful lookups
      # callbackStrategy: strategy used for callback execution
      # useDnsQuerySm: use DnsQuerySm instead of AresolveSm
      # if the result changes
      self.dns_ = Aresolve.Querier( self._dnsResponse, shortTime=1,
                                    longTime=DNS_RETRY_PERIOD,
                                    callbackStrategy=CbStrategy.OnUsedIpMissing,
                                    useDnsQuerySm=True )
      # To keep track of currently used hostnames
      self.syslogHostnames_ = []

      self.rootLoginSev = self.loggingConfig.loggingRootLoginSeverity
      self.rootLoginSevVal = LoggingLib.severityTaccToValue( self.rootLoginSev )

      # TLS global config
      self.configRsyslogOssl = False

      if not self.fixLogFilePerms():
         Logging.log( LOGMGR_CONSOLE_LOGGING_ERROR )

      # Reactor for 'logging source-address [IPv4 Address]' command
      self.srcIpAddrVrfReactor_ = Tac.collectionChangeReactor(
                                    self.loggingConfig.srcIpAddress,
                                    SrcIpAddrVrfReactor,
                                    reactorArgs=( self,
                                                  self.ipStatus.vrfIpIntfStatus ),
                                    reactorTakesKeyArg=True )

      # Reactor for 'logging source-address [IPv6 Address]' command
      self.srcIp6AddrVrfReactor_ = Tac.collectionChangeReactor(
                                    self.loggingConfig.srcIp6Address,
                                    SrcIp6AddrVrfReactor,
                                    reactorArgs=( self,
                                                  self.ip6Status.vrfIp6IntfStatus ),
                                    reactorTakesKeyArg=True )

      # Reactor for 'logging source-interface' command
      self.srcIntfVrfReactor_ = Tac.collectionChangeReactor(
                                    self.loggingConfig.srcIntfName,
                                    SrcIntfVrfReactor,
                                    reactorArgs=( self,
                                                  self.ipStatus.ipIntfStatus,
                                                  self.ip6Status.intf ),
                                    reactorTakesKeyArg=True )

      self.vrfLoggingHostReactor_ = Tac.collectionChangeReactor(
                           self.loggingConfig.vrfLoggingHost,
                           VrfLoggingHostReactor,
                           reactorArgs=( self, ),
                           reactorTakesKeyArg=True )
      self.initMatchListReactor()

      if resync:
         self.sync()

   def _devName( self, intf ):
      intfStatus = self.allIntfStatusDir.intfStatus.get( intf )
      if intfStatus:
         return intfStatus.deviceName
      else:
         return None

   def fixLogFilePerms( self ):
      """Change permissions on the console and buffer log files to allow
      reading by the eosadmin group so that console logging can work for
      non-root users."""
      cfg = self.notifier_
      # Get gid for eosadmin once
      try:
         groupInfo = grp.getgrnam( "eosadmin" )
         gid = groupInfo[ 2 ]
      except KeyError as e:
         qt0( "eosadmin group id not found:", qv( e ) )
         return False

      def _fixFile( filename ):
         qt0( "Fixing file permissions on", qv( filename ), "if necessary" )
         perms = 0o640
         try:
            # Create file if it doesn't exist
            flags = os.O_CREAT | os.O_APPEND
            fd = os.open( filename, flags, perms )
         except OSError as e:
            qt0( "Error occurred opening log file:", qv( e ) )
            return False
         except: # pylint: disable=bare-except
            e = sys.exc_info()[ 1 ]
            qt0( "Error occurred opening log file:", ( e ) )
            return False
         try:
            os.fchown( fd, 0, gid )
            os.fchmod( fd, perms )
         except OSError as e:
            qt0( "Failed to change permissions on log file:", qv( e ) )
            return False
         finally:
            os.close( fd )
         return True

      bufferedFixed = _fixFile( cfg.bufferedLogFilename )
      consoleFixed = _fixFile( LoggingLib.consoleLogFilename )
      consoleSyncFixed = _fixFile( LoggingLib.consoleSyncLogFilename )
      monitorFixed = _fixFile( LoggingLib.monitorLogFilename )
      monitorSyncFixed = _fixFile( LoggingLib.monitorSyncLogFilename )
      persistentFixed = _fixFile( cfg.persistentLogFilename )

      return ( bufferedFixed and consoleFixed and consoleSyncFixed and
               monitorFixed and monitorSyncFixed and persistentFixed )

   def serviceEnabled( self ):
      t5( "serviceEnabled: logging on is", self.loggingConfig.loggingOn )
      return self.loggingConfig.loggingOn

   def maybeLogSystemStart( self ):
      t5( "maybeLogSystemStart enter" )
      def logSystemStart():
         self.act = None
         # Just ad additional check to make sure that if LogMgr has
         # restarted more than once in the last 2 minutes (which would
         # cause more than one activity to be scheduled) I only log
         # the message once.
         t1( "Checking for the status of system restart: %s" % \
               ( self.loggingStatus.systemStartLogged ) )
         if not self.loggingStatus.systemStartLogged:
            Logging.log( SYS_SYSTEM_RESTARTED )
            def opt( var ):
               return var if var is not None else "( unknown )"
            vi = EosVersion.VersionInfo( self.emRoot )
            version = opt( vi.version() )
            modelName = opt( vi.modelNameExtended() )
            serialNumber = opt( vi.serialNum() )
            Logging.log( SYS_SYSTEM_INFO,
                  "Software image version: %s" % ( version ) )
            Logging.log( SYS_SYSTEM_INFO, "Model: %s" % ( modelName ) )
            Logging.log( SYS_SYSTEM_INFO, "Serial number: %s" % ( serialNumber ) )
            self.loggingStatus.systemStartLogged = True

      # If no 'System restarted' message has been logged yet,
      # I'll start an activity to log it. I'm adding a delay
      # of 2 minutes to make sure everything is up.
      if not self.loggingStatus.systemStartLogged:
         t1( "Instantiating a Tac.ClockNotifiee object to register " \
               "the notifiee for later execution" )
         act = Tac.ClockNotifiee( logSystemStart )

         act.timeMin = Tac.now() + 120
         self.act = act

   def startService( self ):
      t5( "startService enter" )
      # For warmupReporting: If the service is already running, then it
      # will not be restarted. Syslog is warm as soon as /dev/log exists.
      self.devLogCtimeBeforeRestart_ = None

      SuperServer.LinuxService.startService( self )
      self.maybeLogSystemStart()

   def restartService( self ):
      # For warmupReporting: remember the ctime of /dev/log before we
      # restart syslogd. Then, we can declare that the system is warm
      # once the ctime has changed.
      self.devLogCtimeBeforeRestart_ = None
      if os.access( self.devLog_, os.F_OK ):
         self.devLogCtimeBeforeRestart_ = os.stat( self.devLog_ ).st_ctime

         if os.access( "/var/run/syslogd.pid", os.F_OK ):
            oldPid = LoggingLib.getRsyslogdPid()
            t3( "Syslog is currently running. mode %s ctime %s pid %s" % \
                ( str( os.stat( self.devLog_ ).st_mode ),
                  str( self.devLogCtimeBeforeRestart_ ),
                  oldPid ) )
         qt0( "Syslog is currently running. %s has ctime %s" % \
             ( qv( self.devLog_ ), qv( str( self.devLogCtimeBeforeRestart_ ) ) ) )
      else:
         qt0( "Syslog is not currently running" )

      SuperServer.LinuxService.restartService( self )
      self.maybeLogSystemStart()

   def serviceProcessWarm( self ):
      # If syslogd is in the process of restarting, then /dev/log can
      # disappear at any time. Therefore, checking for its existence
      # with os.access(), followed by os.stat(), can still result in
      # an OSError exception in os.stat(). We might as well just wrap
      # os.stat() inside a try/except, and assume that the only reason
      # for an exception is the non-existence of /dev/log -- in which
      # case, we're not warm.

      # EOS18: Since /dev/log is now controlled by systemd, it may still exist
      # even if service is disabled. If service is disabled, the we're warm,
      # regardless of /dev/log status.
      if not self.serviceEnabled():
         qt0( "syslog is warm: service is disabled" )
         t6( "syslog is warm: service is disabled" )
         return True
      try:
         curDevLogCtime = os.stat( self.devLog_ ).st_ctime
      except OSError as details:
         if self.serviceEnabled():
            qt0( "syslog is not warm: could not stat '%s'. Reason: %s" % \
                ( qv( self.devLog_ ), qv( details ) ) )
            t6( "syslog is not warm: could not stat ", self.devLog_,
                " because ", details )
            return False

      # EOS18: since /dev/log is now controlled by systemd, we need not check
      # for this status change time.
      #if self.devLogCtimeBeforeRestart_ == curDevLogCtime:
      #   qt0( "syslog is not warm: /dev/log ctime %s has not yet changed" % \
      #       qv( str( self.devLogCtimeBeforeRestart_ ) ) )
      #   return False

      # EOS18: when rsyslogd is restarted, there can be a small delay by the time
      # the pid file is created. Until then wait.
      if not os.path.exists( "/var/run/syslogd.pid" ):
         t6( "syslog is not warm: pid file does not exist" )
         return False

      newPid = LoggingLib.getRsyslogdPid()
      t3( "Syslog has restarted running. mode %s prev %s ctime %s pid %s" % \
             ( str( os.stat( self.devLog_ ).st_mode ),
               str( self.devLogCtimeBeforeRestart_ ),
               str( curDevLogCtime ), newPid ) )
      qt0( "syslog has restarted" )
      return True

   def rsyslogSysConfig( self, netNs='' ):
      confStr = 'SYSLOGD_OPTIONS=""\n'
      if netNs:
         # This macro will allow rsyslogd to get spawned in a network namespace
         # which will allow it to send syslogs to the remote host
         confStr = confStr + 'NSNAME="' + netNs + '"'
      else:
         confStr = confStr + 'NSNAME="default"'
      return confStr

   def syslogVersion( self ):
      return TMPL.syslogVersion( self.notifier_.syslogFormatRFC5424 )

   def syslogPidMsgId( self ):
      return TMPL.syslogPidMsgId( self.notifier_.syslogFormatRFC5424 )

   def timestampFormat( self, v8Syntax=False ):
      cfg = self.notifier_
      if not cfg.syslogFormatRFC5424 and cfg.timestampFormat == 'tsRFC3164':
         year = ''
         timezone = ''
         # Add the year and timezone in the timestamp if enabled.
         if cfg.timestampYear:
            year = '%$year% '
            if v8Syntax:
               year = TMPL.generalProperty( '$year' ) + TMPL.constant( " " )
         if cfg.timestampTimezone:
            timezone = ' ' + self.logMgr.timezoneInfo_.timezoneStr
            if v8Syntax:
               timezone = TMPL.constant( timezone )
         fmt = ( year + '%TIMESTAMP:::date-rfc3164%' + \
                    timezone )
         if v8Syntax:
            fmt = year + TMPL.timestamp( dateFormat='rfc3164' ) + timezone
      elif cfg.syslogFormatRFC5424 or cfg.timestampFormat == 'tsRFC3339':
         fmt = '%TIMESTAMP:::date-rfc3339%'
         if v8Syntax:
            fmt = TMPL.timestamp( dateFormat='rfc3339' )
      else:
         # Warn but ignore unknown enumeration values
         Logging.log( LOGMGR_TIMESTAMP_FORMAT_ERROR, cfg.timestampFormat )
         fmt = '%TIMESTAMP:::date-rfc3164%'
         if v8Syntax:
            df = 'rfc3339' if cfg.syslogFormatRFC5424 else 'rfc3164'
            fmt = TMPL.timestamp( dateFormat=df )
      return fmt

   def ipAddressFormat( self ):
      """ Returns ipv4 address of a host """
      intfStatus = self.ipStatus.ipIntfStatus.get( managementIntf() )
      if intfStatus:
         return intfStatus.activeAddrWithMask.address
      # 0.0.0.0 is the default value for IpAddrWithMask type so it seems like
      # a reasonable default if the interface does not exist for some reason
      return '0.0.0.0'

   def supeLogTag( self ):
      if self.logMgr.active():
         return '[ACTIVE]'
      else:
         return '[STANDBY]'

   def peerSupeLogTag( self ):
      if self.logMgr.active():
         return "[STANDBY]"
      else:
         return "[ACTIVE]"

   def supeAddress( self ):
      return '127.1.0.%d' % Cell.cellId()

   def peerSupeAddress( self ):
      return '127.1.0.%d' % ( 3 - Cell.cellId() )

   def syslogTagFilter( self ):
      return "$syslogtag startswith '%s' and " % self.peerSupeLogTag()

   def eosFileTemplate( self ):
      return ( TMPL.start( 'EOS_FileFormat' )
             + self.timestampFormat( v8Syntax=True )
             + TMPL.hostname()
             + TMPL.syslogTag()
             + TMPL.message( addSpace=True )
             + TMPL.message( dropLastLF=True )
             + TMPL.constant( "\\n" )
             + TMPL.end() )

   def eosForwardTemplate( self, sevSyslog, sevValue ):
      facility = LoggingLib.facilityTaccToValue( self.notifier_.loggingFacility )
      return ( TMPL.start( 'EOS_ForwardFormat_' + sevSyslog )
             + TMPL.constant( '<%d>' % ( facility * 8 + sevValue ) )
             + self.syslogVersion()
             + self.timestampFormat( v8Syntax=True )
             + TMPL.hostname()
             + TMPL.syslogTag( fromPosition="1", toPosition="32" )
             + self.syslogPidMsgId()
             + TMPL.message( addSpace=True )
             + TMPL.message()
             + TMPL.end() )

   def eosRootLoginFileTemplate( self, sshLogin=True, success=True ):
      templateStr = " %%AAA-%d-ROOT_LOGIN:" % self.rootLoginSevVal
      if sshLogin:
         if success:
            templateName = "EOS_FileFormat_Root_SSH_Success"
            templateStr += " root login succeeded"
         else:
            templateName = "EOS_FileFormat_Root_SSH_Fail"
            templateStr += " root login failed"
         regexStr = " from [^[:space:]]+"
      else:
         if success:
            templateName = "EOS_FileFormat_Root_Success"
            templateStr += " root login succeeded"
            regexStr = " ON .*($|FROM [^[:space:]]+)"
         else:
            templateName = "EOS_FileFormat_Root_Fail"
            templateStr += " root login failed"
            regexStr = " FROM [^[:space:]]+"
      return ( TMPL.start( templateName )
             + self.timestampFormat( v8Syntax=True )
             + TMPL.hostname()
             + TMPL.syslogTag()
             + TMPL.message( addSpace=True )
             + TMPL.constant( templateStr )
             + TMPL.msgRegex( regexStr, 0, lowercase=True )
             + TMPL.constant( "\\n" )
             + TMPL.end() )

   def eosRootLoginForwardTemplate( self, sshLogin=True, success=True ):
      templateStr = " %%AAA-%d-ROOT_LOGIN:" % self.rootLoginSevVal
      facility = LoggingLib.facilityTaccToValue( self.notifier_.loggingFacility )
      if sshLogin:
         if success:
            templateName = "EOS_ForwardFormat_Root_SSH_Success"
            templateStr += " root login succeeded"
         else:
            templateName = "EOS_ForwardFormat_Root_SSH_Fail"
            templateStr += " root login failed"
         regexStr = " from [^[:space:]]+"
      else:
         if success:
            templateName = "EOS_ForwardFormat_Root_Success"
            templateStr += " root login succeeded"
            regexStr = " ON .*($|FROM [^[:space:]]+)"
         else:
            templateName = "EOS_ForwardFormat_Root_Fail"
            templateStr += " root login failed"
            regexStr = " FROM [^[:space:]]+"
      return ( TMPL.start( templateName )
             + TMPL.constant( '<%d>' % ( facility * 8 + self.rootLoginSevVal ) )
             + self.timestampFormat( v8Syntax=True )
             + TMPL.hostname()
             + TMPL.syslogTag( fromPosition="1", toPosition="32" )
             + TMPL.message( addSpace=True )
             + TMPL.constant( templateStr )
             + TMPL.msgRegex( regexStr, 0, lowercase=True )
             + TMPL.constant( "\\n" )
             + TMPL.end() )

   def eosPeerForwardTemplate( self ):
      return ( TMPL.start( 'EOS_PeerForwardFormat' )
             + TMPL.priority()
             + self.timestampFormat( v8Syntax=True )
             + TMPL.hostname()
             + TMPL.constant( self.supeLogTag() )
             + TMPL.syslogTag()
             + TMPL.message( addSpace=True )
             + TMPL.message()
             + TMPL.end() )

   def _logLocalTargetsWithSync( self, enabled, level, cfg,
                                 logfile, syncLogfile ):
      # shared by both console and monitor logging
      targets = [ ]
      if enabled:
         syncLevel = cfg.loggingSynchronousSeverity
         if cfg.loggingSynchronousEnabled and cfg.loggingSynchronousAll:
            # All messages that are console logging sever and up are synchronous
            targets.append( ( syncLogfile, level, None ) )
         elif ( not cfg.loggingSynchronousEnabled or
                LoggingLib.lowerTaccSeverity( level, syncLevel ) == syncLevel ):
            # No messages need to be synchronous, either since it's not
            # enabled or the synchronous severity is lower than or equal to the
            # console logging severity
            targets.append( ( logfile, level, None ) )
         elif cfg.loggingSynchronousEnabled:
            # Split up '*.crit' between eos-console and eos-console-sync so
            # messages of severity critical and up aren't duplicated.
            allFacilitiesSync = []
            allFacilitiesAsync = []
            if syncLevel == 'logEmergency':
               allFacilitiesSync.append( ( '*', 'logCritical', 'logAlert' ) )
               allFacilitiesAsync.append( ( '*', 'logEmergency', None ) )
            elif syncLevel == 'logAlert':
               allFacilitiesSync.append( ( '*', 'logCritical', 'logCritical' ) )
               allFacilitiesAsync.append( ( '*', 'logAlert', None ) )
            else:
               allFacilitiesAsync.append( ( '*', 'logCritical', None ) )
            # Messages between console logging severity level inclusive and
            # synchronous severity level exclusive are synchronous
            targets.append( ( syncLogfile, level,
                              LoggingLib.incTaccSeverity( syncLevel, 1 ),
                              allFacilitiesSync ) )
            # Messages equal to or higher than synchronous logging severity
            # are asynchronous
            targets.append( ( logfile, syncLevel, None,
                              allFacilitiesAsync ) )
         # If console logging isn't enabled, there's no point in logging
         # synchronous messages
      return targets

   def _logLocalTargets( self, cfg ):
      # Return a list of 3or4-tuples
      # ( target, severityN, severityM, [loExtraSeverities] ) for local logging
      #
      # Severity is specified as follows to allow for ranges:
      #  ( levelN, None ) - messages of severity level and up
      #  ( levelN, levelN ) - messages of severity levelN only
      #  ( levelN, levelM ) - messages between severity levelN and levelM
      #
      # loExtraSeverities is an optional list of 3-tuples
      # ( facility, severityN, severityM ) to specify capturing messages from other
      # facilities. If this isn't specified, '*.crit' is added in rsyslog for the
      # log destination; if it is, then '*.crit' isn't added.
      targets = [ ]
      if cfg.buffered:
         targets.append( ( cfg.bufferedLogFilename,
                           cfg.bufferedSeverity, None ) )
      # console logging
      targets.extend(
         self._logLocalTargetsWithSync( cfg.loggingConsoleEnabled,
                                        cfg.loggingConsoleSeverity,
                                        cfg,
                                        LoggingLib.consoleLogFilename,
                                        LoggingLib.consoleSyncLogFilename ) )
      # monitor logging
      if cfg.loggingMonitorUseConsole:
         enabled = cfg.loggingConsoleEnabled
         level = cfg.loggingConsoleSeverity
      else:
         enabled = cfg.loggingMonitorEnabled
         level = cfg.loggingMonitorSeverity

      targets.extend(
         self._logLocalTargetsWithSync( enabled, level, cfg,
                                        LoggingLib.monitorLogFilename,
                                        LoggingLib.monitorSyncLogFilename ) )
      return targets

   def getIpAddr( self, ipOrHost ):
      hostname, addrs = self.dns_.handleIpAddrOrHostname( ipOrHost )

      if hostname:
         # Keep track of all used hostnames, so we can do cleanup later.
         self.syslogHostnames_.append( hostname )

      if not addrs:
         # We have a hostname, but it's not resolved yet.
         return None

      if hostname in self.dns_.usedIPs:
         # Hostname is resolved and one of the IPs is already selected, so re-use it.
         addr = self.dns_.usedIPs[ hostname ]
      else:
         # Either IP has been configured manually (no hostname) or we have hostname,
         # but haven't selected IP yet.
         addr = addrs[ 0 ]
         # Select IP if this is the hostname case.
         if hostname:
            self.dns_.usedIPs[ hostname ] = addrs[ 0 ]

      return addr

   def _dnsResponse( self, record ):
      """Handles a DNS response
      Args:
        record: a Aresolve.DnsRecord, the results from Aresolve.
      """
      bt52( "DNS record: %s" % record )

      if record.valid:
         self.sync()

   def _getSSLProfileInfo( self, cfg, vrfName, hostOrIp, protocol ):
      # Return information on ssl profile validitiy , status, and configurations

      profileName = ""
      validProfile = False
      profileStatus = None
      profileConfig = None

      if protocol == "tls":
         # Turn on ossl for rsyslog
         # Connection will fail if profile is invalid
         self.configRsyslogOssl = True
         profileName = \
            cfg.vrfLoggingHost[ vrfName ].loggingHost[ hostOrIp ].sslProfile
         if vrfName not in self.sslReactor:
            self.sslReactor[ vrfName ] = {}
            self.sslReactor[ vrfName ][ hostOrIp ] = \
                  LogMgrSslReactor( self.sslConfig, self.sslStatus,
                                    profileName, self )
         else:
            if hostOrIp not in self.sslReactor[ vrfName ] or \
                  profileName != self.sslReactor[ vrfName ][ hostOrIp ].profileName:
               self.sslReactor[ vrfName ][ hostOrIp ] = \
                  LogMgrSslReactor( self.sslConfig, self.sslStatus,
                                    profileName, self )
      else:
         if vrfName in self.sslReactor:
            profileReactor = self.sslReactor[ vrfName ].pop( hostOrIp, None )
            if profileReactor:
               profileReactor.close()

      if profileName:
         profileStatusList = self.sslStatus.profileStatus
         if ( ( profileName in profileStatusList ) and
               profileStatusList[ profileName ].state == ProfileState.valid ):
            profileConfig = self.sslConfig.profileConfig[ profileName ]
            profileStatus = self.sslStatus.profileStatus[ profileName ]
            if profileConfig.trustedCert:
               validProfile = True
            else:
               t0( f"No trusted certificate configured for {profileName}" )
               Logging.log( LOGMGR_TLS_MISSING_TRUSTED_CERTIFICATE,
                              profileName )

      return validProfile, profileStatus, profileConfig

   def _setSSLProfileActions( self, actionParam, profileStatus, profileConfig,
                             peerName ):
      # Populate the actionParam for tls connections

      actionParam[ "ca" ] = profileStatus.trustedCertsPath
      actionParam[ "peername" ] = peerName

      certName = profileConfig.certKeyPair.certFile
      if certName:
         actionParam[ "certificate" ] = profileStatus.certKeyPath
         actionParam[ "key" ] = profileStatus.certKeyPath
      if profileConfig.crl:
         actionParam[ "crl" ] = profileStatus.crlsPath
      actionParam[ "tlsVersion" ] = profileStatus.tlsVersion
      actionParam[ "cipherSuite" ] = profileStatus.cipherSuite
      actionParam[ "cipherSuiteV1_3" ] = profileStatus.cipherSuiteV1_3

      return actionParam


   def _logRemoteTargetsWithSrcIp( self, cfg, ip, ipIntfName,
                                   ip6, ip6IntfName, vrfName ):
      # return a list of ( target, severity ) for remote logging to udp servers using
      # omfwd with Address param or to tcp servers using omfwd with Device param
      targets = [ ]
      # Do not forward syslogs to remote servers on standby
      if not self.logMgr.active():
         return targets
      if cfg.loggingTrapEnabled and vrfName in cfg.vrfLoggingHost:
         # If the logging 'trap' feature is enabled, send messages to the
         # configured remote syslog servers.

         for ipOrHost in cfg.vrfLoggingHost[ vrfName ].loggingHost:
            protocol = cfg.vrfLoggingHost[ vrfName ].loggingHost[ ipOrHost ].protocol
            peerName = cfg.vrfLoggingHost[ vrfName ].loggingHost[ ipOrHost ]\
               .ipAddrOrHostname

            # Get SSL profile status and config
            validSslProfile, profileStatus, profileConfig = \
               self._getSSLProfileInfo( cfg, vrfName, ipOrHost, protocol )

            addr = self.getIpAddr( ipOrHost )
            if not addr:
               continue

            for port in cfg.vrfLoggingHost[ vrfName ].loggingHost[ ipOrHost ].ports:
               actionParam = {}
               netNs = ''
               if vrfName != defaultVrf:
                  assert self.allVrfStatusLocal.vrf[ vrfName ].state == 'active'
                  netNs = self.allVrfStatusLocal.vrf[ vrfName ].networkNamespace
                  assert netNs != ''

               actionParam[ "protocol" ] = protocol
               actionParam[ "dstAddr" ] = addr
               actionParam[ "port" ] = port

               if protocol in ( "tls", "tcp" ):
                  # Adding source interface as the device in rsyslog forward action
                  t5( f"_logRemoteTargetsWithSrcIp: tcp, netNs {netNs}" )
                  protocolCmd = "@@"

                  actionParam[ "protocol" ] = "tcp"

                  if validSslProfile:
                     actionParam = self._setSSLProfileActions( actionParam,
                                                               profileStatus,
                                                               profileConfig,
                                                               peerName )

                  devName = None
                  if ipIntfName:
                     devName = self._devName( ipIntfName )
                  elif ip6IntfName:
                     devName = self._devName( ip6IntfName )

                  if devName:
                     actionParam[ "device" ] = devName

                  if not netNs:
                     tstr = ( f"_logRemoteTargetsWithSrcIp: {protocolCmd}[{addr}]"
                             f":{port}" )
                     conf = f"{protocolCmd}[{addr}]:{port}"
                  else:
                     tstr = "_logRemoteTargetsWithSrcIp: %s[(%s)%s]:%d" % (
                           protocolCmd, netNs, addr, port )
                     conf = "%s[(%s)%s]:%d" % (
                           protocolCmd, netNs, addr, port )
                     actionParam[ "netNs" ] = netNs
               else:
                  # Adding source address as the address in rsyslog forward action
                  t5( "_logRemoteTargetsWithSrcIp: udp, netNs {}, ip {}".format(
                      netNs, ip ) )
                  protocolCmd = "@"
                  srcAddr = ip

                  if ip6 and isValidIp( addr ) == 6:
                     srcAddr = ip6

                  if srcAddr:
                     actionParam[ "srcAddr" ] = srcAddr
                     t5( "_logRemoteTargetsWithSrcIp:srcAddr: %s " % srcAddr )

                  if not netNs:
                     tstr = "_logRemoteTargetsWithSrcIp: %s[%s]:%d" % (
                        protocolCmd, addr, port )
                     conf = "%s[%s]:%d" % ( protocolCmd, addr, port )
                  else:
                     actionParam[ "netNs" ] = netNs
                     tstr = "_logRemoteTargetsWithSrcIp: %s[(%s)%s]:%d" % (
                        protocolCmd, netNs, addr, port )
                     conf = "%s[(%s)%s]:%d" % (
                        protocolCmd, netNs, addr, port )
               t5( tstr )
               targets.append( ( conf, cfg.loggingTrapSeverity, actionParam ) )

      return targets

   def _logRemoteTargetsWithSrcIntf( self, cfg, intfStatus, intf6Status, vrfName ):
      # return a list of ( target, severity ) for remote logging to udp servers
      # using omudpspoof or to tcp servers using device configuration.
      targets = [ ]
      ipSrcAddr = intfStatus.activeAddrWithMask.address if intfStatus else None
      # Do not forward syslogs to remote servers on standby
      if not self.logMgr.active():
         return targets
      # pylint: disable-next=too-many-nested-blocks
      if cfg.loggingTrapEnabled and vrfName in cfg.vrfLoggingHost:
         # If the logging 'trap' feature is enabled, send messages to the
         # configured remote syslog servers.

         for i, ipOrHost in enumerate( cfg.vrfLoggingHost[ vrfName ].loggingHost ):
            protocol = cfg.vrfLoggingHost[ vrfName ].loggingHost[ ipOrHost ].protocol
            peerName = cfg.vrfLoggingHost[ vrfName ].loggingHost[ ipOrHost ]\
               .ipAddrOrHostname

            # Get SSL profile status and config
            validSslProfile, profileStatus, profileConfig = \
               self._getSSLProfileInfo( cfg, vrfName, ipOrHost, protocol )

            addr = self.getIpAddr( ipOrHost )
            if not addr:
               continue

            for port in cfg.vrfLoggingHost[ vrfName ].loggingHost[ ipOrHost ].ports:
               actionParam = {}
               netNs = ''
               vstr = ""
               if vrfName != defaultVrf:
                  assert self.allVrfStatusLocal.vrf[ vrfName ].state == 'active'
                  netNs = self.allVrfStatusLocal.vrf[ vrfName ].networkNamespace
                  assert netNs != ''

               actionParam[ "protocol" ] = protocol
               actionParam[ "dstAddr" ] = addr
               actionParam[ "port" ] = port

               if protocol in ( "tls", "tcp" ):
                  # Adding source interface as the device in rsyslog forward action
                  t5( "_logRemoteTargetsWithSrcIntf: tcp, netNs %s" % netNs )
                  protocolCmd = "@@"

                  actionParam[ "protocol" ] = "tcp"

                  if validSslProfile:
                     actionParam = self._setSSLProfileActions( actionParam,
                                                               profileStatus,
                                                               profileConfig,
                                                               peerName )

                  devName = None
                  if intfStatus and intfStatus.intfId:
                     devName = self._devName( intfStatus.intfId )
                  elif intf6Status and intf6Status.intfId:
                     devName = self._devName( intf6Status.intfId )

                  if devName:
                     actionParam[ "device" ] = devName

                  if not netNs:
                     tstr = "_logRemoteTargetsWithSrcIntf: %s[%s]:%d" % (
                            protocolCmd, addr, port )
                     conf = "%s[%s]:%d" % ( protocolCmd, addr, port )
                  else:
                     tstr = "_logRemoteTargetsWithSrcIntf: %s[(%s)%s]:%d" % (
                            protocolCmd, netNs, addr, port )
                     conf = "%s[(%s)%s]:%d" % (
                            protocolCmd, netNs, addr, port )
                     actionParam[ "netNs" ] = netNs
               else:
                  t5( "_logRemoteTargetsWithSrcIntf: udp, netNs %s, ipSrcAddr %s" %
                      ( netNs, ipSrcAddr ) )
                  vstr = "_" + vrfName
                  srcAddr = ipSrcAddr
                  srcTplName = "EOS_ForwardSourceAddress%s_%d" % ( vstr, i )

                  if intf6Status and isValidIp( addr ) == 6:
                     ipv6dst = Arnet.IpGenAddr( addr )
                     srcAddr = intf6Status.sourceAddressForDestination(
                           ipv6dst.privateV6Addr )

                  if srcAddr:
                     t5( "_logRemoteTargetsWithSrcIntf:srcAddr: %s " % srcAddr )
                     conf = "\n# Templates for source address spoofing\n"
                     conf += f'$template {srcTplName},"{srcAddr}"\n'
                     conf += "$ActionOMUDPSpoofSourceNameTemplate %s\n" % srcTplName
                     if not netNs:
                        tstr = "_logRemoteTargetsWithSrcIntf: %s[%s]:%d" % (
                               protocol, addr, port )
                        conf += "# Redirect messages to %s:%d\n" % ( addr,
                                                                     port )
                     else:
                        tstr = "_logRemoteTargetsWithSrcIntf: %s[(%s)%s]:%d" % (
                               protocol, netNs, addr, port )
                        conf += "# Redirect messages to (%s)%s:%d\n" % (
                                netNs, addr, port )
                        conf += "$ActionOMUDPSpoofNetNsName %s\n" % netNs
                     conf += "$ActionOMUDPSpoofTargetHost %s\n" % addr
                     conf += "$ActionOMUDPSpoofTargetPort %d\n" % port
                  else:
                     t5( "_logRemoteTargetsWithSrcIntf: no srcAddr, netNs " + netNs )
                     protocolCmd = "@"
                     if not netNs:
                        tstr = "_logRemoteTargetsWithSrcIntf: %s[%s]:%d" % (
                               protocolCmd, addr, port )
                        conf = "%s[%s]:%d" % ( protocolCmd, addr, port )
                     else:
                        actionParam[ "netNs" ] = netNs
                        tstr = "_logRemoteTargetsWithSrcIntf: %s[(%s)%s]:%d" % (
                               protocolCmd, netNs, addr, port )
                        conf = "%s[(%s)%s]:%d" % (
                               protocolCmd, netNs, addr, port )
               t5( tstr )
               targets.append( ( conf, cfg.loggingTrapSeverity, actionParam ) )

      return targets

   def _logRemoteTargets( self, cfg, vrfName ):
      # return a list of ( target, severity ) for remote logging
      targets = [ ]
      # Do not forward syslogs to remote servers on standby
      if not self.logMgr.active():
         return targets

      if cfg.loggingTrapEnabled and vrfName in cfg.vrfLoggingHost:
         # If the logging 'trap' feature is enabled, send messages to the
         # configured remote syslog servers.
         for h in cfg.vrfLoggingHost[ vrfName ].loggingHost:
            addr = cfg.vrfLoggingHost[ vrfName ].loggingHost[ h ].ipAddrOrHostname
            protocol = cfg.vrfLoggingHost[ vrfName ].loggingHost[ h ].protocol
            peerName = addr

            # Get SSL profile status and config
            validSslProfile, profileStatus, profileConfig = \
               self._getSSLProfileInfo( cfg, vrfName, h, protocol )

            t5( "_logRemoteTargets:" + addr )
            addr = self.getIpAddr( addr )
            if not addr:
               continue

            for port in cfg.vrfLoggingHost[ vrfName ].loggingHost[ h ].ports:
               actionParam = {}
               protocolCmd = "@" if protocol == 'udp' else "@@"

               actionParam[ "protocol" ] = "tcp" if protocol == "tls" else protocol
               actionParam[ "dstAddr" ] = addr
               actionParam[ "port" ] = port

               if validSslProfile:
                  actionParam = self._setSSLProfileActions( actionParam,
                                                               profileStatus,
                                                               profileConfig,
                                                               peerName )

               if vrfName == defaultVrf:
                  targets.append( ( "%s[%s]:%d" % ( protocolCmd, addr, port ),
                                 cfg.loggingTrapSeverity, actionParam ) )
               else:
                  netNs = self.allVrfStatusLocal.vrf[ vrfName ].networkNamespace
                  tstr = "_logRemoteTargets: %s[(%s)%s]:%d" % (
                     protocolCmd, netNs, addr, port )
                  t5( tstr )
                  assert netNs != ''
                  actionParam[ "netNs" ] = netNs
                  targets.append( ( "%s[(%s)%s]:%d"
                                    % ( protocolCmd, netNs, addr, port ),
                                    cfg.loggingTrapSeverity, actionParam ) )
      return targets

   def _logPeerSupeTargets( self ):
      template =  'EOS_PeerForwardFormat'
      return """
if not ( $syslogtag startswith '{}' ) and not ( $msg contains '{}' ) then @{}:514;{}
""".format( self.peerSupeLogTag(), self.peerSupeLogTag(),
        self.peerSupeAddress(), template )

   def _logTrapSystemWithUdpSpoof( self, sysCfg ):
      return self._logSystemCfg( sysCfg, ':omudpspoof:' )

   def _logTrapSystemNoSpoof( self, sysCfg, logHosts ):
      # Use actionParam in logHosts to generate action
      action = ''
      for target in logHosts:
         if 'UDPSpoof' not in target[ 0 ]:
            actionParamStr = self.generateActionParamStr( target[ 2 ] )
            actionStr = 'action(' + actionParamStr + ')\n'
            action += actionStr
      if not action:  # udp spoof for all hosts.
         return ''
      # rsyslog to forward syslogs to atleast one host using UDP or TCP
      return self._logSystemCfg( sysCfg, '{' + action + '}' )

   def _logSystemCfg( self, sysCfg, act ):
      lcfg = ''
      for v in sysCfg:
         cmds = []
         if v.flag.facility:
            cmds.append( '$syslogfacility == %d' % \
               LoggingLib.facilityTaccToValue( v.facility ) )
         if v.flag.severity:
            cmds.append( '$syslogseverity == %d' % \
               LoggingLib.severityTaccToValue( v.severity ) )
         if v.tag:
            cmds.append( '$programname == \'%s\'' % v.tag )
         if v.contain:
            cmds.append( '$msg contains \'%s\'' % v.contain )
         assert cmds
         lcfg += 'if {} then {}\n'.format( ' and '.join( cmds ), act )
      return lcfg

   def _saveToLocalSysLog( self, sysCfg ):
      act = '{action(type="omfile" file="/var/log/eos")}'
      return self._logSystemCfg( sysCfg, act )

   def _bgpLogActions( self, bgpSyslog, cfg, targets, syncSyslog ):
      # BGP log actions
      # filter bgp messages for peer supe must go before local to have
      # higher priority
      conf = ""
      sinks = []
      for t in targets:
         if t[ 1 ] == 'logInfo' or t[ 1 ] == 'logDebug' :
            sinks.append( t[ 0 ] )
      if syncSyslog:
         conf += "# Dispatch directives for BGP auth failure logs follow "
         conf += "from peer supervisor\n\n"
         conf += bgpSyslog.generateRules(sinks, self.syslogTagFilter(), True)
         conf += '\n'
      conf += "# Dispatch directives for BGP auth failure logs follow\n\n"
      conf += bgpSyslog.generateRules(sinks)
      conf += '\n'
      return conf

   def _aclLogActions( self, cfg, targets, syncSyslog ):
      # ACL action
      # We put the action last so the raw messages go to /var/log/messages
      # We assume kernel messages are not otherwise handled
      # filter iptables messages for peer supe must go before local to have
      # higher priority
      conf = ""
      if syncSyslog:
         conf += self._aclLogAction( cfg, targets, peer=True )
         conf += "\n"
      conf += self._aclLogAction( cfg, targets )
      conf += "\n"
      return conf

   def _aclLogAction( self, cfg, targets, peer=False ):
      def _logAcl( level ):
         aclLogLevel = self.loggingConfig.facilityLogLevel.get( 'ACL' )
         if aclLogLevel and aclLogLevel != 'logInfo' and aclLogLevel != 'logDebug':
            return False
         # check if we need to log the ACL message (logInfo)
         # pylint: disable-next=consider-using-in
         return level == 'logInfo' or level == 'logDebug'
      def _actionWithTemplate( targets, template, peer ):
         # append template to the target
         targetTemplates = []
         peerStr = '_PEER' if peer else ''
         for target in targets:
            if target.startswith( '@' ) or target == ":omudpspoof:":
               remoteStr = '_REMOTE'
            else:
               remoteStr = ''
            targetTemplates.append( '%s;%s%s%s' % \
                        ( target, template, remoteStr, peerStr ) )
         output = '\n& '.join( targetTemplates )
         output += '\n'
         return output

      # generate the acl actions based on severity
      actions = [ t[ 0 ] for t in targets if _logAcl( t[ 1 ] ) ]
      if not actions:
         return ""

      tagFilter = self.syslogTagFilter() if peer else ''
      # pylint: disable-next=duplicate-string-formatting-argument
      return """

# Filter iptable logging messages {}.
# The first two rules are for TCP/UDP or any protocols have port numbers
# but specifically exclude ICMP as some ICMP messages (like port-unreachable)
# return information about other protocols.
if {}$syslogfacility-text == 'kern' and $msg contains 'ACL:P:' and \
$msg contains ' SPT=' and not ( $msg contains ' PROTO=ICMP ' ) then {}
if {}$syslogfacility-text == 'kern' and $msg contains 'ACL:D:' and \
$msg contains ' SPT=' and not ( $msg contains ' PROTO=ICMP ' ) then {}
if {}$syslogfacility-text == 'authpriv' and $msg contains 'ACL:P:' then {}
if {}$syslogfacility-text == 'authpriv' and $msg contains 'ACL:D:' then {}
if {}$syslogfacility-text == 'kern' and $msg contains 'ACL:d/:' and \
$msg contains 'OUT=et' and \
$msg contains ' SPT=' and not ( $msg contains ' PROTO=ICMP ' ) then {}
if {}$syslogfacility-text == 'kern' and $msg contains 'ACL:d/:' and \
$msg contains 'OUT=po' and \
$msg contains ' SPT=' and not ( $msg contains ' PROTO=ICMP ' ) then {}
if {}$syslogfacility-text == 'kern' and $msg contains 'ACL:d/:' and \
$msg contains 'OUT=vlan' and \
$msg contains ' SPT=' and not ( $msg contains ' PROTO=ICMP ' ) then {}

if {}$syslogfacility-text == 'kern' and $msg contains 'ACL:P:' and \
( not ( $msg contains ' SPT=' ) or ( $msg contains ' PROTO=ICMP ' ) ) then {}
if {}$syslogfacility-text == 'kern' and $msg contains 'ACL:D:' and \
( not ( $msg contains ' SPT=' ) or ( $msg contains ' PROTO=ICMP ' ) ) then {}
if {}$syslogfacility-text == 'kern' and $msg contains 'ACL:d/:' and \
$msg contains 'OUT=et' and \
( not ( $msg contains ' SPT=' ) or ( $msg contains ' PROTO=ICMP ' ) ) then {}
if {}$syslogfacility-text == 'kern' and $msg contains 'ACL:d/:' and \
$msg contains 'OUT=po' and \
( not ( $msg contains ' SPT=' ) or ( $msg contains ' PROTO=ICMP ' ) ) then {}
if {}$syslogfacility-text == 'kern' and $msg contains 'ACL:d/:' and \
$msg contains 'OUT=vlan' and \
( not ( $msg contains ' SPT=' ) or ( $msg contains ' PROTO=ICMP ' ) ) then {}
""".format( 'from peer supervisor' if peer else '',
        tagFilter, _actionWithTemplate( actions, 'ACL_TCP_PERMIT', peer ),
        tagFilter, _actionWithTemplate( actions, 'ACL_TCP_DENY', peer ),
        tagFilter, _actionWithTemplate( actions, 'ACL_SERVICE_PERMIT', peer ),
        tagFilter, _actionWithTemplate( actions, 'ACL_SERVICE_DENY', peer ),
        tagFilter, _actionWithTemplate( actions, 'ACL_TCP_EGRESS_DENY_ETH', peer ),
        tagFilter, _actionWithTemplate( actions, 'ACL_TCP_EGRESS_DENY_PO', peer ),
        tagFilter, _actionWithTemplate( actions, 'ACL_TCP_EGRESS_DENY_VLAN', peer ),
        tagFilter, _actionWithTemplate( actions, 'ACL_IP_PERMIT', peer ),
        tagFilter, _actionWithTemplate( actions, 'ACL_IP_DENY', peer ),
        tagFilter, _actionWithTemplate( actions, 'ACL_IP_EGRESS_DENY_ETH', peer ),
        tagFilter, _actionWithTemplate( actions, 'ACL_IP_EGRESS_DENY_PO', peer ),
        tagFilter, _actionWithTemplate( actions, 'ACL_IP_EGRESS_DENY_VLAN', peer ) )

   def _rbfdLogAction( self ):
      return """
# Filter rbfd logging messages.
if ($syslogfacility-text == 'kern' and $msg contains ' rbfd ') then {
      action(type="omfile" file="/var/log/rbfd.log")
      stop
}
"""

   def _oomKilledProcessAction( self ):
      return """
# Filter OOM Killed messages so they go to EOS log in var directory.
if ($syslogfacility-text == 'kern' and $msg contains 'Killed process') then {
      action(type="omfile" file="/var/log/eos")
}
"""

   def _aclLoggingTemplates( self, priValue, peer=False ):

      # converts et5_2_12[.1] to Ethernet5/2/12[.1]
      aclEthernetIfParse = ( TMPL.constant( "Ethernet" )
                    + TMPL.msgRegex( "OUT=et([0-9]+(\\\\.[0-9]+)?)", 1 )
                    + TMPL.msgRegex( "ACL:d(/)(.*)OUT=et([0-9]+)_", 1 )
                    + TMPL.msgRegex( "OUT=et([0-9]+)_([0-9]+(\\\\.[0-9]+)?)",
                                     2 )
                    + TMPL.msgRegex( "ACL:d(/)(.*)OUT=et([0-9]+)_([0-9]+)_",
                                     1 )
                    + TMPL.msgRegex(
            "OUT=et([0-9]+)_([0-9]+)_([0-9]+(\\\\.[0-9]+)?)",
                                     3 )
                    )

      # converts po43[.1] to Po43[.1]
      aclPoIfParse = ( TMPL.constant( "Port-Channel" )
                       + TMPL.msgRegex( "OUT=po([0-9]+(\\\\.[0-9]+)?)", 1 ) )

      # converts vlan321 to Vlan321
      aclVlanIfParse = ( TMPL.constant( "Vlan" )
                       + TMPL.msgRegex( "OUT=vlan([0-9]+)", 1 ) )

      prefix = """
# Template for converting iptable messages %s
#
# Generic IP protocol
"""
      aclIpPermit = ( TMPL.start( 'ACL_IP_PERMIT%s%s' )
                      + "%s" # for priority string
                      + self.timestampFormat( v8Syntax=True )
                      + TMPL.hostname()
                      + TMPL.constant( "%sAcl: %%ACL-6-IPACCESS: list " )
                      + TMPL.msgField( 3 )
                      + TMPL.constant( ' permitted ' )
                      + TMPL.msgRegex( "PROTO=([A-Za-z0-9]+)", 1, lowercase=True )
                      + TMPL.constant( ' ' )
                      + TMPL.msgRegex( "SRC=([0-9|A-F|a-f|:|\\\\.]+)", 1 )
                      + TMPL.constant( ' -> ' )
                      + TMPL.msgRegex( "DST=([0-9|A-F|a-f|:|\\\\.]+)", 1 )
                      + TMPL.msgRegex( " TYPE=[0-9]+ CODE=[0-9]+",
            0, lowercase=True )
                      + TMPL.constant( "\\n" )
                      + TMPL.end() )

      aclIpDeny = ( TMPL.start( 'ACL_IP_DENY%s%s' )
                    + "%s" # for priority string
                    + self.timestampFormat( v8Syntax=True )
                    + TMPL.hostname()
                    + TMPL.constant( "%sAcl: %%ACL-6-IPACCESS: list " )
                    + TMPL.msgField( 3 )
                    + TMPL.constant( ' denied ' )
                    + TMPL.msgRegex( "PROTO=([A-Za-z0-9]+)", 1, lowercase=True )
                    + TMPL.constant( ' ' )
                    + TMPL.msgRegex( "SRC=([0-9|A-F|a-f|:|\\\\.]+)", 1 )
                    + TMPL.constant( ' -> ' )
                    + TMPL.msgRegex( "DST=([0-9|A-F|a-f|:|\\\\.]+)", 1 )
                    + TMPL.msgRegex( " TYPE=[0-9]+ CODE=[0-9]+", 0, lowercase=True )
                    + TMPL.constant( "\\n" )
                    + TMPL.end() )

      aclTcpPermit = ( TMPL.start( 'ACL_TCP_PERMIT%s%s' )
                       + "%s" # for priority string
                       + self.timestampFormat( v8Syntax=True )
                       + TMPL.hostname()
                       + TMPL.constant( "%sAcl: %%ACL-6-IPACCESS: list " )
                       + TMPL.msgField( 3 )
                       + TMPL.constant( ' permitted ' )
                       + TMPL.msgRegex( "PROTO=([A-Za-z0-9]+)", 1, lowercase=True )
                       + TMPL.constant( ' ' )
                       + TMPL.msgRegex( "SRC=([0-9|A-F|a-f|:|\\\\.]+)", 1 )
                       + TMPL.constant( '(' )
                       + TMPL.msgRegex( "SPT=([0-9]+)", 1 )
                       + TMPL.constant( ')' )
                       + TMPL.constant( ' -> ' )
                       + TMPL.msgRegex( "DST=([0-9|A-F|a-f|:|\\\\.]+)", 1 )
                       + TMPL.constant( '(' )
                       + TMPL.msgRegex( "DPT=([0-9|\\\\.]+)", 1 )
                       + TMPL.constant( ')' )
                       + TMPL.msgRegex( " TYPE=[0-9]+ CODE=[0-9]+",
            0, lowercase=True )
                       + TMPL.constant( "\\n" )
                       + TMPL.end() )

      aclTcpDeny = ( TMPL.start( 'ACL_TCP_DENY%s%s' )
                     + "%s" # for priority string
                     + self.timestampFormat( v8Syntax=True )
                     + TMPL.hostname()
                     + TMPL.constant( "%sAcl: %%ACL-6-IPACCESS: list " )
                     + TMPL.msgField( 3 )
                     + TMPL.constant( ' denied ' )
                     + TMPL.msgRegex( "PROTO=([A-Za-z0-9]+)", 1, lowercase=True )
                     + TMPL.constant( ' ' )
                     + TMPL.msgRegex( "SRC=([0-9|A-F|a-f|:|\\\\.]+)", 1 )
                     + TMPL.constant( '(' )
                     + TMPL.msgRegex( "SPT=([0-9]+)", 1 )
                     + TMPL.constant( ')' )
                     + TMPL.constant( ' -> ' )
                     + TMPL.msgRegex( "DST=([0-9|A-F|a-f|:|\\\\.]+)", 1 )
                     + TMPL.constant( '(' )
                     + TMPL.msgRegex( "DPT=([0-9|\\\\.]+)", 1 )
                     + TMPL.constant( ')' )
                     + TMPL.msgRegex( " TYPE=[0-9]+ CODE=[0-9]+", 0, lowercase=True )
                     + TMPL.constant( "\\n" )
                     + TMPL.end() )

      aclServicePermit = ( TMPL.start( 'ACL_SERVICE_PERMIT%s%s' )
                           + "%s" # for priority string
                           + self.timestampFormat( v8Syntax=True )
                           + TMPL.hostname()
                           + TMPL.constant( "%s" )
                           + TMPL.msgRegex( "PROGRAM=([A-Za-z0-9]+)",
            1, lowercase=True )
                           + TMPL.constant( ": %%ACL-6-IPACCESS: list " )
                           + TMPL.msgField( 3 )
                           + TMPL.constant( ' permitted ' )
                           + TMPL.msgRegex( "PROTO=([A-Za-z0-9]+)",
            1, lowercase=True )
                           + TMPL.constant( ' ' )
                           + TMPL.msgRegex( "SRC=([0-9|A-F|a-f|:|\\\\.]+)", 1 )
                           + TMPL.constant( '(' )
                           + TMPL.msgRegex( "SPT=([0-9]+)", 1 )
                           + TMPL.constant( ')' )
                           + TMPL.constant( ' -> ' )
                           + TMPL.msgRegex( "DST=([0-9|A-F|a-f|:|\\\\.]+)", 1 )
                           + TMPL.constant( '(' )
                           + TMPL.msgRegex( "DPT=([0-9|\\\\.]+)", 1 )
                           + TMPL.constant( ')' )
                           + TMPL.constant( "\\n" )
                           + TMPL.end() )

      aclServiceDeny = ( TMPL.start( 'ACL_SERVICE_DENY%s%s' )
                         + "%s" # for priority string
                         + self.timestampFormat( v8Syntax=True )
                         + TMPL.hostname()
                         + TMPL.constant( "%s" )
                         + TMPL.msgRegex( "PROGRAM=([A-Za-z0-9]+)",
            1, lowercase=True )
                         + TMPL.constant( ": %%ACL-6-IPACCESS: list " )
                         + TMPL.msgField( 3 )
                         + TMPL.constant( ' denied ' )
                         + TMPL.msgRegex( "PROTO=([A-Za-z0-9]+)",
            1, lowercase=True )
                         + TMPL.constant( ' ' )
                         + TMPL.msgRegex( "SRC=([0-9|A-F|a-f|:|\\\\.]+)", 1 )
                         + TMPL.constant( '(' )
                         + TMPL.msgRegex( "SPT=([0-9]+)", 1 )
                         + TMPL.constant( ')' )
                         + TMPL.constant( ' -> ' )
                         + TMPL.msgRegex( "DST=([0-9|A-F|a-f|:|\\\\.]+)", 1 )
                         + TMPL.constant( '(' )
                         + TMPL.msgRegex( "DPT=([0-9|\\\\.]+)", 1 )
                         + TMPL.constant( ')' )
                         + TMPL.constant( "\\n" )
                         + TMPL.end() )

      aclIpEgressDeny = ( TMPL.start( "ACL_IP_EGRESS_DENY%s%s%s" )
                         + "%s" # for priority string
                         + self.timestampFormat( v8Syntax=True )
                         + TMPL.hostname()
                         + TMPL.constant( "%sAcl: %%ACL-6-IPACCESS: egress list " )
                         + TMPL.msgRegex( "ACL:d:([A-Za-z0-9]+)", 1 )
                         + TMPL.constant( " " )
                         + "%s" # interface parsing
                         + TMPL.constant( " denied " )
                         + TMPL.msgRegex( "PROTO=([A-Za-z0-9]+)", 1, lowercase=True )
                         + TMPL.constant( ' ' )
                         + TMPL.msgRegex( "SRC=([0-9|A-F|a-f|:|\\\\.]+)", 1 )
                         + TMPL.constant( ' -> ' )
                         + TMPL.msgRegex( "DST=([0-9|A-F|a-f|:|\\\\.]+)", 1 )
                         + TMPL.msgRegex( " TYPE=[0-9]+ CODE=[0-9]+", 0,
                                          lowercase=True )
                         + TMPL.constant( "\\n" )
                         + TMPL.end() )

      aclTcpEgressDeny = ( TMPL.start( "ACL_TCP_EGRESS_DENY%s%s%s" )
                         + "%s" # for priority string
                         + self.timestampFormat( v8Syntax=True )
                         + TMPL.hostname()
                         + TMPL.constant( "%sAcl: %%ACL-6-IPACCESS: egress list " )
                         + TMPL.msgRegex( "ACL:d:([A-Za-z0-9]+)", 1 )
                         + TMPL.constant( " " )
                         + "%s" # interface parsing
                         + TMPL.constant( " denied " )
                         + TMPL.msgRegex( "PROTO=([A-Za-z0-9]+)", 1, lowercase=True )
                         + TMPL.constant( ' ' )
                         + TMPL.msgRegex( "SRC=([0-9|A-F|a-f|:|\\\\.]+)", 1 )
                         + TMPL.constant( '(' )
                         + TMPL.msgRegex( "SPT=([0-9]+)", 1 )
                         + TMPL.constant( ')' )
                         + TMPL.constant( ' -> ' )
                         + TMPL.msgRegex( "DST=([0-9|A-F|a-f|:|\\\\.]+)", 1 )
                         + TMPL.constant( '(' )
                         + TMPL.msgRegex( "DPT=([0-9|\\\\.]+)", 1 )
                         + TMPL.constant( ')' )
                         + TMPL.msgRegex( " TYPE=[0-9]+ CODE=[0-9]+", 0,
                                          lowercase=True )
                         + TMPL.constant( "\\n" )
                         + TMPL.end() )

      def _aclLoggingTemplateGen( peer, priStr='' ):
         peerStr = '_PEER' if peer else ''
         peerTag = self.peerSupeLogTag() if peer else ''
         remoteStr = '_REMOTE' if priStr else ''

         aclTemplates = aclIpPermit % \
                  ( remoteStr, peerStr, TMPL.constant( priStr ), peerTag )
         aclTemplates += aclIpDeny % \
                  ( remoteStr, peerStr, TMPL.constant( priStr ), peerTag )
         aclTemplates += aclTcpPermit % \
                  ( remoteStr, peerStr, TMPL.constant( priStr ), peerTag )
         aclTemplates += aclTcpDeny % \
                  ( remoteStr, peerStr, TMPL.constant( priStr ), peerTag )
         aclTemplates += aclServicePermit % \
                  ( remoteStr, peerStr, TMPL.constant( priStr ), peerTag )
         aclTemplates += aclServiceDeny % \
                  ( remoteStr, peerStr, TMPL.constant( priStr ), peerTag )

         intfTypes = [ '_ETH', '_PO', '_VLAN' ]
         intfParse = [ aclEthernetIfParse, aclPoIfParse, aclVlanIfParse ]
         for i, j in zip( intfTypes, intfParse ):
            aclTemplates += aclIpEgressDeny % \
                  ( i, remoteStr, peerStr, TMPL.constant( priStr ), peerTag, j )
            aclTemplates += aclTcpEgressDeny % \
                  ( i, remoteStr, peerStr, TMPL.constant( priStr ), peerTag, j )
         return aclTemplates

      aclConf = prefix % ( 'from peer supervisor' if peer else '' )
      aclConf += _aclLoggingTemplateGen( peer )
      aclConf += "\n"
      aclConf += _aclLoggingTemplateGen( peer, "<%d>" % priValue )
      aclConf += "\n"
      return aclConf

   def _logPersistentCfg( self, cfg, sevThreshold ):
      persistentCfg = "\n# Persistent logging is %s\n" % (
                      "enabled" if cfg.persistentLog else "disabled" )
      if not cfg.persistentLog:
         return persistentCfg

      persistentCfg += "# Save all log messages above the severity threshold into "
      persistentCfg += "persistent destination at /mnt/flash/persist/messages.\n"
      persistentCfg += ( "$outchannel persistentLogging, %s, %d, %s\n"
                         % ( cfg.persistentLogFilename, cfg.persistentLogSize,
                             cfg.persistentLogAction ) )
      persistentCfg += self.newSyslogEntry( { "*" : ( sevThreshold, None ),
                                              "daemon" : ( 'logEmergency', None ) },
                                            ":omfile:$persistentLogging\n" )
      t5( "persistent conf is:", persistentCfg )
      return persistentCfg

   def initMatchListReactor( self ):
      loggingMatchListName = self.loggingConfig.loggingMatchList.matchListName
      reactorFilter = lambda x: x == loggingMatchListName
      if loggingMatchListName:
         self.matchListReactor_ = Tac.collectionChangeReactor(
                                     self.matchListConfig.matchStringList,
                                     MatchStringListReactor,
                                     reactorArgs=( self, ),
                                     reactorFilter=reactorFilter )
      else:
         if self.matchListReactor_:
            self.matchListReactor_.close()
         self.matchListReactor_ = None

   @Tac.handler( 'timestampTimezone' )
   def handleTimestampTimezone( self ):
      if self.logMgr.timezoneReactor_:
         self.logMgr.timezoneReactor_.schedule( self.notifier_.timestampTimezone )

   @Tac.handler( 'loggingMatchList' )
   def handleLoggingMatchList( self ):
      self.initMatchListReactor()

   @Tac.handler( 'persistentLog' )
   @Tac.handler( 'persistentLogSize' )
   def handlePersistentLog( self ):
      cfg = self.notifier_
      # Write the output channels action script with file max line limit
      # Note that action-on-max-size is queried BEFORE writing the log message
      # When this size is reached, rsyslogd will execute the action-on-max-size
      # command and then REOPEN the file and retry. Therefore we reserve the length
      # of one message(300 bytes) for the new log to write.
      if cfg.persistentLog:
         t3( "create new persistent logging script" )
         logSize = cfg.persistentLogSize - 300
         # Since we tail the last size_limit bytes of the file,
         # the first line of the cut file might be incomplete,
         # therefore remove the line.
         scriptFt = """#!/bin/sh
maxLine=$(( $( tail -c {logSize} {logFile} | wc -l ) - 1 ))
tail -n $maxLine {logFile} > {logFile}.tmp && mv -f {logFile}.tmp {logFile}
"""
         with open( cfg.persistentLogAction, "w" ) as script:
            script.write( scriptFt.format( logSize=logSize,
                                           logFile=cfg.persistentLogFilename ) )
         st = os.stat( cfg.persistentLogAction )
         os.chmod( cfg.persistentLogAction, st.st_mode | stat.S_IEXEC )
      else:
         t3( "remove persistent logging script" )
         try:
            os.remove( cfg.persistentLogAction )
         except OSError as e:
            t0( "remove persistent logging script failed due to %s" % str( e ) )
      self.sync()

   def conf( self ):
      # Regenerate the contents of the syslog.conf file based on the current
      # LogConfig state. The format of this file is specified in the
      # syslog.conf(5) man page, with any changes/extensions described in
      # the sysklogd(8) man page.

      self.syslogHostnames_.clear()
      cfg = self.notifier_
      status = self.loggingStatus
      protocol = self.logMgr.redundancyProtocol()

      syncSyslog = protocol != 'simplex'

      # Logging is disabled
      if not cfg.loggingOn:
         return """
# Logging is disabled
module( load="imuxsock" )
module( load="imklog" )

# Turn off dumping error messages to stderr
$ErrorMessagesToStderr off

# Drop all messages from EOS agents except canary syslogs
daemon.emerg;kern.notice  /var/log/messages
authpriv.*              /var/log/secure
# All local5 messages go to canary.log
local5.debug            /var/log/canary.log
kern.=debug             /var/log/kernel.debug
*.*                     ~
"""

      t5( "Entering conf()" )

      # Build list containing default vrf at the head of the list
      # (active or not) followed by all active VRF's with hosts
      v = defaultVrf
      vrfList = [ v ]
      loadUdpSpoofMod = False

      if v in cfg.vrfLoggingHost:
         intfStatus = status.srcIntfStatus.get( v )
         intf6Status = status.srcIntf6Status.get( v )
         if ( intfStatus and intfStatus.activeAddrWithMask.address != "0.0.0.0" ) \
            or ( intf6Status and len( intf6Status.addr ) ):
            loadUdpSpoofMod = True
            t5( "vrf " + v + " is spoofing" )
         else:
            t5( "vrf " + v + " is NOT spoofing" )

      for v in sorted( cfg.vrfLoggingHost ):
         if v == defaultVrf:
            continue
         if v not in self.allVrfStatusLocal.vrf:
            continue
         if self.allVrfStatusLocal.vrf[ v ].state == 'active':
            vrfList.append( v )
            intfStatus = status.srcIntfStatus.get( v )
            intf6Status = status.srcIntf6Status.get( v )
            if ( intfStatus and
                 intfStatus.activeAddrWithMask.address != "0.0.0.0" ) or \
                 ( intf6Status and len( intf6Status.addr ) ):
               loadUdpSpoofMod = True
               t5( "vrf " + v + " is spoofing" )
            else:
               t5( "vrf " + v + " is NOT spoofing" )
         else:
            t5( "vrf " + v + " is NOT Active" )

      conf = ""
      assert 'rsyslog' in self.linuxServiceName()
      conf += "# Global directives for rsyslogd\n\n"
      conf += "# Enables local system logging via /dev/log\n"
      conf += "module( load=\"imuxsock\" )\n\n"
      conf += "# If main queue fills up, then drop messages immediately\n"
      conf += "# Don't block the system for 2 seconds\n"
      conf += "$MainMsgQueueTimeoutEnqueue 0\n\n" # LEGACY_CONFIG avail 8.1901.0+
      if not CEosHelper.isCeos() or CEosHelper.isCeosLab():
         conf += "# Enables kernel logging support\n"
         conf += "module( load=\"imklog\" )\n\n"
      else:
         conf += "# Disable Kernel logging in cEOS\n\n"

      if CEosHelper.isCeosLab():
         # Now that we are loading imklog in cEOS-lab, we have to
         # deal with the fact that older cEOS-lab versions sent
         # their systemd output to kmsg/imklog.  If we see a message
         # coming from imklog that contains 'systemd[1]' then skip
         # it altogether.
         conf += "# Filter out systemd messages\n"
         conf += ( "if $inputname == 'imklog' and "
                   "$msg contains 'systemd[1]' then stop\n" )
         conf += "\n"

      logMatchList = cfg.loggingMatchList
      matchListName = logMatchList.matchListName
      if matchListName and matchListName in self.matchListConfig.matchStringList:
         matchList = self.matchListConfig.matchStringList[ matchListName ].matchInfo
         if not logMatchList.inverseResult:
            conf += "# Filter out messages\n"
            for match in matchList.values():
               if match.type == 'regex':
                  # re_match uses ERE regex
                  conf += "if re_match( $rawmsg, '%s' ) then stop\n" % match.string
         else:
            expr = ""
            for match in matchList.values():
               if match.type == 'regex':
                  expr += "re_match( $rawmsg, '%s' ) or \\\n" % match.string
            if expr:
               conf += "# Filter in messages\n"
               conf += "if not ( " + expr.rstrip( " or \\\n"  ) + " ) then stop\n"
            else:
               # Empty match-list so all messages should be discarded
               conf += "# Filter in no messages\n"
               conf += "stop\n"

         conf += "\n"
      if syncSyslog:
         conf += "# Enables syslog peer forwarding support\n"
         conf += "module( load=\"imudp\" )\n"
         conf += "input( type=\"imudp\" port=\"514\" )\n\n"
         # The check is to prevent infinite forwarding loop if the
         # remote host is set to localhost or supervisor. And it guarantees
         # the rsyslogd can only receive peer sync messages and cannot be
         # used as a remote log server.
         conf += "# discard any remote message not from peer supervisor\n"
         conf += "# also discard message containing log tag or "
         conf += "not containing peer log tag\n"
         conf += "if ( $inputname == 'imudp' ) and " \
                    "( ( not ( $fromhost-ip == '%s' ) ) or " \
                       "( $syslogtag startswith '%s' ) or " \
                       "( not ( $syslogtag startswith '%s' ) ) ) then stop\n\n" % \
                 ( self.peerSupeAddress(), self.supeLogTag(), self.peerSupeLogTag() )

      # Load udp spoof module only if configured
      if loadUdpSpoofMod:
         conf += "# Enables source IP address spoofing\n"
         conf += "module( load=\"omudpspoof\" )\n\n"

      conf += "# The default file format\n"
      conf += self.eosFileTemplate()
      conf += "$ActionFileDefaultTemplate EOS_FileFormat\n" # LEGACY_CONFIG
      #conf += "module( load=\"omfile\" template=\"EOS_FileFormat\" )\n"
      conf += "# Eight forwarding formats - one for each severity level\n"
      conf += "# This is used to overwrite the facility to a user defined\n"
      conf += "# facility using the 'logging facility <facility>' cli\n"
      for sevTacc, sevValue in LoggingLib.logSevTaccToValueMap.items():
         sevSyslog = LoggingLib.severityTaccToSyslog( sevTacc )
         conf += self.eosForwardTemplate( sevSyslog, sevValue )

      # Add template for root user messages
      if self.loggingConfig.loggingRootLogin:
         for sshLogin in True, False:
            for loginResult in True, False:
               conf += self.eosRootLoginFileTemplate( sshLogin=sshLogin,
                                                      success=loginResult )
               conf += self.eosRootLoginForwardTemplate( sshLogin=sshLogin,
                                                         success=loginResult )

      if syncSyslog:
         # The Syslog SSO forwarding formats
         conf += "\n# Template for peer supervisor forwarding\n"
         conf += self.eosPeerForwardTemplate()

      facilityName = self.notifier_.loggingFacility
      facilityValue = LoggingLib.facilityTaccToValue( facilityName )
      severityValue = LoggingLib.severityCliToValue( "informational" )
      priValue = ( ( facilityValue * 8 ) + severityValue )
      timestampFormat = self.timestampFormat( )
      conf += self._aclLoggingTemplates( priValue )
      if syncSyslog:
         conf += self._aclLoggingTemplates( priValue, True )

      conf += self._oomKilledProcessAction()
      conf += self._rbfdLogAction()
      tcpBadAuthSyslog = TcpBadAuthSyslogConf()
      conf += "# Templates for translating BGP auth failure logs follow\n\n"
      conf += tcpBadAuthSyslog.generateTemplates("<%d>" % priValue, timestampFormat)
      conf += '\n'
      if syncSyslog:
         conf += "# Templates for translating BGP auth failure logs follow "
         conf += "from peer supervisor\n\n"
         conf += tcpBadAuthSyslog.generateTemplates("<%d>" % priValue,
                                             timestampFormat, True,
                                             self.peerSupeLogTag() )
         conf += '\n'

      conf += "# Turn off dumping error messages to stderr\n"
      conf += "$ErrorMessagesToStderr off\n\n" # LEGACY_CONFIG avail 8.30

      conf += "# Repeated line reduction reduces redundant lines to\n"
      conf += "# 'message repeated n times'\n"
      repeatedMsgReduction = "off" if cfg.repeatMsgs else "on"
      t3( "Setting $RepeatedMsgReduction to %s" % repeatedMsgReduction )
      conf += "$RepeatedMsgReduction %s\n\n" % repeatedMsgReduction

      conf += "# Syslog entries for configured log destinations.\n\n"
      conf += "# For each log destination, we have one filter that captures\n"
      conf += "# the messages from the EOS log "
      conf += "facility %s.\n" % self.eosFacilityName_
      conf += "# In addition, we also have a filter that captures critical\n"
      conf += "# log messages from all facilities. The authpriv facility\n"
      conf += "# is always excluded to prevent password activity from being\n"
      conf += "# logged\n"

      conf += "\n# Buffered logging is %s\n" % (
         "enabled" if cfg.buffered else "disabled" )
      conf += "# Console logging is %s\n" % (
         "enabled" if cfg.loggingConsoleEnabled else "disabled" )
      if cfg.loggingMonitorUseConsole:
         monitorState = "default"
      else:
         monitorState = "enabled" if cfg.loggingMonitorEnabled else "disabled"
      conf += "# Monitor logging is %s\n" % monitorState

      # Forward kernel messages to StorageDevices agent
      conf += "# Forward kernel messages to StorageDevices\n"
      conf += "$ModLoad         omuxsock\n"
      conf += "$OMUxSockSocket  /var/run/kernelmessages\n"
      conf += "kern.info        :omuxsock:\n\n"

      localTargets = self._logLocalTargets( cfg )
      for t in localTargets:
         conf += self.newLogLocalDestEntry( t[ 0 ], t[ 1:3 ],
                                            t[ 3 ] if len(t) > 3 else None )
      # ACL local log actions
      conf += self._aclLogActions( cfg, localTargets, syncSyslog )

      # BGP local log actions
      conf += self._bgpLogActions( tcpBadAuthSyslog, cfg, localTargets, syncSyslog )

      conf += "# Remote logging is %s\n" % (
         "enabled" if cfg.loggingTrapEnabled else "disabled" )

      for vrfName in vrfList:
         t5( "Syslog enabled for VRF " + vrfName )
         conf += "# Syslog enabled for VRF %s\n" % vrfName
         ipIntfName = status.srcIpIntfName.get ( vrfName )
         ip6IntfName = status.srcIp6IntfName.get( vrfName )
         ipSAddr = ipIntfName or ip6IntfName

         intfStatus = status.srcIntfStatus.get( vrfName )
         if intfStatus and intfStatus.activeAddrWithMask.address == "0.0.0.0":
            intfStatus = None
         intf6Status = status.srcIntf6Status.get( vrfName )
         intfSAddr = intfStatus or ( intf6Status and len( intf6Status.addr ) )

         if ipSAddr:
            ip = cfg.srcIpAddress.get( vrfName )
            ip6 = cfg.srcIp6Address.get( vrfName )
            remoteTargets = self._logRemoteTargetsWithSrcIp( cfg, ip, ipIntfName,
                                                             ip6, ip6IntfName,
                                                             vrfName )
         elif intfSAddr:
            remoteTargets = self._logRemoteTargetsWithSrcIntf( cfg, intfStatus,
                                                               intf6Status,
                                                               vrfName )
         else:
            remoteTargets = self._logRemoteTargets( cfg, vrfName )
         for t in remoteTargets:
            udpSpoof = False
            if "ActionOMUDPSpoofSourceNameTemplate" in t[ 0 ]:
               # output the template definition
               conf += t[ 0 ]
               # update the target entry, replacing the template definition
               # with a reference to the template
               t = ( ":omudpspoof:", t[ 1 ], t[ 2 ] )
               udpSpoof = True
            conf += self.newLogRemoteDestEntry( t[ 0 ], t[ 1 ], t[ 2 ],
                                                udpSpoof, cfg )

            # ACL remote log actions
            conf += self._aclLogActions( cfg, [ t ], syncSyslog )

            # BGP remote log actions
            conf += self._bgpLogActions( tcpBadAuthSyslog, cfg, [ t ], syncSyslog )

      if self.configRsyslogOssl:
         # Turn on per host tls configuration
         conf += "$TlsPerHostConfig on\n"

      sevThreshold = LoggingLib.lowerTaccSeverity(
         cfg.bufferedSeverity, cfg.loggingTrapSeverity )
      sevThreshold = LoggingLib.lowerTaccSeverity(
         sevThreshold, cfg.loggingConsoleSeverity )
      if not cfg.loggingMonitorUseConsole:
         sevThreshold = LoggingLib.lowerTaccSeverity(
            sevThreshold, cfg.loggingMonitorSeverity )

      # sevThreshold now reflects the lowest severity of all log
      # destinations, which means that any log messages with severity
      # lower than minSeverity will be filtered out by syslog.
      status.severityThreshold = sevThreshold

      # Program a default entry that sends all log messages
      # above the severity threshold to /var/log/messages. This is the
      # long term store of log messages.
      # final store of log messages. The location of this call should
      # always be at the end, because we assume that after processing all
      # the potential log destinations, we now have a up to date
      # severity threshold
      conf += "\n# All log messages above severity threshold also go\n"
      conf += "# into /var/log/messages\n"
      conf += self.newSyslogEntry( { "*" : ( sevThreshold, None ),
                                     "daemon" : ( 'logEmergency', None ) },
                                   "/var/log/messages" )
      if self.sshConfig.loggingTargetEnabled:
         # send security ssh/sshd log messages also to /var/log/messages
         # for NDcPP FAU_GEN.1 Audit Data Generation requirement
         conf += "\n# Authpriv messages of ssh/sshd also go into /var/log/messages\n"
         for s in [ 'ssh', 'sshd' ]:
            conf += ( "if $syslogfacility-text == 'authpriv' and "
                      "$programname == '%s' then /var/log/messages\n" % s )

      # copy /var/log/messages log to flash disk at /mnt/flash/persist/messages
      conf += self._logPersistentCfg( cfg, sevThreshold )

      conf += "\n# Authpriv messages are logged in /var/log/secure. The permissions "
      conf += "on this file give write permission to root only.\n"
      conf += "authpriv.*              /var/log/secure\n\n"

      conf += "# The local5 facility is reserved for canaries\n"
      conf += "local5.debug            /var/log/canary.log\n\n"

      conf += "# Log output from canaries to a dedicated log file so that\n"
      conf += "# users are not distracted by debug information.\n"
      conf += "kern.=debug              /var/log/kernel.debug\n\n"

      conf += "# Log crond.\n"
      conf += "# cron.service reached its max tasks limit\n"
      conf += "if ($syslogfacility-text == \"cron\") then {\n"
      conf += "   if ($msg contains \"CAN'T FORK\") then {\n"
      conf += "      action(type=\"omfile\"\n"
      conf += "             file=\"/var/log/eos\"\n"
      conf += "             action.execOnlyOnceEveryInterval=\"3600\"\n"
      conf += "             template=\"RSYSLOG_FileFormat\")\n"
      conf += "   }\n"
      conf += "   # All crond generated syslogs go into /var/log/cron\n"
      conf += "   action(type=\"omfile\" file=\"/var/log/cron\")\n"
      conf += "}\n"

      # Syslog SSO action
      if syncSyslog:
         conf += "# All log messages are also sent to rsyslogd "
         conf += "on the peer supervisor"
         conf += self._logPeerSupeTargets()

      # logging trap for system messages
      if remoteTargets:
         conf += "# Logging trap for system messages\n"
         logTrapCfgWithSpoof = self._logTrapSystemWithUdpSpoof(
                                                        cfg.loggingTrapSystem )
         logTrapCfgNoSpoof = self._logTrapSystemNoSpoof( cfg.loggingTrapSystem,
                                                         remoteTargets )
         t5( "cfgWithSpoof: %s" % logTrapCfgWithSpoof )
         t5( "cfgWithNOspoof: %s" % logTrapCfgNoSpoof )

         for tgt in remoteTargets:
            if "OMUDPSpoof" in tgt[ 0 ]:
               conf += tgt[ 0 ]
               conf += logTrapCfgWithSpoof
         conf += '\n' + logTrapCfgNoSpoof
      conf += self._saveToLocalSysLog( cfg.loggingBufferedSystem )

      # replace the #HOSTNAME_PLACEHOLDER# string with the correct string
      if cfg.hostnameFormat == 'fqdn':
         conf = conf.replace ( "#HOSTNAME_PLACEHOLDER#", self.netStatus_.fqdn )
      elif cfg.hostnameFormat == 'ipv4':
         ipv4Addr = self.ipAddressFormat()
         conf = conf.replace ( "#HOSTNAME_PLACEHOLDER#", ipv4Addr )
      else:
         conf = conf.replace ( "#HOSTNAME_PLACEHOLDER#", self.getHostname() )

      # Include external configuration directory
      conf += "\n# Include external configuration directory\n"
      conf += "include( file=\"%s\"" % LoggingDefs.externalConfigDir
      conf += " mode=\"optional\" )\n"

      # Cleanup hostnames that are not in use
      names = set( self.dns_.dnsRecords.keys() ) - set( self.syslogHostnames_ )
      for name in names:
         bt50( f"Stop resolving {name}" )
         self.dns_.finishHost( name )
         self.dns_.removeRecord( name )
         self.dns_.usedIPs.pop( name, None )

      return conf

   def getHostname( self ):
      try:
         hostname = socket.gethostname().split( '.' )[ 0 ]
         return hostname if hostname else 'localhost'
      except OSError:
         return 'localhost'

   def ForwardRootLoginMsg( self, remoteForwarding=False ):
      # Check "logging level AAA"
      aaaLogLevel = self.loggingConfig.facilityLogLevel.get( "AAA" )
      if not aaaLogLevel:
         return True
      sevLimit = self.rootLoginSev
      # We should log this when AAA level is not higher than sevLimit
      # For example, if sevLimit is notice, log when AAA level is
      # notice, info or debug
      return LoggingLib.higherTaccSeverity( aaaLogLevel, sevLimit ) == sevLimit

   #
   # For every new log destination we need to ensure that
   # 1. All messages from the EOS log facility with the
   #    severity passed in are sent to the destination.
   #
   # 2. All log messages from any other facility with severity
   #    greater than critical are sent to the destination
   #    This is used to ensure that important logging activity from
   #    any running process on the system is captured under the
   #    control of the logging CLI
   def newLogLocalDestEntry( self, destination, severity, otherFacilities=None ):
      entry = ""
      facility = {}
      facility[ self.eosFacilityName_ ] = severity
      if otherFacilities is not None:
         for t in otherFacilities:
            facility[ t[0] ] = t[1:]
      else:
         facility[ '*' ] = ( 'logCritical', None )
      entry += self.newSyslogEntry( facility, destination )

      def _ForwardRootLoginLocal():
         if not self.ForwardRootLoginMsg():
            return False
         sevLimit = self.rootLoginSev
         sevN, sevM = severity
         # Severity is a 2-tuple to specify the range:
         #  ( levelN, None ) - messages of severity level and up
         #  ( levelN, levelN ) - messages of severity levelN only
         #  ( levelN, levelM ) - messages between severity levelN and levelM
         if sevM is None or sevM == "logEmergency":
            return LoggingLib.higherTaccSeverity( sevN, sevLimit ) == sevLimit
         elif sevN == sevM:
            return sevN == sevLimit
         else:
            # messages from severity N to M
            if LoggingLib.lowerTaccSeverity( sevN, sevM ) == sevM:
               sevTemp = sevM # pylint: disable=consider-swap-variables
               sevM = sevN
               sevN = sevTemp
            if not LoggingLib.higherTaccSeverity( sevN, sevLimit ) == sevLimit:
               return False
            if not LoggingLib.lowerTaccSeverity( sevM, sevLimit ) == sevLimit:
               return False
            return True

      if self.loggingConfig.loggingRootLogin and _ForwardRootLoginLocal():
         # Adding config for root user messages
         entry += 'if $msg contains "root" and $msg contains "Accepted"'
         entry += ' and $syslogfacility-text == "authpriv" then'
         entry += '              %s;EOS_FileFormat_Root_SSH_Success\n' % destination
         entry += 'if $msg contains "Authentication failure for root"'
         entry += ' and $syslogfacility-text == "authpriv" then'
         entry += '              %s;EOS_FileFormat_Root_SSH_Fail\n' % destination
         entry += 'if $msg contains "ROOT LOGIN ON" then'
         entry += '              %s;EOS_FileFormat_Root_Success\n' % destination
         entry += 'if $msg contains "FAILED LOGIN" then'
         entry += '              %s;EOS_FileFormat_Root_Fail\n' % destination

      return entry

   def generateActionParamStr( self, actionParam ):
      netNsStr = ""
      tlsStr = ""
      deviceStr = ""
      srcAddrStr = ""

      if "netNs" in actionParam:
         netNsStr = 'NetworkNamespace="%s" ' % actionParam[ "netNs" ]
      if "device" in  actionParam:
         deviceStr = 'Device="%s" ' % actionParam[ "device" ]
      if "srcAddr" in actionParam:
         srcAddrStr = 'Address="%s" ' % actionParam[ "srcAddr" ]
      if "ca" in actionParam:
         tlsStr = 'streamdriver="ossl" streamdrivermode="1"' \
                  ' streamdriverauthmode="x509/name"' \
                  ' streamdriverpermittedpeers="{peerAddr}"' \
                  ' streamdrivercafile="{caFile}" '.format(
                        peerAddr=actionParam[ "peername" ],
                        caFile=actionParam[ "ca" ] )
         if "certificate" in actionParam:
            tlsStr += 'streamdrivercertfile="%s" ' % actionParam[ "certificate" ]
         if "key" in actionParam:
            tlsStr += 'streamdriverkeyfile="%s" ' % actionParam[ "key" ]
         if "crl" in actionParam:
            tlsStr += 'streamdrivercrlfile="%s" ' % actionParam[ "crl" ]
         versionList = []
         tlsVersionMap = [ ( SslConstants.tlsv1_3,
                             'TLSv1.3' ),
                           ( SslConstants.tlsv1_2,
                             'TLSv1.2' ),
                           ( SslConstants.tlsv1_1,
                             'TLSv1.1' ),
                           ( SslConstants.tlsv1,
                             'TLSv1' ) ]
         for mask, version in tlsVersionMap:
            if actionParam[ "tlsVersion" ] & mask:
               versionList.append( version )
         versionStr = 'Protocol=-ALL,%s' % ( ','.join( versionList ) )
         cipherStr = 'CipherString=%s' % actionParam[ "cipherSuite" ]
         cipherSuiteStr = "Ciphersuites=%s" % actionParam[ "cipherSuiteV1_3" ]
         priorityString = 'gnutlsPriorityString="{}\n{}\n{}"'.format( versionStr,
                                                                      cipherStr,
                                                                      cipherSuiteStr
                                                                    )
         tlsStr += priorityString

      actionParamStr = 'type="omfwd" target="{dstAddr}"' \
                       ' protocol="{protocol}" port="{port}"' \
                       ' {netNsStr}{deviceStr}{srcAddrStr}{tlsStr}'.format(
                           protocol=actionParam[ "protocol" ],
                           port=actionParam[ "port" ],
                           dstAddr=actionParam[ "dstAddr" ],
                           netNsStr=netNsStr,
                           deviceStr=deviceStr,
                           srcAddrStr=srcAddrStr,
                           tlsStr=tlsStr )
      return actionParamStr

   def newLogRemoteDestEntry( self, destination, severity,
                              actionParam, udpSpoof, cfg ):
      entry = ""

      defaultSeverityValue = 3

      if not udpSpoof:
         actionStr = 'action(' + self.generateActionParamStr( actionParam )

      def _getDest( formatStr ):
         destStr = ''
         if udpSpoof:
            destStr += '              %s;' % destination
            destStr += 'EOS_ForwardFormat_%s\n' % formatStr
         else:
            templateStr = 'template="EOS_ForwardFormat_%s")\n' % formatStr
            actionStrWithTemplate = actionStr + templateStr
            destStr += '              %s' % actionStrWithTemplate
         return destStr

      for severityValue in range( defaultSeverityValue ):
         severityName = LoggingLib.severityValueToTacc( severityValue )
         severityName = LoggingLib.severityTaccToSyslog( severityName )
         entry += '*.=%s;local4.!*;authpriv.!*' % severityName
         entry += _getDest( severityName )

      severityValue = LoggingLib.severityTaccToValue( severity )
      maxSeverityValue = max( severityValue + 1, defaultSeverityValue )

      for severityValue in range( maxSeverityValue ):
         severityName = LoggingLib.severityValueToTacc( severityValue )
         severityName = LoggingLib.severityTaccToSyslog( severityName )
         entry += 'local4.=%s;authpriv.!*' % severityName
         entry += _getDest( severityName )

      def _ForwardRootLoginRemote():
         if not self.ForwardRootLoginMsg():
            return False
         sevLimit = self.rootLoginSev
         return LoggingLib.higherTaccSeverity( cfg.loggingTrapSeverity,
                                               sevLimit ) == sevLimit

      if self.loggingConfig.loggingRootLogin and _ForwardRootLoginRemote():
         # Adding config for root user messages
         entry += 'if $msg contains "root" and $msg contains "Accepted"'
         entry += ' and $syslogfacility-text == "authpriv" then'
         entry += _getDest( "Root_SSH_Success" )
         entry += 'if $msg contains "Authentication failure for root"'
         entry += ' and $syslogfacility-text == "authpriv" then'
         entry += _getDest( "Root_SSH_Fail" )
         entry += 'if $msg contains "ROOT LOGIN ON" then'
         entry += _getDest( "Root_Success" )
         entry += 'if $msg contains "FAILED LOGIN" then'
         entry += _getDest( "Root_Fail" )

      return entry

   def newSyslogEntry( self, facility, dest ):
      entry = ""

      # add all the filters
      for facname in facility:
         # Severity is a 2-tuple to specify the range:
         #  ( levelN, None ) - messages of severity level and up
         #  ( levelN, levelN ) - messages of severity levelN only
         #  ( levelN, levelM ) - messages between severity levelN and levelM
         sevN, sevM = facility[ facname ]
         if sevM is None or sevM == "logEmergency":
            # messages from severity N and up
            sevStrN = LoggingLib.severityTaccToSyslog( sevN )
            entry += f"{facname}.{sevStrN};"
         elif sevN == sevM:
            # messages with severity N
            sevStrN = LoggingLib.severityTaccToSyslog( sevN )
            entry += f"{facname}.={sevStrN};"
         else:
            # messages from severity N to M
            if LoggingLib.lowerTaccSeverity( sevN, sevM ) == sevM:
               sevTemp = sevM # pylint: disable=consider-swap-variables
               sevM = sevN
               sevN = sevTemp
            sevStrN = LoggingLib.severityTaccToSyslog( sevN )
            sevStrM = LoggingLib.severityTaccToSyslog(
                           LoggingLib.incTaccSeverity( sevM, -1 ) )
            entry += f"{facname}.{sevStrN};{facname}.!{sevStrM};"

      # Always exclude authpriv so that password activity is not exposed
      entry += "authpriv.!*"

      # Keep kernel debug messages out of the log because they are pure
      # debug messages that are not of interest to customers.
      entry += ";kern.!=debug"

      # Always exclude the local5 facility at level debug (canaries), so that
      # the user is not alarmed by harmless logs
      entry += ";local5.!=debug"

      # Keep crond messages out of the /var/log/messages.
      # Log archiver is invoked by crond once a minute. If cron is not excluded,
      # /var/log/messages gets updated and archived every minute.
      entry += ";cron.none"

      # Now add the destination"
      entry += "              %s\n" % dest

      return entry

class HostnameChangeReactor( Tac.Notifiee ):
   notifierTypeName = "System::NetStatus"

   def __init__( self, sysMgrNetStatus, configReactor ):
      Tac.Notifiee.__init__( self, sysMgrNetStatus )
      self.configReactor_ = configReactor

   @Tac.handler( 'fqdn' )
   def handleHostname( self ):
      self.configReactor_.forceRestart_ = True
      self.configReactor_.sync()

class LogMgrTimezoneSysdbReactor( Tac.Notifiee ):
   """ Reactor to the 'time/zone/status' object when a new timezone is set
       and populate timezone display string in local entity.
   """
   notifierTypeName = 'Time::Zone::Status'

   def __init__( self, status, timezoneInfo, configReactor ):
      Tac.Notifiee.__init__( self, status )
      self.status_ = status
      self.timezoneInfo_ = timezoneInfo
      self.configReactor_ = configReactor
      self.activity_ = Tac.ClockNotifiee()
      self.activity_.handler = self.updateTimeZoneStr
      self.activity_.timeMin = Tac.endOfTime
      self.timezoneDebugFile_ = os.environ.get( 'TIMEZONE_DEBUG_FILE' )
      self.handleZone()
      # schedule timezone update
      configReactor.handleTimestampTimezone()

   @Tac.handler( 'zone' )
   def handleZone( self ):
      tzStr = None
      if self.timezoneDebugFile_:
         try:
            # pylint: disable-next=consider-using-with
            tzStr = open( self.timezoneDebugFile_ ).read().strip()
         except OSError:
            pass
      if not tzStr:
         tzStr = time.strftime( '%Z' )
      self.timezoneInfo_.timezoneStr = tzStr

   def updateTimeZoneStr( self ):
      self.handleZone()
      # schedule for the next hour
      if self.timezoneDebugFile_:
         interval = 1 # for testing
      else:
         # the next hour
         interval = 3600 - time.time() % 3600
      self.activity_.timeMin = Tac.now() + interval

   def schedule( self, enabled ):
      if enabled:
         qt0( "schedule timezoneStr update" )
         self.updateTimeZoneStr()
      else:
         qt0( "unschedule timezoneStr update" )
         self.activity_.timeMin = Tac.endOfTime

class LogMgrTimezoneInfoReactor( Tac.Notifiee ):
   """ Reactor to local TimeZoneInfo entity when the display string of timezone
       changes.
   """
   notifierTypeName = 'LogMgr::TimeZoneInfo'

   def __init__( self, timezoneInfo, config, configReactor ):
      Tac.Notifiee.__init__( self, timezoneInfo )
      self.timezoneInfo_ = timezoneInfo
      self.logConfig_ = config
      self.configReactor_ = configReactor

   @Tac.handler( 'timezoneStr' )
   def handleTimezoneStr( self ):
      qt0( "Update timezone for rsyslog logging." )
      self.configReactor_.forceRestart_ = True
      self.configReactor_.sync()

class SshLogConfigReactor( Tac.Notifiee ):
   """ Reactor to the 'ssh/config/loggingTargetEnabled' object when writing
       ssh/sshd target syslog locally is set """
   notifierTypeName = 'Mgmt::Ssh::Config'

   def __init__( self, sshConfig, configReactor ):
      Tac.Notifiee.__init__( self, sshConfig )
      self.configReactor_ = configReactor

   @Tac.handler( 'loggingTargetEnabled' )
   def handleSshLogConfig( self ):
      self.configReactor_.forceRestart_ = True
      self.configReactor_.sync()

class LogMgrSslReactor( MgmtSecuritySslStatusSm.SslStatusSm ):
   __supportedFeatures__ = [ SslFeature.sslFeatureCertKey,
                             SslFeature.sslFeatureTrustedCert,
                             SslFeature.sslFeatureChainedCert,
                             SslFeature.sslFeatureCrl,
                             SslFeature.sslFeatureTls,
                             SslFeature.sslFeatureCipher,
                             SslFeature.sslFeatureCipherV1_3 ]

   def __init__( self, config, status, profileName, configReactor ):
      self.profileName = profileName
      self.configReactor_ = configReactor
      super().__init__( status, profileName,
                        'LogMgr' )

   def handleProfileState( self ):
      self.configReactor_.sync()

   def handleProfileDelete( self ):
      self.configReactor_.sync()

class LogMgr( SuperServer.SuperServerAgent ):
   def __init__( self, entityManager ):
      t5( "LogMgr initialized" )
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      mg = entityManager.mountGroup()
      emRoot = entityManager.root()
      loggingConfig = mg.mount( 'sys/logging/config', 'LogMgr::LogConfig', 'r' )
      loggingConfigReq = mg.mount( 'sys/logging/configReq',
                                     'LogMgr::LogConfigReq',
                                     'r' )
      loggingStatus = mg.mount( Cell.path( 'sys/logging/status' ),
                                             'LogMgr::LogStatus', 'wf' )
      sysMgrNetStatus = mg.mount( Cell.path( 'sys/net/status' ),
                                  'System::NetStatus', 'r' )
      sslConfig = mg.mount( 'mgmt/security/ssl/config',
                                    'Mgmt::Security::Ssl::Config', 'r' )
      sslStatus = mg.mount( 'mgmt/security/ssl/status',
                                    'Mgmt::Security::Ssl::Status', 'r' )
      Tac.Type( "Ira::IraIpStatusMounter" ).doMountEntities( mg.cMg_, True, True )
      ipStatus = mg.mount( "ip/status", "Ip::Status", "r" )
      ip6Status = mg.mount( "ip6/status", "Ip6::Status", "r" )
      allVrfStatusLocal = mg.mount( Cell.path( 'ip/vrf/status/local' ),
                                    'Ip::AllVrfStatusLocal', 'r' )
      # Mount timezone status object.
      zoneStatus = mg.mount( Cell.path( 'sys/time/zone/status' ),
                             'Time::Zone::Status', 'r' )

      # Mount ssh config object
      sshConfig = mg.mount( 'mgmt/ssh/config', 'Mgmt::Ssh::Config', 'r' )

      # MatchList config
      matchListConfig = mg.mount( 'matchlist/config/cli',
            'MatchList::Config', 'r' )

      allIntfStatusDir = self.intfStatusAll()

      self.service_ = None
      self.hostnameConfigReactor_ = None
      self.vrfStatusReactor_ = None
      self.statusSysdb_ = None
      self.loggingStatusReactor_ = None
      self.timezoneInfo_ = Tac.newInstance( "LogMgr::TimeZoneInfo" )
      self.timezoneReactor_ = None
      self.timezoneInfoReactor_ = None
      self.configReactorReq = None
      self.vrfStateReactors_ = None
      self.archiveReactors_ = None
      self.sshLogConfigReactor_ = None

      def _finish():
         global defaultVrf
         defaultVrf = Tac.newInstance( 'L3::VrfName', 'temp-status' ).defaultVrf
         self.statusSysdb_ = loggingStatus
         configReactor = ConfigReactor( loggingConfig,
                                        loggingStatus, sysMgrNetStatus,
                                        ipStatus, ip6Status, allVrfStatusLocal,
                                        matchListConfig, sshConfig, sslConfig,
                                        sslStatus,
                                        allIntfStatusDir, weakref.proxy( self ),
                                        emRoot )
         self.service_ = configReactor
         # This reacts to the 'time/zone/status' object and update the
         # the timezone display in 'logging/config' object.
         self.timezoneReactor_ = LogMgrTimezoneSysdbReactor( zoneStatus,
                                                             self.timezoneInfo_,
                                                             configReactor )
         self.timezoneInfoReactor_ = LogMgrTimezoneInfoReactor( self.timezoneInfo_,
                                                                loggingConfig,
                                                                configReactor )

         self.hostnameConfigReactor_ = HostnameChangeReactor( sysMgrNetStatus,
                                                              configReactor )
         # This reacts to the 'mgmt/ssh/config' object and write
         # ssh/sshd syslog to /var/log/messages
         self.sshLogConfigReactor_ = SshLogConfigReactor( sshConfig, configReactor )

         self.configReactorReq = ConfigReactorReq( loggingConfig,
                                                   loggingConfigReq,
                                                   loggingStatus,
                                                   configReactor )
         self.vrfStateReactors_ = Tac.collectionChangeReactor(
                                          allVrfStatusLocal.vrf,
                                          VrfStateReactor,
                                          reactorArgs=( configReactor, ),
                                          reactorTakesKeyArg=True )

         # If we're the Alternate Sup on an SSO we can't update config
         if self.redundancyProtocol() == 'sso' and not self.active():
            t0( 'LogMgrArchive: SSO-Standby no op' )
            return

      mg.close( _finish )

   def onProtocolChange( self, protocol ):
      self.service_.sync()

   def warm( self ):
      if self.service_:
         return self.service_.warm()
      t6( "syslog self.service_ does not exist, so not warm" )
      return False

def Plugin( ctx ):
   ctx.registerService( LogMgr( ctx.entityManager ) )
