#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import QuickTrace
import SuperServer
import Tac
import Tracing
import subprocess
import os
import signal
import shlex
import errno
import fnmatch
# pylint: disable-next=consider-using-from-import
import Toggles.RadiusToggleLib as RadiusToggleLib
from RadsecLib import generateRadsecProxyConf
from Radius import radsecproxyInternalPort

__defaultTraceHandle__ = Tracing.Handle( 'SuperServerPlugin::Radsec' )
th = Tracing.defaultTraceHandle()
t0 = th.trace0

qt0 = QuickTrace.trace0
qt1 = QuickTrace.trace1

def unlinkFile( fileName ):
   try:
      os.unlink( fileName )
   except OSError as e:
      if e.errno != errno.ENOENT:
         raise

# Class which manages the radsecproxy service per vrf
class RadsecVrfService( SuperServer.LinuxService ):
   notifierTypeName = 'Radsec::Status'

   def __init__( self, status ):
      self.status = status
      self.vrfName = status.vrfName
      qt0( 'RadsecrVrfService for vrf:', self.vrfName, 'initialized' )
      self.configFileName = '/etc/radsecproxy-' + self.vrfName + '.conf'
      self.logFile = '/var/log/radsecproxy-' + self.vrfName + '.log'
      self.pidfile = None
      self.initialized = False
      self.radsecproxyProcess = None
      self.timeUpdated = Tac.beginningOfTime
      self.processName = 'radsecproxy'
      self.client = [ ( '127.0.0.1', 'arastra' ), ( '::1', 'arastra' ) ]
      SuperServer.LinuxService.__init__( self, self.processName, self.processName,
                                         status, self.configFileName )

   def conf( self ):
      qt0( 'Generating conf file for vrf:', self.vrfName )
      return generateRadsecProxyConf(
         self.status, self.logFile, self.client,
         listenPort=radsecproxyInternalPort,
         coaEnabled=RadiusToggleLib.toggleRadsecCoaToggleEnabled() )

   def serviceProcessWarm( self ):
      if not self.radsecproxyProcess:
         return False

      if self.isProcessKilled():
         # the process exited
         self.radsecproxyProcess = None
         return False

      return True

   def serviceEnabled( self ):
      return self.status.version > 0

   def isProcessKilled( self ):
      # pylint: disable-next=singleton-comparison
      return self.radsecproxyProcess.poll() != None

   @Tac.handler( 'version' )
   def handleVersion( self ):
      # Version may get updated multiple times in burst leading to calling
      # self.sync like during configure replace which in turn may restart the service
      # multiple number of times.
      # This has been taken care of in parent class self.sync by implementing a
      # codef.
      self.sync()

   @Tac.handler( 'forceRestart' )
   def handleForceRestart( self ):
      self.restartService()

   # Overriding the base class onAttribute function which along with
   # Tac.Notifiee also calls self.sync(). It results in restarting the service
   # whenever any attribute of the notifier changes.
   #
   # We don't want to call self.sync and in inturn restart the service everytime
   # when any attribute changes except Version.
   def onAttribute( self, attr, key ):
      Tac.Notifiee.onAttribute( self, attr, key )

   # Overriding the base class function to check service health. Linux service
   # keeps checking the health of the service periodically i.e every 60 seconds.
   # If the service is not running, it restarts the service.
   def _checkServiceHealth( self ):
      if not self.serviceEnabled():
         return

      doRestart = False

      if self.radsecproxyProcess and self.isProcessKilled():
         self.radsecproxyProcess = None
         doRestart = True
         qt1( 'No radsecproxy process for vrf:', self.vrfName )

      if doRestart:
         self.restartService()

      self.healthMonitorActivity_.timeMin = Tac.now() + self.healthMonitorInterval

   def startService( self ):
      qt0( 'Starting radsecproxy for vrf:', self.vrfName )

      if not self.initialized:
         self.initialized = True

      if self.radsecproxyProcess:
         if not self.radsecproxyProcess.poll():
            qt1( 'radsecproxy process with pid:', self.radsecproxyProcess.pid,
                 'is already running' )
            return
      if not self.serviceEnabled():
         qt1( 'Incomplete Config, hence returning' )
         return

      cmd = 'sudo ip netns exec '
      if self.vrfName == 'default':
         cmd += self.vrfName
      else:
         cmd += 'ns-%s' % self.vrfName # pylint: disable=consider-using-f-string
      cmd += ' /usr/sbin/radsecproxy -c ' + self.configFileName + ' -d 4 -f'
      t0( 'Radsec:', self.vrfName, cmd )
      radsecCmdArgs = shlex.split( cmd )
      # pylint: disable=subprocess-popen-preexec-fn,consider-using-with
      self.radsecproxyProcess = subprocess.Popen( radsecCmdArgs,
                                                  stderr=open( self.logFile, 'a' ),
                                                  preexec_fn=os.setsid )
      # pylint: enable=subprocess-popen-preexec-fn,consider-using-with

      if self.radsecproxyProcess:
         self.timeUpdated = Tac.now()
         self.starts += 1
         self.pidfile = '/var/run/radsecproxy-' + self.vrfName + '.pid'
         # pylint: disable-next=consider-using-with
         open( self.pidfile, 'w' ).write( str( self.radsecproxyProcess.pid ) )
         qt0( 'radsecproxy process for vrf:', self.vrfName, 'started with pid:',
              self.radsecproxyProcess.pid )
      else:
         qt0( 'radsecproxy process for vrf:', self.vrfName, 'is null' )

   def stopService( self ):
      qt0( 'Stopping radsecproxy for vrf:', self.vrfName )
      if self.radsecproxyProcess:
         os.killpg( self.radsecproxyProcess.pid, signal.SIGKILL )
         qt1( 'radsecproxy process with pid:', self.radsecproxyProcess.pid,
              'killed' )
         try:
            # pylint: disable-msg=W0108
            Tac.waitFor( lambda: self.isProcessKilled(),
                         description='radsecproxyProcess to be killed',
                         sleep=True )
         except ( Tac.SystemCommandError, Tac.Timeout ):
            qt1( 'Error/Timeout while trying to kill radsecproxy for vrf:',
                 self.vrfName )
            return
         self.radsecproxyProcess = None
      self.stops += 1

   def restartService( self ):
      qt0( 'Radsec : restartService called' )
      self.stopService()
      self.startService()
      self.stops -= 1
      self.starts -= 1
      self.restarts += 1

class RadsecStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Radsec::StatusDir'

   def __init__( self, notifier ):
      qt0( 'RadsecStatusReactor __init__ called' )
      Tac.Notifiee.__init__( self, notifier )
      self.notifier = notifier
      self.radsecVrfService = {}

      self.cleanup( '/var/run/', 'pid' )

      for vrf in self.notifier.status:
         self.handleStatus( vrf )

   def cleanup( self, directory, fileExt ):
      qt0( 'Radsec cleanup called for directory:', directory, 'fileExt:', fileExt )
      for fl in os.listdir( directory ):
         # pylint: disable-next=consider-using-f-string
         if fnmatch.fnmatch( fl, 'radsecproxy-*.%s' % fileExt ):
            if fileExt == 'pid':
               with open( directory + fl ) as f:
                  pid = f.readline()
                  try:
                     os.killpg( int( pid ), signal.SIGKILL )
                     qt1( 'radsecproxy process with pid:', pid, 'killed' )
                  except: # pylint: disable-msg=bare-except
                     # pylint: disable-next=consider-using-f-string
                     qt1( 'Failed to kill an old process with pid %s' % pid )
            # Delete the file
            try:
               Tac.run( [ 'rm', directory + fl ], asRoot=True )
            except: # pylint: disable-msg=bare-except
               # pylint: disable-next=consider-using-f-string
               qt1( 'Failed to delete file %s' % fl )

   @Tac.handler( 'status' )
   def handleStatus( self, vrfName ):
      qt0( 'handleStatus called for vrf:', vrfName )
      if vrfName in self.notifier.status:
         if vrfName not in self.radsecVrfService:
            radsecService = RadsecVrfService( self.notifier.status.get( vrfName ) )
            self.radsecVrfService[ vrfName ] = radsecService
      else:
         if vrfName in self.radsecVrfService:
            t0( 'Call stop service and delete vrfname:', vrfName,
                'from self.radsecVrfService' )
            self.radsecVrfService[ vrfName ].stopService()
            try:
               unlinkFile( ( self.radsecVrfService[ vrfName ] ).configFileName )
               unlinkFile( ( self.radsecVrfService[ vrfName ] ).pidfile )
               unlinkFile( ( self.radsecVrfService[ vrfName ] ).logFile )
            except:  # pylint: disable-msg=bare-except
               qt1( 'Unable to unlink some file' )
            del self.radsecVrfService[ vrfName ]

class RadiusTLS( SuperServer.SuperServerAgent ):
   def __init__( self, entityManager ):
      qt0( 'RadiusTLS __init__ called' )
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      mg = entityManager.mountGroup()
      self.radiusTlsServerConfigSm = None
      self.config = mg.mount( 'security/aaa/radius/config',
                              'Radius::Config', 'r' )
      self.sslStatus = mg.mount( 'mgmt/security/ssl/status',
                                 'Mgmt::Security::Ssl::Status', 'r' )
      self.ipStatus = mg.mount( 'ip/status', 'Ip::Status', 'r' )
      self.ip6Status = mg.mount( 'ip6/status', 'Ip6::Status', 'r' )
      self.radsecConfig = mg.mount( 'security/aaa/radsec/config',
                                    'Radsec::ConfigDir', 'r' )
      self.statusDir = mg.mount( 'security/aaa/radsec/status',
                                 'Radsec::StatusDir', 'w' )

      def _finished():
         # do not start RadsecTlsServerStatusSm if not on active supe
         if self.active():
            self.onSwitchover( None )
      qt0( '_finished decalaration done' )
      mg.close( _finished )

   def onSwitchover( self, protocol ):
      qt0( 'onSwitchover called for protocol:', protocol )
      # when switchover happens from standby to active, start the reactors
      self.notifiee_ = RadsecStatusReactor( self.statusDir )
      self.radiusTlsServerConfigSm = Tac.newInstance(
         'Radsec::RadiusTlsServerConfigSm', self.config,
         self.sslStatus, self.ipStatus, self.ip6Status, self.radsecConfig,
         self.statusDir )

def Plugin( ctx ):
   qt0( 'Radsec Plugin registered' )
   ctx.registerService( RadiusTLS( ctx.entityManager ) )
