# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import errno
import weakref
import SuperServer
import os
import Cell
import QuickTrace
import Tracing
from IpLibConsts import DEFAULT_VRF
from DeviceNameLib import eosIntfToKernelIntf
import shutil
from SsdpUtils import ssdpSocketFile

# pkgdeps: rpm MiniSsdpd
# pkgdeps: rpm MiniSsdpd-bin

t0 = Tracing.trace0
__defaultTraceHandle__ = Tracing.Handle( 'SsdpServer' )

qv = QuickTrace.Var
qt0 = QuickTrace.trace0
qt1 = QuickTrace.trace1

def vrfNamespace( vrfName ):
   if vrfName == DEFAULT_VRF:
      return ""
   return "ns-%s" % vrfName

def confFile( vrfName ):
   if vrfName == DEFAULT_VRF:
      return "/etc/minissdp.conf"
   return "/etc/minissdp_%s.conf" % vrfNamespace( vrfName )

class MiniSsdpService( SuperServer.LinuxService ):
   serviceName = "minissdpd"
   notifierTypeName = 'Ssdp::VrfStatus'

   _enabled = True

   def __init__( self, vrf, config, vrfStatus ):
      t0( "MiniSssdpService init ", vrf )
      self.config_ = config
      self.vrfStatus_ = vrfStatus
      self.vrfName_ = vrf

      if self.vrfName_ != DEFAULT_VRF:
         defaultInitFile = "/etc/init.d/minissdpd"
         self.serviceName = "{}-{}".format( "minissdpd", self.vrfName_ )
         vrfInitFile = "/etc/init.d/%s" % self.serviceName
         with open( defaultInitFile ) as f:
            with open( vrfInitFile, "w" ) as d:
               for line in f:
                  if line.startswith( 'NS=' ):
                     line = 'NS="%s"\n' % vrfNamespace( self.vrfName_ )
                  elif line.strip().startswith( 'status minissdpd' ):
                     line = '       status -p ${pidfile} minissdpd\n'
                  d.write( line )
         shutil.copymode( defaultInitFile, vrfInitFile )

      SuperServer.LinuxService.__init__( self, self.serviceName,
                                         self.serviceName,
                                         self.config_,
                                         confFile( self.vrfName_ ), sync=False,
                                         asRoot=self.vrfName_ != DEFAULT_VRF )

   def conf( self ):
      cnf = "-s %s\n" % ssdpSocketFile( self.vrfName_ )
      # config interfaces
      for intf in self.vrfStatus_.interface:
         cnf += "-i %s\n" % eosIntfToKernelIntf( intf )
      return cnf

   def serviceProcessWarm( self ):
      return True

   def restartService( self ):
      t0( "MiniSsdpService::restartService", self.vrfName_ )
      self.vrfStatus_.serverState = "disabled"
      SuperServer.LinuxService.restartService( self )
      self.vrfStatus_.serverState = "enabled"

   def startService( self ):
      t0( "MiniSsdpService::startService", self.vrfName_ )
      self.vrfStatus_.serverState = "disabled"
      SuperServer.LinuxService.startService( self )
      self.vrfStatus_.serverState = "enabled"

   # this will decide whether the service will be started/stopped
   # and set the status
   def serviceEnabled( self ):
      enabled = True
      if not self._enabled:
         enabled = False
      elif not self.vrfStatus_.interface:
         enabled = False
      else:
         # disabled for the vrf
         vrfConfig = self.config_.vrfConfig.get( self.vrfName_ )
         if vrfConfig and vrfConfig.serverState == "disabled":
            enabled = False

      if enabled:
         self.vrfStatus_.serverState = "enabled"
      else:
         self.vrfStatus_.serverState = "disabled"
      return enabled

   # this will disable/enable the service regardless of its VrfConfig settings
   # for example when the VRF isn't active or the whole SSDP is disabled
   def disableService( self ):
      t0( "MiniSsdpService::disableService", self.vrfName_ )
      self._enabled = False

   def enableService( self ):
      t0( "MiniSsdpService::enableService", self.vrfName_ )
      self._enabled = True

   def cleanupService( self ):
      t0( "MiniSsdpService::cleanupService", self.vrfName_ )
      SuperServer.LinuxService.cleanupService( self )
      if self.vrfName_ != DEFAULT_VRF:
         # the init script must have been created
         os.unlink( "/etc/init.d/%s" % self.serviceName )
         # the config file may not exist
         try:
            os.unlink( confFile( self.vrfName_ ) )
         except OSError as e:
            if e.errno != errno.ENOENT:
               raise

class L3StatusReactor( Tac.Notifiee ):
   notifierTypeName = "L3::Intf::Status"

   def __init__( self, l3Status, ipIntfStatusReactor ):
      self.ipIntfStatusReactor_ = ipIntfStatusReactor
      Tac.Notifiee.__init__( self, l3Status )

   @Tac.handler( 'vrf' )
   def handleVrf( self ):
      self.ipIntfStatusReactor_.handleVrf()

   @Tac.handler( 'vrfId' )
   def handleVrfId( self ):
      self.ipIntfStatusReactor_.handleVrf()

class IntfStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Ip::IpIntfStatus'

   def __init__( self, notifier, master ):
      t0( "IntfStatusReactor::__init__" )
      self.master_ = weakref.proxy( master )
      Tac.Notifiee.__init__( self, notifier )
      self.l3StatusReactor_ = L3StatusReactor( self.notifier_.l3Status,
                                               weakref.proxy( self ) )
      self.master_.sync()

   def handleVrf( self ):
      t0( "IntfStatusReactor::handleVrf vrf changed: ", self.notifier_.intfId,
            self.notifier_.vrf )
      self.master_.sync()

   @Tac.handler( 'activeAddrWithMask' )
   def handleActiveAddrWithMask( self ):
      t0( "IntfStatusReactor::handleActiveAddrWithMask IP changed: ",
            self.notifier_.intfId,
            self.notifier_.activeAddrWithMask )
      self.master_.sync()

   def close( self ):
      t0( "IntfStatusReactor::close" )
      self.master_.sync()
      Tac.Notifiee.close( self )

class VrfStatusLocalReactor( Tac.Notifiee ):
   notifierTypeName = 'Ip::VrfStatusLocal'

   def __init__( self, notifier, master ):
      self.master_ = weakref.proxy( master )
      Tac.Notifiee.__init__( self, notifier )

   @Tac.handler( 'state' )
   def handleState( self ):
      self.master_.syncService( self.notifier_.vrfName )

class VrfConfigReactor( Tac.Notifiee ):
   notifierTypeName = "Ssdp::VrfConfig"

   def __init__( self, notifier, master ):
      self.master_ = weakref.proxy( master )
      Tac.Notifiee.__init__( self, notifier )

   @Tac.handler( 'serverState' )
   def handleServerState( self ):
      self.master_.syncService( self.notifier_.name )

   @Tac.handler( 'enableIpv6' )
   def enableIpv6( self ):
      self.master_.syncService( self.notifier_.name )


class SsdpServer( SuperServer.SuperServerAgent, SuperServer.GenericService ):

   # Reactor to the 'upnp/ssdp/config' entity
   notifierTypeName = 'Ssdp::Config'

   vrfConfigReactor_ = None
   allVrfStatusLocalReactor_ = None
   intfStatusReactor_ = None

   def __init__( self, entityManager ):
      t0( 'SsdpServer::__init__' )

      SuperServer.SuperServerAgent.__init__( self, entityManager )

      self.service_ = {}

      mg = entityManager.mountGroup()
      mounts = mg.mountProfile( "Ssdp-SuperServer-Plugin" )
      self.config_ = mounts[ 'upnp/ssdp/config' ]
      self.status_ = mounts[ 'upnp/ssdp/status' ]
      self.allVrfStatusLocal_ = mounts[ Cell.path( 'ip/vrf/status/local' ) ]
      self.ipStatus_ = mounts[ 'ip/status' ]
      Tac.Type( "Ira::IraIpStatusMounter" ).doMountEntities( mg.cMg_, True, False )

      def _finished():
         t0( 'SsdpServer::_finished' )
         # This will register the instance as a notifiee for Ssdp::Config changes
         SuperServer.GenericService.__init__( self, "SsdpServer", self.config_,
                                           sync=False )

         self.vrfConfigReactor_ = Tac.collectionChangeReactor(
            self.config_.vrfConfig, VrfConfigReactor,
            reactorArgs=( self, ) )

         self.allVrfStatusLocalReactor_ = Tac.collectionChangeReactor(
            self.allVrfStatusLocal_.vrf, VrfStatusLocalReactor,
            reactorArgs=( self, ) )

         self.intfStatusReactor_ = Tac.collectionChangeReactor(
            self.ipStatus_.ipIntfStatus, IntfStatusReactor,
            reactorArgs=( self, ) )

         self.sync()
         self.updateServerStatus()

      mg.close( _finished )

   def sync( self ):
      t0( 'SsdpServer::sync' )

      self.updateVrfStatus()

      # make sure all services are created
      for vrfName in self.status_.vrfStatus:
         self.createService( vrfName )

   def updateVrfStatus( self ):
      # delete interfaces that are not in the list
      for vrf in self.status_.vrfStatus.values():
         for intf in list( vrf.interface ):
            if not self.config_.interface.get( intf ):
               del vrf.interface[ intf ]
            else:
               # delete interfaces that are not in the current vrf anymore
               intfStatus = self.ipStatus_.ipIntfStatus.get( intf )
               if not intfStatus or intfStatus.vrf != vrf.name:
                  del vrf.interface[ intf ]

      # add interfaces from config to their corresponding
      # vrf status
      for intf in self.config_.interface:
         # the interface must have a status
         intfStatus = self.ipStatus_.ipIntfStatus.get( intf )
         if not intfStatus:
            t0( "intfStatus isn't set for the intf: ", intf )
            continue
         vrf = intfStatus.vrf
         if not intfStatus.activeAddrWithMask:
            t0( "activeAddrWithMask isn't set for the intf: ", intf )
            if self.status_.vrfStatus.get( vrf ):
               del self.status_.vrfStatus[ vrf ].interface[ intf ]
            continue
         if not self.status_.vrfStatus.get( vrf ):
            self.status_.vrfStatus.newMember( vrf )
            self.status_.vrfStatus[ vrf ].interface.add( intf )
         else:
            if not self.status_.vrfStatus[ vrf ].interface.get( intf ):
               self.status_.vrfStatus[ vrf ].interface.add( intf )

      # remove vrf's and services that have not interfaces
      for vrf in list( self.status_.vrfStatus ):
         # no interfaces
         if not self.status_.vrfStatus[ vrf ].interface:
            srv = self.service_.get( vrf )
            if srv:
               srv.stopService()
               srv.cleanupService()
               del self.service_[ vrf ]
               self.updateServerStatus()
            del self.status_.vrfStatus[ vrf ]

   # returns True if a new serice was actually created
   def createService( self, vrfName ):
      if not self.service_.get( vrfName ):
         srv = MiniSsdpService( vrfName, self.config_,
              self.status_.vrfStatus[ vrfName ] )
         self.service_[ vrfName ] = srv
      self.syncService( vrfName )

   def syncService( self, vrfName ):
      t0( 'SsdpServer::syncService', vrfName )

      # ignore if vrf is not present in status
      # this may happen when this function is called by
      # VrfStatusLocalReactor
      if not self.status_.vrfStatus.get( vrfName ):
         t0( 'SsdpServer::syncService service not present for ', vrfName )
         return

      srv = self.service_.get( vrfName )

      vrfEnabled = False
      if vrfName == DEFAULT_VRF:
         vrfEnabled = True
      else:
         vrfStatus = self.allVrfStatusLocal_.vrf.get( vrfName )
         if vrfStatus and vrfStatus.state == "active":
            vrfEnabled = True

      if vrfEnabled:
         srv.enableService()
      else:
         srv.disableService()

      self.updateServerStatus()
      srv.sync()

   def updateServerStatus( self ):
      self.status_.serviceCounter = len( self.service_ )
      activeCnt = 0
      for srv in self.service_.values():
         if srv.serviceEnabled():
            activeCnt += 1
      self.status_.serviceActiveCounter = activeCnt
      self.status_.ssdpServiceStarted = ( activeCnt != 0 )

def Plugin( ctx ):
   t0( "Registering SSDP server service" )
   ctx.registerService( SsdpServer( ctx.entityManager ) )
