# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=use-dict-literal
# pylint: disable=consider-using-f-string
# pylint: disable=consider-using-in

import Tac, SuperServer, Cell, signal, TcpdumpLib, QuickTrace, os, weakref
import SharedMem
import Smash
from SysConstants.if_h import IFF_RUNNING, IFF_UP
from IpLibConsts import DEFAULT_VRF
from Arnet.NsLib import popenMaybeInNetNs, DEFAULT_NS
from TcpdumpLib import fabricName
from pwd import getpwnam
from grp import getgrnam
import gc

# pylint: disable-msg=W0622
qv = QuickTrace.Var
qt0 = QuickTrace.trace0 # SERVICE STATUS: Errors, Start of Tcpdump service
qt1 = QuickTrace.trace1 # SESSION STATUS: Starting/stopping/resuming a session
qt2 = QuickTrace.trace2 # SESSION STATUS CONDITIONS: Timer/Interfaces/Enabled
qt3 = QuickTrace.trace3 # MISC: Reactors initialization and triggering/File handling

DeviceType = Tac.Type( 'Tcpdump::DeviceType' )
sessionPollerTimeout = 5
sessionExitTimeout = 30

###############################
# File Handling functions
###############################

def createDirectory( dirpath ):
   if not os.path.exists( dirpath ):
      # Hardcoded defaults from tcpdump and Eos-initscripts package
      tcpdump_uid = 72
      eosadmin_gid = 88
      try:
         tcpdump_uid = getpwnam( 'tcpdump' ).pw_uid
         eosadmin_gid = getgrnam( 'eosadmin' ).gr_gid
      except KeyError:
         # Live with the defaults, they are correct
         pass
      try:
         qt3( "Creating directory:", qv( dirpath ) )
         os.makedirs( dirpath )
         os.chown( dirpath, tcpdump_uid, eosadmin_gid )
         os.chmod( dirpath, 0o775 )
      except OSError as e:
         qt0( "Failed to create directory:", qv( dirpath ), "exception:", qv( e ) )
         return False
   return True

###############################
# Reactors
###############################

class TcpdumpSessionConfigReactor( Tac.Notifiee ):
   notifierTypeName = "Tcpdump::SessionConfig"

   def __init__( self, notifier, master ):
      qt3( "SessionConfig Reactor - Initialization for session",
           qv( notifier.name ) )
      self.master_ = master
      self.act_ = None
      if notifier.name in self.master_.status_.session:
         self.master_.clearSessionStatus( notifier.name )
      self.master_.status_.session.newMember( notifier.name )
      self.intfStatusLocalReactor_ = Tac.collectionChangeReactor(
         self.master_.intfStatusLocal_.intfStatusLocal, IntfStatusLocalReactor,
         reactorArgs=( weakref.proxy( self ), ),
         reactorFilter=lambda intfId: intfId == notifier.device )
      self.intfStatusLocalReactor_.inCollection = True
      self.ipIntfStatusReactor_ = Tac.collectionChangeReactor(
         self.master_.ipStatus_.ipIntfStatus, IpIntfStatusReactor,
         reactorArgs=( weakref.proxy( self ), ),
         reactorFilter=lambda intfId: intfId == notifier.device )
      self.ipIntfStatusReactor_.inCollection = True
      self.ip6IntfStatusReactor_ = Tac.collectionChangeReactor(
         self.master_.ip6Status_.intf, Ip6IntfStatusReactor,
         reactorArgs=( weakref.proxy( self ), ),
         reactorFilter=lambda intfId: intfId == notifier.device )
      self.ip6IntfStatusReactor_.inCollection = True
      Tac.Notifiee.__init__( self, notifier )
      self.handleEnabled()

   @Tac.handler( 'enabled' )
   def handleEnabled( self ):
      qt2( "SessionConfig Reactor - Handling enabled set to",
           qv( self.notifier_.enabled ) )
      sessionName = self.notifier_.name
      # If session is (re)enabled, (re)start the timer
      if self.notifier_.enabled and self.notifier_.duration != 0:
         timerDuration = self.notifier_.duration
         status = self.master_.status_
         sessionStatus = status.session[ sessionName ]
         sessionStatus.timerActive = True
         self.act_ = Tac.ClockNotifiee( handler=self.timeIsUp )
         self.act_.timeMin = Tac.now() + timerDuration

         qt2( "SessionConfig Reactor - Starting TIMER during",
              qv( timerDuration ), "seconds" )

      self.master_.updateSessionStatus( sessionName )

   def timeIsUp( self ):
      qt2( "SessionConfig Reactor - Time is UP!" )
      status = self.master_.status_
      sessionName = self.notifier_.name
      if sessionName in status.session:
         sessionStatus = status.session[ sessionName ]
         sessionStatus.timerActive = False
         self.master_.updateSessionStatus( sessionName )

   def close( self ):
      qt2( "SessionConfig Reactor - Deleting session",
           qv( self.notifier_.name ) )
      self.intfStatusLocalReactor_.close()
      self.ipIntfStatusReactor_.close()
      self.ip6IntfStatusReactor_.close()
      del self.master_.status_.session[ self.notifier_.name ]
      Tac.Notifiee.close( self )

class TcpdumpSessionStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Tcpdump::SessionStatus"

   def __init__( self, notifier, master ):
      qt3( "SessionStatus Reactor - Initialization for session",
           qv( notifier.name ) )
      self.master_ = master
      Tac.Notifiee.__init__( self, notifier )

   def close( self ):
      qt3( "SessionStatus Reactor - Stopping session",
           qv( self.notifier_.name ) )
      self.master_.stopTcpdump( self.notifier_.name )
      try:
         os.remove( TcpdumpLib.getCtxFileName( self.notifier_.pid ) )
      except OSError:
         pass
      Tac.Notifiee.close( self )


class KniReactor( Tac.Notifiee ):
   notifierTypeName = "KernelNetInfo::Root"

   def __init__( self, netNs, root, master ):
      self.netNs_ = netNs
      self.root_ = root
      self.master_ = master
      self.interfaces_ = {}
      Tac.Notifiee.__init__( self, root )

      for k in self.root_.interfaceByDeviceName:
         self.handleKniIntf( k )

   @Tac.handler( 'interfaceByDeviceName' )
   def handleKniIntf( self, key ):
      if key not in self.root_.interfaceByDeviceName:
         if key in self.interfaces_:
            del self.interfaces_[ key ]
         return

      interface = self.root_.interfaceByDeviceName[ key ]
      if key not in self.interfaces_ or \
         self.interfaces_[ key ].flags != interface.flags or \
         self.interfaces_[ key ].flagChgCount != interface.flagChgCount:
         self.handleFlags( interface )

      self.interfaces_[ key ] = interface

   def handleFlags( self, interface ):
      flags = interface.flags
      qt2( "Kernel Intf Reactor - Intf", qv( interface.deviceName ), "changed to",
           qv( flags ), "(IFF_UP/IFF_RUNNING):", qv( flags & IFF_UP ),
           qv( flags & IFF_RUNNING ) )
      self.master_.handleInterfaceStatus( interface.deviceName )

class MirroringSessionStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Mirroring::SessionStatus"
   def __init__( self, notifier, master ):
      self.master_ = master
      Tac.Notifiee.__init__( self, notifier )
      self.handleMirrorDevice()

   def close( self ):
      self.handleMirrorDevice()
      Tac.Notifiee.close( self )

   @Tac.handler( 'mirrorDeviceName' )
   def handleMirrorDevice( self ):
      qt2( "Mirror device reactor - mirror session", qv( self.notifier_.name ),
           "changed to", qv( self.notifier_.mirrorDeviceName ) )
      self.master_.handleMirroringSessionStatus( self.notifier_.name )

class VrfStatusLocalReactor( Tac.Notifiee ):
   notifierTypeName = "Ip::VrfStatusLocal"

   def __init__( self, vrfStatus, master ):
      self.master_ = master
      self.vrfName_ = vrfStatus.vrfName
      Tac.Notifiee.__init__( self, vrfStatus )

   @Tac.handler( 'state' )
   def handleState( self ):
      qt2( "VrfStatus Reactor - Vrf", qv( self.vrfName_ ), "changed state to",
           qv( self.notifier_.state ) )
      self.master_.handleVrfChange( self.vrfName_, self.notifier_.networkNamespace )
      if self.notifier_.state == 'deleted':
         self.master_.deleteKniReactor( self.vrfName_ )

   def close( self ):
      qt2( "VrfStatus Reactor - Vrf", qv( self.vrfName_ ), "closing" )
      self.master_.handleVrfChange( self.vrfName_, self.notifier_.networkNamespace )
      self.master_.deleteKniReactor( self.vrfName_ )
      Tac.Notifiee.close( self )


class IntfStatusLocalReactor( Tac.Notifiee ):
   notifierTypeName = "Interface::IntfStatusLocal"

   def __init__( self, intfStatusLocal, configReactor ):
      self.configReactor_ = configReactor
      self.intfStatusLocal_ = intfStatusLocal
      Tac.Notifiee.__init__( self, intfStatusLocal )

   @Tac.handler( 'netNsName' )
   def handleNetNsName( self ):
      qt2( "IntfStatus Local Reactor - Intf", qv( self.notifier_.intfId ),
           "changed netNsName to", qv( self.notifier_.netNsName ) )
      self.configReactor_.master_.updateSessionStatus(
         self.configReactor_.notifier_.name )

   def close( self ):
      qt2( "IntfStatus Local Reactor - Intf", qv( self.notifier_.intfId ),
           "closing" )
      Tac.Notifiee.close( self )

class L3StatusReactor( Tac.Notifiee ):
   notifierTypeName = "L3::Intf::Status"
   def __init__( self, l3Status, ipIntfStatusReactor ):
      self.ipIntfStatusReactor_ = ipIntfStatusReactor
      Tac.Notifiee.__init__( self, l3Status )

   @Tac.handler( 'vrf' )
   def handleVrf( self ):
      self.ipIntfStatusReactor_.handleVrf()

class IpIntfStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Ip::IpIntfStatus"

   def __init__( self, notifier, configReactor ):
      self.configReactor_ = configReactor
      self.ipIntfStatus_ = notifier
      Tac.Notifiee.__init__( self, notifier )
      self.l3StatusReactor_ = L3StatusReactor( self.notifier_.l3Status,
                                               weakref.proxy( self ) )

   def handleVrf( self ):
      qt2( "IpIntfStatus Reactor - Intf", qv( self.notifier_.intfId ),
           "changed vrf to", qv( self.notifier_.vrf ) )
      self.configReactor_.master_.updateSessionStatus(
         self.configReactor_.notifier_.name )

   @Tac.handler( 'activeAddrWithMask' )
   def handleIpAddr( self ):
      addr = self.notifier_.activeAddrWithMask.address
      if addr == '0.0.0.0':
         qt2( "IpIntfStatusReactor: IP address deleted" )
      else:
         qt2( "IpIntfStatusReactor: IP address added: ", qv( addr ) )
      self.configReactor_.master_.updateSessionStatus(
         self.configReactor_.notifier_.name )

   @Tac.handler( 'activeSecondaryWithMask' )
   def handleSecondaryIp( self, addr ):
      if addr not in self.notifier_.activeSecondaryWithMask:
         qt2( "IpIntfStatusReactor: secondary IP deleted" )
      else:
         qt2( "IpIntfStatusReactor: secondary IP added: ", qv( addr ) )
      self.configReactor_.master_.updateSessionStatus(
         self.configReactor_.notifier_.name )

   def close( self ):
      qt2( "IpIntfStatus Reactor - Intf", qv( self.notifier_.intfId ),
           "closing" )
      self.configReactor_.master_.updateSessionStatus(
         self.configReactor_.notifier_.name )
      self.l3StatusReactor_.close()
      Tac.Notifiee.close( self )

class Ip6IntfStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Ip6::IntfStatus"

   def __init__( self, notifier, configReactor ):
      qt2( "Ip6IntfStatusReactor init! w notifier ", qv( notifier ) )
      self.configReactor_ = configReactor
      self.ip6IntfStatus_ = notifier
      Tac.Notifiee.__init__( self, notifier )

   @Tac.handler( 'addr' )
   def handleIp6Addr( self, addr ):
      if addr not in self.notifier_.addr:
         qt2( "Ip6IntfStatusReactor: IP deleted" )
      else:
         qt2( "Ip6IntfStatusReactor: IP added: ", qv( addr ) )
      self.configReactor_.master_.updateSessionStatus(
         self.configReactor_.notifier_.name )

   def close( self ):
      self.configReactor_.master_.updateSessionStatus(
         self.configReactor_.notifier_.name )
      Tac.Notifiee.close( self )

###############################
# Tcpdump service
###############################

class TcpdumpService():
   """ Manages the tcpdump sessions based on configuration mounted at 
   /tcpdump/config"""

   def __init__( self, config, status, ipAddressDir, ip6AddressDir, deviceStatusAll,
                 intfStatusLocal, allVrfStatusLocal, ipStatus, ip6Status, em ):
      qt0( 'Creating Tcpdump service' )
      self.config_ = config
      self.status_ = status
      self.ipAddressDir_ = ipAddressDir
      self.ip6AddressDir_ = ip6AddressDir
      self.deviceStatusAll_ = deviceStatusAll
      self.intfStatusLocal_ = intfStatusLocal
      self.allVrfStatusLocal_ = allVrfStatusLocal
      self.ipStatus_ = ipStatus
      self.ip6Status_ = ip6Status
      self.em_ = em
      # started tcpdump processes
      self.subprocesses_ = dict()
      self.startPoller_ = dict()
      self.stopPoller_ = dict()
      # if start happens before stop is completed, queue it here
      self.startQueue_ = dict()
      self.kniIntfInfo_ = dict()
      self.kniReactor_ = dict()

      self.tcpdumpSessionConfigReactor_ = Tac.collectionChangeReactor(
         self.config_.session, TcpdumpSessionConfigReactor, reactorArgs=( self, ) )

      # create KernelNetworkInfo Root and Manager for DEFAULT_VRF
      self.createKniReactor()

      self.tcpdumpSessionStatusReactor_ = Tac.collectionChangeReactor(
         self.status_.session, TcpdumpSessionStatusReactor, reactorArgs=( self, ) )
      self.mirroringSessionStatusReactor_ = Tac.collectionChangeReactor(
         self.deviceStatusAll_[ DeviceType.mirror ], MirroringSessionStatusReactor,
         reactorArgs=( self, ) )
      self.vrfStatusLocalReactor_ = Tac.collectionChangeReactor(
         self.allVrfStatusLocal_.vrf, VrfStatusLocalReactor,
         reactorArgs=( weakref.proxy( self ), ) )
      qt0( 'Done creating Tcpdump service' )

   def createKniReactor( self, vrfName=DEFAULT_VRF ):
      """ Creates KernelNetInfo::InterfaceInfo, and reactor to interface
      collection in KernelNetInfo::Root. Each of these objects are created
      once per vrf """
      if vrfName not in self.kniIntfInfo_:
         # Determine the network namespace
         netNs = DEFAULT_NS
         if vrfName != DEFAULT_VRF:
            netNs = self.allVrfStatusLocal_.vrf[ vrfName ].networkNamespace

         qt3( "Creating kni reactors for vrf", qv( vrfName ) )
         shmemEm = SharedMem.entityManager( sysdbEm=self.em_ )
         kniStatus = shmemEm.doMount( "kni/ns/%s/status" % netNs,
                                      "KernelNetInfo::Status",
                                      Smash.mountInfo( 'keyshadow' ) )
         self.kniIntfInfo_[ vrfName ] = Tac.newInstance(
            'KernelNetInfo::InterfaceInfo', kniStatus )
         self.kniReactor_[ vrfName ] = \
            KniReactor( netNs, self.kniIntfInfo_[ vrfName ].root, self )

   def deleteKniReactor( self, vrfName ):
      """ Deletes KernelNetworkInfo::InterfaceInfo, and reactor to interface
      collection in KernelNetInfo::Root associate with the vrf. """
      if vrfName in self.kniIntfInfo_:
         qt3( "Deleting kni reactors for vrf", qv( vrfName ) )
         netNs = self.kniReactor_[ vrfName ].netNs_
         del self.kniReactor_[ vrfName ]
         del self.kniIntfInfo_[ vrfName ]
         shmemEm = SharedMem.entityManager( sysdbEm=self.em_ )
         shmemEm.doUnmount( "kni/ns/%s/status" % netNs )

   def interfaceUp( self, interfaceName, vrfName, netNsName, sessionConfig,
                    sessionStatus ):
      """ Returns True if interface's kernel status flags is IFF_RUNNING, else False.
      For a non-fabric interface, check if network namespace and corresponing VRF
      are consistent. """
      qt3( "Checking status of interface", qv( interfaceName ) )
      
      deviceType = sessionConfig.deviceType
      if deviceType == DeviceType.any:
         return True
      
      deviceStatusAll = self.deviceStatusAll_
      intfStatusLocal = self.intfStatusLocal_
      allVrfStatusLocal = self.allVrfStatusLocal_
      ipIntfStatus = self.ipStatus_.ipIntfStatus

      # determine the kernel device name
      intfIsUp = False
      try:
         deviceName = TcpdumpLib.deviceName( deviceStatusAll, interfaceName,
                                             deviceType )
         if not deviceName:
            qt3( "Interface", qv( interfaceName ), "not found" )
            raise TcpdumpLib.InvalidDeviceException( deviceType,
                  interfaceName, deviceStatusAll[ deviceType ]  )
      except TcpdumpLib.InvalidDeviceException as e:
         sessionStatus.reason = str( e )
         qt0( "Kernel interface for", qv( interfaceName ), "not found" )
         return False

      # get kernel interface status from KernelNetworkInfo::Root collection
      try:
         kernelIntfStatus = self.kniIntfInfo_[ vrfName ].root.\
                                interfaceByDeviceName[ deviceName ]
      except KeyError:
         qt0( "Kernel interface for", qv( interfaceName ),
              "not found in vrf", qv( vrfName ) )
         return False

      # kernel device is UP if kernel status flags is IFF_RUNNING
      intfIsUp = kernelIntfStatus.flags & IFF_RUNNING != 0

      # for fabric, checking kernel status flags is sufficient
      if deviceName == fabricName:
         qt3( "Fabric interface has IFF_RUNNING set to", qv( intfIsUp ) )
         return intfIsUp

      qt3( "Interface", qv( interfaceName ), "has IFF_RUNNING set to",
           qv( intfIsUp ) )
      if not intfIsUp:
         return intfIsUp

      # check if interface exists
      deviceColl = deviceStatusAll[ deviceType ].intfStatus \
            if deviceType == DeviceType.interface else \
            deviceStatusAll[ deviceType ]
      if deviceType in ( DeviceType.interface, DeviceType.mirror,
            DeviceType.lanz ) and interfaceName not in deviceColl:
         qt0( "Interface", qv( interfaceName ), "not found in",
              "Interface::AllIntfStatusDir" if deviceType == DeviceType.interface \
                    else "Mirroring::Status" )
         return False

      # check if the namespace matches with IntfStatusLocal
      if netNsName != DEFAULT_NS:
         try:
            interfaceNetNsName = TcpdumpLib.netNsName( deviceStatusAll,
                                                       intfStatusLocal,
                                                       allVrfStatusLocal,
                                                       sessionConfig )
         except TcpdumpLib.InvalidTargetException as e:
            self.status_.session[ sessionStatus.name ].reason = str( e )
            qt0( "Interface", qv( interfaceName ), "does not exist" )
            return False

         if interfaceNetNsName != netNsName:
            qt0( "Interface", qv( interfaceName ), "not converged to namespace",
                 qv( netNsName ), "but still in namespace",
                 qv( interfaceNetNsName ) )
            return False

      # check if vrfName matches with IpIntfStatus or VrfStatusLocal
      if vrfName != DEFAULT_VRF:
         if interfaceName in ipIntfStatus and \
            vrfName != ipIntfStatus[ interfaceName ].vrf:
            qt0( "Interface", qv( interfaceName ), "not converged to vrf",
                 qv( ipIntfStatus[ interfaceName ].vrf ), "but still in vrf",
                 qv( vrfName ) )
            return False
         else:
            found = False
            for v in allVrfStatusLocal.vrf:
               if allVrfStatusLocal.vrf[ v ].networkNamespace == netNsName:
                  found = True
                  break
            if not found:
               qt0( "Interface", qv( interfaceName ), "not converged to vrf",
                    qv( vrfName ) )
               return False

      # interface is UP and RUNNING, netNsName and vrfName are consistent
      return True

   def handleMirroringSessionStatus( self, mirroringIntf ):
      for sessionConfig in self.config_.session.values():
         if ( sessionConfig.deviceType == DeviceType.mirror or \
              sessionConfig.deviceType == DeviceType.lanz ) and \
              mirroringIntf == TcpdumpLib.deviceNameFromSessionConfig(
                                                            sessionConfig ):
            self.updateSessionStatus( sessionConfig.name )

   def vrfActive( self, vrfName ):
      """ Returns True if the given vrf exists and its state is 'active', else False.
      If vrfName is DEFAULT_VRF, then the vrf is active. """
      vrfStatusLocal = self.allVrfStatusLocal_.vrf
      if not vrfName:
         return False
      elif vrfName == DEFAULT_VRF:
         return True
      else:
         try:
            state = vrfStatusLocal[ vrfName ].state
            qt3( "Vrf", qv( vrfName ), "is in", qv( state ), "state" )
            return state == 'active'
         except KeyError:
            qt3( "Vrf", qv( vrfName ), " doesn't exist." )
            return False

   def tcpdumpCmdSyntaxFromSession( self, sessionName, lastPid=0 ):
      """ Builds tcpdump command syntax from TcpdumpSession config. """
      sessionConfig = self.config_.session[ sessionName ]
      if sessionConfig.deviceType == 'interfaceAddress':
         ipAddressDir = self.ipAddressDir_
         ip6AddressDir = self.ip6AddressDir_
         dumpFilter = sessionConfig.filter
         addrFilter = TcpdumpLib.computeAddrFilter( sessionConfig.device,
                                                    ipAddressDir, ip6AddressDir )
         self.status_.session[ sessionName ].addrFilter = addrFilter
         filterExpr = TcpdumpLib.combineFilters( addrFilter, dumpFilter )
         device = 'any'
         deviceType = DeviceType.any

      else:
         device = sessionConfig.device
         deviceType = sessionConfig.deviceType
         filterExpr = sessionConfig.filter
      options = dict()
      options[ 'file' ] = TcpdumpLib.dumpFile( sessionConfig )
      options[ 'device' ] = device
      options[ 'size' ] = sessionConfig.size
      options[ 'packetCount' ] = sessionConfig.packetCount
      options[ 'maxFileSize' ] = sessionConfig.maxFileSize
      options[ 'fileCount' ] = sessionConfig.fileCount
      options[ 'filter' ] = filterExpr
      return TcpdumpLib.tcpdumpCmdSyntax( self.deviceStatusAll_, options,
                                          deviceType, lastPid )

   def handleInterfaceStatus( self, ifName ):
      for sessionConfig in self.config_.session.values():
         try:
            if TcpdumpLib.deviceName( self.deviceStatusAll_,
                                      sessionConfig.device,
                                      sessionConfig.deviceType ) == ifName:
               self.updateSessionStatus( sessionConfig.name )
         except TcpdumpLib.InvalidTargetException as e:
            self.status_.session[ sessionConfig.name ].reason = str( e )

   def handleVrfChange( self, vrfName, netNsName ):
      for sessionConfig in self.config_.session.values():
         try:
            if TcpdumpLib.netNsName( self.deviceStatusAll_,
                                     self.intfStatusLocal_,
                                     self.allVrfStatusLocal_,
                                     sessionConfig ) == netNsName:
               self.updateSessionStatus( sessionConfig.name )
         except TcpdumpLib.InvalidTargetException:
            continue

   def updateSessionStatus( self, sessionName ):
      """ Starts/stops the tcpdump session by checking if config is enabled and
      timer is active, if interface is up and running in the right network namespace,
      and if vrf is active. If all the conditions are satisfied, but tcpdump session
      is already running, then the session is ignored. """
      qt3( "Updating session status of", qv( sessionName ) )

      status = self.status_
      config = self.config_
      deviceStatusAll = self.deviceStatusAll_
      intfStatusLocal = self.intfStatusLocal_
      ipIntfStatus = self.ipStatus_.ipIntfStatus
      ip6IntfStatus = self.ip6Status_.intf
      allVrfStatusLocal = self.allVrfStatusLocal_

      if sessionName not in config.session:
         qt0( "Error: config does not hold an instance for the", 
              qv( sessionName ), "session" )
         # This can happen when config is deleted and we are triggered
         # through various close() function.
         return
      if sessionName not in status.session:
         qt0( "Error: status does not hold an instance for the", 
              qv( sessionName ), "session" )
         return
      sessionStatus = status.session[ sessionName ]
      sessionConfig = config.session[ sessionName ]
      interfaceName = TcpdumpLib.deviceNameFromSessionConfig( sessionConfig )

      # running is true if session is enabled and active,
      # interface is up and in the right vrf and namespace, and
      # the corresponding vrf is active
      running = False

      # determine netNsName and vrfName
      try:
         netNsName = TcpdumpLib.netNsName( deviceStatusAll, intfStatusLocal, 
                                           allVrfStatusLocal, sessionConfig )
         vrfName = None
         if sessionConfig.deviceType in ( DeviceType.mirror, DeviceType.lanz ):
            vrfName = DEFAULT_VRF
         elif sessionConfig.deviceType == DeviceType.any:
            vrfName = sessionConfig.vrfName
            if vrfName and vrfName != DEFAULT_VRF:
               if not allVrfStatusLocal.vrf.get( vrfName ):
                  raise TcpdumpLib.InvalidVrfException( 
                                     sessionConfig.deviceType, vrfName ) 
            else:
               vrfName = DEFAULT_VRF
         elif interfaceName in ipIntfStatus:
            vrfName = ipIntfStatus[ interfaceName ].vrf
         elif netNsName != DEFAULT_NS:
            for v in allVrfStatusLocal.vrf:
               if allVrfStatusLocal.vrf[ v ].networkNamespace == netNsName:
                  vrfName = v
                  break
         else:
            vrfName = DEFAULT_VRF

         # if an interface in DEFAULT_VRF is in ipIntfStatus,
         # then vrfName is ''
         if not vrfName and netNsName == DEFAULT_NS:
            vrfName = DEFAULT_VRF

         # update KernelNetworkInfo objects
         if vrfName and vrfName != DEFAULT_VRF and \
            vrfName in allVrfStatusLocal.vrf and \
            allVrfStatusLocal.vrf[ vrfName ].networkNamespace == netNsName:
            self.createKniReactor( vrfName )

      except TcpdumpLib.InvalidTargetException:
         qt1( "Session", qv( sessionName ),
              "is not eligible to run because the interface does not exist" )

      else:
         if not sessionConfig.enabled:
            qt1( "Session", qv( sessionName ), "is not eligible to run because",
                 "enabled is false" )

         elif not sessionStatus.timerActive:
            qt1( "Session", qv( sessionName ), "is not eligible to run because",
                 "timerActive is false" )

         elif not self.vrfActive( vrfName ):
            qt1( "Session", qv( sessionName ), "is not eligible to run because vrf",
                 qv( vrfName ), "is not active" )

         elif not self.interfaceUp( interfaceName, vrfName, netNsName,
                                    sessionConfig, sessionStatus ):
            qt1( "Session", qv( sessionName ), "is not eligible to run because",
                 qv( interfaceName if interfaceName else fabricName ),
                 "in namespace", qv( netNsName ), "is not up" )
         elif sessionConfig.deviceType == 'interfaceAddress':
            if interfaceName in ipIntfStatus or interfaceName in ip6IntfStatus:
               running = True
         else:
            # all necessary conditions are True
            running = True

      if running != sessionStatus.running:
         if running:
            # start tcpdump now
            self.startTcpdump( sessionName, netNsName )
            sessionStatus.reason = "" 
         else:
            # stop tcpdump
            self.stopTcpdump( sessionName, signal.SIGUSR2 )
         qt3( "Session", qv( sessionName ), "running updated to", qv( running ) )
      elif running and netNsName != sessionStatus.netNsName:
         self.stopTcpdump( sessionName, signal.SIGUSR2 )
         self.startTcpdump( sessionName, netNsName )
         qt3( "Session", qv( sessionName ), "restarted in namespace",
              qv( netNsName ) )
         sessionStatus.reason = ""
      elif ( sessionStatus.addrFilter and
             TcpdumpLib.computeAddrFilter( sessionConfig.device, self.ipAddressDir_,
                                           self.ip6AddressDir_ )
             != sessionStatus.addrFilter
             and running ):
         self.stopTcpdump( sessionName, signal.SIGUSR2 )
         self.startTcpdump( sessionName, netNsName )
         sessionStatus.reason = ""
      sessionStatus.running = running
      qt3( "sessionStatus.running set to ", qv( sessionStatus.running ) )

   def startTcpdump( self, sessionName, netNsName=DEFAULT_NS ):
      """ Starts tcpdump in the given network namespace. """
      if sessionName in self.stopPoller_:
         # We are already stopping this process, but it's not dead yet.
         qt1( "Queue startTcpdump for", qv( sessionName ), "in", netNsName )
         self.startQueue_[ sessionName ] = netNsName
         return

      sessionConfig = self.config_.session[ sessionName ]
      sessionStatus = self.status_.session[ sessionName ]
      try:
         # Building the cmd for tcpdump
         cmd = self.tcpdumpCmdSyntaxFromSession( sessionName,
                                                 lastPid=sessionStatus.pid )
      except TcpdumpLib.InvalidTargetException as e:
         qt0( "Unable to start Tcpdump session", qv( sessionName ), "because",
               str( e ) )
         return
      qt1( "Starting Tcpdump session", qv( sessionName ), "in", netNsName )
      # QuickTrace only traces the first 24 bytes of each string, so give it
      # a bunch of chunks of the command so that we have better hope of seeing
      # the whole thing.
      qtcmd = ' '.join( cmd )
      qt2( 'Running', qv( qtcmd[ 0:20 ] ), qv( qtcmd[ 20:40 ] ),
                      qv( qtcmd[ 40:60 ] ), qv( qtcmd[ 60:80 ] ),
                      qv( qtcmd[ 80:100 ] ), qv( qtcmd[ 100: ] ) )
      createDirectory( os.path.dirname( TcpdumpLib.dumpFile( sessionConfig ) ) )
      # force garbage collection before forking so child doesnt attempt to 
      # garbage collect fds that affect the parent. 
      # BUG836696
      gc.collect()
      tcpdumpSubprocess = popenMaybeInNetNs( netNsName, cmd,
                                             prefix=f'[tcpdump {sessionName}] ' )
      self.subprocesses_[ sessionName ] = tcpdumpSubprocess
      sessionStatus.pid = tcpdumpSubprocess.grandchild
      sessionStatus.netNsName = netNsName

      def tcpdumpIsAlive():
         try:
            return tcpdumpSubprocess.wait( block=False ) is None
         except OSError:
            return False

      def handleTcpdumpExit( condition ):
         self.startPoller_.pop( sessionName, None )
         sessionStatus.reason = 'tcpdump returned exit status: %s' % \
                                tcpdumpSubprocess.returncode
         qt0( qv( sessionStatus.reason ) )

      def handlePollerTimeout():
         qt0( "tcpdump did not exit within", qv( sessionPollerTimeout ),
              "seconds, assuming session", qv( sessionName ), "is ok" )
         self.startPoller_.pop( sessionName, None )

      tcpdumpPoller = Tac.Poller(
         lambda: not tcpdumpIsAlive(),
         handleTcpdumpExit,
         timeoutHandler=handlePollerTimeout,
         timeout=sessionPollerTimeout,
         description='tcpdump to be running for more than %ds' %
                     sessionPollerTimeout )
      self.startPoller_[ sessionName ] = tcpdumpPoller
      qt1( "tcpdump started (", qv( sessionStatus.pid ), ")" )

   def stopTcpdump( self, sessionName, sgno=signal.SIGTERM ):
      """ Stops the given tcpdump session and the given signal.
      If sgno=signal.SIGUSR2, then tcpdump process is killed with SIGUSR2 to have
      it generate a context we can resume from. """
      qt1( "Stopping Tcpdump session", qv( sessionName ), "with signal", qv( sgno ) )
      # clear any pending start requests
      self.startQueue_.pop( sessionName, None )
      poller = self.startPoller_.pop( sessionName, None )
      if poller:
         poller.cancel()
      if sessionName in self.stopPoller_:
         # nothing to do, already being stopped
         return

      tcpdumpSubprocess = self.subprocesses_.pop( sessionName, None )
      if not tcpdumpSubprocess:
         qt1( "No tcpdump process for", qv( sessionName ) )
         return

      try:
         tcpdumpSubprocess.kill( sgno )
      except OSError:
         qt1( "tcpdump process for", qv( sessionName ), "already exited" )
         tcpdumpSubprocess.wait()
         return

      def _maybeStartTcpdump():
         nsName = self.startQueue_.pop( sessionName, None )
         if nsName:
            self.startTcpdump( sessionName, nsName )

      def tcpdumpIsAlive():
         try:
            return tcpdumpSubprocess.wait( block=False ) is None
         except OSError:
            return False

      def handleTcpdumpExit( condition ):
         qt1( "tcpdump exited" )
         self.stopPoller_.pop( sessionName, None )
         _maybeStartTcpdump()

      def handleExitTimeout():
         self.stopPoller_.pop( sessionName, None )
         qt1( "Killing Tcpdump session", qv( sessionName ) )
         try:
            tcpdumpSubprocess.kill( signal.SIGKILL )
         except OSError as e:
            qt0( "failed to kill tcpdump", qv( str( e ) ) )
         tcpdumpSubprocess.wait()
         _maybeStartTcpdump()

      tcpdumpStopPoller = Tac.Poller(
         lambda: not tcpdumpIsAlive(),
         handleTcpdumpExit,
         timeoutHandler=handleExitTimeout,
         timeout=sessionExitTimeout,
         description='tcpdump to exit gracefully' )
      self.stopPoller_[ sessionName ] = tcpdumpStopPoller

   def clearSessionStatus( self, sessionName ):
      """ Clears tcpdump session status to resume previously killed tcpdump session
      after SuperServer recovers due to a crash. """
      qt1( "Clearing Tcpdump session", qv( sessionName ) )

      def _parseTcpdumpCmd( cmd ):
         tcpdumpArgs = []
         if type( cmd ) == str: # pylint: disable=unidiomatic-typecheck
            cmd = cmd.split()
         for word in cmd:
            if word[ 0 ] == '-' or not tcpdumpArgs:
               # Option flag or non-option
               tcpdumpArgs.append( [ word ] )
            else:
               prev = tcpdumpArgs[ -1 ]
               if prev[ 0 ][ 0 ] == "-" and len( prev ) == 1:
                  # Option argument
                  prev.append( word )
               else:
                  # Previous was not a flag or already has its argument.
                  tcpdumpArgs.append( [ word ] )
         return tcpdumpArgs

      sessionPid = self.status_.session[ sessionName ].pid
      # Look for the name of a running process matching sessionPid
      output = Tac.run( [ 'ps', '-p', str( sessionPid ), '-o', 'cmd', 'h' ],
                        stdout=Tac.CAPTURE, stderr=Tac.DISCARD, asRoot=True,
                        ignoreReturnCode=True )
      if output:
         # Remove whitespaces
         output = output.strip()
         try:
            # Generate Tcpdump command syntax to check if any stale tcpdump
            # process with sessionPid is still running.
            cmd = self.tcpdumpCmdSyntaxFromSession( sessionName )
            if sorted( _parseTcpdumpCmd( output ) ) == \
                  sorted( _parseTcpdumpCmd( cmd ) ):
               # Kill the lingering tcpdump process using SIGTERM as we're resetting
               # Tcpdump session status and a new tcpdump instance will be started.
               qt3( "Killing Tcpdump Session", qv( sessionName ), "using SIGTERM" )
               Tac.run( [ 'kill', '-SIGTERM', str( sessionPid ) ], asRoot=True,
                        stderr=Tac.DISCARD )
            else:
               qt3( "Ignoring pid %d belongs as it to a different process %s" % \
                     ( sessionPid, output ) )
         except TcpdumpLib.InvalidTargetException:
            pass
      qt3( "Clearing Session", qv( sessionName ), "PID and running flag" )
      self.status_.session[ sessionName ].pid = 0
      self.status_.session[ sessionName ].running = False

class Tcpdump( SuperServer.SuperServerAgent ):
   def __init__( self, entityManager ):
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      mg = entityManager.mountGroup()
      # Mount tcpdump config
      self.config_ = mg.mount( 'sys/tcpdump/config', 'Tcpdump::Config', 'r' )
      self.status_ = mg.mount( 'sys/tcpdump/status', 'Tcpdump::Status', 'w' )

      self.ipAddressDir_ = mg.mount( 'ip/status', 'Ip::Status', 'r' )
      self.ip6AddressDir_ = mg.mount( 'ip6/status', 'Ip6::Status', 'r' )
      mirroringStatus = mg.mount( 'mirroring/status',
                                  'Mirroring::Status', 'r' )
      # ip/vrf/status/local maps VRF name -> VRF state, network namespace
      self.allVrfStatusLocal_ = mg.mount( Cell.path( 'ip/vrf/status/local' ),
                                       'Ip::AllVrfStatusLocal', 'r' )
      Tac.Type( "Ira::IraIpStatusMounter" ).doMountEntities( mg.cMg_, True, False )
      self.ipStatus_ = mg.mount( 'ip/status', 'Ip::Status', 'r' )
      self.ip6Status_ = mg.mount( 'ip6/status', 'Ip6::Status', 'r' )

      # Get the local interface entities, created by SuperServer
      # interface/status/all maps interface -> kernel device name (used
      # with KernelNetworkInfo)
      intfStatusAll = self.intfStatusAll()
      # interface/status/local maps interface -> network namespace
      self.intfStatusLocal_ = self.intfStatusLocal()
      self.em_ = entityManager
      self.deviceStatusAll_ = None
      self.service_ = None

      def _finished():
         # do not run on sso-standby
         self.deviceStatusAll_ = { DeviceType.interface : intfStatusAll, 
                                   DeviceType.interfaceAddress : intfStatusAll,
                                   DeviceType.mirror : mirroringStatus.sessionStatus,
                                   DeviceType.lanz : mirroringStatus.sessionStatus
                                 }
         if self.redundancyProtocol() == 'sso' and not self.active():
            qt0( 'Current role is not active, skipping Tcpdump service creation' )
            return
         self.service_ = TcpdumpService( self.config_, self.status_,
                                         self.ipAddressDir_,
                                         self.ip6AddressDir_,
                                         self.deviceStatusAll_,
                                         self.intfStatusLocal_,
                                         self.allVrfStatusLocal_,
                                         self.ipStatus_,
                                         self.ip6Status_,
                                         self.em_ )

      mg.close( _finished )

   def onSwitchover( self, protocol ):
      qt0( 'Tcpdump switchover handler' )
      if self.service_ is None:
         self.service_ = TcpdumpService( self.config_, self.status_,
                                         self.ipAddressDir_,
                                         self.ip6AddressDir_,
                                         self.deviceStatusAll_,
                                         self.intfStatusLocal_,
                                         self.allVrfStatusLocal_,
                                         self.ipStatus_,
                                         self.ip6Status_,
                                         self.em_ )
      else:
         qt0( 'Tcpdump service already created' )

def Plugin( ctx ):
   ctx.registerService( Tcpdump( ctx.entityManager ) )
