# Copyright (c) 2007-2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, os, QuickTrace, SuperServer, Cell

qv = QuickTrace.Var
qt0 = QuickTrace.trace0

def _zonefile( zone ):
   return os.path.join( "/usr/share/zoneinfo/posix", zone )

class ZoneNotifiee( Tac.Notifiee ):
   """ Reactor to the 'time/zone/config' object that symlinks
   /etc/localtime to point to the zonefile indicated in the config, or
   to UTC if no such file exists.  time/zone/status indicates the
   actual value in use."""

   notifierTypeName = 'Time::Zone::Config'

   def __init__( self, config, status ):
      Tac.Notifiee.__init__( self, config )
      self.status_ = status
      self.handleZone()

   @Tac.handler('zone')
   def handleZone( self ):
      z = self.notifier_.zone

      # Find the new zone file, defaulting to UTC if not found
      if z and os.path.isfile( _zonefile( z ) ):
         newzone = z
      else:
         newzone = 'UTC'
      newzf = _zonefile( newzone )

      # Only make a new link if it is changing.
      if not (os.path.islink( '/etc/localtime' ) \
              and os.readlink( '/etc/localtime' ) == newzf ):
         # QuickTrace only traces the first 24 characters, but the
         # first 25 characters of zonef are constant, so we take the
         # last 24 so that we get the more useful portion of the
         # string.
         qt0( "using timezone file ending", qv( newzf[ -24: ] ), "for",
               qv( newzone ) )
         os.unlink( '/etc/localtime' )
         os.symlink( newzf, '/etc/localtime' )
      self.status_.zone = newzone
      
class Timezone( SuperServer.SuperServerAgent ):
   """ This agent reacts to the 'time/zone/config' object and copies updates the 
   /etc/localtime symlink to point to the zonefile indicated, or to UTC if no such
   file exists. """
   def __init__( self, entityManager ):
      SuperServer.SuperServerAgent.__init__( self, entityManager )
      mg = entityManager.mountGroup()
      zoneConfig = mg.mount( 'sys/time/zone/config', 'Time::Zone::Config', 'r' )
      zoneStatus = mg.mount( Cell.path( 'sys/time/zone/status' ),
                             'Time::Zone::Status', 'wf' )
      def _finished():
         # pylint: disable-next=attribute-defined-outside-init
         self.zoneStatusSysdb_ = zoneStatus
         # pylint: disable-next=attribute-defined-outside-init
         self.notifiee_ = ZoneNotifiee( zoneConfig, zoneStatus )
      mg.close( _finished )

def Plugin( ctx ):
   ctx.registerService( Timezone( ctx.entityManager ) )
