# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import argparse
import atexit
import os
import sys
import tempfile
import zipfile

import ChecksumsGenLib
import Swi
import SwimHelperLib

def checksumVerify( swiPath, optimization, jsonHashes ):
   print( 'Calculating hashes of ' + optimization + ' image' )
   imageHashes = ChecksumsGenLib.getHashes( swiPath )
   for jsonCsumType, jsonCsumVal in jsonHashes.items():
      imageChecksumVal = imageHashes[ jsonCsumType ]
      if imageChecksumVal != jsonCsumVal:
         print( jsonCsumType + ' checksum mismatch!\n' +
                'Image checksum is:\n\t' + imageChecksumVal + '\n' +
                'Checksum from file is:\n\t' + jsonCsumVal )
         sys.exit( 1 )

def checkPassedArguments( args, parser ):
   if not os.path.exists( args.swiPath ):
      parser.error( 'Provided SWI image ' + args.swiPath +
                    ' does not exists.' )

   if not zipfile.is_zipfile( args.swiPath ):
      parser.error( args.swiPath + ' is not a zipfile' )

   if not args.checksumPath:
      args.checksumPath = args.swiPath + '.checksums.json'
      print( 'Missing --checksum argument. Assuming: ' + 
             args.checksumPath )

   if not os.path.exists( args.checksumPath ):
      parser.error( 'Provided checksum file ' + args.checksumPath +
                    ' does not exist.' )

def prepareTempDir():
   tempDir = tempfile.mkdtemp()
   def cleanupTempDir():
      Swi.run( [ 'sudo', 'rm', '-rf', tempDir ] )
   atexit.register( cleanupTempDir )

   return tempDir

def extractSwiAdapt( swiPath, tempDir ):
   with zipfile.ZipFile( swiPath ) as swiFile:
      swadaptPath = swiFile.extract( 'swadapt' , tempDir )

   Swi.run( [ 'chmod', 'u+x', swadaptPath ] )
   return swadaptPath

def prepareOptimization( swiFile, tempDir, swadaptPath, optimization ):
   if optimization == 'Full':
      return swiFile

   try:
      tmpSwiFile = os.path.join( tempDir, optimization + '.swi' )
      print( 'Preparing ' + optimization + ' image optimization' )
      Swi.run( [ swadaptPath, swiFile, tmpSwiFile, optimization ] )
   # pylint: disable=broad-except
   except Exception:
      print( 'Failed to adapt optimization: ' + optimization )
      sys.exit( 1 )

   return tmpSwiFile

def checksumverifyHandler( args=None ):
   args = sys.argv[ 1: ] if args is None else args
   parser = argparse.ArgumentParser( description='Verifies checksum of a swi image',
                                     prog='swi checksumverify',
                                     usage='<swiPath> <checksumPath>' )
   parser.add_argument( 'swiPath', type=str,
                        help='Path to SWI image to be verified' )
   parser.add_argument( 'checksumPath', nargs='?', type=str,
                        help='Path to checksums json file' )

   args = parser.parse_args( args )
   checkPassedArguments( args, parser )

   tempDir = prepareTempDir()
   jsonHashes = ChecksumsGenLib.getHashesFromFile( args.checksumPath )

   optimizations = [ 'Full' ] + \
         SwimHelperLib.getSupportedOptimizationsFromSwi( args.swiPath )
   if len( optimizations ) > 1:
      swadaptPath = extractSwiAdapt( args.swiPath, tempDir )
      for optimization in optimizations:
         if optimization.startswith( 'Default' ):
            continue
         swi = prepareOptimization( args.swiPath, tempDir, swadaptPath,
                                    optimization )
         checksumVerify( swi, optimization, jsonHashes[ optimization ] )
         if optimization != 'Full':
            Swi.run( [ 'sudo', 'rm', swi ] )
   else:
      checksumVerify( args.swiPath, 'Full', jsonHashes[ 'Full' ] )

   print( "\nChecksum verification successful!" )

if __name__ == "__main__":
   checksumverifyHandler()
