#!/usr/bin/env python3
# Copyright (c) 2008-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import errno
import os
import re
import sys
import glob
import optparse # pylint: disable=deprecated-module
import subprocess
from EosVersion import ( swimHdrFile as hdrFile, sidToOptimizationFile,
                         swimHdrFileV2 as hdrFileV2, extensionHdrFile )
from Swi import installrootfs
from Swi.extract import TIMESTAMP_FILE_NAME
import platform

if platform.machine() == 'aarch64':  # A4NOCHECK
   farch = "aarch64"
   rootfsDir = 'rootfs-aarch64.dir'
else:
   farch = "i386"
   rootfsDir = 'rootfs-i386.dir'

def mksquashfs( f, outfilename, fast, rootfsRpmDbOnly, zstd=False ):
   # fast implies gzip, zstd implies zstd, so only one can be specified
   assert not ( fast and zstd )
   # Run mksquashfs as root so that it has permissions to read all the
   # files in the rootfs directory.
   squashfsArgs = [ "sudo", "mksquashfs", f, outfilename,
                    "-noappend" ]
   if not sys.stdout.isatty():
      # Only show progress bar if output device is terminal
      squashfsArgs += [ "-no-progress" ]
   if fast:
      squashfsArgs += [ "-comp", "gzip", "-Xcompression-level", "9" ]
   if zstd:
      squashfsArgs += [ "-comp", "zstd", "-Xcompression-level", "15" ]
   # If fast or zstd is not specified, compress the swi with xz (this is slow!)
   if not fast and not zstd:
      squashfsArgs += [ "-comp", "xz", "-Xbcj", "x86" ]

   # Don't compress inodes to alleviate slow boot times for SWIM.
   # There's slowdown (30-60s) in boot-time when we have dense directory
   # overlap in multiple layers of the OverlayFS such as /usr/lib.
   # We trade-off around 5MB increase of SWI size to alleviate that slow-down.
   squashfsArgs += [ "-noI" ]

   # Discard dnf cache
   if os.path.exists( "%s/var/cache/dnf" % f ):
      subprocess.check_call( [ "sudo", "rm", "-rf", "%s/var/cache/dnf" % f ] )
   # Only include rpmdb files for rootfs squashes or the <optim>.rpmdb.sqsh.
   # We don't expect rpmdbs to be found in rootfs squashes in trunk, but
   # we handle both rootfs & rpmdb squashes here to keep backwards
   # compatibility swi create.
   if rootfsRpmDbOnly and ( "rootfs" not in f and ".rpmdb." not in f ) and \
         os.path.exists( "%s/var/lib/rpm" % f ):
      subprocess.check_call( [ "sudo", "mv", "%s/var/lib/rpm" % f,
                               "%s.rpmdir" % f ] )
   subprocess.check_call( squashfsArgs )
   if rootfsRpmDbOnly and ( "rootfs" not in f and ".rpmdb." not in f ) and \
         os.path.exists( "%s.rpmdir" % f ):
      subprocess.check_call( [ "sudo", "mv", "%s.rpmdir" % f,
                               "%s/var/lib/rpm" % f ] )
   # An unfortunate side-effect of running mksquashfs as root is that the
   # output file is owned by root, so chown it to the user running the
   # script.
   uid = os.geteuid()
   gid = os.getegid()
   subprocess.check_call( [ "sudo", "chown", "%d:%d" % ( uid, gid ),
                            outfilename ] )

def shouldIResquash( filename ):
   # If the timestamp file does not exist, than this directory needs to be resquashed
   if not os.path.isfile( TIMESTAMP_FILE_NAME ):
      return True

   if not subprocess.check_output( [ "sudo", "find", filename,
                                     "-newer", TIMESTAMP_FILE_NAME, "-or",
                                     "-cnewer", TIMESTAMP_FILE_NAME ] ):
      # There are no newer files than timestamp. Nothing to resquash
      return False

   return True

def refreshVersionFile( versionFilePath, buildDate ):
   if not os.path.exists( versionFilePath ):
      return

   with open( versionFilePath ) as f:
      versionContent = f.read()

   with open( versionFilePath, 'w' ) as f:
      for line in versionContent.splitlines():
         if 'BUILD_DATE' in line:
            f.write( 'BUILD_DATE=%s\n' % buildDate )
         else:
            f.write( '%s\n' % line )

def refreshVersionFiles( _dir, allFiles ):
   newBuildDate = installrootfs.getBuildDate()

   for f in allFiles:
      if f == 'version' or f.endswith( '.version' ):
         refreshVersionFile( os.path.join( _dir, f ), newBuildDate )

# pylint: disable-next=inconsistent-return-statements
def createSquashfs( _dir, isSwim, opts ):
   installrootfs.removeDocs( _dir )
   installrootfs.removePyiStubs( _dir )
   try:
      oldcwd = os.getcwd()
   except OSError as e:
      if e.errno != errno.ENOENT:
         raise
      oldcwd = "/tmp"
   os.chdir( _dir )
   filesToZip = set()
   filesToExclude = set()

   assert os.path.exists( 'version' )

   for f in os.listdir( "." ):
      if f == TIMESTAMP_FILE_NAME:
         # Don't add timestamp file to final .swi(m)
         continue

      if f.endswith( ".dir" ):
         if not opts.force_resquash and not shouldIResquash( f ):
            print( f'Not resquashing {f}' )
            continue
         if not os.path.exists( "%s/bin/bash" % f ):
            if ( opts and opts.force ) or ( f != rootfsDir ):
               # For SWIM, bash isn't required to be installed in
               # every ".dir" file.
               pass
            else:
               print( f"{_dir}/{f}/bin/bash does not exist;",
                      "is this really a suitable image?" )
               print( "Use --force to force." )
               return

         base = f[ : -4 ]
         # In the case where you extract a swi containing rootfs-i386 and then
         # use that extracted data as the source to create a swi with a
         # squashfs rootfs you don't want the new swi to contain the original
         # rootfs-i386.
         filesToExclude.add( base )
         # squash content of <variant/flavor>.rootfs-i386.dir into
         # <variant/flavor>.rootfs-i386.sqsh
         outfilename = base + ".sqsh"
         try:
            os.unlink( outfilename )
         except OSError as e:
            if e.errno != errno.ENOENT:
               sys.stderr.write( "%s: %s\n" % ( outfilename, e ) )
               sys.exit( 1 )
         if not isSwim or os.path.exists( "%s/boot" % f ):
            for f1 in os.listdir( "%s/boot" % f ):
               m = re.match( r"^vmlinuz-EosKernel(-kdump|)$", f1 )
               if m:
                  dst = "linux-" + farch + m.group( 1 )
                  # Only copy the file if it doesn't already exist or if
                  # the file is found in a rootfs, likely meaning the file was added
                  # through a `Swi rpm` update command
                  if ( not os.path.exists( dst ) or rootfsDir in f or
                       'rpmdb.dir' in f ):
                     subprocess.check_call( [ "sudo", "cp", f"{f}/boot/{f1}",
                                              dst ] )
                     filesToZip.add( dst )
         # Now add files as listed in swi-meta dir. We expect files to
         # contain just 2 lines with first line being path of the file in the rootfs
         # and second line listing the destination file name as it would appear in
         # the swi file. The original file is also contained in the embedded
         # squashfs.
         if not isSwim or os.path.exists( "%s/etc/swi-metadata" % f ):
            for specFileName in glob.glob( "%s/etc/swi-metadata/*" % f ):
               with open( specFileName ) as specFile:
                  contents = specFile.read().strip().split( '\n' )
                  assert len( contents ) == 2
                  filePath = os.path.normpath( '/' + contents[ 0 ] )
                  nameInZip = os.path.normpath( contents[ 1 ] )
                  subprocess.check_call( [ "sudo", "cp", f + filePath, nameInZip ] )
                  filesToZip.add( nameInZip )
         mksquashfs( f, outfilename, opts and opts.fast, isSwim,
                     opts and opts.zstd )
         filesToZip.add( outfilename )
      else:
         if os.path.isdir( f ):
            for root, _, files in os.walk( f ):
               for name in files:
                  filesToZip.add( os.path.join( root, name ) )
         elif not isSwim or f is not hdrFile:
            filesToZip.add( f )
   os.chdir( oldcwd )
   return filesToZip - filesToExclude

def create( filename, opts=None ):
   if opts and opts.dirname:
      _dir = opts.dirname
   else:
      _dir = os.path.basename( filename ).rsplit( '.', 1 )[ 0 ]

   isSwim = not os.path.exists( os.path.join( _dir, rootfsDir ) )
   if isSwim:
      print( "No %s found." % rootfsDir )
      print( "Assume we're creating a SWIM Formatted image" )

   # Store files in EOS.swi sorted by size to speed extraction of individual files
   # (stage 0 Aboot needs only initrd and linux)
   files = sorted( createSquashfs( _dir, isSwim, opts ),
                   key=lambda f: os.stat( os.path.join( _dir, f ) ).st_size )

   def modifySqshHdrFile( headerFile ):
      # header file is formatted for use by swim build process with lines
      # in the following format
      #   <flavor>=<path>/<moduleA>.dir[:<path>/<moduleN>.dir]*
      # rename .dir to .sqsh here for use by swim adaptor and boot process with lines
      # in the following format
      #   <flavor>=<moduleA>.sqsh[:<moduleN>.sqsh]*
      oldcwd = os.getcwd()
      os.chdir( _dir )
      subprocess.check_call( [ "rm", "-f", "%s.tmp" % headerFile ] )
      subprocess.check_call( [ "mv", headerFile, "%s.tmp" % headerFile ] )
      # Update the input hdrfile's extensions from dir => sqsh
      # This is required since swimSqshMap initially contains directory
      # information, but those directories get sqshed in createSquashFs().
      with open( headerFile, "w" ) as outhdrfile:
         with open( headerFile + ".tmp" ) as inhdrfile:
            for line in inhdrfile:
               key, value = line.split( '=', 1 )
               dirs = value.split( ':' )
               sqshes = []
               for d in dirs:
                  sqshes += [ os.path.basename( d ).replace( 'dir', 'sqsh' ) ]
               value = ':'.join( sqshes )
               line = f'{key}={value}'
               outhdrfile.write( line )
      os.unlink( headerFile + ".tmp" )
      os.chdir( oldcwd )

   initialFiles = []

   # If bootstrap.swix is available, include it as the first file in the zip
   foundArchives = [ f for f in os.listdir( _dir )
                     if f.startswith( "bootstrap" ) and f.endswith( ".swix" ) ]
   bootstrapSwix = foundArchives[ 0 ] if foundArchives else None
   if bootstrapSwix:
      initialFiles += [ bootstrapSwix ]

   # include SWIM metadata files as first files in the zip
   if os.path.exists( os.path.join( _dir, hdrFile ) ):
      # First file in swi maps sid (product name) to optimizations, the next
      # file maps that optimizations to a list of .sqshes to overlay mount.
      initialFiles += [ sidToOptimizationFile, hdrFile ]
      modifySqshHdrFile( hdrFile )

   if os.path.exists( os.path.join( _dir, hdrFileV2 ) ):
      initialFiles += [ hdrFileV2 ]
      modifySqshHdrFile( hdrFileV2 )

   if os.path.exists( os.path.join( _dir, extensionHdrFile ) ):
      initialFiles += [ extensionHdrFile ]

   versionFiles = [ f for f in files if ( f.endswith( ".version" ) or
                                          f == "version" ) ]

   # Filter out metadata and version files since they'll be prepended
   # to the front of the list
   files = [ f for f in files if ( f not in initialFiles and
                                   f not in versionFiles ) ]

   files = initialFiles + versionFiles + files

   if not opts.installfs_only:
      try:
         # Don't unlink since creating new file needs write permission to parent
         # directory. write mode anyway truncates the file for us.
         outfile = open( filename, "w" ) # pylint: disable=consider-using-with
      except Exception as e: # pylint: disable=broad-except
         sys.stderr.write( "while writing to %s: %s\n" % ( filename, e ) )
         sys.exit( 1 )

      if opts.update_version:
         refreshVersionFiles( _dir, files )

      subprocess.check_call( [ "zip", "-", "-0" ] + files, stdout=outfile, cwd=_dir )

def createHandler( args=None ):
   args = sys.argv[ 1 : ] if args is None else args
   op = optparse.OptionParser(
      prog="swi create",
      usage="usage: %prog [options] EOS.swi" )
   op.add_option( '-d', '--dirname', action='store' )
   op.add_option( '-f', '--force', action='store_true' )
   op.add_option( '-t', '--trace', action='store_true',
                  help="trace mode, no cleaning up of intermediate files" )
   op.add_option( '--fast', action='store_true',
                  help="Compress the squashfs with gzip" )
   op.add_option( '--installfs-only', action='store_true',
                  help="Only install rpms, but do not generate squashfs and image." )
   op.add_option( '-s', '--squashfs', action='store_true',
                  help="Create the root filesystem in squashfs format.  This "
                       "is the default, and thus this option is useless." )
   op.add_option( '--zstd', action='store_true',
                  help="Compress the squashfs with zstd" )
   op.add_option( '--force-resquash', action='store_true',
                  help='force resquash sqsh dirs even if'
                  ' no change has been made to them. To be used with swi extract.' )
   op.add_option( '--update-version', action='store_true',
                  help='refresh build date in all of the version files' )
   op.set_defaults( squashfs=True )
   op.set_defaults( fast=False )
   op.set_defaults( zstd=False )
   opts, args = op.parse_args( args )

   if not opts.installfs_only and len( args ) != 1:
      op.error( 'Please give me exactly one swi file!' )

   if opts.fast and opts.zstd:
      op.error( 'argument --zstd: not allowed with argument --fast' )

   create( args[ 0 ], opts )

if __name__ == "__main__":
   createHandler()
