#!/usr/bin/env python3
# Copyright (c) 2008, 2009 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import atexit
import zipfile
import optparse # pylint: disable=deprecated-module
import sys
import shutil
import os
import tempfile
import Swi
import SwiSignLib
import Tac
import EosVersion
import SwimHelperLib
import subprocess

def zipInfo( _file, opts ):
   zippedFile = zipfile.ZipFile( _file ) # pylint: disable=consider-using-with
   fileList = zippedFile.namelist()

   zipVersion( zippedFile, fileList, opts )

   if opts.printSwimSqshMap:
      zipSwimSqshMap( zippedFile, fileList )

def extractFromZip( swiFile, targetDir, targetFile ):
   with zipfile.ZipFile( swiFile ) as _swiFile:
      return _swiFile.extract( targetFile, targetDir )

def getVersionFilesFromSwi( zippedFile, fileList, opts ):
   # Make a list of all version files
   versionFiles = []
   if 'version' in fileList:
      versionFiles = [ 'version' ]
   if opts.printAllVersions:
      versionFiles += [ i for i in fileList if i.endswith( '.version' ) ]
   # When joining two or more swis into one using 'swi zip',
   # version.* files are generated
   versionFiles += [i for i in fileList if i.startswith('version.')]
   if not versionFiles:
      raise Exception(
         "%s does not appear to be a swi image (no version)." \
         % zippedFile.filename )

   # Determine image type
   imageType = "swim" if EosVersion.swimHdrFile in fileList else "swi"
   # Read version files and store contents in a dictionary
   versionDict = {}
   for i in versionFiles:
      version = zippedFile.read( i ).decode( "utf-8" )
      versionDict[ i ] = version

   return versionDict, imageType

def zipVersion( zippedFile, fileList, opts):
   versionFileDict, imageType = getVersionFilesFromSwi( zippedFile, fileList, opts )
   print( "Image type: %s\n" % imageType )
   # Always print 'version' first
   print( "version:" )
   print( versionFileDict[ 'version' ] )
   # Print the rest, remove version as it was already printed
   versionFileDict.pop( 'version' )
   for version, info in versionFileDict.items():
      print( "%s:" % version )
      print( info ) 

def zipSwimSqshMap( zippedFile, fileList ):
   if EosVersion.swimHdrFile not in fileList:
      raise Exception(
            "Cannot find {} in provided {}. Is this a legacy image?". \
            format( EosVersion.swimHdrFile, zippedFile.fileName ) )

   print( f"{EosVersion.swimHdrFile}:" )
   lines = zippedFile.read( EosVersion.swimHdrFile ).decode( 'utf-8' ).split( '\n' )
   for line in sorted( lines ):
      if not line.rstrip():
         continue

      optimizationAndDirs = line.split( '=' )
      optimization = optimizationAndDirs[ 0 ]
      dirs = optimizationAndDirs[ 1 ]
      dirs = dirs.split( ':' )
      print( f"   {optimization}:" )
      for d in dirs:
         if d in fileList:
            # Directory is present
            print( f"      {d}" )
         else:
            # Directory not present, signalize this
            print( f"    * {d}" )

   # Just to separate output nicely
   print()

def printInfo( _file, opts ):
   # For cEOS case. Unfortunately tarfile library does not have the --occurrence
   # option which allows tar to extract only the first occurrence of a file and
   # returning immediately. This is 100X faster when dealing with a 3.5 GB tar.
   tarCommand = subprocess.run( [ 'tar', '--extract', f'--file={_file}',
                                  '--to-stdout', '--occurrence',
                                  './etc/swi-version' ],
                                  check=False, capture_output=True )
   if tarCommand.returncode == 0:
      print( tarCommand.stdout.decode(), end='' )
   elif zipfile.is_zipfile( _file ):
      zipInfo( _file, opts )
   else:
      raise Exception( "Swi file is tar or zip archive, " +
                       f"{_file} does not seem to be any of those." )

def verifySwi( swiFile, opts, prefix="" ):
   if SwiSignLib.swiSignatureExists( swiFile ):
      blessed, _ = Swi.getBlessedAndVersion( swiFile )
      if blessed or opts.verifySignature:
         _, verifySigMessage, _ = SwiSignLib.verifySwiSignature( swiFile )
         print( prefix + verifySigMessage )
      else:
         print( "%sSWI is signed but not blessed. Use the --verify-signature "
                "option to force verification" % prefix )
   else:
      if prefix:
         print( "%sSWI is not signed." % prefix )
      else:
         print( "%s is not signed." % swiFile )

def verifySwimOptimizations( swiFile, opts ):
   optimizations = SwimHelperLib.getSupportedOptimizationsFromSwi( swiFile )
   if len( optimizations ) <= 1:
      # image has been adapted or is a single sqsh image,
      # so version/swi-signature would have been verified at this point
      return

   print( "Verifying internal SWIM images..." )
   tempDir = tempfile.mkdtemp()
   def cleanupTempDir():
      Swi.run( [ "sudo", "rm", "-rf", tempDir ] )
   atexit.register( cleanupTempDir )

   swadaptPath = extractFromZip( swiFile, tempDir, "swadapt" )
   if not os.path.exists( swadaptPath ):
      print( "Warning: Unable to verify all signatures due to missing "
             "swadapt file" )
      return
   Swi.run( [ "chmod", "777", swadaptPath ] )

   tmpSwiFile = os.path.join( tempDir, "EOS-tmp.swi" )

   for optimization in optimizations:
      optimizationFormatStr = "   %s: " % optimization

      if optimization.startswith( "Default" ):
         continue

      try:
         Swi.run( [ swadaptPath, swiFile, tmpSwiFile, optimization ] )

         verifySwi( tmpSwiFile, opts, prefix=optimizationFormatStr )
      # pylint: disable=broad-except
      except Exception:
         # Print generic failure message for any type of exception. Since
         # customers may run this command, the command should not
         # fail in a ugly way ( raising exceptions )
         print( "%sfailed to verify image" % optimizationFormatStr )
      finally:
         Swi.run( [ "rm", "-f", tmpSwiFile ] )

def verifyBootstrapSwix( swiFile ):
   with zipfile.ZipFile( swiFile, "r" ) as z:
      swixFile = next( ( f for f in z.namelist() if f.endswith( '.swix' ) ),
                       None )
      if not swixFile:
         return

      workDir = tempfile.mkdtemp()
      atexit.register( shutil.rmtree, workDir )
      tempSwix = os.path.join( workDir, swixFile )
      z.extract( swixFile, workDir )
      ( _, msg, _ ) = SwiSignLib.verifySwixSignature( tempSwix, [ None ] )
      print( f"Verify {swixFile}: {msg}" )

def infoHandler( args=None ):
   args = sys.argv[1:] if args is None else args
   op = optparse.OptionParser(
         prog="swi info",
         usage="usage: %prog EOS.swi" )
   op.add_option( "--verify-signature", action="store_true", 
         dest="verifySignature", default=False,
         help="Force verifying SWI signature if SWI is not blessed" )
   op.add_option( "--all", action="store_true",
         dest="printAllVersions", default=False,
         help="Print all version files found in swi" )
   op.add_option( "--verify-all", action="store_true",
         dest="verifyAll", default=False,
         help="Verify whether all SWIM images have been blessed and signed" )
   op.add_option( "--swimSqshMap", action="store_true",
         dest="printSwimSqshMap", default=False,
         help="Print swim squash map" )
   op.add_option( "--debug", action="store_true",
         dest="debug", default=False,
         help="Print debugging information" )
   opts, args = op.parse_args( args )

   if not args:
      op.error( 'Please give me at least one swi file!' )

   for _file in args:
      try:
         printInfo( _file, opts )
      except Exception as e: # pylint: disable=broad-except
         if opts.debug: # pylint: disable=no-else-raise
            raise e
         else:
            print( e )
            sys.exit( 1 )

      verifySwi( _file, opts )
      if opts.verifyAll:
         verifySwimOptimizations( _file, opts )
         verifyBootstrapSwix( _file )

if __name__ == "__main__":
   infoHandler()
