#!/usr/bin/env python3
# Copyright (c) 2008-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import argparse
import glob
from itertools import islice
import os
import os.path
import re
import socket
import subprocess
import sys
import tempfile
import time

from AddInOrder import AddEnableDisableReposInOrder

from RepoArg import RepoArgOp, createRepoArgs
from EosVersion import (
   swiFlavorDefault,
   swiFlavorDPE,
   swiLatestImageFormatVersion,
   swimLatestImageFormatVersion,
)
import EpochConsts

# Function taken from the Qube Plugin to detect OS version
def getOsVersion():
   p = subprocess.run( [ 'rpm', '--eval', '%{rhel}' ], stdout=subprocess.PIPE,
                       check=False )
   return int( p.stdout )

def run( argv, asRoot=False, captureStdout=False, verbose=True,
         ignoreReturnCode=False, printOutput=True,
         combineStdoutStderr=True ):
   if asRoot:
      argv = ["sudo"] + list( argv )
   if verbose:
      sys.stdout.write( "+ %s\n" % " ".join( argv ) )

   with tempfile.NamedTemporaryFile( mode='w' ) as stdout, \
        tempfile.NamedTemporaryFile( mode='w' ) as stderr:
      # pylint: disable-next=consider-using-with
      p = subprocess.Popen( argv, stdout=stdout, stderr=stderr,
                            universal_newlines=True )
      p.wait()
      retCode = p.returncode
      with open( stdout.name ) as outRead, open( stderr.name ) as errRead:
         out = outRead.read()
         err = errRead.read()
         if printOutput:
            print( out, end='' )
            print( err, end='' )
         if not ignoreReturnCode:
            if retCode != 0:
               if not printOutput: # in case of error, always print out/err
                  print( out, end='' )
                  print( err, end='' )
               assert False, "'%s' returned code %d" % ( " ".join( argv ),
                                                         retCode )
            # it looks like in some cases shell command can fail but return 0!?
            if ( re.search( "exit status [1-9]", out ) or
                 re.search( "exit status [1-9]", err ) ):
               if not printOutput:
                  print( out, end='' )
                  print( err, end='' )
               assert False, "'%s' output contained a non-zero exit status." \
                             "To find failure cause, search above output for '" \
                             "'exit status'" % ( " ".join( argv ) )
         if captureStdout:
            if combineStdoutStderr:
               return out + err
            return out
   return None

def getDocsToRemove( rootdir ):
   docsToRemove = run( [ "find", "%s/usr/share/doc" % rootdir,
                         "-iname", "change*", "-o",
                         "-iname", "news*", "-o",
                         "-iname", "demo*", "-o",
                         "-iname", "example*", "-o",
                         "-iname", "*faq*" ],
                       asRoot=True, captureStdout=True,
                       ignoreReturnCode=True ).splitlines()
   docsToRemove.append( "%s/usr/share/man" % rootdir )
   return docsToRemove

def getGitFilesToRemove( rootdir ):
   files = [
         "git-sh-i18n--envsubst",
         "git-http-backend",
         "git-http-fetch",
         "git-imap-send",
         "git-http-push",
         "git-remote-http"
         ]
   filesToRemove = []
   for file in files:
      filesToRemove.append( "%s/usr/libexec/git-core/%s" % ( rootdir, file ) )
   return filesToRemove

def chunkIter( iterable, chunkSize ):
   it = iter( iterable )
   chunk = list( islice( it, chunkSize ) )
   while chunk:
      yield chunk
      chunk = list( islice( it, chunkSize ) )

def batchRmFiles( files, batchSize=100 ):
   for batch in chunkIter( files, batchSize ):
      run( [ "rm" ] + list( batch ), asRoot=True )

def removePyiStubsByDir( rootdir ):
   batchRmFiles(
      glob.iglob(
         f"{rootdir}/usr/lib/python3.*/site-packages/**/*.pyi",
         recursive=True
      )
   )

def removePyiStubs( rootdir ):
   oldcwd = os.getcwd()
   os.chdir( rootdir )
   for f in os.listdir( "." ):
      if f.endswith( ".dir" ):
         removePyiStubsByDir( f )
   os.chdir( oldcwd )

def removeDocs( rootdir ):
   oldcwd = os.getcwd()
   os.chdir( rootdir )
   for f in os.listdir( "." ):
      if f.endswith( ".dir" ):
         docsToRemove = getDocsToRemove( f )
         run( [ 'rm', '-rf' ] + docsToRemove, asRoot=True )
   os.chdir( oldcwd )

def unmountProcPath( procPath ):
   # <dir>/proc sometimes fails to be unmounted, so try to force unmount and lazy
   # unmount if that fails
   if os.path.ismount( procPath ):
      try:
         run( [ "umount", "-f", procPath ], asRoot=True )
      except Exception as e: # pylint: disable=broad-except
         print( f"Force umount of {procPath} failed with:" )
         print( e )

      if os.path.ismount( procPath ):
         try:
            time.sleep( 5 )
            # last attempt - lazy unmount to detach from fs hierarchy and have
            # references cleaned up later
            run( [ "umount", "-l", procPath ], asRoot=True )
         except Exception as e: # pylint: disable=broad-except
            print( f"Lazy umount of {procPath} failed with:" )
            print( e )

def installrootfs( rootdir, arch, packages, yumConfig, erasePattern,
                   repoArgs, keepdoc=False, verifyprovenance=False ):
   assert os.path.exists( rootdir )
   repoArgs = repoArgs if repoArgs else []
   rootdir = os.path.abspath( rootdir )
   tempdir = tempfile.mkdtemp()

   try:
      # Suppress glibc locales other than en_US.utf8 to save space
      with open( os.path.join( tempdir, ".rpmmacros" ), "w" ) as f:
         f.write( "%_install_langs C:en_US.utf8" )

      # Create essential device nodes and mountpoints in rootfs
      for d in ["/dev", "/proc", "/.deltas"]:
         run( ["mkdir", "-p", rootdir + d], asRoot=True )
      for n, a in [("/dev/null", ["c", "1", "3"]),
                   ("/dev/console", ["c", "5", "1"]),
                   ("/dev/random", ["c", "1", "8"]),
                   ("/dev/urandom", ["c", "1", "9"])]:
         if not os.path.exists( rootdir + n ):
            run( ["mknod", "--mode=0666", rootdir + n] + a, asRoot=True )
      run( ["mount", "-t", "proc", "proc", rootdir + "/proc"], asRoot=True )

      # yum install packages into rootfs
      if yumConfig is None:
         cmd = ["env", "A4_YUM_NATIVE_ARCH=1", "HOME=%s" % tempdir, "a4", "yum" ]
      else:
         cmd = ["env", "HOME=%s" % tempdir, "yum", "-c", yumConfig]
      cmd.append( "--setopt=install_weak_deps=False" )
      if verifyprovenance:
         cmd += [ "--enableplugin=verifyprovenance" ]
      for repoArg in repoArgs:
         cmd += [ f"--{ repoArg.op.value }repo={ repoArg.repo }" ]

      # Original swi format (still used for DiagsImage) has only one rootfs and
      # rootdir starts with 'rootfs'. Swim format (used by EosImage) can have
      # multiple rootdirs and 'eos.dir'/'Base.dir' suffix indicates the "base"
      # rootfs.
      if arch == 'i686' and ( rootdir.split('/')[-1].startswith( 'rootfs' ) or
                              rootdir.endswith( 'eos.dir' ) or
                              rootdir.endswith( 'Base.dir' ) ):
         # Explicitly install 64-bit glibc in 32-bit images. This is needed for
         # GbEPROM in DiagsImage and for 64-bit Docker for EosImage

         # Start by install glibc.i686 to avoid pulling in extra dependencies like
         # 64-bit libselinux, libsepol and pcre
         run( cmd + [ "-y", "--installroot=%s" % rootdir, "install", "glibc.i686" ],
              asRoot=True )

         # Now install 64-bit glibc. This also pulls in nss-softokn-freebl.
         # Note that this yum command installs from the prebuilt repos.
         run( [ "setarch", "x86_64" ] + cmd + [
            "-y", "--installroot=%s" % rootdir, "install", "glibc.x86_64" ],
              asRoot=True )

         # Make sure that 64-bit glibc RPM was really installed
         run( [ "setarch", "x86_64", "rpm", "--root=%s" % rootdir, "-q",
                "glibc.x86_64" ] )

         # Now install 64-bit qemu..
         # Note that this yum command installs from the prebuilt repos.
         # In 32-bit el9 based EOS, we are using 64-bit qemu. So, add that as well.
         if getOsVersion() >= 9:
            run( [ "setarch", "x86_64" ] + cmd + [
               "-y", "--installroot=%s" % rootdir, "install", "qemu.x86_64" ],
                 asRoot=True )

            # Make sure that 64-bit qemu RPM was really installed
            run( [ "setarch", "x86_64", "rpm", "--root=%s" % rootdir, "-q",
                   "qemu.x86_64" ] )

      # package list too long may cause python pipe error, so break it into
      # shorter list
      for i in range( 0, len( packages ), 30 ):
         pkgs = packages[ i : i + 30 ]
         setArchCmd = []
         if arch == 'i686' and getOsVersion() >= 9:
            setArchCmd = [ "setarch", 'x86_64' ]
         run( setArchCmd + cmd + [ "-y", "--installroot=%s" % rootdir,
                                   "install" ] + pkgs, asRoot=True )
         # Make sure packages were really installed
         run( ["rpm", "--root=%s" % rootdir, "-q"] + pkgs )

      if arch == 'i686' and ( rootdir.split('/')[-1].startswith( 'rootfs' ) or
                              rootdir.endswith( 'eos.dir' ) or
                              rootdir.endswith( 'Base.dir' ) ):
         # Additional 64-bit multilib dependencies to enable running 64-bit docker
         # on 32-bit EOS
         docker64Deps = [
            'audit-libs', 'cracklib', 'device-mapper-libs', 'glib2',
            'libattr', 'libblkid', 'libcap', 'libcap-ng', 'libdb', 'libffi',
            'libgcc', 'libgcrypt', 'libgpg-error', 'libmount', 'libseccomp',
            'libselinux', 'libsepol', 'libuuid', 'lz4', 'pam', 'pcre',
            'systemd-libs', 'xz-libs', 'zlib' ]

         # Note that this yum command only installs from the prebuilt repos.
         run( [ "setarch", "x86_64" ] + cmd + [
            # We need to disable local repo because it has higher priority and has
            # only 32-bit rpms
            "--disablerepo=local",
            "-y", "--installroot=%s" % rootdir, "install" ] + [
               d + '-[0-9]*.x86_64' for d in docker64Deps ], asRoot=True )
         # Make sure packages were really installed
         run( [ "setarch", "x86_64", "rpm", "--root=%s" % rootdir, "-q" ] +
              [ p + ".x86_64" for p in docker64Deps ] )

      if arch == 'i686' and os.path.exists( rootdir + "/etc/rpm" ):
         with open( os.path.join( tempdir, "rpmplatform" ), "w" ) as f:
            # set rpm platform to x86_64 because customers want to install 64-bit
            # extensions on 32-bit EOS.
            f.write( "x86_64-redhat-linux\n" )
            f.flush()
         run( [ "cp", tempdir + "/rpmplatform", rootdir + "/etc/rpm/platform" ],
              asRoot=True )

      # Remove erasePackages
      erasePackages = [ p for p in run(
         [ "rpm", "--root=%s" % rootdir, "-qa", "--qf=%{NAME}.%{ARCH} " ],
         captureStdout=True ).split() if re.match( erasePattern, p ) ]
      if erasePackages:
         run( ["rpm", "--root=%s" % rootdir, "-e", "--nodeps"] + erasePackages,
               asRoot=True )

      # remove some docs to free up space
      docsToRemove = getDocsToRemove( rootdir )
      if not keepdoc:
         run( [ 'rm', '-rf' ] + docsToRemove, asRoot=True )

      # remove bloat files from git rpm package
      gitFilesToRemove = getGitFilesToRemove( rootdir )
      run( [ 'rm', '-rf' ] + gitFilesToRemove, asRoot=True )

      # Remove .pyi stubs
      removePyiStubs( rootdir )

      # Clean up yum cache
      run( ["sh", "-c", "rm -rf %s/var/cache/yum/*" % rootdir], asRoot=True )

      # Force arista-python to python3 inside SWI
      arPyConfigPath = "%s/etc/alternatives/arista-python" % rootdir
      py3 = os.path.realpath( '%s/usr/bin/python3' % rootdir )

      try:
         target = os.readlink( arPyConfigPath )
      except OSError:
         print( 'Unable to resolve target link' )
         target = ''

      if target and 'python3' not in target and os.path.exists( py3 ):
         py3 = py3.replace( rootdir, '' )
         run( [ "ln", "-sf", py3, arPyConfigPath ], asRoot=True )
   finally:
      unmountProcPath( rootdir + "/proc" )

      run( ["rm", "-rf", tempdir], asRoot=True, verbose=False )

def installmodfs( moddir, arch, lowerdirs, pkgs, yumConfig, erasePattern,
                  repoArgs, trace, keepdoc=False, verifyprovenance=False ):
   run( [ "mkdir", "-p", moddir ], asRoot=True )
   workdir = ""
   uniondir = ""
   try:
      if lowerdirs:
         workdir = '%s-work' % moddir
         uniondir = '%s-union' % moddir
         run( [ "mkdir", "-p", workdir, uniondir ], asRoot=True )
         run( [ "mount", "-t", "overlay", "overlay-%s" % moddir, "-o",
                "lowerdir={},upperdir={},workdir={}".format( lowerdirs, moddir,
                                                         workdir ), uniondir ],
              asRoot=True )
         # workaround for following yum issue with overlayfs
         #    Rpmdb checksum is invalid: dCDPT(pkg checksums): package_name
         for d in os.listdir( "%s/var/lib/rpm" % uniondir ):
            run( [ "touch", f"{uniondir}/var/lib/rpm/{d}" ], asRoot=True )
         moddir = uniondir

      if pkgs:
         installrootfs( moddir, arch, pkgs, yumConfig, erasePattern,
                        repoArgs, keepdoc=keepdoc,
                        verifyprovenance=verifyprovenance )
   finally:
      # show mount points to help with debugging
      run( [ "mount" ], asRoot=True )
      if lowerdirs and not trace:
         if os.path.ismount( uniondir ):
            run( [ "umount", "-R", uniondir ], asRoot=True )
         if workdir or uniondir:
            run( [ "rm", "-rf", workdir, uniondir ], asRoot=True )

def updateImageFormatVersion( versionFile, imageFormatVersion ):
   run( [ "sed", "-i",
          r"s|\(IMAGE_FORMAT_VERSION=\)\(.*\)$|\1%s|" % imageFormatVersion,
          versionFile ], asRoot=True )

def updateSwiOptimization( versionFile, swiOptimization):
   run( [ "sed", "-i",
          r"s|\(SWI_OPTIMIZATION=\)\(.*\)$|\1%s|" % swiOptimization,
          versionFile ], asRoot=True )

def getBuildDate():
   return time.strftime( "%Y%m%dT%H%M%SZ", time.gmtime() )

def genSerialNum():
   with open( "/proc/sys/kernel/random/uuid" ) as fd:
      return fd.read().strip()

def writeSwiVersion( swiDir, version, arch, release, swiVariant='US',
                     swiFlavor=swiFlavorDefault,
                     swiOptimization=None,
                     swiMaxHwEpoch=EpochConsts.SwiMaxHwEpoch,
                     imageFormatVersion=swimLatestImageFormatVersion,
                     buildDate=None,
                     serialNum=None,
                     swiVersionPath=None ):
   f = tempfile.NamedTemporaryFile( mode="w") # pylint: disable=consider-using-with
   f.write( "SWI_VERSION=%s\n" % version )
   f.write( "SWI_ARCH=%s\n" % arch )
   f.write( "SWI_RELEASE=%s\n" % release )
   f.write( "BUILD_DATE=%s\n" % ( buildDate or getBuildDate() ) )
   f.write( "BUILD_HOST=%s\n" % socket.gethostname() )
   f.write( "SERIALNUM=%s\n" % ( serialNum or genSerialNum() ) )
   f.write( "IMAGE_FORMAT_VERSION=%s\n" % imageFormatVersion )
   f.write( "SWI_MAX_HWEPOCH=%d\n" % swiMaxHwEpoch )
   f.write( "SWI_VARIANT=%s\n" % swiVariant )
   f.write( "SWI_FLAVOR=%s\n" % swiFlavor )
   f.write( "SWI_OPTIMIZATION=%s\n" % swiOptimization )
   f.flush()
   with open( f.name ) as fd:
      sys.stdout.write( fd.read() )

   if not swiVersionPath:
      assert os.path.exists( swiDir )
      swiDir = os.path.abspath( swiDir )
      if swiOptimization is None: # Most likely a single rootfs swi
         versionFileName = 'version'
      else:
         versionFileName = swiOptimization + '.version'

      swiVersionPath = os.path.join( swiDir, versionFileName )

   run( [ "cp", f.name, swiVersionPath ], asRoot=True )
   run( [ "chmod", "0644", swiVersionPath ], asRoot=True )

def setSwiVersion( rootdir, swi_version, swi_arch, swi_release, imageFormatVersion,
                   buildDate=None, serialNum=None ):
   arch = swi_arch or run( [ "arch" ], captureStdout=True ).rstrip( "\n" )
   # For swi, base name of rootdir is 'rootfs-i386.dir'
   # For swim, base name of rootdir is '<flavor>.rootfs-i386.dir'
   optimization = os.path.basename( rootdir ).split( '.' )[ 0 ]
   isDpe = optimization.endswith( '-DPE' ) or optimization == 'DPE'
   swiFlavor = swiFlavorDPE if isDpe else swiFlavorDefault
   # Image with just rootfs-i386.dir cannot have an optimization
   # pylint: disable-next=consider-using-in
   if optimization == 'rootfs-i386' or optimization == 'rootfs-aarch64':
      optimization = None 

   swiDir = os.path.join( rootdir, '..' )
   writeSwiVersion( swiDir, swi_version, arch, swi_release,
                    swiVariant='US', swiFlavor=swiFlavor,
                    swiOptimization=optimization,
                    imageFormatVersion=imageFormatVersion,
                    buildDate=buildDate, serialNum=serialNum )

def installrootfsHandler( args=None ):
   args = sys.argv[1:] if args is None else args

   op = argparse.ArgumentParser(
         prog="swi installrootfs",
         usage="usage: %(prog)s [OPTIONS] ROOTFSDIR" )

   op.add_argument( "-l", "--lowerdirs", action="store",
                    help="directories to be union-mounted as lowerdirs, "
                         "separated by ':', must be in full path, "
                         "with top layer on left and bottom layer on right" )
   op.add_argument( '-f', '--force', action='store_true',
                    help="clean up existing installation" )
   op.add_argument( "-t", "--trace", action="store_true",
                    help="trace mode, no cleaning up intermediate files." )
   op.add_argument( "--keepdoc", action="store_true",
                    help="keep doc directories" )
   op.add_argument( "-p", "--package", action="append", default=[],
                    help="install PACKAGE (may be specified multiple times)" )
   op.add_argument( "-c", "--yum-config", action="store", metavar="CONFIG",
                    help="use yum configuration file CONFIG" )
   op.add_argument( "-e", "--erase-pattern", action="store", metavar="REGEX",
                    help="erase packages matching REGEX after yum install "
                    "(default=%(default)s)" )
   op.add_argument( "--swi-version", action="store",
                    help="set the SWI_VERSION in /etc/swi-version" )
   op.add_argument( "--swi-arch", action="store",
                    help="set the SWI_ARCH in /etc/swi-version" )
   op.add_argument( "--swi-release", action="store",
                    help="set the SWI_RELEASE in /etc/swi-version" )
   op.add_argument( "--enablerepo", action=AddEnableDisableReposInOrder,
                    help="enable repository (may be specified multiple times)" )
   op.add_argument( "--disablerepo", action=AddEnableDisableReposInOrder,
                    help="disable repository (may be specified multiple times)" )
   op.add_argument( "--verify-provenance", action="store_true",
                    help="verify provenance documents" )
   op.set_defaults( erase_pattern="$" )

   opts, args = op.parse_known_args( args )

   if len( args ) != 1:
      op.error( "Missing ROOTFSDIR argument" )
   if not opts.package:
      op.error( "Must specify at least one package to install" )
   if len( args ) == 1:
      rootdir = args[0]

   # Provide defaults for enablerepo and disablerepo, don't use
   # op.set_defaults(), as then anything provided by the user gets
   # appended to the defaults, the user specified ones does not
   # replace the defaults.
   #
   # XXX - 'local' repo doesn't exist outside of workspace,
   #       i.e. in Jenkins.
   try:
      repoArgs = createRepoArgs( opts.ordered_args )
   except AttributeError as _: # User did not provide enablerepo or disablerepo
      repoArgs = None

   if repoArgs is None:
      repoArgs = createRepoArgs( [ ( RepoArgOp.ENABLE, "local" ) ] )

   arch = opts.swi_arch or run( [ "arch" ], captureStdout=True ).rstrip( "\n" )
   imageFormatVersion = None
   if opts.lowerdirs:
      if opts.force:
         # make sure to start with clean slate
         run( [ "rm", "-rf", rootdir ], asRoot=True )
      installmodfs( rootdir, arch, opts.lowerdirs, opts.package,
                    opts.yum_config, opts.erase_pattern, repoArgs,
                    opts.trace, keepdoc=opts.keepdoc,
                    verifyprovenance=opts.verify_provenance )
      imageFormatVersion = swimLatestImageFormatVersion
   else :
      installrootfs( rootdir, arch, opts.package, opts.yum_config,
                     opts.erase_pattern, repoArgs,
                     keepdoc=opts.keepdoc, verifyprovenance=opts.verify_provenance )
      imageFormatVersion = swiLatestImageFormatVersion

   if opts.swi_version and rootdir:
      setSwiVersion( rootdir, opts.swi_version, opts.swi_arch, opts.swi_release,
                     imageFormatVersion )

if __name__ == "__main__":
   installrootfsHandler()
