#!/usr/bin/env python3
# Copyright (c) 2008 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import zipfile
import optparse # pylint: disable=deprecated-module
import sys
import SwiSignLib

outfiles = {}
def combineZipFiles( args, opts=None ):
   """zip up the contents of zip files args[1:] into a new zip file
   named args[0]. If a filename occurs in more than one input zip file
   then it is renamed each later time that it appears into name.input2
   where the second input file is called 'input2.swi'."""
   outfile = args[0]
   # pylint: disable-next=consider-using-with
   zout = zipfile.ZipFile( outfile, 'w', zipfile.ZIP_DEFLATED )
   print( "creating", outfile )
   for filename in args[1:]:
      basename = filename
      if '.' in basename:
         basename = basename.rsplit( '.', 1)[0]
      try:
         zi = zipfile.ZipFile( filename ) # pylint: disable=consider-using-with
      except zipfile.BadZipfile:
         print( filename, "does not appear to be a valid zipfile." )
         return
      except OSError:
         print( filename, 'could not be opened -- does it exist?' )
         return
      for zipInfo in zi.infolist():
         name = zipInfo.filename
         if name == SwiSignLib.SIG_FILE_NAME:
            # Skip swi-signature files, the resulting zipped SWI should be resigned
            # separately
            continue
         arcname = name
         if arcname in outfiles:
            arcname = name + "." + basename
         if arcname in outfiles:
            print( "I tried to rename", name,
                  "to avoid a conflict but the name", arcname,
                  "that I picked is already used in the target.  I'm giving up." )
            return
         if arcname != name:
            print( f"*** renaming {basename}/{name} to {arcname}" )
         outfiles[arcname] = True
         data = zi.read( name )
         # pylint: disable-next=consider-using-f-string
         print( 'adding %s/%s as %s (%d bytes)' % ( basename, name,
                                                   arcname, len( data ) ) )
         zipInfo.filename = arcname
         zout.writestr( zipInfo, data )
   zout.close()

# zipHandler is a special name that is found dynamically by
# SwiCommand.run.  It is invoked with args set to the arguments
# passed to swi zip.
def zipHandler( args=None ):
   args = sys.argv[1:] if args is None else args
   op = optparse.OptionParser(
         prog="swi zip",
         usage="usage: %prog out.swi in1.swi in2.swi ..." )
   if not args: 
      op.usage()
   opts, args = op.parse_args( args )

   if len( args ) <=2 :
      op.error( 'swi zip requires at least three arguments!' )

   combineZipFiles( args, opts )

if __name__ == "__main__":
   zipHandler()
